import 'modules/setup/js/services/setup.svc.js';
import 'dlo/js/services/nameServers.svc.js';
import 'modules/servers/js/services/servers.svc.js';

import { commonSetupModule } from 'common/js/modules';

/**
 * A factory for producing a new Setup object.
 */
commonSetupModule.factory('coreSetup', [
	'$state',
	'$location',
	'$log',
	'serverService',
	'cvLoc',
	'cvToaster',
	'cvUtil',
	'nameServersService',
	'setupService',
	'cvNavigationFactory',
	'$stateParams',
	function(
		$state,
		$location,
		$log,
		serverService,
		cvLoc,
		cvToaster,
		cvUtil,
		nameServersService,
		setupService,
		cvNavigationFactory,
		$stateParams
	) {
		var factory = {};

		factory.planData = {
			createdPlan: {},
			serverPlanMap: []
		};

		/**
		 * Initialize a new setup for the factory.
		 *
		 * @param {Ojbect[]}
		 *            steps - An array of steps used for the setup.
		 * @param {boolean[]}
		 *            setupSteps - An array that specifies what steps have been completed.
		 * @param {number=}
		 *            setupStep - The step the setup should start on.
		 * @param {string}
		 *            stepMap - The index of the setupStepsMap object in the cv.sessionContext instance used
		 *            in the setup.
		 * @param {string}
		 *            defaultPageState - The name of the state of the page that the user should be taken to if
		 *            he/she finishes the setup, or chooses to finish it later.
		 */
		factory.newSetup = function(steps, setupSteps, setupStep, stepMap, defaultPageState) {
			factory.stepMap = stepMap;
			factory.proxyList = [];
			factory.defaultPage = defaultPageState;
			factory.model = {};
			factory.setupStep = 0;
			factory.data = {};
			factory.data.steps = steps;
			factory.setupStep = -1;
			factory.setupSteps = setupSteps;
			factory.setupComplete = factory.setupSteps.every(function(finished) {
				return finished;
			});

			factory.canMarkAsComplete = !factory.setupComplete;

			setupStep = parseInt(setupStep);
			if (factory.canGoToStep(setupStep)) {
				factory.setupStep = setupStep;
			} else {
				/*
				 * The user is not allowed to go to the step that was intended yet. Therefore, we revert by
				 * going to the first incomplete step.
				 */
				factory.setupStep = 0;
				for (var i in setupSteps) {
					if (!setupSteps[i]) {
						factory.setupStep = +i;
						break;
					}
				}
			}

			factory.currItem = factory.data.steps[factory.setupStep];
			factory.model = factory.currItem.model;

			if (factory.currItem.name === cvLoc('title.addLibrary') && factory.isStepCompleted(factory.setupStep)) {
				factory.storagePresent = true;
				factory.storagePresentMessage = cvUtil.infoMsgLoc('info.storagePoolPresent');
				factory.formStorage = {
					hide: true
				};
				factory.changeStorage = function() {
					factory.formStorage.hide = false;
					factory.storagePresentMessage = cvUtil.emptyMsg();
				};
			}
		};

		factory.getPlanCreatedInGettingStarted = function() {
			return factory.planData.createdPlan;
		};

		/*
		 * Set Plan created in Create File system plan [Getting Started]
		 *
		 */
		factory.setPlan = function(plan) {
			factory.planData.createdPlan = plan;
		};

		factory.getServerPlanMapping = function() {
			return factory.planData.serverPlanMap;
		};

		/*
		 * Set client Plan mapping saved in Add file server step [Getting Started]
		 *
		 */

		factory.setServerPlanMapping = function(serverPlanMap) {
			factory.planData.serverPlanMap = serverPlanMap;
		};

		/**
		 * Skip the current step and move on to the next one.
		 */
		factory.skip = function() {
			factory.handleNext(true);
		};

		/**
		 * Handle going to the next step in the setup. If there is no step page for the current step, the user
		 * will be redirected to the default page.
		 *
		 * @param boolean
		 *            skip - Whether or not the current step has been skipped. If it has been skipped, then
		 *            the current step will not be set as complete, the session context will not refresh, and
		 *            the setup will move on to the next step.
		 */
		factory.handleNext = function(skip) {
			let completeStepPromise;
			if (!skip) {
				completeStepPromise = factory.completeStep(factory.setupStep);
			} else {
				completeStepPromise = Promise.resolve();
			}

			completeStepPromise.then(() => {
				if (factory.setupStep + 1 >= factory.data.steps.length) {
					factory.goToDefaultPage(true, factory.stepMap);
				} else {
					factory.reloadCurrentState(
						{
							forceStep: factory.setupStep + 1
						},
						{
							reload: parseInt($stateParams.forceStep) === factory.setupStep + 1 ? true : false
						}
					);
				}
			});
		};

		/**
		 * Refresh the user's current session so that if he/she refreshes the page, the setup steps will
		 * remain in sync.
		 */
		factory.refreshSession = function() {
			return setupService.refreshSession();
		};

		/**
		 * Mark the current setup as complete. When the steps have been marked as complete, a call to the
		 * back-end will be made to refresh the user's session context. This is done so that in the event the
		 * user refreshes the page, the setup steps will remain in sync.
		 */
		factory.markAsComplete = function() {
			setupService.markSetupAsComplete(factory.stepMap).success(function() {
				for (var i in factory.setupSteps) {
					factory.setupSteps[i] = true;
				}
				factory.refreshSession();
				cv.sessionContext.completedSetups[factory.stepMap] = true;

				factory.goToDefaultPage(true);
			});
		};

		/**
		 * Set up a new name server instance.
		 */
		factory.setUpNameServer = function() {
			factory.model = {
				id: 0,
				netBiosName: '',
				domainName: '',
				username: '',
				enableSSO: '',
				secureLDAP: '',
				resourceDomain: '',
				proxyClientId: 0,
				accessADProxy: ''
			};

			nameServersService
				.getActiveProxyList()
				.success(function(data) {
					factory.proxyList = cvUtil.sortAscending(data, 'client.clientName');
					if (factory.proxyList && factory.proxyList.length > 0) {
						factory.model.proxyClientId = factory.proxyList[0].client.clientId;
						factory.showProxy = true;
					}
				})
				.error(function(data) {
					factory.proxyList = [];
					factory.nameServerWindowMessage = cvUtil.errMsgLoc('error.failedLoadingServers');
				});
		};

		/**
		 * Returns whether or not a user is allowed to go to a given step.
		 *
		 * @return boolean True if the user can go to the step; False otherwise.
		 */
		factory.canGoToStep = function(step) {
			if (factory.setupStep != step && angular.isDefined(factory.data.steps[step])) {
				let stepToGo = factory.data.steps[step];

				if (stepToGo.isDisabled === true) {
					return false;
				}

				if (factory.setupComplete || step === 0) {
					return true;
				} else if (angular.isDefined(factory.setupSteps[step - 1])) {
					return factory.showWarning(step - 1) || factory.isStepCompleted(step - 1);
				}
			}
			return false;
		};

		/**
		 * Got to a given step in the setup.
		 *
		 * @param {number}
		 */
		factory.goToStep = function(step) {
			if (factory.canGoToStep(step)) {
				$state.go($state.current.name, {
					forceStep: step
				});
			}
		};

		/**
		 * Return whether or not a given step is the current step.
		 *
		 * @param {number}
		 *            step The step to check for whether or not it's the current step.
		 * @return {boolean} True if the current step is the given step; False otherwise.
		 */
		factory.isCurrentStep = function(step) {
			return factory.setupStep === step;
		};

		/**
		 * Set the current step in the setup.
		 *
		 * @param {number}
		 *            step The step to set as the current step.
		 */
		factory.setCurrentStep = function(step) {
			factory.setupStep = step;
			factory.currItem = factory.data.steps[factory.setupStep];
		};

		/**
		 * Get the current step in the setup.
		 *
		 * @return {number} The current step in the setup.
		 */
		factory.getCurrentStep = function() {
			return factory.setupStep;
		};

		/**
		 * Whether or not the setup is on the first step.
		 *
		 * @return {boolean} True if the setup is on the first step; False otherwise.
		 */
		factory.isFirstStep = function() {
			return factory.setupStep === 0;
		};

		/**
		 * Whether or not the setup is on the last step.
		 *
		 * @return {boolean} True if the setup is on the last step; False otherwise.
		 */
		factory.isLastStep = function() {
			return factory.setupStep === factory.data.steps.length - 1;
		};

		/**
		 * Get the next step's label.
		 *
		 * @return {string} The next step's label.
		 */
		factory.getNextLabel = function() {
			return factory.data.steps[factory.getCurrentStep()].actionName;
		};

		/**
		 * Submit an action for the current step the user is on.
		 */
		factory.submit = function() {
			if (angular.isDefined(factory.data.steps[factory.getCurrentStep()])) {
				factory.data.steps[factory.getCurrentStep()].action();
			}
		};

		/**
		 * Skip the entire setup. A cookie is also set so that the setup is not loaded in this session.
		 */
		factory.skipSetup = function() {
			factory.goToDefaultPage(false);
		};

		/**
		 * Return whether or not to show a warning. A warning is only showed only if a step after this is
		 * already done.
		 *
		 * @return {boolean} Whether or not to show a warning.
		 */
		factory.showWarning = function(index) {
			for (var i = index + 1; i < factory.setupSteps.length; i++) {
				if (!factory.setupSteps[index] && factory.setupSteps[i]) {
					return true;
				}
			}
			return false;
		};

		/**
		 * Go to the default page when leaving the setup.
		 */
		factory.goToDefaultPage = function(bMarkSetupAsComplete, stepMap) {
			/**
			 * This is required because when the setup is marked as complete and that state is not available
			 * to Tenant we need to load that state which is handled in NavService.java Use Case is , Tenant
			 * has hidden FS from nav preferences and setup was never completed, once the tenant marks that
			 * setup complete we need to load state for FS servers.which will be taken care in navService,
			 * where we append the state if the setup is complete.
			 */

			if (!cvNavigationFactory.isStateAvailable(factory.defaultPage)) {
				if (bMarkSetupAsComplete) {
					// This is when setup is being marked as complete. if the corresponding states are part of denied navs then, we removed the states from denied navs and we need to reload the page to reconstruct the navs
					location.reload();
				} else {
					if (
						cv.sessionContext.completedSetups[factory.stepMap] === true &&
						!cv.sessionContext.deniedNavStates.includes(factory.defaultPage)
					) {
						//If some setup is clicked for tfirst time and all its steps are marked complete, then setup gets completed through JAVA layer, and when we complete the state, the corresponding state is removed from denied navs but a reload is necessary for reconstructing the nav.
						location.reload();
					} else {
						// Even after completing the setup if the user has denied the defaultPage through navigation preferences we redirect to landing page.
						$location.url('/');
					}
				}
			} else {
				$state.go(factory.defaultPage);
			}
		};

		/**
		 * Reload the current state with the given parameters and options.
		 *
		 * @param {Object}
		 *            params - The parameters for the reload.
		 * @param {Object}
		 *            options The options for the reload.
		 */
		factory.reloadCurrentState = function(params, options) {
			$state.go($state.current.name, params, options);
		};

		/**
		 * Check whether or not the given step number has been completed.
		 *
		 * @param {number}
		 *            stepNumber - The step number to see if it has been completed.
		 * @return {boolean} True if the step was completed; False otherwise.
		 */
		factory.isStepCompleted = function(stepNumber) {
			return (
				angular.isDefined(cv.sessionContext.setupStepsMap) &&
				angular.isDefined(cv.sessionContext.setupStepsMap[factory.stepMap]) &&
				cv.sessionContext.setupStepsMap[factory.stepMap][stepNumber]
			);
		};

		/**
		 * Check whether or not the current step has been completed.
		 *
		 * @return {boolean} True if the step was completed; False otherwise.
		 */
		factory.isCurrentStepCompleted = function() {
			return factory.isStepCompleted(factory.setupStep);
		};

		/**
		 * Specify that the given step has been completed. When a step has been completed, a call to the
		 * back-end will be made to refresh the user's session context. This is done so that if the user
		 * refreshes the page, the setup steps will remain in sync.
		 *
		 * @param number
		 *            step - The step that was completed.
		 */
		factory.completeStep = function(step) {
			if (angular.isDefined(cv.sessionContext.setupStepsMap[factory.stepMap][step])) {
				cv.sessionContext.setupStepsMap[factory.stepMap][step] = true;
				if (
					cv.sessionContext.setupStepsMap[factory.stepMap].every(function(finished) {
						return finished;
					})
				) {
					/*
					 * If every step in the setup has been completed, then we will mark the setup as being
					 * completed in the system so that other users can be aware of the setup's status.
					 */
					return setupService.markSetupAsComplete(factory.stepMap).then(function() {
						cv.sessionContext.completedSetups[factory.stepMap] = true;
						return factory.refreshSession();
					});
				} else {
					return factory.refreshSession();
				}
			}
			return Promise.resolve();
		};

		/**
		 * Check to see if the current session context has a setupStepsMap of the given name.
		 *
		 * @param {string}
		 *            map - The map to check if it exists in the current sessionContext.
		 * @return {boolean} True if the map exists; False otherwise.
		 */
		factory.hasMap = function(map) {
			return (
				angular.isDefined(cv.sessionContext.setupStepsMap) && angular.isDefined(cv.sessionContext.setupStepsMap[map])
			);
		};

		/**
		 * Set the setupStepsMap of the given name to the given setupSteps.
		 *
		 * @param {string}
		 *            map The map to set.
		 * @param {boolean[]}
		 *            setupSteps The setupSteps to set.
		 */
		factory.setMap = function(map, setupSteps) {
			cv.sessionContext.setupStepsMap[map] = setupSteps;
		};

		/**
		 * Get the setupStepsMap of the given name in the current session context.
		 *
		 * @param {string}
		 *            map - The map to get from the session context.
		 */
		factory.getMap = function(map) {
			return factory.hasMap(map) ? cv.sessionContext.setupStepsMap[map] : null;
		};

		/**
		 * Go to the user's most recently made server.
		 */
		factory.goToServer = function() {
			serverService
				.getServers()
				.success(function(data) {
					$location.path('/servers/' + data[data.length - 1].clientId);
				})
				.error(function() {
					cvToaster.showInfoMessage({
						ttl: 10000,
						message: cvLoc('error.navigation')
					});
				});
		};

		/**
		 * Start a backup for the user's most recently made server.
		 */
		factory.startBackup = function() {
			serverService.backupRecentServer(
				function(wr) {
					$location.path('jobs/' + wr.jobIds[0]);
				},
				function() {
					cvToaster.showInfoMessage({
						ttl: '10000', //10 sec
						message: cvLoc('error.startBackup')
					});
				}
			);
		};

		return factory;
	}
]);

export default commonSetupModule;
