import { reportsDashboardModule } from 'common/js/modules';
import 'modules/reports/js/services/reports.svc';
import 'modules/reports/js/factory/reports.factory.js';
import 'modules/reports/js/constants/reports.constants.js';

var app = reportsDashboardModule;

app.directive('cvReportsLaptopLocation', function() {
	return {
		restrict: 'E',
		scope: {},
		templateUrl: appUtil.appRoot + 'modules/reports/partials/directives/cvReportsLaptopLocation.jsp',
		bindToController: {
			dashboardType: '='
		},
		controller: 'DashboardLaptopLocationController',
		controllerAs: 'locationCtrl'
	};
});
app.controller('DashboardLaptopLocationController', [
	'cvLoc',
	'reportsDashboardService',
	'reportsDashboardFactory',
	'cvToaster',
	function DashboardLocationController(cvLoc, reportsDashboardService, reportsDashboardFactory, cvToaster) {
		var self = this;
		this.$onInit = function() {
			var MARKER_COLOR_SAFE = '#228b22';

			self.data = {};
			self.data.title = cvLoc('pane.laptopLocation.title');

			getLaptopLocationData();

			function getLaptopLocationData() {
				self.data.showLoader = true;
				reportsDashboardService
					.getLaptopLocation()
					.success(function(data) {
						onSuccess(data);
					})
					.error(function(err) {
						onError(err);
					});
			}

			function onSuccess(rawData) {
				self.data.showLoader = false;
				var data = reportsDashboardFactory.getJsonFromDSResult(rawData);

				if (!data || data.length == 0) {
					self.noData = { text: cvLoc('dashboard.msg.noDataAvailable') };
				} else {
					var result = normalizeData(data);
					var mapData = {
						type: 'FeatureCollection',
						crs: {
							type: 'name',
							properties: {
								name: 'urn:ogc:def:crs:OGC:1.3:CRS84'
							}
						},
						features: result.geoJson
					};

					mapboxgl.accessToken = 'pk.eyJ1IjoiY29tbXZhdWx0LW1hcHMiLCJhIjoiNk81djMzYyJ9.1fAPKqwczcfXkTEBZevStA';
					var map = new mapboxgl.Map({
						container: 'dashboard-mapbox-wrapper',
						style: 'mapbox://styles/mapbox/streets-v10',
						center: [0, 30],
						maxZoom: 15,
						zoom: 0.8
					});
					map.on('load', function() {
						// Add a new source from our GeoJSON data and set the
						// 'cluster' option to true. GL-JS will add the point_count property to your source data.
						map.addSource('laptops', {
							type: 'geojson',
							data: mapData,
							cluster: true,
							clusterMaxZoom: 15, // Max zoom to cluster points on
							clusterRadius: 50 // Radius of each cluster when clustering points (defaults to 50)
						});
						var nav = new mapboxgl.NavigationControl();
						map.addControl(nav, 'bottom-right');
						//map.addControl(new mapboxgl.FullscreenControl());
						map.addLayer({
							id: 'clusters',
							type: 'circle',
							source: 'laptops',
							filter: ['has', 'point_count'],
							paint: {
								// Use step expressions (https://docs.mapbox.com/mapbox-gl-js/style-spec/#expressions-step)
								// with four steps to implement four types of circles:
								'circle-color': [
									'step',
									['get', 'point_count'],
									'#F5A9E1',
									10,
									'#F79F81',
									100,
									'#81F781',
									500,
									'#F4FA58'
								],
								'circle-radius': ['step', ['get', 'point_count'], 15, 10, 20, 100, 30, 500, 40]
							}
						});

						map.addLayer({
							id: 'cluster-count',
							type: 'symbol',
							source: 'laptops',
							filter: ['has', 'point_count'],
							layout: {
								'text-field': '{point_count_abbreviated}',
								'text-size': 12
							}
						});

						map.addLayer({
							id: 'unclustered-point',
							type: 'circle',
							source: 'laptops',
							filter: ['!=', 'cluster', true],
							paint: {
								'circle-color': '#F79F81',
								'circle-radius': 10
							}
						});
						map.addLayer({
							id: 'uncluster-count',
							type: 'symbol',
							source: 'laptops',
							filter: ['!=', 'cluster', true],
							layout: {
								'text-field': '1',
								'text-size': 12
							}
						});

						// inspect a cluster on click
						map.on('click', 'clusters', function(e) {
							var features = map.queryRenderedFeatures(e.point, { layers: ['clusters'] });
							var clusterId = features[0].properties.cluster_id;
							map.getSource('laptops').getClusterExpansionZoom(clusterId, function(err, zoom) {
								if (err) {
									return;
								}

								map.easeTo({
									center: features[0].geometry.coordinates,
									speed: 0.5,
									zoom: zoom - 3
								});
							});
						});

						map.on('click', 'unclustered-point', function(e) {
							var features = map.queryRenderedFeatures(e.point, { layers: ['unclustered-point'] });
							map.flyTo({
								center: features[0].geometry.coordinates,
								zoom: 12
							});
						});

						map.on('mouseenter', 'clusters', function(e) {
							map.getCanvas().style.cursor = 'pointer';

							var features = map.queryRenderedFeatures(e.point, { layers: ['clusters'] });
							var clusterId = features[0].properties.cluster_id,
								point_count = features[0].properties.point_count,
								clusterSource = map.getSource('laptops');

							// Get all points under a cluster
							clusterSource.getClusterLeaves(clusterId, point_count, 0, function(err, aFeatures) {
								var clients = '';
								var col = 0,
									max = 15;
								for (var i = 0; i < aFeatures.length; i++) {
									if (i >= max) {
										clients += '+' + (point_count - max) + ' more.';
										break;
									}
									clients = clients + aFeatures[i].properties.title + ', ';
									col++;
									if (col == 3) {
										clients += '<br>';
										col = 0;
									}
								}
								if (point_count < max) {
									clients = clients.substr(0, clients.length - 2);
								}
								var coordinates = e.features[0].geometry.coordinates.slice();
								while (Math.abs(e.lngLat.lng - coordinates[0]) > 180) {
									coordinates[0] += e.lngLat.lng > coordinates[0] ? 360 : -360;
								}
								popup
									.setLngLat(coordinates)
									.setHTML(clients)
									.addTo(map);
							});
						});
						map.on('mouseleave', 'clusters', function() {
							map.getCanvas().style.cursor = '';
							popup.remove();
						});
						// Create a popup, but don't add it to the map yet.
						var popup = new mapboxgl.Popup({
							closeButton: false,
							closeOnClick: false
						});

						map.on('mouseenter', 'unclustered-point', function(e) {
							// Change the cursor style as a UI indicator.
							map.getCanvas().style.cursor = 'pointer';

							var coordinates = e.features[0].geometry.coordinates.slice();
							var description = e.features[0].properties.title;

							// Ensure that if the map is zoomed out such that multiple
							// copies of the feature are visible, the popup appears
							// over the copy being pointed to.
							while (Math.abs(e.lngLat.lng - coordinates[0]) > 180) {
								coordinates[0] += e.lngLat.lng > coordinates[0] ? 360 : -360;
							}

							// Populate the popup and set its coordinates
							// based on the feature found.
							popup
								.setLngLat(coordinates)
								.setHTML(description)
								.addTo(map);
						});

						map.on('mouseleave', 'unclustered-point', function() {
							map.getCanvas().style.cursor = '';
							popup.remove();
						});
					});
				}
			}

			function normalizeData(data) {
				var result = {
					geoJson: []
				};
				if (data.length > 0) {
					data.forEach(function(loc) {
						// Merging laptops in same location to a single point value
						if (loc.latitude && loc.longitude) {
							var temp = {
								type: 'Feature',
								geometry: {
									type: 'Point',
									coordinates: [loc.longitude, loc.latitude]
								},
								properties: {
									//"id": loc.clientId,
									title: loc.clientName,
									'marker-color': MARKER_COLOR_SAFE
								}
							};
							result.geoJson.push(temp);
						}
					});
				}
				result.centeringData = getCenterandZoomForMap(data);

				return result;
			}

			function getCenterandZoomForMap(locationData) {
				var result = {
					center: [0, 0],
					zoom: 2
				};

				var minLatitude = Infinity;
				var maxLatitude = -Infinity;
				var minLongitude = Infinity;
				var maxLongitude = -Infinity;

				Object.keys(locationData).forEach(function(key) {
					var value = locationData[key];
					if (value.latitude && value.longitude) {
						value.latitude = parseFloat(value.latitude);
						value.longitude = parseFloat(value.longitude);
						if (value.latitude < minLatitude) {
							minLatitude = value.latitude;
						}
						if (value.latitude > maxLatitude) {
							maxLatitude = value.latitude;
						}
						if (value.longitude < minLongitude) {
							minLongitude = value.longitude;
						}
						if (value.longitude > maxLongitude) {
							maxLongitude = value.longitude;
						}
					}
				});

				if (
					minLatitude != Infinity &&
					maxLatitude != -Infinity &&
					minLongitude != Infinity &&
					maxLongitude != -Infinity
				) {
					var centerLatitude = (minLatitude + maxLatitude) / 2;
					var centerLongitude = (minLongitude + maxLongitude) / 2;

					var maxdistanceFromCenter = 0;

					Object.keys(locationData).forEach(function(key) {
						var value = locationData[key];
						if (value.latitude && value.longitude) {
							value.latitude = parseFloat(value.latitude);
							value.longitude = parseFloat(value.longitude);
							var distance = Math.sqrt(
								Math.pow(centerLatitude - value.latitude, 2) + Math.pow(centerLongitude - value.longitude, 2)
							); // Distance between two points in a plane

							if (distance > maxdistanceFromCenter) {
								maxdistanceFromCenter = distance;
							}
						}
					});

					result.center = [centerLatitude, centerLongitude];

					if (maxdistanceFromCenter == 0) {
						result.zoom = 7;
					} else if (maxdistanceFromCenter < 10) {
						result.zoom = 5;
					} else if (maxdistanceFromCenter < 50) {
						result.zoom = 3;
					} else if (maxdistanceFromCenter < 100) {
						result.zoom = 2;
					} else {
						result.zoom = 1;
					}
				}
				return result;
			}

			function onError(err) {
				self.data.showLoader = false;
				self.noData = { text: cvLoc('error.loadingLocationData') };
				console.error(err);
				reportsDashboardFactory.showErrorMessage(err, self.data.title);
			}
		};
	}
]);

export default app;
