import 'modules/reports/js/constants/paneEntityTypes.constants.js';
import 'modules/reports/js/directives/cv-reports-pane-orchestration-directive.js';
import 'modules/reports/js/factory/reports.factory.js';
import 'modules/reports/js/services/reports.svc.js';
import 'modules/reports/js/directives/cv-reports-commcell-info-directive.js';
import 'modules/reports/js/directives/cv-dashboard-title-dropdown.js';
import 'vsa/js/factories/replicationGroups.factory.js';

import { reportsDashboardModule } from 'common/js/modules';

/**
 * Shows the Dashboard for Orchestration
 */

reportsDashboardModule.controller('orchestrationDashboard', [
	'cvLoc',
	'setupFactory',
	'reportsDashboardFactory',
	'reportsDashboardService',
	'PaneEntityTypes',
	'replicationGroupFactory',
	function(
		cvLoc,
		setupFactory,
		reportsDashboardFactory,
		reportsDashboardService,
		PaneEntityTypes,
		replicationGroupFactory
	) {
		var self = this;
		self.dashboardType = 'orchestration';
		self.showCommCellInfo = cv.userRole == 'Role_Msp_Admin';
		const StatsPaneLinks = {
			drjobs: {
				complete:
					'#/jobs?jobType=126&view=finishedJobs30&status=Completed,Completed%20w%2F%20one%20or%20more%20errors&showAdminJobs=true',
				failed: '#/jobs?jobType=126&view=finishedJobs30&status=Failed&showAdminJobs=true'
			},
			cloudMigrationsReport: {
				databases: '#/reportDetails?name=Cloud%20Migrations&app=ADMIN_CONSOLE&input.i_Type=Databases',
				applications: '#/reportDetails?name=Cloud%20Migrations&app=ADMIN_CONSOLE&input.i_Type=Applications',
				vms: '#/reportDetails?name=Cloud%20Migrations&app=ADMIN_CONSOLE&input.i_Type=VMs'
			},
			liveMountsReport: {
				fileServers: '#/reportDetails?name=Live%20Mount%20Jobs&app=ADMIN_CONSOLE&input.i_Type=FileServers',
				vms: '#/reportDetails?name=Live%20Mount%20Jobs&app=ADMIN_CONSOLE&input.i_Type=VMs'
			}
		};

		/* ---- CLASSES START ---- */
		class OrchestrationPaneTitle {
			constructor(title, href) {
				this._title = {
					text: title,
					href: href
				};
			}

			get title() {
				return this._title;
			}
		}

		class OrchestrationEntity {
			constructor(value, title, imgSrc, addSeperator, type, entityClass, href, tooltip) {
				this._entity = {
					value: value,
					title: title,
					imgSrc: imgSrc,
					addSeperator: addSeperator,
					type: type,
					entityClass: entityClass,
					detailsLink: href,
					tooltip: tooltip
				};
			}

			get entity() {
				return this._entity;
			}
		}

		class OverviewEntity extends OrchestrationEntity {
			constructor(title, value, img, href) {
				super(value, title, img, false, PaneEntityTypes.EVENT, 'overviewentity', href);
			}
		}

		class StatsEntity extends OrchestrationEntity {
			constructor(value, title, addSeperator, href) {
				super(value, title, null, addSeperator, PaneEntityTypes.EVENT, 'statsentity', href);
			}
		}

		class OverviewPane {
			constructor(title, value, img, href) {
				this._pane = {
					title: cvLoc('dashboard.overview'),
					showLoader: true,
					noData: null,
					gridClasses: 'col-xs-12 col-sm-12 col-md-6 col-lg-2'
				};

				if (title && value && img) {
					this._pane.entities = [new OverviewEntity(title, value, img, href).entity];
				}
			}

			get pane() {
				return this._pane;
			}

			static entities(title, value, img, href) {
				return [new OverviewEntity(title, value, img, href).entity];
			}
		}

		class ApplicationOverviewPane extends OverviewPane {
			constructor(genericAppValue, genericAppImg, mailboxesValue, mailboxesImg) {
				super();
				this._pane.gridClasses = 'col-xs-12 col-sm-12 col-md-5 col-lg-4';
				if (genericAppValue && mailboxesValue) {
					this._pane.entities = [
						new OrchestrationEntity(
							genericAppValue,
							cvLoc('label.genericApps'),
							genericAppImg,
							true,
							PaneEntityTypes.EVENT,
							'overviewentity',
							'#/genApps',
							cvLoc('label.genericApps')
						).entity,
						new OrchestrationEntity(
							mailboxesValue,
							cvLoc('label.mailboxes'),
							mailboxesImg,
							false,
							PaneEntityTypes.EVENT,
							'overviewentity',
							'#/exchange',
							cvLoc('label.mailboxes')
						).entity
					];
				}
			}

			static entities(genericAppValue, genericAppImg, mailboxesValue, mailboxesImg) {
				return [
					new OrchestrationEntity(
						genericAppValue,
						cvLoc('label.genericApps'),
						genericAppImg,
						true,
						PaneEntityTypes.EVENT,
						'overviewentity',
						'#/genApps',
						cvLoc('label.genericApps')
					).entity,
					new OrchestrationEntity(
						mailboxesValue,
						cvLoc('label.mailboxes'),
						mailboxesImg,
						false,
						PaneEntityTypes.EVENT,
						'overviewentity',
						'#/exchange',
						cvLoc('label.mailboxes')
					).entity
				];
			}
		}

		class SLAPane {
			constructor(dashboardType, appGroup) {
				this._pane = {
					title: cvLoc('dashboard.sla'),
					titleDetailsLink:
						appUtil.appRoot + '#/reportDetails?name=Recovery Readiness&app=ADMIN_CONSOLE&input.AppGroup=' + appGroup,
					showLoader: false,
					noData: null,
					data: {
						showTitle: false,
						showDocLink: false,
						dashboardType: dashboardType,
						appGroup: appGroup,
						showAsSmaller: true
					},
					gridClasses: 'col-xs-12 col-sm-12 col-md-6 col-lg-3'
				};
			}

			get pane() {
				return this._pane;
			}
		}

		class ApplicationSLAPane extends SLAPane {
			constructor() {
				super('Orchestration_Applications', 'Orchestration_Applications');
				this._pane.gridClasses = 'col-xs-12 col-sm-12 col-md-5 col-lg-4';
			}
		}

		class StatsPane {
			constructor(liveMountsValue, cloudMigrationsValue, failoverRunsValue) {
				this._pane = {
					title: cvLoc('label.lastMonthStats'),
					showLoader: false,
					noData: null,
					gridClasses: 'col-xs-12 col-sm-12 col-md-12 col-lg-5'
				};
				if (liveMountsValue && cloudMigrationsValue && failoverRunsValue) {
					this._pane.entities = [
						new StatsEntity(liveMountsValue, cvLoc('label.liveMounts'), true).entity,
						new StatsEntity(cloudMigrationsValue, cvLoc('entity.cloudmigrations.title'), true).entity,
						new StatsEntity(
							failoverRunsValue,
							cvLoc('entity.failoverruns.title'),
							false,
							StatsPaneLinks.drjobs.complete
						).entity
					];
				}
			}

			get pane() {
				return this._pane;
			}

			static entities(liveMountsValue, cloudMigrationsValue, failoverRunsValue) {
				return [
					new StatsEntity(liveMountsValue, cvLoc('label.liveMounts'), true).entity,
					new StatsEntity(cloudMigrationsValue, cvLoc('entity.cloudmigrations.title'), true).entity,
					new StatsEntity(failoverRunsValue, cvLoc('entity.failoverruns.title'), false, StatsPaneLinks.drjobs.complete)
						.entity
				];
			}
		}

		class DatabaseStatsPane extends StatsPane {
			constructor(clonesValue, cloudMigrationsValue, failoverRunsValue) {
				super();
				if (clonesValue && cloudMigrationsValue && failoverRunsValue) {
					this._pane.entities = [
						new StatsEntity(clonesValue, cvLoc('label.clones'), true, '#/dbInstancesClone').entity,
						new StatsEntity(
							cloudMigrationsValue,
							cvLoc('entity.cloudmigrations.title'),
							true,
							StatsPaneLinks.cloudMigrationsReport.databases
						).entity,
						new StatsEntity(
							failoverRunsValue,
							cvLoc('entity.failoverruns.title'),
							false,
							StatsPaneLinks.drjobs.complete
						).entity
					];
				}
			}

			static entities(clonesValue, cloudMigrationsValue, failoverRunsValue) {
				return [
					new StatsEntity(clonesValue, cvLoc('label.clones'), true, '#/dbInstancesClone').entity,
					new StatsEntity(
						cloudMigrationsValue,
						cvLoc('entity.cloudmigrations.title'),
						true,
						StatsPaneLinks.cloudMigrationsReport.databases
					).entity,
					new StatsEntity(failoverRunsValue, cvLoc('entity.failoverruns.title'), false, StatsPaneLinks.drjobs.complete)
						.entity
				];
			}
		}

		class FileServersStatsPane extends StatsPane {
			constructor(liveMountsValue, failoverRunsValue) {
				super();
				if (liveMountsValue && failoverRunsValue) {
					this._pane.entities = [
						new StatsEntity(
							liveMountsValue,
							cvLoc('label.liveMounts'),
							true,
							StatsPaneLinks.liveMountsReport.fileServers
						).entity,
						new StatsEntity(
							failoverRunsValue,
							cvLoc('entity.failoverruns.title'),
							false,
							StatsPaneLinks.drjobs.complete
						).entity
					];
				}
			}

			get pane() {
				return this._pane;
			}

			static entities(liveMountsValue, failoverRunsValue) {
				return [
					new StatsEntity(liveMountsValue, cvLoc('label.liveMounts'), true, StatsPaneLinks.liveMountsReport.fileServers)
						.entity,
					new StatsEntity(failoverRunsValue, cvLoc('entity.failoverruns.title'), false, StatsPaneLinks.drjobs.complete)
						.entity
				];
			}
		}

		class ApplicationStatsPane extends StatsPane {
			constructor(cloudMigrationsValue) {
				super();
				this._pane.gridClasses = 'col-xs-12 col-sm-12 col-md-2 col-lg-4';
				if (cloudMigrationsValue) {
					this._pane.entities = [
						new StatsEntity(
							cloudMigrationsValue,
							cvLoc('entity.cloudmigrations.title'),
							false,
							StatsPaneLinks.cloudMigrationsReport.applications
						).entity
					];
				}
			}

			static entities(cloudMigrationsValue) {
				return [
					new StatsEntity(
						cloudMigrationsValue,
						cvLoc('entity.cloudmigrations.title'),
						false,
						StatsPaneLinks.cloudMigrationsReport.applications
					).entity
				];
			}
		}

		class VMsStatsPane extends StatsPane {
			constructor(liveMountsValue, cloudMigrationsValue, failoverRunsValue, failoverFailedValue) {
				super();
				if (liveMountsValue && cloudMigrationsValue && failoverRunsValue && failoverFailedValue) {
					this._pane.entities = [
						new StatsEntity(liveMountsValue, cvLoc('label.liveMounts'), true, StatsPaneLinks.liveMountsReport.vms)
							.entity,
						new StatsEntity(
							cloudMigrationsValue,
							cvLoc('entity.cloudmigrations.title'),
							true,
							StatsPaneLinks.cloudMigrationsReport.vms
						).entity,
						new StatsEntity(
							failoverRunsValue,
							cvLoc('entity.failovercompleted.title'),
							true,
							StatsPaneLinks.drjobs.complete
						).entity,
						new StatsEntity(
							failoverFailedValue,
							cvLoc('entity.failoverfailed.title'),
							false,
							StatsPaneLinks.drjobs.failed
						).entity
					];
				}
			}

			static entities(liveMountsValue, cloudMigrationsValue, failoverRunsValue, failoverFailedValue) {
				return [
					new StatsEntity(liveMountsValue, cvLoc('label.liveMounts'), true, StatsPaneLinks.liveMountsReport.vms).entity,
					new StatsEntity(
						cloudMigrationsValue,
						cvLoc('entity.cloudmigrations.title'),
						true,
						StatsPaneLinks.cloudMigrationsReport.vms
					).entity,
					new StatsEntity(failoverRunsValue, cvLoc('entity.failoverruns.title'), true, StatsPaneLinks.drjobs.complete)
						.entity,
					new StatsEntity(
						failoverFailedValue,
						cvLoc('entity.failoverfailed.title'),
						false,
						StatsPaneLinks.drjobs.failed
					).entity
				];
			}
		}

		class BarChart {
			constructor(title, metValue, missedValue) {
				var metPercent = 0;
				if (metValue > 0 || missedValue > 0) {
					metPercent = Math.floor(parseFloat((metValue / (metValue + missedValue)) * 100).toFixed(2));
				}
				var missedPercent = 100 - metPercent;

				this._config = {
					options: {
						chart: {
							type: 'bar'
						},
						exporting: false,
						plotOptions: {
							series: {
								stacking: 'normal'
							},
							bar: {
								pointWidth: 10
							}
						},
						colors: ['#1ebe50', '#ee4743'],
						legend: {
							reversed: true,
							itemStyle: {
								fontWeight: 'normal',
								fontSize: '10px'
							},
							x: 0,
							y: 10,
							align: 'right',
							verticalAlign: 'middle',
							layout: 'vertical'
						}
					},
					title: {
						text: title,
						style: {
							fontSize: '10px',
							fontWeight: 'normal',
							color: '#428BCA'
						}
					},
					xAxis: {
						categories: [''],
						gridLineWidth: 0,
						minorTickLength: 0,
						tickLength: 0,
						lineColor: 'transparent',
						lineWidth: 0
					},
					yAxis: {
						min: 0,
						title: {
							text: null
						},
						gridLineWidth: 0,
						minorTickLength: 0,
						tickLength: 0,
						lineColor: 'transparent',
						lineWidth: 0,
						labels: {
							enabled: false
						}
					},
					series: [
						{
							name: cvLoc('sla.label.met') + ' ' + metValue + ' (' + metPercent + '%)',
							data: [metValue]
						},
						{
							name: cvLoc('sla.label.missed') + ' ' + missedValue + ' (' + missedPercent + '%)',
							data: [missedValue]
						}
					]
				};
			}

			get config() {
				return this._config;
			}
		}

		class ReplicationGroupsPane {
			constructor(config) {
				this._pane = {
					title: cvLoc('label.replicationGroups'),
					showLoader: true,
					noData: null,
					gridClasses: 'col-xs-12 col-sm-12 col-md-12 col-lg-5'
				};
				if (config) {
					this._pane.config = config;
				}
			}

			get pane() {
				return this._pane;
			}

			static config(rpo) {
				return [new BarChart(cvLoc('label.liveSyncReplication'), rpo.metrpo, rpo.missedrpo).config];
			}
		}

		class VMsReplicationGroupsPane extends ReplicationGroupsPane {
			constructor(config) {
				super();
				this._pane.title = 'Replication monitor';
			}

			static config(rpoLiveSync, rpoLiveSyncDirect) {
				var charts = [];
				if (!isSLADataZeroes(rpoLiveSync)) {
					charts.push(new BarChart(cvLoc('label.replicationGroups'), rpoLiveSync.metrpo, rpoLiveSync.missedrpo).config);
				}
				return charts;
			}
		}

		class OrchestrationPane {
			constructor(title, name, cssclass, noData, overviewPane, slaPane, statsPane, replicationGroupsPane) {
				this._paneData = {
					title: title,
					name: name,
					class: cssclass,
					noData: noData,
					overview: overviewPane,
					//sla: slaPane,
					stats: statsPane,
					replicationgroups: replicationGroupsPane
				};
			}

			get paneData() {
				return this._paneData;
			}
		}

		/* ---- CLASSES END ---- */

		/* ---- INIT FUNCTIONS START ---- */

		/*
		 * Overview data for all the IDA's are fetched with one dataset query.
		 */
		function fetchOverviewData() {
			reportsDashboardService
				.getOverviewData('orchestration')
				.success(function(data) {
					if (data && data.columns && data.columns.length > 0 && data.records && data.records.length > 0) {
						var propertyTypeIndex = -1;
						var propertyCountIndex = -1;
						var expectedRecords = {
							databases: {
								value: 0,
								img: 'common/img/Databases.svg'
							},
							fileservers: {
								value: 0,
								img: 'common/img/FileServers.svg'
							},
							genericapps: {
								value: 0,
								img: 'common/img/GenericApps.svg'
							},
							mailboxes: {
								value: 0,
								img: 'common/img/Mailboxes.svg'
							},
							vms: {
								value: 0,
								img: 'common/img/VMs.svg'
							}
						};
						data.columns.forEach(function(column, index) {
							if (column.name === 'PropertyType') {
								propertyTypeIndex = index;
							} else if (column.name === 'PropertyCount') {
								propertyCountIndex = index;
							}
						});

						data.records.forEach(function(record) {
							var expectedRecordName = record[propertyTypeIndex].toLowerCase();
							var expectedRecordValue = record[propertyCountIndex];
							expectedRecords[expectedRecordName].value = expectedRecordValue;
						});

						/*
						 * Remove the loader for the overview panes. Create entity inputs for the overview panes. If
						 * the values are 0, show no data for the entire OrchestrationPane
						 */
						removeLoader('overview');
						var expectedRecord = null;
						if ((expectedRecord = expectedRecords['databases']).value === 0) {
							getDatabasePane().noData = {
								text: cvLoc('noData.database'),
								href: {
									text: cvLoc('dashboard.configure'),
									href: self.paneLinks.databases
								}
							};
						} else {
							getDatabasePane().overview.entities = OverviewPane.entities(
								cvLoc('entity.label.servers'),
								expectedRecord.value,
								expectedRecord.img,
								'#/instances'
							);
						}
						if ((expectedRecord = expectedRecords['fileservers']).value === 0) {
							getFileServersPane().noData = {
								text: cvLoc('noData.fileServers'),
								href: {
									text: cvLoc('dashboard.configure'),
									href: self.paneLinks.fileServers
								}
							};
						} else {
							getFileServersPane().overview.entities = OverviewPane.entities(
								cvLoc('entity.label.fileServers'),
								expectedRecord.value,
								expectedRecord.img,
								'#/fsServers'
							);
						}
						if (expectedRecords['genericapps'].value === 0 && expectedRecords['mailboxes'].value === 0) {
							getApplicationsPane().noData = {
								text: cvLoc('noData.applications'),
								href: {
									text: cvLoc('dashboard.configure'),
									href: self.paneLinks.applications
								}
							};
						} else {
							getApplicationsPane().overview.entities = ApplicationOverviewPane.entities(
								expectedRecords['genericapps'].value,
								expectedRecords['genericapps'].img,
								expectedRecords['mailboxes'].value,
								expectedRecords['mailboxes'].img
							);
						}
						if ((expectedRecord = expectedRecords['vms']).value === 0) {
							getVMsPane().noData = {
								text: cvLoc('noData.vmsBackedup'),
								href: {
									text: cvLoc('dashboard.configure'),
									href: self.paneLinks.vms
								}
							};
						} else {
							getVMsPane().overview.entities = OverviewPane.entities(
								cvLoc('pane.vms.title'),
								expectedRecord.value,
								expectedRecord.img,
								'#/virtualization/vms?vmStatus=ALL'
							);
						}
					} else {
						var noData = {
							text: cvLoc('noData.overview')
						};
						showErrorMessage(noData, 'overview');
					}
				})
				.error(function(e) {
					var noData = {
						text: cvLoc('error.overview')
					};
					showErrorMessage(noData, 'overview', true, e);
				});
		}

		/*
		 * Separate dataset queries need to be fired for each IDA Stats data.
		 */
		function fetchStatsData() {
			var statsPanes = [
				{
					dataSet: 'Databases',
					orchestrationPaneName: 'databases',
					expectedColumns: {
						livemounts: {
							value: 0
						},
						cloudmigrations: {
							value: 0
						},
						failoverruns: {
							value: 0
						}
					},
					buildEntities: function() {
						return DatabaseStatsPane.entities(
							this.expectedColumns['livemounts'].value,
							this.expectedColumns['cloudmigrations'].value,
							this.expectedColumns['failoverruns'].value
						);
					}
				},
				{
					dataSet: 'FileServers',
					orchestrationPaneName: 'fileServers',
					expectedColumns: {
						livemounts: {
							value: 0
						},
						failoverruns: {
							value: 0
						}
					},
					buildEntities: function() {
						return FileServersStatsPane.entities(
							this.expectedColumns['livemounts'].value,
							this.expectedColumns['failoverruns'].value
						);
					}
				},
				{
					dataSet: 'Applications',
					orchestrationPaneName: 'applications',
					expectedColumns: {
						cloudmigrations: {
							value: 0
						}
					},
					buildEntities: function() {
						return ApplicationStatsPane.entities(this.expectedColumns['cloudmigrations'].value);
					}
				},
				{
					dataSet: 'VMs',
					orchestrationPaneName: 'vms',
					expectedColumns: {
						livemounts: {
							value: 0
						},
						cloudmigrations: {
							value: 0
						},
						failoverruns: {
							value: 0
						},
						failoverfailed: {
							value: 0
						}
					},
					buildEntities: function() {
						return VMsStatsPane.entities(
							this.expectedColumns['livemounts'].value,
							this.expectedColumns['cloudmigrations'].value,
							this.expectedColumns['failoverruns'].value,
							this.expectedColumns['failoverfailed'].value
						);
					}
				}
			];

			statsPanes.forEach(function(pane) {
				reportsDashboardService
					.getLastMonthStats(pane.dataSet)
					.success(function(data) {
						if (data && data.columns && data.columns.length > 0 && data.records && data.records.length > 0) {
							data.columns.forEach(function(column, index) {
								var found;
								if ((found = pane.expectedColumns[column.name.toLowerCase()])) {
									found.value = found.value || data.records[0][index];
								}
							});
							self.orchestrationPanes.forEach(function(orchestrationPane) {
								if (orchestrationPane.paneData.name === pane.orchestrationPaneName) {
									orchestrationPane.paneData.stats.showLoader = false;
									orchestrationPane.paneData.stats.entities = pane.buildEntities();
								}
							});
						} else {
							var noData = {
								text: cvLoc('error.lastMonthStats')
							};
							showNoData('stats', noData, pane.orchestrationPaneName);
							removeLoader('stats', pane.orchestrationPaneName);
						}
					})
					.error(function(e) {
						var noData = {
							text: cvLoc('error.lastMonthStats')
						};
						showNoData('stats', noData, pane.orchestrationPaneName);
						removeLoader('stats', pane.orchestrationPaneName);
						reportsDashboardFactory.showErrorMessage(e && e.errorMessage ? e.errorMessage : cvLoc('generic_error'));
					});
			});
		}

		/*
		 * One dataset query fetches the replication group stats data
		 */
		function fetchReplicationGroupStats() {
			reportsDashboardService
				.getDataset('ReplicationGroupStats')
				.success(function(data) {
					var noData = getReplicationGroupsNoData();
					var noDataFS = getFSReplicationGroupsNoData();
					var noDataVSA = getVSAReplicationGroupsNoData();

					if (data && data.columns && data.columns.length > 0 && data.records && data.records.length > 0) {
						var expectedColumns = {
							type: -1,
							replicationtype: -1,
							metrpo: -1,
							missedrpo: -1
						};
						var expectedRecords = {
							databases: {
								livesync: {
									metrpo: 0,
									missedrpo: 0
								}
							},
							fileservers: {
								livesync: {
									metrpo: 0,
									missedrpo: 0
								}
							},
							vms: {
								livesync: {
									metrpo: 0,
									missedrpo: 0
								},
								livesyncdirect: {
									metrpo: 0,
									missedrpo: 0
								}
							}
						};
						data.columns.forEach(function(column, index) {
							var columnName = column.name.toLowerCase();
							if (expectedColumns[columnName]) {
								expectedColumns[columnName] = index;
							}
						});

						data.records.forEach(function(record) {
							var type = record[expectedColumns.type];
							var replicationtype = record[expectedColumns.replicationtype];
							var metrpo = record[expectedColumns.metrpo];
							var missedrpo = record[expectedColumns.missedrpo];

							var rpo = expectedRecords[type.toLowerCase()][replicationtype.toLowerCase()];
							rpo.metrpo = metrpo;
							rpo.missedrpo = missedrpo;
						});

						/*
						 * If the met and missed rpo data is zero, then show no data message
						 */
						if (isSLADataZeroes(expectedRecords.databases.livesync)) {
							getDatabasePane().replicationgroups.noData = noData;
						} else {
							getDatabasePane().replicationgroups.config = ReplicationGroupsPane.config(
								expectedRecords.databases.livesync
							);
						}
						if (isSLADataZeroes(expectedRecords.fileservers.livesync)) {
							getFileServersPane().replicationgroups.noData = noDataFS;
						} else {
							getFileServersPane().replicationgroups.config = ReplicationGroupsPane.config(
								expectedRecords.fileservers.livesync
							);
						}
						if (isSLADataZeroes(expectedRecords.vms.livesync) && isSLADataZeroes(expectedRecords.vms.livesyncdirect)) {
							getVMsPane().replicationgroups.noData = noDataVSA;
						} else {
							getVMsPane().replicationgroups.config = VMsReplicationGroupsPane.config(
								expectedRecords.vms.livesync,
								expectedRecords.vms.livesyncdirect
							);
						}
						removeLoader('replicationgroups');
					} else {
						getDatabasePane().replicationgroups.noData = noData;
						getFileServersPane().replicationgroups.noData = noDataFS;
						getVMsPane().replicationgroups.noData = noDataVSA;
						removeLoader('replicationgroups');
					}
				})
				.error(function(e) {
					var noData = {
						text: cvLoc('error.replicationGroups')
					};
					showErrorMessage(noData, 'replicationgroups', true, e);
				});
		}

		function _init_() {
			self.paneLinks = getDefaultLinksForPanes();

			self.orchestrationPanes = [
				new OrchestrationPane(
					new OrchestrationPaneTitle(cvLoc('label.virtualMachines'), self.paneLinks.vms).title,
					'vms',
					'',
					null,
					new OverviewPane().pane,
					new SLAPane('virtualization', 'VMS').pane,
					new StatsPane().pane,
					new VMsReplicationGroupsPane().pane
				),
				new OrchestrationPane(
					new OrchestrationPaneTitle(cvLoc('label.databases'), self.paneLinks.databases).title,
					'databases',
					'databasesPane',
					null,
					new OverviewPane().pane,
					new SLAPane('Orchestration_Databases', 'Orchestration_Databases').pane,
					new DatabaseStatsPane().pane,
					new ReplicationGroupsPane().pane
				),
				new OrchestrationPane(
					new OrchestrationPaneTitle(cvLoc('label.fileServers'), self.paneLinks.fileServers).title,
					'fileServers',
					'fileSeversPane',
					null,
					new OverviewPane().pane,
					new SLAPane('Orchestration_Fileservers', 'Orchestration_Fileservers').pane,
					new FileServersStatsPane().pane,
					new ReplicationGroupsPane().pane
				),
				new OrchestrationPane(
					new OrchestrationPaneTitle(cvLoc('label.applications'), self.paneLinks.applications).title,
					'applications',
					'applicationsPane',
					null,
					new ApplicationOverviewPane().pane,
					new ApplicationSLAPane().pane,
					new ApplicationStatsPane().pane
				)
			];

			fetchOverviewData();
			fetchStatsData();
			fetchReplicationGroupStats();
		}

		_init_();

		/* ---- INIT FUNCTIONS END ---- */

		/* ---- HELPER FUNCTIONS BEGIN ---- */

		/*
		 * Checks if the setup is complete for the IDA's and determines the links that is used for headers and
		 * configure button.
		 */
		function getDefaultLinksForPanes() {
			var links = {};
			links.databases = setupFactory.isAtLeastOneSetupComplete('DBS') ? '#/instances' : '#/gettingStarted';
			links.fileServers = setupFactory.isAtLeastOneSetupComplete('ADMIN_CONSOLE')
				? '#/fsServers/overview'
				: '#/gettingStarted';
			links.applications = '#/genApps';
			links.vms = setupFactory.isAtLeastOneSetupComplete('VSA') ? '#/virtualization/hypervisors' : '#/gettingStarted';

			return links;
		}

		/*
		 * If orchestrationPaneName is provided, removes the loader from a specific pane under a specific
		 * OrchestrationPane. Otherwise, removes the loader from a specific pane from all the OrchestrationPanes.
		 */
		function removeLoader(pane, orchestrationPaneName) {
			self.orchestrationPanes.forEach(function(orchestrationPane) {
				if (orchestrationPaneName) {
					if (orchestrationPane.paneData.name === orchestrationPaneName && orchestrationPane.paneData[pane]) {
						orchestrationPane.paneData[pane].showLoader = false;
					}
				} else {
					if (orchestrationPane.paneData[pane]) {
						orchestrationPane.paneData[pane].showLoader = false;
					}
				}
			});
		}

		/*
		 * If orchestrationPaneName is provided, adds the no data message to specific pane under a specific
		 * OrchestrationPane Otherwise, adds the no data message to a specific pane in all the OrchestrationPanes
		 */
		function showNoData(pane, noDataMsg, orchestrationPaneName) {
			self.orchestrationPanes.forEach(function(orchestrationPane) {
				if (orchestrationPaneName) {
					if (orchestrationPane.paneData.name === orchestrationPaneName) {
						orchestrationPane.paneData[pane].noData = noDataMsg;
					}
				} else {
					if (orchestrationPane.paneData[pane]) {
						orchestrationPane.paneData[pane].noData = noDataMsg;
					}
				}
			});
		}

		/*
		 * Checks if the met and missed data is 0
		 */
		function isSLADataZeroes(obj) {
			return obj.metrpo === 0 && obj.missedrpo === 0;
		}

		function showErrorMessage(noData, pane, showToaster, e) {
			removeLoader(pane);
			showNoData(pane, noData);
			showToaster && reportsDashboardFactory.showErrorMessage(e ? e.errorMessage : cvLoc('generic_error'));
		}

		function getDatabasePane() {
			return self.orchestrationPanes[1].paneData;
		}

		function getFileServersPane() {
			return self.orchestrationPanes[2].paneData;
		}

		function getApplicationsPane() {
			return self.orchestrationPanes[3].paneData;
		}

		function getVMsPane() {
			return self.orchestrationPanes[0].paneData;
		}

		function getReplicationGroupsNoData() {
			return {
				text: cvLoc('noData.replicationGroups'),
				href: {
					text: cvLoc('dashboard.configure'),
					href: '#/replicationGroups'
				}
			};
		}

		function getFSReplicationGroupsNoData() {
			return {
				text: cvLoc('noData.replicationGroups'),
				onClick: {
					text: cvLoc('dashboard.configure'),
					onClick: function() {
						replicationGroupFactory.configureFSReplicationGroup();
					}
				}
			};
		}

		function getVSAReplicationGroupsNoData() {
			return {
				text: cvLoc('noData.replicationGroups'),
				onClick: {
					text: cvLoc('dashboard.configure'),
					onClick: function() {
						replicationGroupFactory.configureVSAReplicationGroup();
					}
				}
			};
		}

		/* ---- HELPER FUNCTIONS END ---- */
	}
]);

export default reportsDashboardModule;
