import { policyModule } from 'common/js/modules';
import 'modules/policies/schedulePolicies/js/constants/freqType.constant.js';
import 'modules/policies/schedulePolicies/js/filters/backupInterval.filter.js';
import 'modules/schedule/js/factory/schedule.factory.js';

var app = policyModule;

app.factory('schedulePatternFactory', ['$filter', 'cvLoc', 'FREQ_TYPE', 'scheduleUIFactory', schedulePatternFactory]);

function schedulePatternFactory($filter, cvLoc, FREQ_TYPE, scheduleUIFactory) {
	var factory = {
		editPatternEmitListener: 'EDIT_PATTERN_LISTENER',
		patternInfoInitialization: {
			Daily: initPatternForRepeatedSchedules,
			Weekly: initPatternForRepeatedSchedules,
			Monthly: initPatternForRepeatedSchedules,
			Monthly_Relative: initPatternForRepeatedSchedules,
			One_Time: initPatternForOneTimeSchedules,
			Automatic_Schedule: initPatternForAutomaticSchedules,
			After_Job_Completes: initPatternForContinuousSchedules
		}
	};

	factory.getPatternInfo = getPatternInfo;
	factory.isAutomaticScheduleSupported = isAutomaticScheduleSupported;
	factory.getDefaultOptionsForAutomaticSchedules = getDefaultOptionsForAutomaticSchedules;
	factory.editSchedulePattern = editSchedulePattern;
	factory.formatRepeatSettingsObject = formatRepeatSettingsObject;
	factory.formatAutomaticSchedules = formatAutomaticSchedules;

	function getPatternInfo(subTaskInfo) {
		let freqType = _.get(subTaskInfo, 'pattern.freq_type');
		let schedulePatternInfo = {
			frequencyDisplayStr: cvLoc(FREQ_TYPE[freqType])
		};
		if (factory.patternInfoInitialization.hasOwnProperty(freqType))
			factory.patternInfoInitialization[freqType](schedulePatternInfo, subTaskInfo);
		return schedulePatternInfo;
	}

	function initPatternForRepeatedSchedules(schedulePatternInfo, subTaskInfo) {
		let displayStr = scheduleUIFactory.scheduleDisplayString(subTaskInfo.pattern, false);
		schedulePatternInfo.showTimeString = true;
		schedulePatternInfo.timeStr = displayStr.timeStr;
		schedulePatternInfo.repeatInfoString = displayStr.repeatStr;
		if (schedulePatternInfo.repeatInfoString) {
			schedulePatternInfo.showRepeatString = true;
		}
	}

	function initPatternForOneTimeSchedules(schedulePatternInfo, subTaskInfo) {
		let startTime = _.get(subTaskInfo, 'pattern.active_start_time', 0);
		let startDateInSeconds = _.get(subTaskInfo, 'pattern.active_start_date', 0);
		schedulePatternInfo.showTimeString = true;
		schedulePatternInfo.showStartDate = true;
		schedulePatternInfo.timeStr = $filter('getTime')(startTime);
		schedulePatternInfo.startDate = $filter('date')(startDateInSeconds * 1000, 'MMM dd yyyy', 'UTC');
	}

	function initPatternForAutomaticSchedules(schedulePatternInfo, subTaskInfo) {
		let minBackupHour = _.get(subTaskInfo, 'options.commonOpts.automaticSchedulePattern.minBackupInterval', 0);
		let minBackupMinutes = _.get(
			subTaskInfo,
			'options.commonOpts.automaticSchedulePattern.minBackupIntervalMinutes',
			0
		);
		let maxBackupHour = _.get(subTaskInfo, 'options.commonOpts.automaticSchedulePattern.maxBackupInterval', 0);
		let maxBackupMinutes = _.get(
			subTaskInfo,
			'options.commonOpts.automaticSchedulePattern.maxBackupIntervalMinutes',
			0
		);
		schedulePatternInfo.showMinBackupInterval = true;
		schedulePatternInfo.showMaxBackupInterval = true;
		schedulePatternInfo.minBackupInterval = $filter('backupInterval')(minBackupHour, minBackupMinutes);
		schedulePatternInfo.maxBackupInterval = $filter('backupInterval')(maxBackupHour, maxBackupMinutes);
	}

	function initPatternForContinuousSchedules(schedulePatternInfo, subTaskInfo) {
		let intervalStr = _.get(subTaskInfo, 'pattern.freq_interval', 0);
		schedulePatternInfo.showIntervalBetweenTwoJobs = true;
		schedulePatternInfo.intervalBetweenTwoJobs = intervalStr + ' ' + cvLoc('label.minutes');
	}

	function isAutomaticScheduleSupported(taskType, policyType) {
		return taskType == 'SCHEDULE_POLICY' && policyType != 'DATA_PROTECTION';
	}

	function getDefaultOptionsForAutomaticSchedules() {
		return {
			maxBackupIntervalHours: 0,
			maxBackupIntervalMinutes: 0,
			minBackupIntervalHours: 0,
			minBackupIntervalMinutes: 15,
			daysBetweenSyntheticBackup: 30,
			minSyncIntervalHours: 0,
			minSyncIntervalMinutes: 2,
			ignoreAtMaxInterval: false,
			useOnlyWiredWork: false,
			minNetworkBandwidth: false,
			networkBandwidth: 128,
			useSpecificNetwork: false,
			specificNetworkIp: '0.0.0.0',
			specificNetworkNumber: 24,
			startOnlyOnAc: false,
			stopIfBatteryMode: false,
			preventSleep: false,
			cpuBelowThresholdEnabled: false,
			cpuBelowThreshold: 10,
			startOnlyfileBackUp: false,
			changeStopTask: function() {
				stopIfBatteryMode = false;
			}
		};
	}

	function editSchedulePattern(taskInfo, subTaskInfo) {
		let scheduleObj = {},
			attributes = getCustomAttributes(taskInfo, subTaskInfo);
		scheduleObj.model = subTaskInfo.schedulePattern;
		scheduleObj.model.opType = 'EDIT';
		scheduleObj.model.hideDelete = true;
		initDefaultOptionsForAutomaticSchedules(scheduleObj.model);
		scheduleUIFactory.addSchedule(
			scheduleObj,
			true,
			'Server',
			false,
			undefined,
			undefined,
			attributes,
			undefined,
			factory.editPatternEmitListener
		);
	}

	function getCustomAttributes(taskInfo, subTaskInfo) {
		let taskType = _.get(taskInfo, 'task.taskType'),
			policyType = _.get(taskInfo, 'task.policyType');
		return {
			hideBackupLevelOptions: true,
			operationType: _.get(subTaskInfo, 'subTask.operationType'),
			subTaskType: _.get(subTaskInfo, 'subTask.subTaskType'),
			customTitle: cvLoc('label.editSchedulePattern'),
			hideAutomaticSchedules: isAutomaticScheduleSupported(taskType, policyType),
			scheduleNameNotRequired: true,
			dismissModal: true,
			context: 'SCHEDULE_POLICY',
			hideTimeInCommServeTimezone: true
		};
	}

	function initDefaultOptionsForAutomaticSchedules(schedulePattern) {
		var freqType = _.get(schedulePattern, 'freqType', 'DAILY');
		if (freqType === 'AUTOMATIC_SCHEDULE') {
			return;
		}
		angular.extend(schedulePattern, getDefaultOptionsForAutomaticSchedules());
	}

	function formatRepeatSettingsObject(scheduleObj) {
		scheduleObj.repeat = {
			duration: scheduleObj.repeatDuration,
			hrs: scheduleObj.repeatHrs,
			mins: scheduleObj.repeatMins,
			time: scheduleObj.repeatTime,
			enabled: scheduleObj.repeatEnabled
		};

		if (scheduleObj.repeat.time instanceof Date && !isNaN(scheduleObj.repeat.time.valueOf())) {
			scheduleObj.repeat.time = 86340; //default repeat time: 11:59 PM in seconds
		}
	}

	function formatAutomaticSchedules(subTaskInfo, scheduleObj) {
		const freqType = _.get(scheduleObj, 'freqType', 'DAILY');
		if (freqType !== 'AUTOMATIC_SCHEDULE') {
			return;
		}
		const automaticSchedulePattern = {
			minBackupInterval: scheduleObj.minBackupIntervalHours,
			minBackupIntervalMinutes: scheduleObj.minBackupIntervalMinutes,
			maxBackupInterval: scheduleObj.maxBackupIntervalHours,
			maxBackupIntervalMinutes: scheduleObj.maxBackupIntervalMinutes,
			minSyncInterval: scheduleObj.minSyncIntervalHours,
			minSyncIntervalMinutes: scheduleObj.minSyncIntervalMinutes,
			ignoreOpWindowPastMaxInterval: scheduleObj.ignoreAtMaxInterval,
			wiredNetworkConnection: {
				enabled: scheduleObj.useOnlyWiredWork
			},
			minNetworkBandwidth: {
				enabled: scheduleObj.minNetworkBandwidth,
				threshold: scheduleObj.networkBandwidth
			},
			specfificNetwork: {
				enabled: scheduleObj.useSpecificNetwork,
				ipAddress: {
					address: scheduleObj.specificNetworkIp,
					subnet: scheduleObj.specificNetworkNumber
				}
			},
			stopIfOnBattery: {
				enabled: scheduleObj.stopIfBatteryMode
			},
			acPower: {
				enabled: scheduleObj.startOnlyOnAc
			},
			stopSleepIfBackUp: {
				enabled: scheduleObj.preventSleep
			},
			cpuUtilization: {
				enabled: scheduleObj.cpuBelowThresholdEnabled,
				threshold: scheduleObj.cpuBelowThreshold
			},
			diskSpace: {
				enabled: scheduleObj.spaceBelowThresholdEnabled,
				threshold: scheduleObj.spaceBelowThreshold
			},
			newOrModifiedFile: {
				enabled: scheduleObj.startOnlyfileBackUp
			},
			logFileNum: {
				enabled: scheduleObj.logFileNumEnabled
			},
			logPercent: {
				enabled: scheduleObj.logPercentEnabled,
				threshold: scheduleObj.logPercentEnabled
			},
			diskUsedPercent: {
				enabled: scheduleObj.logPercentEnabled,
				threshold: scheduleObj.logPercentEnabled
			}
		};
		angular.extend(_.get(subTaskInfo, 'options.commonOpts.automaticSchedulePattern'), automaticSchedulePattern);
	}

	return factory;
}

export default app;
