import 'modules/plan/js/directives/exchangePlanDirectives.js';
import 'vsa/js/controllers/collections.ctrl.js';
import 'dlo/js/services/profile.svc.js';
import 'common/js/cv-time-relative-picker.js';

/**
 * A factory for creating new base Office365 Plan instances.
 */
class exchangePlanFactory {
	constructor(
		$q,
		$timeout,
		cvLoc,
		cvUtil,
		profileService,
		name,
		subType,
		exchangeConstants,
		ediscoveryUtility,
		$modal,
		$dialogs
	) {
		this.$q = $q;
		this.cvLoc = cvLoc;
		this.cvUtil = cvUtil;
		this.$modal = $modal;
		this.$timeout = $timeout;
		this.$dialogs = $dialogs;
		this.profileService = profileService;
		this.exchangeConstants = exchangeConstants;
		this.ediscoveryUtility = ediscoveryUtility;
		var _this = this;
		/**
		 * Declare a new instance of a Plan.
		 *
		 * @param {string}
		 *            defaultName - The default name of the plan if the name is not already taken.
		 */
		this.defaultName = name;
		this.planInfo = {
			summary: {
				plan: {
					planName: ''
				},
				type: this.exchangeConstants.plan.TYPE,
				subtype: subType,
				eePolicy: {},
				exchange: {}
			}
		};

		if (subType === this.exchangeConstants.plan.SUBTYPE.EXCHANGEUSER) {
			this.planInfo.exchange = {
				mbArchiving: {
					policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
					detail: {
						emailPolicy: {
							emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.ARCHIVE,
							archivePolicy: {
								primaryMailbox: true,
								archiveMailbox: false,
								disabledMailbox: true,
								backupStubs: true,
								includeMsgsLargerThan: 0,
								includeMsgsOlderThan: 0,
								contentIndexProps: {
									deferredDays: 0,
									enableContentIndex: false,
									enablePreviewGeneration: false,
									previewPathDir: {
										userAccount: {
											password: [],
											confirmPassword: [],
											set stringPassword(value) {
												_this.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.password = angular.isString(
													value
												)
													? _this.string2Bin(value)
													: [];
											},

											get stringPassword() {
												return _this.bin2String(
													_this.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps
														.previewPathDir.userAccount.password
												);
											},

											set stringConfirmPassword(value) {
												_this.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.confirmPassword = angular.isString(
													value
												)
													? _this.string2Bin(value)
													: [];
											},

											get stringConfirmPassword() {
												return _this.bin2String(
													_this.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps
														.previewPathDir.userAccount.confirmPassword
												);
											}
										}
									}
								}
							}
						}
					}
				},
				mbRetention: {
					policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
					detail: {
						emailPolicy: {
							emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.RETENTION,
							retentionPolicy: {
								type: '',
								numOfDaysForMediaPruning: -1,
								numOfDaysForDeletedMediaPruning: -1
							}
						}
					}
				},
				mbCleanup: {
					policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
					detail: {
						emailPolicy: {
							emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.CLEANUP,
							cleanupPolicy: {
								primaryMailbox: true,
								archiveMailbox: false,
								disabledMailbox: true,
								createStubs: true,
								collectMsgsLargerThan: 0,
								collectMsgsDaysAfter: 0,
								numOfDaysForSourcePruning: 730,
								truncateBody: false,
								truncateBodyToBytes: 1024
							}
						}
					}
				}
			};
		}

		if (subType === this.exchangeConstants.plan.SUBTYPE.EXCHANGEJOURNAL) {
			this.planInfo.exchange = {
				mbRetention: {
					policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
					detail: {
						emailPolicy: {
							emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.RETENTION,
							retentionPolicy: {
								numOfDaysForMediaPruning: -1
							}
						}
					}
				},
				mbJournal: {
					policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
					detail: {
						emailPolicy: {
							emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.JOURNAL,
							journalPolicy: {
								contentIndexProps: {
									deferredDays: 0,
									enableContentIndex: false,
									enablePreviewGeneration: false,
									enableExactSearch: false,
									previewPathDir: {
										userAccount: {
											password: [],
											confirmPassword: [],
											set stringPassword(value) {
												_this.planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.previewPathDir.userAccount.password = angular.isString(
													value
												)
													? _this.string2Bin(value)
													: [];
											},

											get stringPassword() {
												return _this.bin2String(
													_this.planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps
														.previewPathDir.userAccount.password
												);
											},

											set stringConfirmPassword(value) {
												_this.planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.previewPathDir.userAccount.confirmPassword = angular.isString(
													value
												)
													? _this.string2Bin(value)
													: [];
											},

											get stringConfirmPassword() {
												return _this.bin2String(
													_this.planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps
														.previewPathDir.userAccount.confirmPassword
												);
											}
										}
									}
								},
								exchMailboxOptions: {
									saveConversationMetaData: false
								}
							}
						}
					}
				}
			};
		}
	}
	/**
	 * Checks to see if the given default name is already taken. If it's not already taken, then it will be
	 * set as the name of the plan.
	 *
	 * @return {HttpPromise}
	 */
	checkDefaultName() {
		return this.profileService.checkPlanName(encodeURIComponent(this.defaultName)).then(resp => {
			if (_.isUndefined(resp.data.id)) {
				this.planInfo.summary.plan.planName = this.defaultName;
				this.autoselect = true;
			} else {
				this.planInfo.summary.plan.planName = '';
			}
		});
	}
	/**
	 * Loads the master template of the plan to populate default values.
	 *
	 * @return {HttpPromise}
	 */
	loadMaster() {
		var deferred = this.$q.defer();
		this.profileService.getMasterProfile(this.planInfo.summary.type, this.planInfo.summary.subtype).then(
			resp => {
				this.planInfo = _.get(resp, 'data.profile', {});
				deferred.resolve(this.planInfo);
			},
			resp => {
				deferred.reject(resp);
			}
		);
		return deferred.promise;
	}

	setPlanInfo(planInfo) {
		this.planInfo = planInfo;
	}

	/**
	 * Converts a given array to a string.
	 *
	 * @param {array}
	 *            array - The array of bytes.
	 * @return {string} The resulting string.
	 */
	bin2String(array) {
		var result = '';
		if (angular.isArray(array)) {
			for (var i = 0; i < array.length; i++) {
				result += String.fromCharCode(parseInt(array[i], 2));
			}
		}
		return result;
	}

	/**
	 * Converts a given string to an array of bytes.
	 *
	 * @param {string}
	 *            str - The string to convert.
	 * @return {array} The resulting array.
	 */
	string2Bin(str) {
		var result = [];
		if (angular.isString(str)) {
			for (var i = 0; i < str.length; i++) {
				result.push(str.charCodeAt(i).toString(2));
			}
		}
		return result;
	}

	/**
	 * Retrieves a single exchange plan from the API and loads it into the current instance.
	 *
	 * @return {HttpPromise}
	 */
	load(id) {
		var deferred = this.$q.defer();
		this.profileService.getProfileDetails(id).then(resp => {
			this.permissions = _.get(resp, 'data.permissions', {});
			this.planInfo = _.get(resp, 'data.profile', {});
			if (this.planInfo.summary.subtype === this.exchangeConstants.plan.SUBTYPE.EXCHANGEUSER) {
				if (
					_.get(this.planInfo, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.type', '') ===
					this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
				) {
					_.set(
						this.planInfo,
						'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
						_.get(this.planInfo, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning', 0)
					);
					_.set(this.planInfo, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning', -1);
				} else if (
					_.get(this.planInfo, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.type', '') ===
					this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
				) {
					_.set(
						this.planInfo,
						'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
						-1
					);
				} else if (
					_.get(
						this.planInfo,
						'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning',
						''
					) <= 0
				) {
					_.set(
						this.planInfo,
						'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
						-1
					);
				}
				_.set(this.planInfo, 'exchange.mbArchiving.detail.emailPolicy.archivePolicy.showArchiveCustomSettings', false);
				if (
					_.get(this.planInfo, 'exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan', 0) > 0 ||
					_.get(this.planInfo, 'exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsLargerThan', 0) > 0 ||
					_.get(
						this.planInfo,
						'exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeOnlyMsgsWithAttachemts',
						false
					) === true
				) {
					_.set(this.planInfo, 'exchange.mbArchiving.detail.emailPolicy.archivePolicy.showArchiveCustomSettings', true);
				}
				_.set(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsDaysAfterValueLabel',
					`${_.get(
						this.planInfo,
						'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsDaysAfter',
						0
					)} ${this.cvLoc('days')}`
				);
				_.set(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsLargerThanValueLabel',
					`${_.get(
						this.planInfo,
						'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsLargerThan',
						0
					)} ${this.cvLoc('labelKB')}`
				);

				let entityDetails = _.get(this.planInfo, 'eePolicy.detail.eePolicy.extractionPolicy.extractionPolicyList', []);
				_.set(
					this.planInfo,
					'exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
					false
				);
				for (let index = 0; index < entityDetails.length; index++) {
					for (let i = 0; i < entityDetails[index].tags.length; i++) {
						_.set(
							this.planInfo,
							'exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
							true
						);
						break;
					}
					if (
						_.get(
							this.planInfo,
							'exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
							false
						) === true
					) {
						break;
					}
				}

				_.set(
					this.planInfo,
					'exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeFolderFilter.folderPatternsSelected',
					_.get(
						this.planInfo,
						'exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeFolderFilter.folderPatternsSelected',
						[]
					)
				);
				_.set(
					this.planInfo,
					'exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeFolderFilter.folderPatternsSelected',
					_.get(
						this.planInfo,
						'exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeFolderFilter.folderPatternsSelected',
						[]
					)
				);
				_.set(
					this.planInfo,
					'exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeMessageClassFilter.folderPatternsSelected',
					_.get(
						this.planInfo,
						'exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeMessageClassFilter.folderPatternsSelected',
						[]
					)
				);

				_.set(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.includeFolderFilter.folderPatternsSelected',
					_.get(
						this.planInfo,
						'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.includeFolderFilter.folderPatternsSelected',
						[]
					)
				);
				_.set(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.excludeFolderFilter.folderPatternsSelected',
					_.get(
						this.planInfo,
						'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.excludeFolderFilter.folderPatternsSelected',
						[]
					)
				);
				_.set(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.excludeMessageClassFilter.folderPatternsSelected',
					_.get(
						this.planInfo,
						'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.excludeMessageClassFilter.folderPatternsSelected',
						[]
					)
				);
			} else {
				let entityDetails = _.get(this.planInfo, 'eePolicy.detail.eePolicy.extractionPolicy.extractionPolicyList', []);
				_.set(
					this.planInfo,
					'exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.enableEntitySearch',
					false
				);
				for (let index = 0; index < entityDetails.length; index++) {
					for (let i = 0; i < entityDetails[index].tags.length; i++) {
						_.set(
							this.planInfo,
							'exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.enableEntitySearch',
							true
						);
						break;
					}
					if (
						_.get(
							this.planInfo,
							'exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.enableEntitySearch',
							false
						) === true
					) {
						break;
					}
				}
			}
			this.originalPlanInfo = _.cloneDeep(this.planInfo);
			deferred.resolve(resp);
		});
		return deferred.promise;
	}

	preProcessData() {
		var sendData = _.cloneDeep(this.planInfo);
		sendData.errorProcessing = false;
		if (
			_.get(sendData, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning', 0) === 0 ||
			_.get(sendData, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning', 0) ===
				0
		) {
			sendData.errorProcessing = true;
			return sendData;
		}
		sendData.eePolicy = this.ediscoveryUtility.getEntityObjectForPlanCreation(this.planInfo.eePolicy);

		var stringPassword =
			sendData.summary.subtype === this.exchangeConstants.plan.SUBTYPE.EXCHANGEJOURNAL
				? _.get(
						sendData,
						'exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.previewPathDir.userAccount.stringPassword',
						''
				  )
				: _.get(
						sendData,
						'exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.stringPassword',
						''
				  );
		var stringConfirmPassword =
			sendData.summary.subtype === this.exchangeConstants.plan.SUBTYPE.EXCHANGEJOURNAL
				? _.get(
						sendData,
						'exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.previewPathDir.userAccount.stringConfirmPassword',
						''
				  )
				: _.get(
						sendData,
						'exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.stringConfirmPassword',
						''
				  );

		if (angular.isString(stringPassword) && stringPassword.length > 0) {
			if (sendData.summary.subtype === this.exchangeConstants.plan.SUBTYPE.EXCHANGEJOURNAL) {
				sendData.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.previewPathDir.userAccount.password = this.cvUtil.getBytes(
					stringPassword
				);
			} else {
				sendData.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.password = this.cvUtil.getBytes(
					stringPassword
				);
			}
		}

		if (angular.isString(stringConfirmPassword) && stringConfirmPassword.length > 0) {
			if (sendData.summary.subtype === this.exchangeConstants.plan.SUBTYPE.EXCHANGEJOURNAL) {
				sendData.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.previewPathDir.userAccount.confirmPassword = this.cvUtil.getBytes(
					stringConfirmPassword
				);
			} else {
				sendData.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.confirmPassword = this.cvUtil.getBytes(
					stringConfirmPassword
				);
			}
		}
		if (sendData.summary.subtype === this.exchangeConstants.plan.SUBTYPE.EXCHANGEUSER) {
			if (
				_.get(
					sendData,
					'exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
					false
				) === false
			) {
				_.set(sendData, 'eePolicy.detail.eePolicy.extractionPolicy.extractionPolicyList', []);
			}
			if (
				_.get(sendData, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning', -1) > -1
			) {
				_.set(
					this.planInfo,
					'exchange.mbRetention.detail.emailPolicy.retentionPolicy.type',
					this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
				);
				_.set(
					sendData,
					'exchange.mbRetention.detail.emailPolicy.retentionPolicy.type',
					this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
				);
				delete sendData.exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning;
			}
			if (
				_.get(sendData, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning', -1) >
				-1
			) {
				_.set(
					sendData,
					'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning',
					_.get(sendData, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning', 0)
				);
				_.set(
					sendData,
					'exchange.mbRetention.detail.emailPolicy.retentionPolicy.type',
					this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
				);
				_.set(
					this.planInfo,
					'exchange.mbRetention.detail.emailPolicy.retentionPolicy.type',
					this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
				);
				delete sendData.exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning;
			}
			if (
				_.get(sendData, 'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning', -1) ===
					-1 &&
				_.get(
					sendData,
					'exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
					-1
				) === -1
			) {
				delete sendData.exchange.mbRetention.detail.emailPolicy.retentionPolicy.type;
				delete sendData.exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning;
			}
			_.set(
				this.planInfo,
				'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsDaysAfterValueLabel',
				`${_.get(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsDaysAfter',
					0
				)} ${this.cvLoc('days')}`
			);
			_.set(
				this.planInfo,
				'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsLargerThanValueLabel',
				`${_.get(
					this.planInfo,
					'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsLargerThan',
					0
				)} ${this.cvLoc('labelKB')}`
			);
		} else {
			if (
				_.get(
					sendData,
					'exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps.enableEntitySearch',
					false
				) === false
			) {
				_.set(sendData, 'eePolicy.detail.eePolicy.extractionPolicy.extractionPolicyList', []);
			}
			delete sendData.exchange.mbRetention.detail.emailPolicy.retentionPolicy.type;
		}
		return sendData;
	}

	/**
	 * Saves the current plan instance in the back-end.
	 *
	 * @return {Promise} Since this method returns a regular Promise, be sure to use the "then" method when
	 *         waiting for the function to complete, rather than the "success" or "error" methods. The reason
	 *         a regular promise is returned is because an HTTP call WILL NOT be called if no data for the
	 *         plan was changed.
	 */
	save() {
		var deferred = this.$q.defer();
		this.error = '';
		if (!angular.equals(this.planInfo, this.originalPlanInfo)) {
			if (
				this.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan !==
				this.originalPlanInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan
			) {
				const callBackFunctions = {
					noFunction: () => {
						this.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan = this.originalPlanInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan;
						return;
					},
					yesFunction: () => {
						this.saveData(deferred);
					}
				};
				this.$dialogs.confirm(
					this.cvLoc('header.changedArchiveRules'),
					this.cvLoc('label.archiveRuleChangeWarningMessage'),
					callBackFunctions
				);
			} else {
				this.saveData(deferred);
			}
		} else {
			deferred.resolve();
		}
		return deferred.promise;
	}

	saveData(deferred) {
		const sendData = this.preProcessData();
		if (sendData.errorProcessing) {
			return deferred.reject(sendData.errorProcessing);
		}
		delete sendData.errorProcessing;
		var UpdatePlanEEPolicyInfo = {
			eePolicy: sendData.eePolicy
		};
		this.profileService
			.editExchangePlan(this.planInfo.summary.plan.planId, sendData.exchange, UpdatePlanEEPolicyInfo)
			.then(
				resp => {
					this.originalPlanInfo = _.cloneDeep(this.planInfo);
					deferred.resolve();
				},
				resp => {
					deferred.reject(resp.data);
				}
			);
	}

	/**
	 * Opens the modal used for editing the archive mailbox types of the current User Mailbox Plan.
	 *
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditArchiveSettingsModal() {
		const showData = {
			accordianTitle: 'label.archiveMessages',
			labelCustomArchiveSettings: 'label.customArchiveSettings'
		};
		return this.openEditModal(
			this.cvLoc('label.editBackupSettings'),
			this.exchangeConstants.pathTemplate.ARCHIVE,
			showData
		);
	}
	/**
	 * Open the modal for editing the Messages section of the cleanup policy for the current plan.
	 *
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditCleanupMessagesModal(data, cleanupPropName) {
		this.planInfo.exchange.mbCleanup.detail.emailPolicy.cleanupPolicy[cleanupPropName] = data;
		if (
			_.get(this.planInfo, 'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.enableMessageRules', false) === true
		) {
			return this.openEditModal(this.cvLoc('label.editCleanupSettings'), this.exchangeConstants.pathTemplate.CLEAUP)
				.result.then(resp => {
					this.originalPlanInfo = _.cloneDeep(this.planInfo);
				})
				.catch(resp => {
					_.set(
						this.planInfo,
						'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.enableMessageRules',
						_.get(
							this.originalPlanInfo,
							'exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.enableMessageRules',
							false
						)
					);
				});
		} else {
			this.save();
		}
	}

	openEditCleanupQuotaModal() {
		return this.openEditModal(this.cvLoc('label.cleanupQuota'), this.exchangeConstants.pathTemplate.CLEANUP_QUOTA);
	}

	openEditGeneralSettingsModal() {
		return this.openEditModal(this.cvLoc('label.generalSettings'), this.exchangeConstants.pathTemplate.GENERAL);
	}

	openEditExchJournalSearchSettingsModal(showContent, showExactSearch) {
		const showData = {};
		showData.showExactSearch = showExactSearch;
		if (_.isUndefined(showContent)) {
			showData.showAllSearchEntity = true;
			showData.modalWindowType = '';
		} else {
			showData[showContent] = true;
			showData.modalWindowType = 'small-size';
		}
		return this.openEditModal(
			this.cvLoc('label.editSearchSettings'),
			this.exchangeConstants.pathTemplate.SEARCH_SETTINGS,
			showData
		)
			.result.then(resp => {
				this.originalPlanInfo = _.cloneDeep(this.planInfo);
			})
			.catch(resp => {
				this.planInfo = _.cloneDeep(this.originalPlanInfo);
			});
	}

	/**
	 * Open the modal for editing the retention rules of the plan.
	 *
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditRetentionRulesModal(isArchiveMailbox) {
		const showData = {
			isArchiveMailbox: isArchiveMailbox,
			isOffice365: false,
			modalWindowType: 'small-size'
		};
		return this.openEditModal(
			this.cvLoc('label.editRetentionSettings'),
			this.exchangeConstants.pathTemplate.RETENTION,
			showData
		);
	}

	openEditExchUserSearchSettingsModal(showContent, showExactSearch) {
		const showData = {};
		showData.showExactSearch = showExactSearch;
		if (_.isUndefined(showContent)) {
			showData.showAllSearchEntity = true;
			showData.modalWindowType = '';
		} else {
			showData[showContent] = true;
			showData.modalWindowType = 'small-size';
		}
		return this.openEditModal(
			this.cvLoc('label.editSearchSettings'),
			this.exchangeConstants.pathTemplate.SEARCH_SETTINGS,
			showData
		)
			.result.then(resp => {
				this.originalPlanInfo = _.cloneDeep(this.planInfo);
			})
			.catch(resp => {
				this.planInfo = _.cloneDeep(this.originalPlanInfo);
			});
	}

	/**
	 * Opens a modal to edit content for an exchange plan.
	 *
	 * @param {string}
	 *            modalTitle - The title of the modal that will be displayed.
	 * @param {string}
	 *            templateUrl - The templateUrl to load within the body of the modal.
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditModal(modalTitle, templateUrl, showData) {
		return this.$modal.open({
			templateUrl: 'modules/plan/partials/exchange/editModal.jsp',
			backdrop: 'static',
			windowClass: _.get(showData, 'modalWindowType', ''),
			controllerAs: 'exchangeEditModalController',
			controller: [
				'$uibModalInstance',
				'plan',
				function($uibModalInstance, plan) {
					this.inModal = true;
					this.showData = showData || {};
					this.isArchiveMailbox = _.get(showData, 'isArchiveMailbox', false);
					this.isOffice365 = _.get(showData, 'isOffice365', false);
					this.plan = _.cloneDeep(plan);
					this.modalTitle = modalTitle;
					this.templateUrl = templateUrl;
					this.exchangeDetails = _.get(this.plan.planInfo, 'exchange', {});

					function isCompleteJobOptionsSaved(jobOptions) {
						return _.get(jobOptions, 'jobMessagesProtected', 0) > 0 || _.get(jobOptions, 'jobHoursRun', 0) > 0;
					}

					this.exchCompleteJobOptions = isCompleteJobOptionsSaved(
						_.get(
							this.plan,
							'originalPlanInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.exchMailboxOptions',
							{}
						)
					);
					this.contentStoreCompleteJobOptions = isCompleteJobOptionsSaved(
						_.get(
							this.plan,
							'originalPlanInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentStoreOptions',
							{}
						)
					);

					this.toggleJournalOptions = function(completeJobOptions) {
						if (!completeJobOptions) {
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.exchMailboxOptions.jobMessagesProtected',
								null
							);
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.exchMailboxOptions.jobHoursRun',
								null
							);
						} else {
							this.exchCompleteJobOptions = true;
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.exchMailboxOptions.jobMessagesProtected',
								10000
							);
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.exchMailboxOptions.jobHoursRun',
								5
							);
						}
					};

					this.toggleContentStoreOptions = function(completeJobOptions) {
						if (!completeJobOptions) {
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentStoreOptions.jobMessagesProtected',
								null
							);
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentStoreOptions.jobHoursRun',
								null
							);
						} else {
							this.contentStoreCompleteJobOptions = true;
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentStoreOptions.jobMessagesProtected',
								10000
							);
							_.set(
								this.plan,
								'planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentStoreOptions.jobHoursRun',
								5
							);
						}
					};

					this.toggleCustomArchiveSettings = function(data) {
						if (!data) {
							_.set(
								this.plan,
								'planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan',
								0
							);
							_.set(
								this.plan,
								'planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsLargerThan',
								0
							);
							_.set(
								this.plan,
								'planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeOnlyMsgsWithAttachemts',
								false
							);
						}
					};

					this.addFilterValues = function(name, value) {
						this[value] = this[value].trim();
						let dataArray = _.get(this, name[0], {});
						if (this[value].length > 0) {
							if (dataArray.folderPatternsSelected.length === 1 && dataArray.folderPatternsSelected[0] === '*') {
								dataArray.folderPatternsSelected = [];
							}
							dataArray.folderPatternsSelected.push(this[value]);
						}
						let pos = dataArray.folderPatternsAvailable.indexOf(this[value]);
						if (pos >= 0) {
							dataArray.folderPatternsAvailable.splice(pos, 1);
						}
						if (name.length > 1) {
							let updateArray = _.get(this, name[1], {});
							pos = updateArray.folderPatternsAvailable.indexOf(this[value]);
							if (pos >= 0) {
								updateArray.folderPatternsAvailable.splice(pos, 1);
								_.set(this, name[1], updateArray);
							}
						}
						_.set(this, name[0], dataArray);
						this[value] = '';
					};

					this.removeFilterValues = function(name, value, deleteAll) {
						let dataArray = _.get(this, name[0], {});
						if (deleteAll === false) {
							dataArray.folderPatternsSelected.splice(dataArray.folderPatternsSelected.indexOf(value), 1);
							dataArray.folderPatternsAvailable.push(value);
							if (name.length > 1) {
								let updateArray = _.get(this, name[1], {});
								updateArray.folderPatternsAvailable.push(value);
							}
						} else {
							_.set(
								dataArray,
								'folderPatternsAvailable',
								dataArray.folderPatternsAvailable.concat(dataArray.folderPatternsSelected)
							);
							if (name.length > 1) {
								let updateArray = _.get(this, name[1], {});
								//updateArray.folderPatternsAvailable.concat(dataArray.folderPatternsSelected);

								_.set(
									updateArray,
									'folderPatternsAvailable',
									updateArray.folderPatternsAvailable.concat(dataArray.folderPatternsSelected)
								);
								_.set(this, name[1], updateArray);
							}
						}

						if (deleteAll === true || dataArray.folderPatternsSelected.length === 0) {
							dataArray.folderPatternsSelected = [];
						}
						_.set(this, name[0], dataArray);
					};

					/**
					 * When the cancel button is pressed.
					 */
					this.cancel = function() {
						$uibModalInstance.dismiss();
					};

					/**
					 * When the save button is pressed.
					 */
					this.save = function() {
						_.set(this.plan.planInfo, 'exchange', this.exchangeDetails);

						//only when a user clicks the save button, the changed planinfo moves to original plan object
						plan.planInfo = _.cloneDeep(this.plan.planInfo);
						plan.save().then(() => {
							$uibModalInstance.close(this.plan);
						});
					};
				}
			],
			resolve: {
				plan: this
			}
		});
	}
}
export { exchangePlanFactory };
