import 'vsa/js/controllers/collections.ctrl.js';
import 'modules/plan/js/directives/exchCleanupQuotaDirective.js';
import 'modules/plan/js/directives/exchJournalJobOptions.js';
import 'modules/plan/js/directives/exchangePlanDirectives.js';
import 'setup/js/directives/setup-steps.js';
import 'adminConsole/js/controllers/security.ctrl.js';
import { exchangePlanFactory } from 'modules/plan/js/factory/exchangePlan.factory.js';
import { exchangeConstants } from 'exchangeV2/js/constants/exchangeConstants.js';
import { acAppExchangeModule } from 'common/js/modules';
import 'ediscovery/common/js/services/ediscovery.util.svc.js';

/**
 * A controller for showing the details of a user mailbox plan and journal mailbox plan.
 */
class exchangeMailboxPlanDetailsController {
	constructor(
		$q,
		$state,
		$modal,
		$dialogs,
		$timeout,
		$stateParams,
		cvBreadcrumbsTabsFactory,
		cvLoc,
		cvUtil,
		cvToaster,
		profileService,
		securityFactory,
		ediscoveryUtility,
		tabService,
		profileUIFactory
	) {
		this.cvBreadcrumbsTabsFactory = cvBreadcrumbsTabsFactory;
		this.exchangeConstants = exchangeConstants;
		this.securityFactory = securityFactory;
		this.profileService = profileService;
		this.cvToaster = cvToaster;
		this.$dialogs = $dialogs;
		this.$timeout = $timeout;
		this.$state = $state;
		this.cvLoc = cvLoc;
		this.tabService = tabService;
		this.profileUIFactory = profileUIFactory;
		this.isArchiveMailbox = true;
		this.type = this.exchangeConstants.plan.SUBTYPE.EXCHANGEUSER;
		if (this.$state.current.name === this.exchangeConstants.navigationStateNames.PLAN.JOURNALMAILBOXPLANDETAILS) {
			this.type = this.exchangeConstants.plan.SUBTYPE.EXCHANGEJOURNAL;
			this.isArchiveMailbox = false;
		}
		this.isTenantAdmin = cv.isTenantAdmin;
		this.plan = new exchangePlanFactory(
			$q,
			$timeout,
			cvLoc,
			cvUtil,
			profileService,
			'',
			this.type,
			exchangeConstants,
			ediscoveryUtility,
			$modal,
			$dialogs
		);
		this.genericEntity = {
			entityId: $stateParams.planId
		};
		this.entityType = this.exchangeConstants.entityTypes.PLAN_ENTITY;
		this.mailboxTypes = [
			{
				displayName: this.cvLoc('label.primaryMailbox'),
				key: 'primaryMailbox'
			},
			{
				displayName: this.cvLoc('label.archiveMailbox'),
				key: 'archiveMailbox'
			},
			{
				displayName: this.cvLoc('label.disabledMailbox'),
				key: 'disabledMailbox'
			}
		];

		this.cvBreadcrumbsTabsFactory.addBreadCrumbs([
			{
				title: this.cvLoc('label.nav.profile'),
				link: '#profile'
			}
		]);
		this.showExactSearch = _.get(cv, 'additionalSettings.Console.showExactSearch', false);
		this.getInitData($stateParams.planId);
	}

	getInitData(planId) {
		this.plan.load(planId).then(
			response => {
				this.formattedSecurityAssociations = this.securityFactory.formatSecurityAssociationsObject(
					_.get(response, 'data.profile.securityAssociations', {})
				);
				this.planLoaded = true;
				this.associationsLoaded = true;
				this.canEdit = _.get(this.plan, 'permissions.Plan_Edit', false);
				this.canDelete = _.get(this.plan, 'permissions.Plan_Delete', false);
				this.changeNameObj = {
					id: this.plan.originalPlanInfo.summary.plan.planId,
					name: this.plan.originalPlanInfo.summary.plan.planName
				};
				this.initTabs(_.get(response, 'data.profile.summary'));
			},
			() => {
				this.$state.go('profile');
				this.cvToaster.showErrorMessage({
					message: this.cvLoc('error.planNotFound')
				});
			}
		);
	}

	initTabs(planSummary) {
		const tabs = this.profileUIFactory.getProfileDetailsTabs(planSummary, false);
		if (tabs.length > 1) {
			this.hasTabs = true;
			this.tabService.init({ tabs: tabs, hasParent: true });
		}
	}

	/**
	 * Deletes the current plan being viewed, and redirects the user back to the list of existing plans.
	 */
	deletePlan() {
		this.$dialogs.confirm(
			this.cvLoc('label.confirmDeleteProfile'),
			this.cvLoc('label.confirmMessageProfile', '<strong>' + this.plan.planInfo.summary.plan.planName + '</strong>'),
			{
				noFunction: () => {},
				yesFunction: () => {
					this.profileService.deleteProfile(this.plan.planInfo.summary.plan.planId).then(
						() => {
							this.$state.go('profile');
							this.cvToaster.showSuccessMessage({
								message: this.cvLoc('label.deleteSuccess')
							});
						},
						resp => {
							this.cvToaster.showErrorMessage({
								ttl: '10000',
								message: _.get(resp, 'data', this.cvLoc('generic_error'))
							});
						}
					);
				}
			}
		);
	}

	openEditSearchSettingsModal(data, showContent) {
		if (this.type === this.exchangeConstants.plan.SUBTYPE.EXCHANGEUSER) {
			this.plan.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps[showContent] = data;
			if (
				(data === true &&
					(showContent === this.exchangeConstants.plan.SEARCHSETTINGS.ENABLEPREVIEWGENERATION ||
						showContent === this.exchangeConstants.plan.SEARCHSETTINGS.ENABLEENTITYSEARCH)) ||
				_.isUndefined(showContent)
			) {
				this.plan.openEditExchUserSearchSettingsModal(showContent, this.showExactSearch);
			} else {
				this.plan.save();
			}
		} else {
			this.plan.planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.contentIndexProps[showContent] = data;
			if (
				(data === true &&
					(showContent === this.exchangeConstants.plan.SEARCHSETTINGS.ENABLEPREVIEWGENERATION ||
						showContent === this.exchangeConstants.plan.SEARCHSETTINGS.ENABLEENTITYSEARCH)) ||
				_.isUndefined(showContent)
			) {
				this.plan.openEditExchJournalSearchSettingsModal(showContent, this.showExactSearch);
			} else {
				this.plan.save();
			}
		}
	}
	saveArchiveData(data, archiveDataName) {
		this.plan.planInfo.exchange.mbArchiving.detail.emailPolicy.archivePolicy[archiveDataName] = data;
		this.plan.save();
	}

	journalGeneralPropSave(data) {
		this.plan.planInfo.exchange.mbJournal.detail.emailPolicy.journalPolicy.deleteArchivedMessages = data;
		this.plan.save();
	}

	saveInlineEdit() {
		_.set(
			this.plan,
			'planInfo.exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsDaysAfter',
			_.parseInt(
				_.get(this.plan, 'planInfo.exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsDaysAfter', 0)
			)
		);
		_.set(
			this.plan,
			'planInfo.exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsLargerThan',
			_.parseInt(
				_.get(this.plan, 'planInfo.exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.collectMsgsLargerThan', 0)
			)
		);
		this.plan.save();
	}
}

exchangeMailboxPlanDetailsController.$inject = [
	'$q',
	'$state',
	'$uibModal',
	'$dialogs',
	'$timeout',
	'$stateParams',
	'cvBreadcrumbsTabsFactory',
	'cvLoc',
	'cvUtil',
	'cvToaster',
	'profileService',
	'securityFactory',
	'ediscoveryUtility',
	'tabService',
	'profileUIFactory'
];
acAppExchangeModule.controller('exchangeMailboxPlanDetailsController', exchangeMailboxPlanDetailsController);
export default acAppExchangeModule;
