import { NAS_CONSTANTS } from 'modules/nas/js/constants';

import { acAppNasModule } from 'common/js/modules';

class AccessNodesDropdownController {
	constructor(
		cvLoc,
		clientGroupService,
		nasService,
		$q,
		cvUtil,
		AppTypesUtil,
		AppTypes,
		clusterTypeConstants,
		nasFactory
	) {
		this.cvLoc = cvLoc;
		this.clientGroupService = clientGroupService;
		this.nasService = nasService;
		this.$q = $q;
		this.cvUtil = cvUtil;
		this.AppTypesUtil = AppTypesUtil;
		this.AppTypes = AppTypes;
		this.clusterTypeConstants = clusterTypeConstants;
		this.nasFactory = nasFactory;

		this.localLang = this.cvUtil.getIStevenLocLabels();
		this.dropdownLoaded = false;
	}

	$onInit() {
		this.initDropdown();
	}

	initDropdown() {
		if (this.accessnodeoptions.type) {
			this.clusterType = this.accessnodeoptions.type;
		} else if (this.accessnodeoptions.appId) {
			this.clusterType = this._getClusterTypeFromApplicationId(this.accessnodeoptions.appId);
		}

		this.applicationId = this.accessnodeoptions.appId;
		this.clientAppType = this.accessnodeoptions.clientAppType;

		this._initMapsForPreselection();

		this._fetchAccessnodes();
	}

	_fetchAccessnodes() {
		let serviceRequests = [
			this.clusterType === NAS_CONSTANTS.NDMP
				? this.nasService.getNasProxyList(0, this.AppTypes.DISTRIBUTED_APPS, this.clusterTypeConstants.WINFS, 2)
				: this.nasService.getProxyNodes(this.clusterType, this.clientAppType)
		];

		if (!this.AppTypesUtil.isClusterTypeFileSystem(this.applicationId, this.clusterType)) {
			serviceRequests.push(this.clientGroupService.getClientGroups());
		}

		this.$q
			.all(serviceRequests)
			.then(
				responses => {
					let proxyNodes = _.get(responses, '[0].data');
					let proxyNodeGroups = _.get(responses, '[1].data');

					if (!_.isArray(proxyNodes) && !_.isUndefined(proxyNodes)) {
						proxyNodes = proxyNodes.clients;
					}

					this.inputModel = [];

					if (proxyNodeGroups) {
						this.inputModel = [
							{
								msGroup: true,
								name: this.cvLoc('label.proxyGroups')
							},
							...proxyNodeGroups.map(clientGroup => ({
								clientGroupId: clientGroup.Id,
								name: clientGroup.name,
								selected: this.accessnodeGroupMap[clientGroup.Id]
							})),
							{
								msGroup: false
							}
						];
					}

					this.inputModel = this.inputModel.concat([
						{
							msGroup: true,
							name: this.cvLoc('label.accessNodes')
						},
						...proxyNodes.map(client => ({
							clientId: client.clientId,
							name: client.clientName,
							selected: this.accessnodeMap[client.clientId]
						})),
						{
							msGroup: false
						}
					]);
				},
				error => {
					// Call this function (if defined) when there's an error loading API
					if (_.isFunction(this.accessnodeoptions.onError)) {
						this.accessnodeoptions.onError(error);
					} else {
						console.error(error);
					}
				}
			)
			.finally(() => {
				this.dropdownLoaded = true;
			});
	}

	onDropdownClose() {
		this.accessnodeoptions.backupDataAccessNodes = this.nasFactory.filterAccessnodeAttributes(this.selectedAccessNodes);

		if (this.selectedAccessNodes && this.selectedAccessNodes.length) {
			this.accessnodeoptions.showerror = false;
		}

		if (!this.accessnodeoptions.accessnodeRequired && this.selectedAccessNodes.length == 0) {
			this.accessnodeoptions.showHelpText = true;
		} else {
			this.accessnodeoptions.showHelpText = false;
		}

		if (_.isFunction(this.accessnodeoptions.successFn)) {
			this.accessnodeoptions.successFn(this.accessnodeoptions.backupDataAccessNodes || []);
		}
	}

	_getClusterTypeFromApplicationId(applicationId) {
		applicationId = parseInt(applicationId);

		switch (applicationId) {
			case this.AppTypes.WINDOWS_FILESYSTEM:
				return NAS_CONSTANTS.CIFS;
			case this.AppTypes.UNIX_FILESYSTEM:
				return NAS_CONSTANTS.NFS;
			case this.AppTypes.DISTRIBUTED_APPS:
				return NAS_CONSTANTS.GPFS;
			default:
				return NAS_CONSTANTS.NDMP;
		}
	}

	_initMapsForPreselection() {
		this.accessnodeMap = {};
		this.accessnodeGroupMap = {};

		if (this.accessnodeoptions.backupDataAccessNodes) {
			this.selectedAccessNodes = this.accessnodeoptions.backupDataAccessNodes;
			_.forEach(this.accessnodeoptions.backupDataAccessNodes, accessnode => {
				if (accessnode.clientGroupId) {
					this.accessnodeGroupMap[accessnode.clientGroupId] = true;
				}

				if (accessnode.clientId) {
					this.accessnodeMap[accessnode.clientId] = true;
				}
			});
		}
	}
}

AccessNodesDropdownController.$inject = [
	'cvLoc',
	'clientGroupService',
	'nasService',
	'$q',
	'cvUtil',
	'AppTypesUtil',
	'AppTypes',
	'clusterType',
	'nasFactory'
];

acAppNasModule.directive('cvAccessnodeDropdown', () => ({
	restrict: 'E',
	controller: AccessNodesDropdownController,
	controllerAs: 'accessnodesCtrl',
	bindToController: true,
	replace: true,
	scope: {
		accessnodeoptions: '=',
		directiveId: '@'
	},
	template: `
        <div>
            <isteven-multi-select ng-if="accessnodesCtrl.dropdownLoaded"
                directive-id="{{ accessnodesCtrl.directiveId }}"
                input-model="accessnodesCtrl.inputModel"
                output-model="accessnodesCtrl.selectedAccessNodes"
                translation="accessnodesCtrl.localLang"
                on-close="accessnodesCtrl.onDropdownClose()"
                selection-mode="multiple"
                tick-property="selected"
                item-label="name"
                button-label="name"
                group-property="msGroup"
                class="isteven-multi-select user-group-select"
            />
            <span ng-show="accessnodesCtrl.accessnodeoptions.showerror" class="cv-help-block" ng-bind="accessnodesCtrl.accessnodeoptions.customErrorMessage || ('error.proxy' | cvLoc)"></span>
            <div class="text-center" ng-show="!accessnodesCtrl.dropdownLoaded"><img src="img/loading.gif"></div>
        </div>
    `
}));

export default acAppNasModule;
