import 'modules/nas/js/services/nas.svc.js';
import 'modules/snapArrays/js/services/snapArray.svc.js';
import 'modules/nas/js/factory/nas.factory.js';
import 'modules/credentialManager/js/directives/cv-credential-manager-directive.js';
import 'modules/credentialManager/js/constants/credentialManager.constant.js';
import { NAS_CONSTANTS } from 'modules/nas/js/constants';
import { acAppClientModule, acAppcredentialManagerModule } from 'common/js/modules';

/**
 * Generic code to create the NAS (different vendors like ISILON | NET APP) and File Server (NUTANIX | QUMULO)
 * clients.
 */

export class nasFileServerClientController {
	constructor(
		clientParams,
		$modal,
		snapArrayService,
		nasService,
		cvLoc,
		credentialManagerService,
		$modalInstance,
		nasFactory,
		cvToaster,
		$state,
		cvUtil,
		AppTypesUtil,
		CREDENTIAL_TYPE
	) {
		this.$modal = $modal;
		this.$modalInstance = $modalInstance;
		this.nasFactory = nasFactory;
		this.nasService = nasService;
		this.credentialManagerService = credentialManagerService;
		this.snapArrayService = snapArrayService;
		this.CREDENTIAL_TYPE = CREDENTIAL_TYPE;
		this.cvToaster = cvToaster;
		this.cvUtil = cvUtil;
		this.clientParams = clientParams;
		this.AppTypesUtil = AppTypesUtil;
		this.cvLoc = cvLoc;
		this.$state = $state;
		this.selectedArray = []; // array to hold selected array
		this.selectedPlan = []; // array to hold selected plan
		this.planList = [];
		this.arrayList = [];
		this.savedCredentials = [];
		this.selectedCIFSUser = ''; // cifs username to be displayed
		this.vendorList = [];
		this.editMode = this.clientParams.editMode;
		this.serverType = this.clientParams.serverType;
		this.selectedVendor = []; // vendor for Nas servers
		this.isNasServer = this.clientParams.isNasServer;
		let isNasServer = this.isNasServer;
		this.errorMessage = {};
		this.arrayRequest = null;
		this.showFileServerHost = true;
		this.hostNamPlaceholderText = '';
		//clientAppType to be used for access nodes selection. Should not be sent for NAS.
		let clientAppType = !this.isNasServer ? this.clientParams.clientInfo.clientAppType : undefined;
		this.model = {
			selectedCredential: [],
			createAccountType: ['STORAGE_ARRAY_ACCOUNT', 'WINDOWSACCOUNT', 'LINUXACCOUNT'],
			sourceLabel: cvLoc('label.qumuloCrdentials'),
			sourceLabelHelpText: cvLoc('label.qumuloAccountHelpText'),
			accountType: [
				this.CREDENTIAL_TYPE.STORAGE_ARRAY_ACCOUNT,
				this.CREDENTIAL_TYPE.WINDOWSACCOUNT,
				this.CREDENTIAL_TYPE.LINUXACCOUNT
			]
		};
		this.qumuloAttributes = ['userName', 'password'];

		this.selectedCredentials = [
			{
				credentialId: 0,
				credentialName: cvLoc('label.enterCredential'),
				selected: true
			}
		];
		this.arrayListLabel = {
			nothingSelected: this.isNasServer ? cvLoc('label.selectArray') : cvLoc('label.pleaseSelectHost')
		};
		this.planListLabel = {
			nothingSelected: cvLoc('label.selectPlan')
		};
		this.vendorListLabel = {
			nothingSelected: cvLoc('label.vendor.selectVendorType')
		};

		this.fileServerLabel = cvLoc('label.fileServerHost');

		if (this.isNasServer) {
			this.fileServerLabel = cvLoc('label.manageArray');
		}

		if (this.clientParams.serverType === NAS_CONSTANTS.QUMULO) {
			this.hostNamPlaceholderText = cvLoc('label.errorFQDN');
		}

		let entity = this.clientParams.entity || { clientName: '' };

		this.status = {
			networkShareCollapsed: true,
			ndmpCollapsed: true,
			migrationCollapsed: true
		};

		// Config object for filer migration
		this.filerMigration = {
			isFeatureAvailable: cvConfig.enableFilerMigration,
			isEnabled: false,
			showConfigPreview: false,
			migrationConfig: {},
			configuredProtocol: undefined
		};

		// the request object to be used for saving the client
		this.createPseudoClientRequest = {
			entity: entity,
			clientInfo: this.clientParams.clientInfo
		};

		function initBackupConfig(appId) {
			let self = this;
			let backupConfig = {
				idaInfo: {
					idaEntity: {
						applicationId: appId
					},
					backupConfiguration: {
						backupDataAccessNodes: []
					}
				},
				subClient: {
					subClientEntity: {
						applicationId: appId
					},
					commonProperties: {},
					content: []
				}
			};

			if (appId === 33) {
				backupConfig.subClient.impersonateUser = {
					userName: '',
					password: ''
				};

				backupConfig.subClient.commonProperties = {
					impersonateUserCredentialinfo: {
						credentialId: 0,
						credentialName: ''
					}
				};
			}

			if (appId === 29) {
				backupConfig.subClient.fsSubClientProp = { enableNetworkShareAutoMount: true };
			}

			return backupConfig;
		}

		function updateNasSubClientProperties(appId, backupConfig) {
			if (appId === 13) {
				backupConfig.subClient.commonProperties.numberOfBackupStreams = 4;
				// Must NOT set allowMultipleDataReaders - that is defaulted in the backend based on vendor and model
			} else {
				// The backend AppIdInterfaceDB.cpp first properly defaults numberOfBackupStreams to 0, the incorrectly defaults it to 4
				// For now keep passing 0 here since the changes are least regressive.
				// 0 means the agent figures out the best number of streams - so it is important for performance.
				backupConfig.subClient.commonProperties.numberOfBackupStreams = 0;
				backupConfig.subClient.commonProperties.allowMultipleDataReaders = true;
			}
		}

		function initBackupType(protocol) {
			let backupConfig = initBackupConfig(protocol.appId);

			if (isNasServer) {
				updateNasSubClientProperties(protocol.appId, backupConfig);
			}
			let key = protocol.label;
			let osType = protocol.osType;
			return {
				key: key,
				enabled: false,
				disabled: protocol.disabled || false,
				hidden: protocol.hidden,
				osType: protocol.osType,
				protocolLabel: protocol.contentLabel,
				showErrors: false,
				appId: protocol.appId,
				backupConfig: backupConfig,
				contents: {
					path: {
						label: cvLoc('label.backupContent'),
						paths: [protocol.contentLabel]
					}
				},
				accessNodeOptions: {
					type: key,
					backupDataAccessNodes: [],
					isNasOnlySolution: true,
					showHelpText: true,
					customErrorMessage: cvLoc('label.noAccessNodes'),
					clientAppType: clientAppType,
					successFn: function(backupDataAccessNodes) {
						backupConfig.idaInfo.backupConfiguration.backupDataAccessNodes = backupDataAccessNodes;
					}
				}
			};
		}

		// initialize the backupconfig based on the client protocols
		this.backupTypes = this.clientParams.protocols.map(protocol => initBackupType(protocol));

		this.loadPlanList();
		// for nutanix
		if (!this.editMode && this.clientParams.serverType === NAS_CONSTANTS.NUTANIX) {
			this.selectedVendor.push(
				_.find(this.nasFactory.getVendorList(), { clientAppTypeName: NAS_CONSTANTS.NUTANIX_AFS })
			);
			this.loadArrayList(NAS_CONSTANTS.NUTANIX_LABEL);
			this.createPseudoClientRequest.clientInfo.fileServerInfo = this.selectedArray[0];
		}
		// for Qumulo
		if (!this.editMode && this.clientParams.serverType === NAS_CONSTANTS.QUMULO) {
			this.loadCredentails();
			//this.createPseudoClientRequest.clientInfo.fileServerInfo = this.selectedArray[0];
		}

		let ndmpProtoCol = _.find(this.backupTypes, { key: NAS_CONSTANTS.NDMP });
		if (ndmpProtoCol) {
			this.initNasClientProperties();
		}

		if (this.editMode) {
			for (var i in this.backupTypes) {
				let backupType = this.backupTypes[i];
				if (backupType.enabled) {
					this.updateDefaultPath(backupType);
				}

				if (this.isNasServer && _.get(this.clientParams, 'clientInfo.fileServerInfo.arrayName')) {
					this.showFileServerHost = false;
				}
			}
		}
	}

	initNasClientProperties() {
		this.vendorList = this.nasFactory.getVendorList().filter(vendor => vendor.serverType === NAS_CONSTANTS.NAS);
		let genericVendor = this.vendorList[0]; // get generic vendor and set it selected
		genericVendor.selected = true;
		this.selectedVendor = [genericVendor]; // init with generic vendor
		this.arrayName = this.cvLoc('label.noArrayAdded');
		this.arrayActionLabel = this.cvLoc('Add');
		this.createPseudoClientRequest.clientInfo.nasClientProperties = {
			listenPort: 10000,
			vendorType: genericVendor.vendorType,
			ndmpServerDetails: {
				ndmpServerClientName: '',
				ndmpServerHostName: '',
				ndmpCredentials: {
					userName: '',
					password: ''
				}
			}
		};
	}

	updateVendor(arrayName) {
		this.createPseudoClientRequest.clientInfo.nasClientProperties.vendorType = this.selectedVendor[0].vendorType;
		this.updateFileServerHost({ arrayId: 0, arrayName: arrayName });
	}

	impersonateCIFSUser(backupConfig) {
		var self = this;
		this.$modal
			.open({
				templateUrl: appUtil.appRoot + 'adminConsole/partials/userCredentials.jsp',
				controller: 'manageUserCredentialsController',
				resolve: {
					controllerParams: {
						impersonateUser: backupConfig.subClient.impersonateUser,
						impersonateUserCredentialinfo: backupConfig.subClient.commonProperties.impersonateUserCredentialinfo,
						recordTypeFilter: NAS_CONSTANTS.WINDOWSACCOUNT
					}
				}
			})
			.result.then(function() {
				let backupMethod = _.find(self.backupTypes, { key: NAS_CONSTANTS.CIFS });
				if (backupMethod.backupConfig.subClient.commonProperties.impersonateUserCredentialinfo.credentialName) {
					self.selectedCIFSUser =
						backupMethod.backupConfig.subClient.commonProperties.impersonateUserCredentialinfo.credentialName;
				} else {
					self.selectedCIFSUser = backupMethod.backupConfig.subClient.impersonateUser.userName;
				}
			});
	}

	// updates the user credentials for the client.
	updateClientCredentials() {
		let credential = this.model.selectedCredential[0];
		if (credential) {
			this.createPseudoClientRequest.clientInfo.userSavedCredentials = credential;
			this.createPseudoClientRequest.clientInfo.userCredentials = undefined;
		} else {
			this.createPseudoClientRequest.clientInfo.userSavedCredentials = undefined;
			this.createPseudoClientRequest.clientInfo.userCredentials = {
				userName: this.model.userName,
				password: this.encryptPassword(this.model.password)
			};
		}
	}

	updatePlan() {
		let plan = this.selectedPlan[0];
		this.createPseudoClientRequest.clientInfo.plan = { planId: plan.planId, planName: plan.planName };
	}

	updateQumuloPassword(pwd) {
		this.createPseudoClientRequest.clientInfo.userCredentials.password = this.encryptPassword(this.qumuloPassword);
	}

	updateNdmpPassword() {
		this.createPseudoClientRequest.clientInfo.nasClientProperties.ndmpServerDetails.ndmpCredentials.password = this.encryptPassword(
			this.ndmpPassword
		);
	}

	validateHostName() {
		return this.nasFactory.isIpAddress(this.createPseudoClientRequest.entity.hostName, this.clientParams.serverType);
	}

	updateHostName() {
		if (this.isNasServer && !this.createPseudoClientRequest.entity.hostName) {
			this.createPseudoClientRequest.entity.hostName = this.createPseudoClientRequest.entity.clientName;
		}
	}

	/**
	 * Common function to be used in various places to check the host name validity.
	 */
	getFileServerHost() {
		let arrayName = '';
		if (this.clientParams.serverType === NAS_CONSTANTS.NUTANIX) {
			arrayName = _.get(this.createPseudoClientRequest, 'clientInfo.fileServerInfo.arrayName');
		} else {
			arrayName = _.get(this.createPseudoClientRequest, 'entity.hostName');
		}
		return arrayName;
	}

	/**
	 * Function will set the default path to the array root. For CIFS it will \\<arrayName>\\ and the label
	 * should be shown as All CIFS shares For NFS it should be <arrayName>:/ and the label should be shown as
	 * All NFS shares. This functionality is true when no content is selected irrespective of whether
	 * exclusions and exceptions are selected.
	 */
	updateDefaultPath(item) {
		let arrayName = this.getFileServerHost();
		if (!arrayName) {
			return;
		}
		let type = item.key;
		let backupConfig = item.backupConfig;
		let defaultPath =
			type === NAS_CONSTANTS.NDMP ? '/' : type === NAS_CONSTANTS.CIFS ? '\\\\' + arrayName + '\\' : arrayName + ':/';
		if (!_.find(backupConfig.subClient.content, 'path')) {
			backupConfig.subClient.content.push({ path: defaultPath });
			item.contents['path'] = {
				label: this.cvLoc('label.content'),
				paths: [item.protocolLabel]
			};
		} else {
			if (backupConfig.subClient.content.length > 1 && backupConfig.subClient.content[0].path === defaultPath) {
				backupConfig.subClient.content.splice(0, 1);
				item.contents.path.paths.splice(0, 1);
			}
		}
	}

	/**
	 * Update content selection when ever the array host is changed
	 */
	updateFileServerHost(fileServerInfo) {
		if (fileServerInfo) {
			this.createPseudoClientRequest.clientInfo.fileServerInfo = fileServerInfo;
		}

		this.backupTypes.map(item => {
			this.updateDefaultPath(item);
		});

		this.showHostNameFQDNError = this.validateHostName();
	}

	updateNDMPServerDetails(item) {
		this.createPseudoClientRequest.clientInfo.nasClientProperties.ndmpServerDetails.ndmpServerHostName = this.getFileServerHost();
		this.createPseudoClientRequest.clientInfo.nasClientProperties.ndmpServerDetails.ndmpServerClientName = this.createPseudoClientRequest.entity.clientName;
	}

	validateContentSelection(backupMethod) {
		backupMethod.showErrors = false;
		this.selectHostNameError = false;
		this.showPlanError = false;
		this.showHostNameFQDNError = this.validateHostName();
		if (backupMethod.enabled) {
			let arrayName = this.getFileServerHost();
			if (!arrayName) {
				this.selectHostNameError = true;
			}

			if (!this.editMode) {
				this.showPlanError = this.selectedPlan.length === 0;
			}

			// for CIFS check if user is selected.
			if (backupMethod.key === NAS_CONSTANTS.CIFS && !this.selectedCIFSUser) {
				backupMethod.showErrors = true;
			}
		}

		return this.showPlanError || backupMethod.showErrors || this.selectHostNameError || this.showHostNameFQDNError;
	}

	encryptPassword(pwd) {
		//pwd ? this.cvUtil.getBytes(Base64.encode(pwd)) : '';
		return pwd ? Base64.encode(pwd) : '';
	}

	getNasContentOptions(backupMethod) {
		let fsHost = this.getFileServerHost();
		let ndmpRequest = '';
		let nasOptions = {
			appId: backupMethod.appId,
			fsHost: fsHost,
			enableNetworkShareAutoMount: true,
			planEntity: this.clientParams.clientInfo.plan,
			hideShowAsUser: backupMethod.key === NAS_CONSTANTS.CIFS,
			isFileStorageServer: !this.isNasServer,
			backupAccessNodes: backupMethod.accessNodeOptions.backupDataAccessNodes //backupMethod.backupConfig.idaInfo.backupConfiguration.backupDataAccessNodes
		};

		if (backupMethod.key === NAS_CONSTANTS.NDMP) {
			let request = this.validateNDMPRequest(true, true);
			if (!request.isValid) {
				return false;
			}
			ndmpRequest = request;
			nasOptions.ndmpRequest = ndmpRequest;
			if (_.get(this.arrayRequest, 'info')) {
				let userInfo = Object.assign({}, this.arrayRequest.info.userPswd);
				userInfo.password = this.cvUtil.getBytes(Base64.decode(userInfo.password));

				let arrayInfo = Object.assign({}, this.arrayRequest.info, {
					userPswd: userInfo
				});
				nasOptions.arrayInfo = arrayInfo;
			}
		}
		let nasContentOptions = this.nasFactory.getContentOptionsForBrowse(nasOptions);

		if (this.AppTypesUtil.isWindowsFileSystem(backupMethod.appId)) {
			let impersonateUser = backupMethod.backupConfig.subClient.impersonateUser;
			let impersonateUserCredentialinfo =
				backupMethod.backupConfig.subClient.commonProperties.impersonateUserCredentialinfo;

			if (impersonateUser && impersonateUser.userName && impersonateUser.password) {
				nasContentOptions.impersonateUser = impersonateUser;
				nasContentOptions.nasBrowseOptions.userName = impersonateUser.userName;
				nasContentOptions.nasBrowseOptions.password = impersonateUser.password;
			} else if (impersonateUserCredentialinfo && impersonateUserCredentialinfo.credentialId) {
				nasContentOptions.impersonateUserCredentialinfo = impersonateUserCredentialinfo;
				nasContentOptions.nasBrowseOptions.credentialId = impersonateUserCredentialinfo.credentialId;
			}
		}
		return nasContentOptions;
	}

	selectContent(backupMethod) {
		if (!this.validateContentSelection(backupMethod)) {
			let self = this;
			let nasContentOptions = this.getNasContentOptions(backupMethod);

			if (!nasContentOptions) {
				return;
			}

			this.$modal.open({
				templateUrl: appUtil.appRoot + 'modules/nas/partials/fileStorageEditContent.jsp',
				backdrop: 'static',
				resolve: {
					nasContentOptions: nasContentOptions
				},
				controller: [
					'$scope',
					'cvUtil',
					'nasContentOptions',
					'$uibModalInstance',
					'cvLoc',
					function($scope, cvUtil, nasContentOptions, $modalInstance, cvLoc) {
						$scope.contentOptions = {
							isForSubclientPolicy: false,
							forNewSubclientCreation: true
						};

						if (backupMethod.contentDetails) {
							$scope.addedEntities = backupMethod.contentDetails.detailedContentEntityList.contentEntities.filter(
								path => !path.isReadOnly
							); // global exception gets added with the backend call.

							$scope.contentOptions.forNewSubclientCreation = false;
							$scope.contentOptions.useGlobalFilters = backupMethod.contentDetails.useGlobalFilters;
							$scope.contentOptions.includePolicyFilters = backupMethod.contentDetails.includePolicyFilters;
							$scope.contentOptions.typeWiseCounterArray =
								backupMethod.contentDetails.detailedContentEntityList.typeWiseCounterArray;
						}

						$scope.contentOptions = { ...$scope.contentOptions, ...nasContentOptions };

						$scope.ok = function() {
							backupMethod.contentDetails = $scope.contentOptions.getContentDetails();
							let selectedContent = $scope.contentOptions.getContentDetails().contentEntities;
							let type = '',
								path = '',
								backendPath = '',
								backendKey = '',
								key = '',
								pathObj = {};
							let backendContents = [];
							let contents = {};
							selectedContent.map(content => {
								type = content.type;
								path = content.path;
								backendPath = content.path;
								(backendKey = ''), (key = '');
								if (self.AppTypesUtil.isUnixFileSystem(backupMethod.appId) && backendPath.indexOf(':') !== -1) {
									backendPath = '/' + backendPath.replace(':', ''); // for nas, browse content with in the request ( not in the gui) should have a leading forward slash('/') and also : should be removed with in the browse content.
								}

								if (type === 0) {
									key = cvLoc('label.content');
									backendKey = 'path';
									pathObj = { path: backendPath };
								} else if (type === 1) {
									backendKey = 'excludePath';
									key = cvLoc('label.Exclusions');
									pathObj = { excludePath: backendPath };
								} else if (type === 2) {
									backendKey = 'includePath';
									key = cvLoc('label.Exceptions');
									pathObj = { includePath: backendPath };
								}
								backendContents.push(pathObj);
								if (contents.hasOwnProperty(backendKey)) {
									contents[backendKey].paths.push(path);
								} else {
									contents[backendKey] = {
										label: key,
										paths: [path]
									};
								}
							});
							backupMethod.backupConfig.subClient.content = backendContents;
							backupMethod.contents = contents;
							self.updateDefaultPath(backupMethod);
							$modalInstance.dismiss();
						};

						$scope.cancel = function() {
							$modalInstance.dismiss();
						};
					}
				]
			});
		}
	}

	openAddArrayModal() {
		let self = this;
		let clientType = this.isNasServer ? NAS_CONSTANTS.NAS : NAS_CONSTANTS.NUTANIX;

		let selectedVendor = { name: this.selectedVendor[0].vendorName, id: this.selectedVendor[0].vendorType };
		let plan = this.selectedPlan[0];
		let arrayInfo = {
			arrayName: {
				name: _.get(this.createPseudoClientRequest, 'entity.hostName')
			}
		};
		if (self.arrayRequest && self.arrayRequest.info) {
			arrayInfo = self.arrayRequest.info;
		}

		let modalInstance = this.nasFactory.addArray({
			vendor: selectedVendor,
			clientType: clientType,
			planId: plan && plan.planId,
			arrayInfo: arrayInfo,
			autoSave: !this.isNasServer // For Nas do not save the array , wait untill client is created.
		});

		modalInstance.result.then(function(arrayName) {
			let add = false;
			if (self.isNasServer && arrayName) {
				self.arrayRequest = arrayName; // keep copy of request that is to be saved after the client
				// for nas add the array which is not save and make sure it is selected.
				self.selectedVendor = arrayName.selectedVendor;
				self.arrayName = arrayName.info.arrayName.name;
				self.arrayActionLabel = self.cvLoc('action.edit');
				self.updateVendor(arrayName.info.arrayName.name);
			} else {
				self.loadArrayList(selectedVendor.name, arrayName);
			}
		});
	}

	async loadArrayList(vendorName, arrayName) {
		let serverType = this.isNasServer ? NAS_CONSTANTS.NAS : NAS_CONSTANTS.NUTANIX;
		const { data: arrays } = await this.nasService.getNonAssociatedArrays(serverType);
		let self = this;
		this.arrayList = [];

		if (arrays.arrayList && arrays.arrayList.length > 0) {
			this.arrayList = arrays.arrayList
				.filter(item => item.vendor.name === vendorName)
				.map((item, index) => {
					let currentArray = { arrayId: item.arrayName.id, arrayName: item.arrayName.name };
					if (arrayName && arrayName === item.arrayName.name) {
						currentArray.selected = true;
						self.updateFileServerHost(currentArray);
					}
					return currentArray;
				});
		}
	}

	async loadPlanList() {
		let self = this;
		const { data: plans } = await this.nasService.getEligiblePlansForNASSubClient();
		this.planList = plans.map((planSummary, index) => {
			let _planObj = {
				planName: planSummary.plan.planName,
				planId: planSummary.plan.planId,
				planType: planSummary.type,
				type: planSummary.subtype,
				rpoInMinutes: planSummary.rpoInMinutes,
				numAssocEntities: planSummary.numAssocEntities,
				numCopies: planSummary.numCopies,
				planSummary: _.get(planSummary, 'plan.planSummary')
			};

			if (_.get(self.clientParams, 'planEntity.planName') === _planObj.planName) {
				_planObj.selected = true;
				self.selectedPlan = [_planObj];
				self.updatePlan();
			}

			return _planObj;
		});
	}

	async loadCredentails() {
		const { data: credentials } = await this.credentialManagerService.getCredentialRecord();
		this.savedCredentials = credentials.map(item => item.credentialRecord);
		this.savedCredentials.unshift({
			credentialId: 0,
			credentialName: this.cvLoc('label.enterCredential'),
			selected: true
		});
	}

	contentPopOver = function(key, paths) {
		let html = `<div class='list-call-out'>
            <div class='call-out-title'><div>${key}</div></div>
            <div class='list-container'>
            	${paths
								.map(
									path =>
										`<div class='list'>
                    <div class='list-text crop' title='${path}'>${path}</div>
                </div>
                `
								)
								.join('')}
            </div>
        </div>`;
		return html;
	};

	cancel() {
		this.$modalInstance.dismiss();
	}

	isNDMPProtocolEnabled(item) {
		item.showNDMPUserLoginError = item.showNDMPPasswordError = false;
		let userName = _.get(
			this.createPseudoClientRequest,
			'clientInfo.nasClientProperties.ndmpServerDetails.ndmpCredentials.userName'
		);

		let password = _.get(
			this.createPseudoClientRequest,
			'clientInfo.nasClientProperties.ndmpServerDetails.ndmpCredentials.password'
		);

		item.enabled = userName || password;

		return item.enabled;
	}

	validateRequest() {
		let errors = false;
		this.selectBackupMethodError = false;
		this.errorMessage = {};
		let self = this;
		errors = !this.createPseudoClientRequest.entity.clientName;

		if (!this.editMode) {
			errors = errors || !this.getFileServerHost();
		}

		if (!this.editMode) {
			errors = errors || this.selectedPlan.length === 0;
		}

		//at least one backupMethod should be slected.

		let backupMethodSelecetd = false;

		this.backupTypes.map(item => {
			if (item.enabled) {
				errors = self.validateContentSelection(item) || errors;
			}

			if (item.key === NAS_CONSTANTS.NDMP && this.isNDMPProtocolEnabled(item)) {
				let ndmpRequst = self.validateNDMPRequest(true);
				errors = !ndmpRequst.isValid || errors;
			}
			backupMethodSelecetd = backupMethodSelecetd || item.enabled;
		});

		if (!backupMethodSelecetd) {
			this.selectBackupMethodError = true;
			errors = true;
		}

		if (this.filerMigration.isEnabled && (this.isInvalidMigrationPrerequisites || this.isMismatchedMigrationProtocol)) {
			errors = true;
		}

		return errors;
	}

	validateNDMPRequest = function(showErrors, isBrowse) {
		let ndmpProtoCol = _.find(this.backupTypes, { key: NAS_CONSTANTS.NDMP });

		let userName = _.get(
			this.createPseudoClientRequest,
			'clientInfo.nasClientProperties.ndmpServerDetails.ndmpCredentials.userName'
		);

		let password = _.get(
			this.createPseudoClientRequest,
			'clientInfo.nasClientProperties.ndmpServerDetails.ndmpCredentials.password'
		);

		// if user name or  password is entered then ndmp is enabled
		if (userName && password) {
			ndmpProtoCol.enabled = true;
		}

		if (ndmpProtoCol.enabled) {
			this.updateNDMPServerDetails(ndmpProtoCol);
		}

		let isValidRequest = true;

		let hostName = _.get(
			this.createPseudoClientRequest,
			'clientInfo.nasClientProperties.ndmpServerDetails.ndmpServerHostName'
		);

		let planId = _.get(this.createPseudoClientRequest, 'clientInfo.plan.planId');

		if (this.editMode) {
			planId = this.clientParams.planEntity.planId;
		}

		if (isBrowse) {
			password = Base64.decode(password); // xml request should contain plain text password.
			password = this.cvUtil.getBytes(password);
		}

		if (!hostName) {
			this.selectHostNameError = showErrors && true;
			isValidRequest = false;
		}

		if (!userName) {
			ndmpProtoCol.showNDMPUserLoginError = showErrors && true;
			isValidRequest = false;
		}

		if (!password) {
			ndmpProtoCol.showNDMPPasswordError = showErrors && true;
			isValidRequest = false;
		}

		let request = {
			isValid: isValidRequest,
			listenPort: this.createPseudoClientRequest.clientInfo.nasClientProperties.listenPort,
			planId: planId,
			ndmpServerDetails: {
				ndmpServerHostName: hostName,
				ndmpCredentials: {
					userName: userName,
					password: password
				}
			}
		};

		ndmpProtoCol.enabled = isValidRequest;

		return request;
	};

	updateNdmpProperties(ndmpServerProperties) {
		this.createPseudoClientRequest.clientInfo.nasClientProperties.listenPort = ndmpServerProperties.listenPort;
		let vendorType = this.cvUtil.lookupEnumConstant(ndmpServerProperties.vendorType, 'value'); // get int value and update
		this.createPseudoClientRequest.clientInfo.nasClientProperties.vendorType = vendorType;
	}

	previewNDMPServer() {
		let request = this.validateNDMPRequest(true);
		if (!request.isValid) {
			return;
		}
		let self = this;
		self.previewErrorMessage = {};
		this.nasService
			.detectNDMPServer(request)
			.success(function(successData) {
				if (successData && successData.ndmpServerProperties) {
					let modalResult = self.nasFactory.openNDMPDetails(successData.ndmpServerProperties);
					modalResult.result.then(function() {
						self.updateNdmpProperties(successData.ndmpServerProperties);
					});
				} else if (successData && successData.errorMessage) {
					self.previewErrorMessage = self.cvUtil.errMsg(successData.errorMessage);
				}
			})
			.error(function(errorMessage) {
				self.previewErrorMessage = self.cvUtil.errMsg(errorMessage);
			});
	}

	handleCreateSuccess(success) {
		var self = this;
		self.cvToaster.showInfoMessage({
			ttl: '10000', //10 sec
			message: success.clientName + ' ' + success.message
		});

		if (!self.editMode) {
			self.$state.go('nasClientDetails', {
				clientId: success.clientId
			});
		} else {
			self.nasService.refreshCache(success.clientId, 0).success(function() {
				self.$state.reload();
			});
		}
		self.cancel();
	}

	createFileStorageServer() {
		// update Qumulo credential to the request.
		if (this.clientParams.serverType === NAS_CONSTANTS.QUMULO) {
			this.updateClientCredentials();
		}

		this.showErrors = this.validateRequest();
		if (this.showErrors) {
			return;
		}
		var self = this;
		let backupConfigList = [];
		let osTypes = [];
		let isNDMPEnabled = false;
		this.backupTypes.map(item => {
			if (item.enabled) {
				if (item.osType) {
					osTypes.push(item.osType);
				}
				backupConfigList.push(item.backupConfig);
			}

			if (item.key === NAS_CONSTANTS.CIFS) {
				// encode the password
				item.backupConfig.subClient.impersonateUser.password = self.encryptPassword(
					item.backupConfig.subClient.impersonateUser.password
				);
			}

			// remove the NDMP properties from the clientInfo
			if (item.key === NAS_CONSTANTS.NDMP) {
				if (!item.enabled) {
					this.createPseudoClientRequest.clientInfo.nasClientProperties = undefined;
					this.createPseudoClientRequest.clientInfo.clientType = 18; // IF NDMP is not enabled , it becomes NON NDMP client
				} else {
					isNDMPEnabled = true;
					this.createPseudoClientRequest.clientInfo.clientType = 2;
					this.updateNDMPServerDetails(item);
				}
			}
		});

		// update the os types.
		if (this.createPseudoClientRequest.clientInfo.clientType === 18 && osTypes.length === 1) {
			this.createPseudoClientRequest.clientInfo.nonNDMPClientProperties = {
				osType: osTypes[0]
			};
		}

		// update the os types.
		if (this.createPseudoClientRequest.clientInfo.clientType === 2 && osTypes.length > 0) {
			this.createPseudoClientRequest.clientInfo.nasClientProperties.osTypes = osTypes;
		}

		// add filer migration details
		if (this.filerMigration.isEnabled) {
			this.createPseudoClientRequest.clientInfo.lookupMigrateServerDetails = true;
			this.createPseudoClientRequest.clientInfo.migrateServerDetails = this.nasFactory.getMigrationConfigPayload(
				this.filerMigration.migrationConfig
			);
		}

		let createPseudoClientRequest = this.createPseudoClientRequest;
		if (this.editMode) {
			//remove the fileserverInfo from the request as it is not need in the edit request.
			createPseudoClientRequest.clientInfo.fileServerInfo = undefined;
		}

		let request = {
			createPseudoClientRequest: createPseudoClientRequest,
			backupConfigList: backupConfigList
		};

		var message = this.clientParams.createSuccessMessage;
		let clientId = this.editMode ? this.createPseudoClientRequest.entity.clientId : 0;
		let params = {
			clientId: clientId,
			request: request,
			serverType: this.clientParams.serverType
		};

		let clientName = self.createPseudoClientRequest.entity.clientName;
		if (this.isNasServer && this.showFileServerHost) {
			// in case of NAS and when an array is added , a deconfigured client is created with the array name and is merged.
			clientName = this.getFileServerHost();
		}

		let success = {
			clientName: clientName,
			message: message,
			clientId: clientId
		};

		this.nasService
			.addOrUpdateFiLeServerClient(params)
			.success(function(successData) {
				if (successData.errorMessage != null) {
					self.errorMessage = self.cvUtil.errMsg(successData.errorMessage);
					return;
				}

				if (!self.editMode) {
					success.clientId = successData.response.entity.clientId;
				}
				// if it is NAS server and arrayRequest object is initialized then update the clientId and save the array. This will create the array and associate the client with array.
				if (self.isNasServer && self.arrayRequest) {
					self.arrayRequest.clientId = success.clientId;
					if (!self.editMode) {
						// update the planId
						self.arrayRequest.planId = _.get(self.createPseudoClientRequest, 'clientInfo.plan.planId');
					}

					self.snapArrayService.createArray(JSON.stringify(self.arrayRequest)).success(data => {
						self.handleCreateSuccess(success);
					});
				} else {
					self.handleCreateSuccess(success);
				}
			})
			.error(function(e) {
				if (e.errorMessage != null) {
					self.errorMessage = self.cvUtil.errMsg(e.errorMessage);
				}
			});
	}

	get isInvalidMigrationPrerequisites() {
		const configuredShareTypes = this.getConfiguredShareTypesForMigration();
		return (
			configuredShareTypes.length !== 1 || // Should have only one selected protocol
			!this.getFileServerHost() || // Valid host name
			(configuredShareTypes[0].key === NAS_CONSTANTS.CIFS && !this.selectedCIFSUser) // Valid credentials (if CIFS)
		);
	}

	/**
	 * Checks if the configured network share and migration use the same protocol
	 */
	get isMismatchedMigrationProtocol() {
		const configuredShareTypes = this.getConfiguredShareTypesForMigration();
		return (
			configuredShareTypes.length &&
			this.filerMigration.isEnabled &&
			this.filerMigration.configuredProtocol &&
			this.filerMigration.configuredProtocol !== configuredShareTypes[0].key
		);
	}

	get migrationConfigErrorMessage() {
		const configuredShareTypes = this.getConfiguredShareTypesForMigration();
		if (configuredShareTypes.length > 1) {
			return this.cvLoc('error.multipleSharesConfigured');
		} else if (this.isMismatchedMigrationProtocol) {
			return this.cvLoc(
				'error.migrationProtocolMismatch',
				this.filerMigration.configuredProtocol,
				configuredShareTypes[0].key
			);
		} else {
			return this.cvLoc('error.missingPrerequisites');
		}
	}

	getConfiguredShareTypesForMigration() {
		return this.backupTypes.filter(
			type =>
				(type.key === NAS_CONSTANTS.CIFS || type.key === NAS_CONSTANTS.NFS) &&
				!type.hidden &&
				!type.disabled &&
				type.enabled
		);
	}

	toggleFilerMigration() {
		if (!this.filerMigration.isEnabled) {
			// Turn on filer migration
			if (this.isInvalidMigrationPrerequisites) {
				return;
			}
			this.filerMigration.isEnabled = true;
			if (_.isUndefined(this.filerMigration.migrationConfig) || _.isEmpty(this.filerMigration.migrationConfig)) {
				this.openMigrationConfigModal();
			}
		} else if (this.filerMigration.isEnabled) {
			// Turn off filer migration
			this.filerMigration.isEnabled = false;
		}
	}

	openMigrationConfigModal() {
		if (this.isInvalidMigrationPrerequisites) {
			return;
		}
		const shareTypes = this.getConfiguredShareTypesForMigration(); // Will be an array with single item. Invalidated in prerequisites check if multiple items are present
		const applicationId = shareTypes[0].key === NAS_CONSTANTS.CIFS ? 33 : 29;
		const browseConfig = this.getNasContentOptions(shareTypes[0]);
		const migrationModal = this.nasFactory.openMigrationDialog(
			applicationId,
			this.filerMigration.migrationConfig,
			true,
			{
				browseConfig: angular.copy(browseConfig)
			}
		);
		migrationModal.result.then(
			updatedMigrationConfig => {
				this.filerMigration.migrationConfig = updatedMigrationConfig;
				this.filerMigration.showConfigPreview = true;
				this.filerMigration.configuredProtocol = shareTypes[0].key;
			},
			dismissalReason => {
				if (_.isUndefined(this.filerMigration.migrationConfig) || _.isEmpty(this.filerMigration.migrationConfig)) {
					this.filerMigration.isEnabled = false;
					this.filerMigration.showConfigPreview = false;
				}
			}
		);
	}

	showMigrationConfigEditButton() {
		return (
			this.filerMigration.isEnabled &&
			!this.isInvalidMigrationPrerequisites &&
			!_.isUndefined(this.filerMigration.migrationConfig) &&
			!_.isEmpty(this.filerMigration.migrationConfig)
		);
	}
}
nasFileServerClientController.$inject = [
	'clientParams',
	'$uibModal',
	'snapArrayService',
	'nasService',
	'cvLoc',
	'credentialManagerService',
	'$uibModalInstance',
	'nasFactory',
	'cvToaster',
	'$state',
	'cvUtil',
	'AppTypesUtil',
	'CREDENTIAL_TYPE'
];
acAppClientModule.controller('nasFileServerClientController', nasFileServerClientController);
export default acAppClientModule;
