import { commonAllAgentsModule } from 'common/js/modules';
import 'machineBrowse/js/services/machineBrowse.svc.js';
import 'modules/ida/js/services/idaService.svc.js';
import 'modules/ida/js/services/idas.factory.js';
import 'adminConsole/js/services/instances.svc.js';
import 'adminConsole/js/services/subclients.svc.js';

/**
 * Use this directive for Oracle RAC database instances management.
 */
const app = commonAllAgentsModule;

app.directive('cvRacManageStreams', function() {
	return {
		restrict: 'E',
		templateUrl: `${appUtil.appRoot}modules/ida/partials/cv-rac-manage-streams.jsp`,
		bindToController: {
			entity: '=',
			hideTitle: '=',
			isRestore: '=',
			getStreamsFunc: '=',
			refreshFunc: '='
		},
		controller: 'racManageStreamsController',
		controllerAs: 'racMS'
	};
});

app.controller('racManageStreamsController', [
	'$timeout',
	'cvUtil',
	'cvToaster',
	'cvLoc',
	'idasFactory',
	'cvTableOptions',
	'$uibModal',
	'instanceService',
	'subclientService',
	function(
		$timeout,
		cvUtil,
		cvToaster,
		cvLoc,
		idasFactory,
		cvTableOptions,
		$uibModal,
		instanceService,
		subclientService
	) {
		const self = this;
		this.$onInit = function() {
			/* GRID CONTSTRUCTION */
			const globalGridOptions = angular.copy(cvTableOptions.commonNgGridOptions);

			const streamsForExistingSubclient = !self.isRestore && self.entity.subclientId > 0;

			self.instances = [];

			function init(entity) {
				if (entity) {
					self.entity = entity;
					self.grid.grid.dataSource.read();
				}
			}

			function fetchData(options) {
				if (self.instances && self.instances.length > 0) {
					$timeout(options.success(self.instances));
				} else {
					instanceService
						.getInstanceDetails(self.entity.instanceId)
						.success(function(data) {
							self.instances = [];
							const instanceMap = {};

							data.oracleRACInstance.racDBInstance.forEach(function(racI) {
								const instance = { entity: {} };
								instance.entity.clientName = racI.instancePhysicalClient.clientName;
								instance.entity.instanceName = racI.instanceOracleSID;

								instance.entity.instanceId = racI.racDbInstanceId;
								instanceMap[instance.entity.instanceId] = instance;
								if (!streamsForExistingSubclient) {
									// Defult stream values for new subclient
									instance.dataStream = 2;
									instance.logStream = 1;
									self.instances.push(instance);
								}
							});
							if (streamsForExistingSubclient) {
								// Get existing stream values
								subclientService
									.getSubClientDetails(self.entity.subclientId)
									.success(function(subclientInfo) {
										const dataStreams = subclientInfo.oracleSubclientProp.oracleDataBackupStreams;
										dataStreams.sort(function(a, b) {
											return a.priority - b.priority;
										});
										dataStreams.forEach(function(dataStream) {
											self.instances.push(instanceMap[dataStream.instanceId]);
											instanceMap[dataStream.instanceId].dataStream = dataStream.streamNumber;
										});
										subclientInfo.oracleSubclientProp.oracleLogBackupStreams.forEach(function(logStream) {
											instanceMap[logStream.instanceId].logStream = logStream.streamNumber;
										});
										options.success(self.instances);
									})
									.error(function(err) {
										options.error(err);
									});
							} else {
								options.success(self.instances);
							}
						})
						.error(function(err) {
							options.error(err);
						});
				}
			}

			/* Set ui-grid options */
			//		angular.extend(globalGridOptions, {
			//			data: 'instances',
			//			enableGridMenu: false,
			//			paginationPageSize: 10,
			//			columnDefs: [{
			//				field: 'entity.clientName',
			//				displayName: cvLoc('label.server'),
			//				cellTemplate: '<span class="crop" title="{{row.entity.entity.clientName}}">{{row.entity.entity.clientName}}</span>',
			//				width: '*',
			//				enableSorting: false,
			//				enableHiding: false
			//			},{
			//				field: 'entity.instanceName',
			//				displayName: cvLoc('tableHeading.instance'),
			//				cellTemplate: '<span class="crop" title="{{row.entity.entity.instanceName}}">{{row.entity.entity.instanceName}}</span>',
			//				width: '*',
			//				enableSorting: false,
			//				enableHiding: false
			//			},{
			//				name: 'dataSteam',
			//				displayName: self.isRestore?cvLoc('label.stream'): cvLoc('label.data'),
			//				cellTemplate: '<span data-ng-if="!row.entity.editing" class="crop" title="{{row.entity.dataStream}}">{{row.entity.dataStream}}</span>\
			//					<span data-ng-if="row.entity.editing"><input type="number" min="0" max="50" step="1" data-ng-model="row.entity.dataStreamNew"/></span>',
			//				width: '*',
			//				enableFiltering: false,
			//				enableSorting: false,
			//			},{
			//				name: 'logSteam',
			//				displayName: cvLoc('label.archiveLogs'),
			//				cellTemplate: '<span data-ng-if="!row.entity.editing" class="crop" title="{{row.entity.logStream}}">{{row.entity.logStream}}</span>\
			//					<span data-ng-if="row.entity.editing"><input type="number" min="0" max="50" step="1" data-ng-model="row.entity.logStreamNew"/></span>',
			//				width: '*',
			//				enableFiltering: false,
			//				enableSorting: false,
			//			},{
			//				name: 'action',
			//				displayName: cvLoc('header.actions'),
			//				cellTemplate: '<span data-ng-disabled="grid.appScope.sendingRequest"><span data-ng-if="!row.entity.editing"><a title=' + cvLoc('action.edit') + ' data-ng-click="grid.appScope.startEditing(row.entity)"><i class="glyphicon glyphicon-pencil"></i></a><span data-ng-if="grid.appScope.instances.length > 1"><a title="' + cvLoc('label.moveUp') + '" class="padding-left-20" data-ng-click="grid.appScope.moveUp(row.entity)"><i class="glyphicon glyphicon-arrow-up"></i></a><a title="' + cvLoc('label.moveDown') + '" class="padding-left-20" data-ng-click="grid.appScope.moveDown(row.entity)"><i class="glyphicon glyphicon-arrow-down"></i></a></span></span>\
			//					<span data-ng-if="row.entity.editing"><a title=' + cvLoc('Save') + ' data-ng-click="grid.appScope.stopEditing(row.entity, true)"><i class="glyphicon glyphicon-ok"></i></a> <a title=' + cvLoc('Cancel') + ' class="padding-left-20" data-ng-click="grid.appScope.stopEditing(row.entity, false)"><i class="glyphicon glyphicon-remove"></i></a></span></span>',
			//				width: '*',
			//				enableFiltering: false,
			//				enableSorting: false,
			//			}]
			//		});

			/* Remove log for restore case */
			//		if(self.isRestore) {
			//			globalGridOptions.columnDefs.splice(3,1);
			//		}

			/* Editing in place in grid cell */
			self.startEditing = function(rowEntity) {
				rowEntity.dataStreamNew = rowEntity.dataStream;
				rowEntity.logStreamNew = rowEntity.logStream;
			};

			function refreshGrid() {
				self.grid.grid.dataSource.read();
				self.grid.grid.clearSelection();
				// self.grid.refreshGrid();
			}

			/* Convert data to request object */
			function getRacStreams(newInstances) {
				let tempInstances = self.instances;
				if (newInstances) {
					tempInstances = newInstances;
				}
				const streamsObject = {
					oracleDataBackupStreams: [],
					oracleLogBackupStreams: []
				};
				const streamsObjectForRestore = {
					racDataStreamAllcation: []
				};
				let priority = 1;
				tempInstances.forEach(function(inst) {
					const dataStream = {
						instanceId: inst.entity.instanceId,
						streamInstanceName: inst.entity.instanceName,
						priority: priority++,
						streamNumber: inst.dataStream
					};

					const logStream = angular.copy(dataStream);
					logStream.streamNumber = inst.logStream;

					streamsObject.oracleDataBackupStreams.push(dataStream);
					streamsObject.oracleLogBackupStreams.push(logStream);
					// Restore uses a list of strings for streams
					streamsObjectForRestore.racDataStreamAllcation.push(`${inst.entity.instanceId} ${inst.dataStream}`);
				});
				if (self.isRestore) {
					return streamsObjectForRestore;
				}
				return streamsObject;
			}

			/* Send request to update for existing subclient only */
			function sendRequest(newInstances, callBack) {
				self.sendingRequest = true;
				const streamsObject = getRacStreams(newInstances);
				const subclientInfo = {
					oracleSubclientProp: {}
				};
				angular.extend(subclientInfo.oracleSubclientProp, streamsObject);
				subclientService
					.updateSubclient({
						subClientEntity: angular.toJson(self.entity),
						subclientInfo: angular.toJson(subclientInfo)
					})
					.success(function() {
						if (callBack) {
							callBack();
						}
						self.sendingRequest = false;
					})
					.error(function(err) {
						cvToaster.showErrorMessage({ ttl: 10000, message: err });
						self.sendingRequest = false;
					});
			}

			/* Move up row */
			self.moveUp = function(rowEntity) {
				const newInstances = [];
				for (let i = 0; i < self.instances.length; ++i) {
					if (i > 0 && self.instances[i] === rowEntity) {
						const last = newInstances.pop();
						newInstances.push(self.instances[i]);
						newInstances.push(last);
					} else {
						newInstances.push(self.instances[i]);
					}
				}
				// Send update request for existing subclient
				if (streamsForExistingSubclient) {
					sendRequest(newInstances, function() {
						self.instances = newInstances;
						refreshGrid();
					});
				} else {
					self.instances = newInstances;
					refreshGrid();
				}
			};
			/* Move down row */
			self.moveDown = function(rowEntity) {
				const newInstances = [];
				for (let i = 0; i < self.instances.length; ++i) {
					if (i < self.instances.length - 1 && self.instances[i] === rowEntity) {
						const next = self.instances[i + 1];
						newInstances.push(next);
						newInstances.push(self.instances[i]);
						++i;
					} else {
						newInstances.push(self.instances[i]);
					}
				}
				// Send update request for existing subclient
				if (streamsForExistingSubclient) {
					sendRequest(newInstances, function() {
						self.instances = newInstances;
						refreshGrid();
					});
				} else {
					self.instances = newInstances;
					refreshGrid();
				}
			};

			/* Check if stream is valid */
			function isValidStream(stream) {
				return (stream || stream === 0) && Math.round(stream) === stream && stream >= 0 && stream <= 50;
			}

			/* Check if all nodes have 0 stream */
			function hasPositiveStream(rowEntity) {
				let hasPositiveDataStream = false;
				let hasPositiveLogStream = false;
				for (let i = 0; i < self.instances.length; ++i) {
					const dataStream =
						rowEntity === self.instances[i] ? self.instances[i].dataStreamNew : self.instances[i].dataStream;
					const logStream =
						rowEntity === self.instances[i] ? self.instances[i].logStreamNew : self.instances[i].logStream;

					if (dataStream > 0) hasPositiveDataStream = true;
					if (logStream > 0) hasPositiveLogStream = true;
				}

				return self.isRestore ? hasPositiveDataStream : hasPositiveDataStream && hasPositiveLogStream;
			}

			/* Stop editing */
			self.stopEditing = function(rowEntity, callBackFunc) {
				if (!isValidStream(rowEntity.dataStreamNew) || !isValidStream(rowEntity.logStreamNew)) {
					cvToaster.showErrorMessage({
						message: cvLoc('error.streamRange')
					});
					return;
				}
				if (!hasPositiveStream(rowEntity)) {
					cvToaster.showErrorMessage({
						message: cvLoc('error.allNodesZeroStream')
					});
					return;
				}
				// Send update request for existing subclient
				if (streamsForExistingSubclient) {
					const newInstances = [];
					for (let i = 0; i < self.instances.length; ++i) {
						const newInst = angular.copy(self.instances[i]);
						if (rowEntity === self.instances[i]) {
							newInst.dataStream = rowEntity.dataStreamNew;
							newInst.logStream = rowEntity.logStreamNew;
							newInst.editing = false;
						}
						newInstances.push(newInst);
					}
					sendRequest(newInstances, function() {
						self.instances = newInstances;
						refreshGrid();
						callBackFunc();
					});
				} else {
					rowEntity.dataStream = rowEntity.dataStreamNew;
					rowEntity.logStream = rowEntity.logStreamNew;
					rowEntity.editing = false;
					refreshGrid();
					callBackFunc();
				}
			};

			/* Pass back function for parent controller to call */
			if (self.getStreamsFunc) {
				self.getStreamsFunc.getRacStreams = getRacStreams;
			}
			if (self.refreshFunc) {
				self.refreshFunc.refresh = init;
			}

			/* Set cv-grid options */
			//		self.gridOptions = {
			//				cvHasTitle : !self.hideTitle,
			//				cvAppScope : self,
			//				cvIsPageTitle : false,
			//				cvHasViews: false,
			//				cvOnGridEmpty : {
			//					'message' : cvLoc('label.noInstances')
			//				},
			//				gridOptions : globalGridOptions,
			//				cvIsSearchable : false,
			//				cvGridCssClass: {
			//					'users-grid': true
			//				}
			//		};
			//
			//		if(self.gridOptions.cvHasTitle) {
			//			self.gridOptions.cvGridTitle = cvLoc('label.noOfStreams');
			//		}

			function getColumns() {
				const columns = {
					'entity.clientName': {
						title: cvLoc('label.server'),
						type: 'string',
						width: '30%',
						template: `<span class="crop" cv-toggle-content="#=entity.clientName#" cv-toggle="tooltip">#=entity.clientName#</span>`
					},
					'entity.instanceName': {
						title: cvLoc('tableHeading.instance'),
						type: 'string',
						width: '30%',
						template: `<span class="crop" cv-toggle-content="#=entity.instanceName#" cv-toggle="tooltip">#=entity.instanceName#</span>`
					},
					dataStream: {
						title: self.isRestore ? cvLoc('label.stream') : cvLoc('label.data'),
						type: 'number',
						width: '20%',
						cellTemplate: `<span class="crop">#=entity.dataStream#</span>`
					},
					logStream: {
						title: cvLoc('label.archiveLogs'),
						type: 'number',
						width: '20%',
						cellTemplate: `<span class="crop">#=entity.logStream#</span>`
					}
				};
				if (self.isRestore) delete columns.logStream;
				return columns;
			}

			function actionClicked(e) {
				const row = e.selectedRowValues[0];
				let selectedInst;
				self.instances.forEach(inst => {
					if (
						inst.entity.clientName === row.entity.clientName &&
						inst.entity.instanceName === row.entity.instanceName
					) {
						selectedInst = inst;
					}
				});
				switch (e.optionId) {
					case 'Edit':
						self.openEditModal(selectedInst);
						break;
					case 'MoveUp':
						self.moveUp(selectedInst);
						break;
					case 'MoveDown':
						self.moveDown(selectedInst);
						break;
					default:
						break;
				}
			}

			self.openEditModal = function(inst) {
				$uibModal.open({
					templateUrl: `${appUtil.appRoot}modules/ida/partials/racEditStream.jsp`,
					windowClass: 'small-size',
					backdrop: 'static',
					controllerAs: 'racESM',
					controller: [
						'$uibModalInstance',
						function($uibModalInstance) {
							let modSelf = this;
							modSelf.inst = inst;
							modSelf.streamsForExistingSubclient = self.streamsForExistingSubclient;
							modSelf.isRestore = self.isRestore;

							self.startEditing(modSelf.inst);
							modSelf.submit = function() {
								self.stopEditing(modSelf.inst, function() {
									$uibModalInstance.close();
								});
							};

							modSelf.cancel = function() {
								$uibModalInstance.dismiss();
							};
						}
					]
				});
			};

			function onGridSelectionChange(e) {
				if (e.rows.length === 1) {
					self.grid.showActionMenuOption('Edit');
					if (e.rows[0].rowIndex === 0) {
						self.grid.hideActionMenuOption('MoveUp');
					} else {
						self.grid.showActionMenuOption('MoveUp');
					}

					if (e.rows[0].rowIndex === self.instances.length - 1) {
						self.grid.hideActionMenuOption('MoveDown');
					} else {
						self.grid.showActionMenuOption('MoveDown');
					}
				} else {
					self.grid.hideActionMenuOption('Edit');
					self.grid.hideActionMenuOption('MoveUp');
					self.grid.hideActionMenuOption('MoveDown');
				}
			}

			// Use Kendo grid
			function initGrid() {
				self.gridOptions = {
					enableCheckBoxColumn: false,
					enableColumnResizing: true,
					enableVirtualization: true,
					enableColumnMenu: true,
					enableFiltering: false,
					enableSorting: false,
					searchable: false,
					tableName: 'racManageStreamsTable',
					//					idField: "entity.clientName",
					columns: getColumns(),
					url: fetchData,
					gridTitle: cvLoc('label.noOfStreams'),
					hasViews: false,
					height: 250,
					hasDefaultView: false,
					onGridSelectionChange,
					beforeGridInitialize: ({ grid }) => {
						self.grid = grid;
					},
					afterGridInitialize: () => {
						$('#streams .cv-k-grid-header').hide();
					}
				};
				self.gridOptions.actionMenu = [
					{
						id: 'Edit',
						label: cvLoc('label.inlineEdit'),
						onSelect: actionClicked
					},
					{
						id: 'MoveUp',
						label: cvLoc('label.moveUp'),
						onSelect: actionClicked
					},
					{
						id: 'MoveDown',
						label: cvLoc('label.moveDown'),
						onSelect: actionClicked
					}
				];
			}
			initGrid();
			init();
		};
	}
]);

export default app;
