import { commonAllAgentsModule } from 'common/js/modules';

var mod = commonAllAgentsModule;

/**
 * A directive used to handle the functionality of searching while browsing items to restore.
 *
 * @param {function=}
 *            onSearch - An optional function to be called when the search button is pressed. The argument
 *            "searchTerm" will be passed to the function.
 */
mod.directive('cvBrowseMailsSearch', [
	'$document',
	function($document) {
		return {
			restrict: 'E',
			scope: {
				onSearch: '&',
				searchTerm: '='
			},
			templateUrl: 'modules/ida/partials/searchBox.jsp',
			link: function($scope, element, attr, controller) {
				var document = angular.element($document);
				var searchFilterDropdown = element.find('#search-filter-dropdown');

				/**
				 * Called when the Received filter in the search box has changed. Handles scrolling the search
				 * filter to the bottom so the user can see the new options.
				 */
				controller.receivedChanged = function() {
					element.find('.scrollable').animate({
						scrollTop: element.find('.scrollable').prop('scrollHeight')
					});
				};

				/**
				 * Handles the "Click" event on the document.
				 *
				 * The handler is used when only when the dropdown for displaying the search filters is open.
				 * The handler checks to see if the click is off the dropdown itself, as well as not being
				 * inside one of the datetime picker elements used as filters. If both of those conditions are
				 * met, then the dropdown will be manually closed.
				 *
				 * @param {Object}
				 *            event - The event being handled for the click.
				 */
				var onClick = function(event) {
					var clickedOffElement =
						!searchFilterDropdown.is(event.target) && searchFilterDropdown.has(event.target).length <= 0;

					if (
						clickedOffElement &&
						angular.element(event.target).closest('.datetime-picker-dropdown').length <= 0 &&
						!element.find('input.search-field').is(event.target)
					) {
						controller.dropdownOpen = false;
					}
				};

				/*
				 * Only if the dropdown is open should we apply the "Click" handler to the document.
				 * Otherwise, we unbind it.
				 */
				$scope.$watch(
					function() {
						return controller.dropdownOpen;
					},
					function(dropdownOpen) {
						if (dropdownOpen) {
							document.bind('click', onClick);
						} else {
							document.unbind('click', onClick);
						}
					}
				);

				/*
				 * On the scope being destroyed, we want to unbind the "Click" handler so that it won't run
				 * anymore.
				 */
				$scope.$on('$destroy', function() {
					document.unbind('click', onClick);
				});
			},
			controllerAs: 'cvBrowseSearchController',
			controller: [
				'$scope',
				'$filter',
				'cvLoc',
				function($scope, $filter, cvLoc) {
					var self = this;
					self.onSearch = $scope.onSearch;

					self.timeRangeLabel = cvLoc('label.dateRange');
					self.receivedOptions = [
						cvLoc('label.today'),
						cvLoc('label.yesterday'),
						cvLoc('label.thisWeek'),
						cvLoc('label.thisMonth'),
						cvLoc('label.thisYear'),
						cvLoc('label.lastWeek'),
						cvLoc('label.lastMonth'),
						cvLoc('label.lastYear'),
						self.timeRangeLabel
					];

					self.sizeOptions = [cvLoc('labelKB'), cvLoc('labelMB'), cvLoc('labelGB')];

					self.fromDateOptions = {
						dateFormat: 'yyyy-MM-dd',
						showWeeks: false,
						maxDate: new Date()
					};

					self.toDateOptions = {
						dateFormat: 'yyyy-MM-dd',
						showWeeks: false,
						maxDate: new Date()
					};

					/**
					 * Called when the "From time" input has changed when the user is selecting a time range for
					 * the search box.
					 */
					self.onFromDateChanged = function() {
						self.toDateOptions.minDate = angular.isDefined(self.searchParams.fromTime)
							? self.searchParams.fromTime
							: null;
					};

					/**
					 * Called when the "To time" input has changed when the user is selecting a time range for the
					 * search box.
					 */
					self.onToDateChanged = function() {
						self.fromDateOptions.maxDate = angular.isDefined(self.searchParams.toTime)
							? self.searchParams.toTime
							: new Date();
					};

					/**
					 * Formats a simple search field for a search term and returns it.
					 *
					 * @param {string}
					 *            prepend - The filter to prepend before the term.
					 * @param {string}
					 *            term - The search term
					 * @return {string} The resulting formatted search term.
					 */
					function formatSearchField(prepend, term) {
						if (angular.isString(term) && term.length > 0) {
							term = term.trim();
							//Java Console Behavior, no additional quotes required
							//						if (term.indexOf(' ') > 0) {
							//							term = '"' + term + '"';
							//						}
							self.searchTerms.push(prepend + ':' + term);
						}
						return '';
					}

					/**
					 * Initializes the search parameters to the default state.
					 */
					self.initializeSearchParams = function() {
						self.searchParams = {
							operation: '>',
							units: cvLoc('labelKB')
						};
					};
					self.initializeSearchParams();

					/**
					 * Fornats the given GMT date into a date that can be displayed to the user.
					 *
					 * @param {string}
					 *            date - A GMT date structed like the following: Wed Jun 07 2017 15:26:15 GMT-0400
					 *            (Eastern Daylight Time)
					 * @return {string} A date that can be displayed to the user.
					 */
					var formatDisplayDate = function(date) {
						return $filter('date')(date, 'MM/dd/yyyy');
					};

					/**
					 * Formats the search term based on the available search parameters.
					 */
					self.formatSearch = function() {
						self.searchTerms = [];
						formatSearchField(cvLoc('label.from'), self.searchParams.from);
						formatSearchField(cvLoc('label.to'), self.searchParams.to);
						formatSearchField(cvLoc('label.subject'), self.searchParams.subject);
						formatSearchField(cvLoc('label.folder'), self.searchParams.folder);
						if (self.searchParams.hasAttachment) {
							self.searchTerms.push(cvLoc('label.hasAttachmentTrue'));
						}
						if (angular.isDefined(self.searchParams.amount) && self.searchParams.amount !== null) {
							self.searchTerms.push(
								cvLoc('label.mailSizeParameter') +
									':' +
									self.searchParams.operation +
									self.searchParams.amount +
									self.searchParams.units
							);
						}
						formatSearchField(cvLoc('label.contains'), self.searchParams.contains);

						if (self.searchParams.received !== self.timeRangeLabel) {
							if (angular.isString(self.searchParams.received) && self.searchParams.received.length > 0) {
								self.searchTerms.push(cvLoc('label.received') + ':' + self.searchParams.received.trim());
							}
						} else {
							var timeRangeFilter = [];
							if (angular.isDefined(self.searchParams.fromTime)) {
								timeRangeFilter.push(formatDisplayDate(self.searchParams.fromTime));
							}
							timeRangeFilter.push(cvLoc('label.to').toLowerCase());
							if (angular.isDefined(self.searchParams.toTime)) {
								timeRangeFilter.push(formatDisplayDate(self.searchParams.toTime));
							}
							if (timeRangeFilter.length > 1) {
								self.searchTerms.push(cvLoc('label.received') + ':' + timeRangeFilter.join(' '));
							}
						}

						$scope.searchTerm = self.searchTerms.join(' ');
					};

					/**
					 * Prepares the search term before a search can be made. If there are valid search parameters
					 * provided by the user, then we need to format the search. Otherwise, we do nothing.
					 */
					self.prepareForSearch = function() {
						angular.forEach(self.searchParams, function(param, key) {
							if (['operation', 'units'].indexOf(key) < 0) {
								if (angular.isObject(param) || (angular.isString(param) && param.trim().length > 0)) {
									self.formatSearch();
								}
							}
						});
					};
				}
			]
		};
	}
]);

export default mod;
