import 'modules/ida/js/directives/cv-browse-mails-search.js';
import 'modules/ida/js/directives/cv-browse-directives.js';
import 'modules/ida/js/services/idaService.svc.js';
import 'office365Exchange/js/controllers/office365ExchMailboxRestoreOptions.ctrl.js';
import 'exchangeV2/js/services/exchange.svc.js';
import { exchangeConstants } from 'exchangeV2/js/constants/exchangeConstants.js';
import { ExchangeO365TabsClass } from 'exchangeV2/js/baseClasses/ExchangeTabsClass.js';

import { commonAllAgentsModule } from 'common/js/modules';

/**
 * A controller for handling the functionality on the exchange browse page.
 */
commonAllAgentsModule.controller('office365ExchangeBrowseController', [
	'$q',
	'$uibModal',
	'idaService',
	'$stateParams',
	'cvLoc',
	'cvUtil',
	'cvBreadcrumbsTabsFactory',
	'cvToaster',
	'uiGridConstants',
	'$state',
	'tabService',
	'exchangeServiceV2',
	'AppTypes',
	function(
		$q,
		$uibModal,
		idaService,
		$stateParams,
		cvLoc,
		cvUtil,
		cvBreadcrumbsTabsFactory,
		cvToaster,
		uiGridConstants,
		$state,
		tabService,
		exchangeService,
		AppTypes
	) {
		var self = this;
		var subclient = null;
		self.ExchangeTabsClass = new ExchangeO365TabsClass(exchangeConstants, exchangeService, cvLoc);
		self.entityType = 'SUBCLIENT_ENTITY';
		self.entityId = $stateParams.subclientId;

		self.mailSortFields = [
			{
				key: 'ModifiedDate',
				label: cvLoc('label.date')
			},
			{
				key: 'From',
				label: cvLoc('label.from')
			},
			{
				key: 'MailSize',
				label: cvLoc('label.size')
			},
			{
				key: 'Subject',
				label: cvLoc('label.subject')
			}
		];

		self.browseOptions = {
			getEntity: idaService.getNodeEntity.bind(undefined, self.entityType, self.entityId),
			getBrowseData: idaService.getIdaBrowseData.bind(undefined, self.entityId, self.entityType),
			/*
			 * Shows the button for collapsing the left tree.
			 */
			showCollapseTreeButton: true,

			/*
			 * Filtering by the date of backups is not supported for the Exchange One Pass app type.
			 */
			showDateFilter: false,

			/**
			 * Hook for customizing the root label's name.
			 *
			 * @param {Object}
			 *            root - The root entity retrieved for the browse page.
			 */
			rootLabel: function(root) {
				return root.backupsetName;
			},

			/**
			 * Called when the user selects the "Restore" button above the browse table.
			 *
			 * @param {Object[]}
			 *            items - An array of items that the user wants to restore.
			 */
			onRestore: function(restoreParams) {
				var fromSubclient = angular.copy(subclient);
				delete fromSubclient.children;

				var selectedPaths = [];
				angular.forEach(restoreParams.items, function(item) {
					selectedPaths.push(item.userObject.path);
				});

				var modalInstance = $uibModal.open({
					templateUrl: appUtil.appRoot + 'office365Exchange/partials/office365ExchMailboxRestoreOptions.jsp',
					controller: 'office365ExchMailboxRestoreOptionsController as exchMailboxRestoreOptionsController',
					backdrop: 'static',
					resolve: {
						subclient: fromSubclient,
						itemsToRestore: function() {
							return selectedPaths;
						}
					}
				});
			},

			/**
			 * Called when the entity from the cv-browse directive is retrieved. We can use this hook to build
			 * our breadcrumbs.
			 *
			 * @param {Object}
			 *            The entity that was retrieved.
			 */
			onEntityRetrieval: function(entity) {
				subclient = entity;
				if ($state.current.name === '365exchangeBrowse') {
					cvBreadcrumbsTabsFactory.addBreadCrumbs([
						{
							title: cvLoc('label.nav.office365'),
							link: '#/office365Dashboard/true'
						},
						{
							title: cvLoc('label.exchangeServers'),
							link: '#/365exchange'
						},
						{
							title: entity.clientName,
							link: `#/365exchangeDetails/${subclient.clientId}`
						},
						{
							title: entity.appName,
							link: `#/365mailboxDetails/${subclient.clientId}`
						},
						{
							title: entity.backupsetName,
							link: `#/365mailboxBackupSetDetails/${subclient.clientId}/${subclient.backupsetId}/${subclient.subclientId}/${subclient.backupsetName}`
						}
					]);
				} else if ($state.current.name === 'O365exchangeBrowseV2') {
					var breadCrumbs = [
						{
							title: cvLoc('label.office365Servers'),
							link: '#/office365V2'
						}
					];
					if (angular.isDefined($stateParams.forMailbox)) {
						breadCrumbs.push({
							title: entity.clientName,
							link: `#office365/clientDetails/${subclient.clientName}/${subclient.clientId}/${subclient.subclientId}/users?backupsetId=${subclient.backupsetId}`
						});
					}
					cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
					self.ExchangeTabsClass.exchangeBrowse(subclient, $stateParams.forMailbox).then(
						values => {
							let tabData = values;
							tabData.onActionItemSelect = value => {
								self.redirectPage(value);
							};
							tabService.init(tabData);
						},
						e => {
							cvToaster.showErrorMessage({
								ttl: '5000', //5 sec
								message: e.data
							});
						}
					);
				}
			},

			/**
			 * Called before a request is sent to the Browse service for populating the left-hand tree in the
			 * browse directive.
			 *
			 * @param {Object}
			 *            item - An optional item that the user has clicked on and wants to browse.
			 * @return {Object} The parameters to be sent to the browse request.
			 */
			onBeforeBrowseTreeRequest: function(item) {
				var params = {
					subclientId: $stateParams.subclientId,
					entityType: self.entityType
				};

				/*
				 * If the item has a "userObject" property, then it's not the root node.
				 */
				if (angular.isObject(item.userObject)) {
					return angular.extend(params, {
						path: item.userObject.path,
						criteria: 'ParentObjectGUID:' + item.id,
						sortingInfo: 'asc:FileName',
						fileOrFolder: 'FOLDER'
					});
				}
				return angular.extend(params, {
					path: '\\',
					sortingInfo: 'asc:MailboxName',
					fileOrFolder: 'MAILBOX',
					criteria: 'ParentObjectGUID:00000000000000000000000000000001'
				});
			},

			/**
			 * Called before a request is sent to the Browse service for populating the right-hand table in
			 * the browse directive.
			 *
			 * @param {Object}
			 *            item - An optional item that the user has clicked on and wants to browse.
			 * @param {boolean}
			 *            forSearch - Whether or not the request was for a search.
			 * @return {Object} The parameters to be sent to the browse request.
			 */
			onBeforeBrowseTableRequest: function(item, forSearch) {
				var params = {
					subclientId: $stateParams.subclientId,
					entityType: self.entityType
				};

				/*
				 * If the item has a "userObject" property, then it's not the root node.
				 */
				if (angular.isObject(item.userObject)) {
					angular.extend(params, {
						path: item.userObject.path,
						criteria: 'ParentObjectGUID:' + item.id
					});
					if (
						item.userObject.advancedData.browseMetaData.indexing.parentObjectGUID === '00000000000000000000000000000001'
					) {
						return angular.extend(params, {
							sortingInfo: forSearch ? 'desc:ModifiedDate' : 'asc:FileName',
							fileOrFolder: forSearch ? 'FILE' : 'FOLDER'
						});
					} else {
						return angular.extend(params, {
							sortingInfo: 'desc:ModifiedDate',
							fileOrFolder: 'FILE'
						});
					}
				}
				return angular.extend(params, {
					path: '\\',
					sortingInfo: forSearch ? 'desc:ModifiedDate' : 'asc:MailboxName',
					fileOrFolder: forSearch ? 'FILE' : 'MAILBOX',
					criteria: 'ParentObjectGUID:00000000000000000000000000000001'
				});
			},

			/**
			 * Called after the request sent to the Browse service for populating the right-hand table in the
			 * browse directive has completed.
			 *
			 * @param {Object}
			 *            item - The item that was selected by the user to browse.
			 * @param {Object[]}
			 *            data - The data that was retrieved from the request.
			 */
			onAfterBrowseTableRequest: function(item, data) {
				if (!angular.isObject(item.userObject)) {
					/*
					 * If the item being browsed does not have a userObject property, then it must be the root
					 * node, so this is the only circumstance where we want to alter the display data.
					 */
					angular.forEach(data, function(item) {
						var mailbox = item.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name;
						mailbox = mailbox.substring(mailbox.indexOf('(') + 1, mailbox.indexOf(')'));
						item.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name = mailbox;
					});
				}
			},

			/**
			 * Called when the browse table on the right hand side is going to be built.
			 *
			 * @param {Object}
			 *            params - The parameters for the function
			 * @param {Object}
			 *            params.tableEntity - The entity that was selected to browse for the table.
			 * @param {Object}
			 *            params.gridOptions - The grid options to manipulate if you see fit.
			 * @param {boolean}
			 *            params.forSearch - Whether or not the table will be used for a search response.
			 *
			 *
			 */
			onBeforeBrowseTableBuilt: function(params) {
				var item = params.tableEntity;
				var gridOptions = params.gridOptions;
				var forSearch = params.forSearch;

				if (forSearch || angular.isObject(item.userObject)) {
					if (
						!forSearch &&
						item.userObject.advancedData.browseMetaData.indexing.parentObjectGUID === '00000000000000000000000000000001'
					) {
						/*
						 * The item's parent is the root node. If this is the case, then the next set of data
						 * we will be showing is folders within the root.
						 */
						gridOptions.gridOptions.columnDefs = [
							{
								field: 'label',
								displayName: cvLoc('Name'),
								headerTooltip: true,
								cellTemplate:
									'<a data-ng-click="grid.appScope.performTableBrowse(row.entity)" class="crop" title="{{row.entity.label}}">{{row.entity.label}}</a>',
								width: '*',
								sortParam: 'FileName',
								sort: {
									direction: uiGridConstants.ASC
								}
							}
						];
					} else {
						/*
						 * The item's parent is a folder. If this is the case, then the next set of data we
						 * will be showing is individual emails.
						 */
						gridOptions.gridOptions.columnDefs = [
							{
								field: 'mail',
								displayName: cvLoc('label.email'),
								enableSorting: false,
								headerTooltip: false,
								cellTemplate:
									'<div class="gmailRow">' +
									'<span class="emailFrom" data-ng-attr-title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.from.name}}">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.from.name}}<i data-ng-if="row.entity.userObject.advancedData.browseMetaData.exchDataV2.hasAttachment" title="' +
									cvLoc('label.hasAttachment') +
									'" class="ion-paperclip"></i></span>' +
									'<span class="emailDate" data-ng-attr-title="{{row.entity.userObject.modificationTime * 1000 | date : "medium"}}">{{row.entity.userObject.modificationTime * 1000 | date : "medium"}}</span>' +
									'<br />' +
									'<span class="emailSubject crop" data-ng-attr-title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.subject}}"> <span class="ui-grid-cell-contents">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.subject}}</span></span>' +
									'<span class="emailSize" data-ng-attr-title="{{row.entity.userObject.size|capacity}}">{{row.entity.userObject.size|capacity}}</span>' +
									'</div>',
								width: '*'
							}
						];
					}
				} else {
					gridOptions.gridOptions.columnDefs = [
						{
							field: 'label',
							displayName: cvLoc('Name'),
							headerTooltip: true,
							cellTemplate:
								'<a data-ng-click="grid.appScope.performTableBrowse(row.entity)" class="crop" title="{{row.entity.label}}">{{row.entity.label}}</a>',
							width: '*',
							sortParam: 'MailboxName',
							sort: {
								direction: uiGridConstants.ASC
							}
						},
						{
							field: 'userObject.displayName',
							displayName: cvLoc('label.emailAddress'),
							headerTooltip: true,
							cellTemplate:
								'<div class="crop" title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name}}">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name}}</div>',
							width: '*',
							enableSorting: false
						}
					];
				}
			}
		};

		self.redirectPage = function(action) {
			switch (action) {
				case exchangeConstants.actions.SETTINGS:
					$state.go(exchangeConstants.navigationStateNames.O365.SETTINGS, {
						clientId: subclient.clientId,
						backupsetId: subclient.backupsetId,
						subclientId: subclient.subclientId,
						backupsetName: subclient.backupsetName
					});
					break;
				case exchangeConstants.actions.VIEW_JOBS:
					$state.go(exchangeConstants.navigationStateNames.JOBS, {
						serverId: subclient.clientId,
						serverName: subclient.clientName,
						applicationId: AppTypes.EXCHANGE_MAILBOX,
						backupsetId: subclient.backupsetId,
						activeJobs: 0
					});
					break;
			}
		};

		if (angular.isDefined($stateParams.forMailbox)) {
			try {
				self.browseOptions.forEntity = angular.fromJson($stateParams.forMailbox);
				self.browseOptions.forEntity = angular.extend(
					{
						userObject: {
							size: 0,
							path: '\\MB\\{' + self.browseOptions.forEntity.id + '}',
							advancedData: {
								browseMetaData: {
									indexing: {
										parentObjectGUID: '00000000000000000000000000000001'
									},
									exchDataV2: {
										mbxInfo: {
											name: self.browseOptions.forEntity.emailAddress
										}
									}
								}
							}
						}
					},
					self.browseOptions.forEntity
				);
			} catch (e) {}
		}
	}
]);

export default commonAllAgentsModule;
