import { emailTemplatesModule } from 'common/js/modules';

import {
	EMAIL_TEMPLATES_API_ENDPOINT,
	ADMIN_CONSOLE_PROXY_ENDPOINT,
	DEFAULT_HEADERS,
	EMAIL_TYPE_VM_TO_DATA,
	EMAIL_TEMPLATE_VM_TO_DATA,
	EMAIL_HEADER_FOOTER_VM_TO_DATA,
	ORGANIZATION_API_ENDPOINT,
	CURRENT_COMPANY_ID,
	CURRENT_LOCALE
} from '../constants';

import { cloneObjectInstance, DataModelToViewModel } from './../util';

emailTemplatesModule.factory('EmailTemplatesService', [
	'$http',
	'$log',
	'$q',
	'cvLoc',
	'$rootScope',
	function EmailTemplatesService($http, $log, $q, cvLoc, $rootScope) {
		const headers = DEFAULT_HEADERS;
		const transformRequest = data => {
			return JSON.stringify(data);
		};

		const BASE_URL = EMAIL_TEMPLATES_API_ENDPOINT;
		const ADMINCONSOLE_BASE_URL = ADMIN_CONSOLE_PROXY_ENDPOINT;

		//    const USER_LOCALE_ID = supportedLocales.filter(locale => {
		//        return CURRENT_LOCALE === locale.locale;
		//    }).pop().id;

		const USER_LOCALE_ID = CURRENT_LOCALE.id;

		/**
		 * EmailTemplates class which is returned as the base of the EmailTemplatesService service
		 */
		class EmailTemplates {
			// ideally supportedLocales and getLocaleName should be in a separate lightweight service but for now this will do…
			// TODO:[email customization] when implementing the new locales service, refactor so that the property is replaced with
			// a new method (eg. getSupportedLocales), you can model the getEmailTypes approach
			//        get supportedLocales() {
			//            // less than ideal looping over this every time it's referenced, trying to cache this output in another property or variable
			//            // creates a pass by reference conflict and negates the benefit of using `get` keyword
			//            return supportedLocales.map(locale => {
			//                locale.name = this.getLocaleName(locale.locale);
			//                return locale;
			//            });
			//        }

			get userLocaleId() {
				return USER_LOCALE_ID;
			}

			getLocaleNameById(localeId) {
				if (localeId === undefined) {
					throw new Error('locale id required');
				}
				let locale = supportedLocales
					.filter(locale => {
						return locale.id === localeId;
					})
					.pop();
				return this.getLocaleName(locale.locale);
			}

			getLocaleName(locale) {
				if (locale === undefined) {
					throw new Error('locale required');
				}
				return cvLoc(locale);
			}

			createTemplate(emailTemplate) {
				if (emailTemplate === undefined) {
					return new EmailTemplate();
				} else {
					return new EmailTemplate(undefined, { emailTemplate });
				}
			}

			getEmailHeaderFooter(companyId = CURRENT_COMPANY_ID) {
				return new EmailHeaderFooter(companyId).promise;
			}

			/**
			 * method to get all available email templates
			 *
			 * @returns {Promise} $http promise
			 */
			getEmailTemplates(params) {
				let endpoint = BASE_URL;
				//params = Object.assign({ propertyLevel: 30 }, params);
				//  EmailTemplates?templateTypeId=1&ownerCompanyId=0&propertyLevel=30&localeId=0&templateId=2
				return (
					$http
						.get(endpoint, { params })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						// return instances of EmailTemplate
						.then(res => {
							if (res.emailTemplatesList !== undefined) {
								return res.emailTemplatesList.map(
									emailTemplate => new EmailTemplate(emailTemplate.emailTemplateId, { emailTemplate })
								);
							} else {
								return [];
							}
						})
				);
			}

			/**
			 * get email template by ID
			 *
			 * @param {string} -
			 *            id of email template
			 * @returns {Promise} $http promise
			 */
			getEmailTemplateById(id) {
				if (id === undefined) {
					throw new Error('id required');
				}
				return new EmailTemplate(id).promise;
			}

			getEmailTypes(config) {
				return EmailTemplates.getEmailTypes(config);
			}
			getSupportedLocales() {
				return EmailTemplates.getSupportedLocales();
			}

			getEmailTypeById(id, localeId, propertyLevel) {
				return EmailTemplates.getEmailTypeById(id, localeId, propertyLevel);
			}

			getEmailTokens(templateTypeId) {
				return EmailTemplates.getEmailTokens(templateTypeId);
			}

			static getEmailTypes(config = {}) {
				let endpoint = `${BASE_URL}/Types`;
				let { templateTypeId, localeId = USER_LOCALE_ID, propertyLevel = 30 } = config;
				// EmailTemplates/Types?propertyLevel=30&localeId=5

				// safe to cache as these won't change?
				return (
					$http
						.get(endpoint, { cache: true, params: { templateTypeId, localeId, propertyLevel } })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						// return instances of EmailType
						.then(res => {
							if (res.emailTemplateTypesList !== undefined) {
								return res.emailTemplateTypesList.map(
									emailType => new EmailType(emailType.typeId, { emailType, localeId, propertyLevel })
								);
							} else {
								return [];
							}
						})
				);
			}
			static getSupportedLocales() {
				let endpoint = `${ADMINCONSOLE_BASE_URL}/CommServ/locales`; //CommServ/locales
				// safe to cache as these won't change as often
				return (
					$http
						.get(endpoint, { cache: true })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						// return instances of locales
						.then(res => {
							return res.localeList ? res.localeList : [];
						})
				);
			}

			/**
			 * get email type by id
			 *
			 * @param {string} -
			 *            id of email type
			 * @returns {Promise} $http promise
			 */
			static getEmailTypeById(id, localeId, propertyLevel) {
				return EmailTemplates.getEmailTypes({ templateTypeId: id, localeId, propertyLevel }).then(emailTypes =>
					emailTypes.pop()
				);
			}

			static getEmailTokens(templateTypeId) {
				let endpoint = `${BASE_URL}/Tokens`;
				// /EmailTemplates/Tokens?templateTypeId={TEMPLATETYPEID}

				return (
					$http
						.get(endpoint, { cache: true, params: { templateTypeId } })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						// return instances of EmailType
						.then(res => {
							return res.emailTemplateTokensList ? res.emailTemplateTokensList : [];
						})
				);
			}

			getTemplate() {
				return $http.get(appUtil.appRoot + 'common/partials/editor.jsp');
			}
		}

		// The classes below are not directly exposed by the service but are support of the EmailTemplates class

		/**
		 * EmailType class used for email type instances within the EmailTemplates service
		 */
		class EmailType extends DataModelToViewModel {
			constructor(id, config) {
				if (id === undefined) {
					throw new Error('id required');
				}
				// destructuring of config object
				let { localeId, propertyLevel, emailType = {}, vmMapping = EMAIL_TYPE_VM_TO_DATA } = config;

				super(vmMapping, emailType);

				this.id = id;
				this.localeId = localeId;
				this.propertyLevel = propertyLevel;
			}

			getEmailTokens() {
				return EmailTemplates.getEmailTokens(this.id);
			}

			createTemplate(emailTemplate) {
				if (emailTemplate === undefined) {
					emailTemplate = new EmailTemplate();
				} else {
					emailTemplate = new EmailTemplate(undefined, { emailTemplate });
				}
				emailTemplate.typeId = this.id;
				return emailTemplate;
			}
		}

		/**
		 * EmailTemplate class used for email template instances within the EmailTemplates service
		 */
		class EmailTemplate extends DataModelToViewModel {
			/**
			 * creates a instance of EmailTemplate
			 *
			 * @param {string}
			 *            [id] - optional id for the EmailTemplate instance
			 * @param {object}
			 *            [config] - configuration object containing baseUrl and page properties
			 */
			constructor(id, config = {}) {
				// destructuring of config object
				let { baseUrl = BASE_URL, emailTemplate, vmMapping = EMAIL_TEMPLATE_VM_TO_DATA } = config;

				let fetchTemplate = id !== undefined && emailTemplate === undefined ? true : false;

				super(vmMapping, emailTemplate);

				this._baseUrl = baseUrl;

				// promise which can be used
				this._deferred = $q.defer();
				this.promise = this._deferred.promise;
				this.resolved = false;

				// if id is not supplied and email template is set to DEFAULT_EMAIL_TEMPLATE we need to fetch email template details
				if (fetchTemplate) {
					this.id = id;
					this._getEmailTemplate();
				} else {
					this._setupDependencies();
				}
			}

			// we need to fetch the email type when type or lang are updated
			get typeId() {
				return this._typeId;
			}

			set typeId(typeId) {
				this._typeId = typeId;
				this._getEmailType().then(emailType => {
					this._emailType = emailType;
				});
			}

			get locale() {
				return this._locale;
			}

			set locale(localeId) {
				this._locale = localeId;
				this._getEmailType().then(emailType => {
					this._emailType = emailType;
				});
			}

			get baseUrl() {
				return this._baseUrl;
			}

			get typeName() {
				// unfortunately this could happen where the type is not set
				return this._emailType !== undefined ? this._emailType.label : undefined;
			}

			//        get localeName() {
			//            let locale = supportedLocales.filter(locale => this.locale === locale.id).pop();
			//            return cvLoc(locale.locale);
			//        }

			_setupDependencies() {
				return this._getEmailType().then(emailType => {
					this._emailType = emailType;
					this._deferred.resolve(this);
					this.resolved = true;
					return this;
				});
			}

			_getEmailType() {
				return EmailTemplates.getEmailTypeById(this.typeId);
			}

			/**
			 * method used by constructor to fetch email template
			 *
			 * @private
			 */
			_getEmailTemplate() {
				let endpoint = this.baseUrl;
				//  EmailTemplates?templateTypeId=1&ownerCompanyId=0&propertyLevel=30&localeId=0&templateId=2
				return (
					$http
						.get(endpoint, { params: { templateId: this.id, propertyLevel: 30 } })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						.then(
							res => {
								// this service returns an array, we are only looking for a specific one
								let emailTemplate = res.emailTemplatesList.pop();
								this._dataModel = emailTemplate;
								this.mapDataToVm(this._vmMapping, emailTemplate);
								return this._setupDependencies();
							},
							err => {
								this._deferred.reject(err);
							}
						)
				);
			}

			getEmailTokens() {
				return EmailTemplates.getEmailTokens(this.typeId);
			}

			generateEmailPreview(emailTokens) {
				let promise;
				let endpoint = `${this.baseUrl}/Email/Action/GeneratePreview`;

				let tokenValues = emailTokens.map(token => {
					return { name: token[0].slice(1, -1), value: '[' + token[1] + ']' };
				});

				const payload = {
					template: {
						emailTemplateTypeId: this.typeId,
						emailTemplateId: this.id,
						emailBody: this.emailBody,
						subject: this.subject
					},
					company: {
						providerId: CURRENT_COMPANY_ID
					},
					tokenValues: tokenValues
				};
				promise = $http.post(endpoint, payload, { headers, transformRequest });
				return promise.then(onRequestSuccess, onRequestFailure);
			}

			sendTestEmail(testEmailReceiver, emailTokens) {
				let promise;

				let endpoint = `${this.baseUrl}/Email/Action/Send`;

				let toEmailsArray = [];
				if (testEmailReceiver && testEmailReceiver !== '') {
					toEmailsArray.push(testEmailReceiver);
				}

				let tokenValues = emailTokens.map(token => {
					return { name: token[0].slice(1, -1), value: '[' + token[1] + ']' };
				});

				const payload = {
					templateTypeId: this.typeId,
					template: {
						emailTemplateId: this.id
					},
					isTest: true,
					toEmails: toEmailsArray,
					tokenValues: tokenValues
				};
				promise = $http.post(endpoint, payload, { headers, transformRequest });
				// basic handlers
				return promise.then(onRequestSuccess, onRequestFailure);
			}

			save() {
				let promise;
				let endpoint = this.baseUrl;
				if (this.id === undefined) {
					// create new
					promise = $http.post(endpoint, this.data, { headers, transformRequest });
				} else {
					// update existing
					endpoint = `${this.baseUrl}/${this.id}`;
					promise = $http.put(endpoint, this.data, { headers, transformRequest });
				}
				return promise.then(onRequestSuccess, onRequestFailure);
			}

			clone() {
				let emailTemplate = cloneObjectInstance(this);
				emailTemplate.id = undefined;
				emailTemplate.isDefault = false;
				return emailTemplate;
			}

			remove() {
				let endpoint = `${this.baseUrl}/${this.id}`;
				return $http.delete(endpoint).then(onRequestSuccess, onRequestFailure);
			}
		}

		/**
		 * EmailType class used for email type instances within the EmailTemplates service
		 */
		class EmailHeaderFooter extends DataModelToViewModel {
			constructor(companyId, config = {}) {
				if (companyId === undefined) {
					throw new Error('company id required');
				}

				// destructuring of config object
				const {
					baseUrl = ORGANIZATION_API_ENDPOINT,
					emailHeaderFooter,
					vmMapping = EMAIL_HEADER_FOOTER_VM_TO_DATA
				} = config;

				const fetchHeaderFooter = emailHeaderFooter === undefined ? true : false;

				super(vmMapping, emailHeaderFooter);

				this._baseUrl = baseUrl;
				this.companyId = companyId;

				// promise which can be used
				this._deferred = $q.defer();
				this.promise = this._deferred.promise;
				this.resolved = false;

				// if id is not supplied and email template is set to DEFAULT_EMAIL_TEMPLATE we need to fetch email template details
				if (fetchHeaderFooter) {
					this._getEmailHeaderFooter();
				}
			}

			get baseUrl() {
				return this._baseUrl;
			}

			_getEmailHeaderFooter() {
				let endpoint = `${this.baseUrl}/${this.companyId}/EmailCustomization`;

				return (
					$http
						.get(endpoint)
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						.then(
							res => {
								this._dataModel = res;
								this.mapDataToVm(this._vmMapping, res);
								this._deferred.resolve(this);
								this.resolved = true;
								return this;
							},
							err => {
								this._deferred.reject(err);
							}
						)
				);
			}

			getCompanyEmailSettings() {
				let endpoint = `${this.baseUrl}/${this.companyId}/EmailCustomization`;
				return (
					$http
						.get(endpoint)
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						.then(
							res => {
								this._dataModel = res;
								this.mapDataToVm(this._vmMapping, res);
								this._deferred.resolve(this);
								this.resolved = true;
								return this;
							},
							err => {
								this._deferred.reject(err);
							}
						)
				);
			}

			save() {
				let endpoint = `${this.baseUrl}/${this.companyId}/EmailCustomization`;
				// /Organization/{OrganizationId}/EmailCustomization
				return $http
					.put(endpoint, this.data.companySettings, { headers, transformRequest })
					.then(onRequestSuccess, onRequestFailure);
			}
		}

		function onRequestSuccess(res) {
			// basic transform to simplify service response handling
			return res.data;
		}

		function onRequestFailure(err) {
			// TODO: [email-templates] better error handler
			$log.error(err);
			return $q.reject(err);
		}

		return new EmailTemplates();
	}
]);

export default emailTemplatesModule;
