import 'vsa/js/controllers/collections.ctrl.js';
import 'vsa/js/controllers/configureReplicationGroup.ctrl.js';
import 'vsa/js/controllers/replication.ctrl.js';
import 'vsa/js/services/replicationGroups.svc.js';
import 'vsa/js/services/restore.svc.js';
import 'vsa/js/directives/cv-replication-groups-summary-directive.js';
import 'vsa/js/factories/vmGroups.factory.js';
import 'dbs/js/services/server.svc.js';
import 'adminConsole/js/directives/cv-replication-monitor-directive.js';
import 'vsa/js/factories/replication.factory.js';
import 'vsa/js/factories/replicationGroups.factory.js';
import 'adminConsole/js/directives/cv-replication-operation-window.js';
import 'adminConsole/js/directives/cv-replication-frequency-directive.js';
import 'vsa/js/factories/collections.factory.js';
import 'vsa/js/controllers/restoreOptions.ctrl.js';
import 'vsa/js/controllers/blockLevelReplication.ctrl.js';
import 'modules/servers/js/services/servers.svc.js';

import { vsaAppServerModule } from 'common/js/modules';

const CONSTANT = {
	enableActivity: 'enabled-activity',
	disableActivity: 'disabled-activity',
	powerOnVmAfterRestore: 'powerOnVmAfterRestore',
	deployVmWhenFailover: 'deployVmWhenFailover',
	validateOnly: 'validateOnly',
	passUnconditionalOverride: 'passUnconditionalOverride',
	unconditialOverwrite: 'unconditialOverwrite',
	createVmsDuringFailover: 'createVmsDuringFailover',
	transportMode: 'transportMode',
	diskProvision: 'diskProvision',
	accessNode: 'accessNode',
	copyPrecedence: 'copyPrecedence',
	VSA: 'Virtual Server',
	replicationGroupsDetail: 'replicationGroupsDetail',
	replicationGroupsDetailConfiguration: 'replicationGroupsDetailConfiguration',
	PERIODIC: 'PERIODIC',
	replicationCopy: 'replicationCopy',
	SET_TRUE: 'SET_TRUE',
	snapShotEngine: 'snapShotEngine',
	nMaxSnapshotsPerDRVM: 'nMaxSnapshotsPerDRVM',
	ROLE_MSP_USER: 'Role_MSP_User',
	ROLE_TENANT_USER: 'Role_Tenant_User'
};

const {
	enableActivity,
	disableActivity,
	powerOnVmAfterRestore,
	unconditialOverwrite,
	deployVmWhenFailover,
	transportMode,
	accessNode,
	copyPrecedence,
	passUnconditionalOverride,
	VSA,
	replicationGroupsDetail,
	replicationGroupsDetailConfiguration,
	PERIODIC,
	replicationCopy,
	SET_TRUE,
	snapShotEngine,
	nMaxSnapshotsPerDRVM,
	ROLE_MSP_USER,
	ROLE_TENANT_USER
} = CONSTANT;

export class ReplicationGroupVSADirectiveController {
	constructor(
		$scope,
		$state,
		$dialogs,
		AppTypes,
		cvLoc,
		cvUtil,
		$log,
		replicationGroupsService,
		collectionFactory,
		cvToaster,
		VENDORS,
		restoreOptionFactory,
		replicationGroupFactory,
		$stateParams,
		tabService,
		replicationFactory,
		restoreService,
		subclientService,
		subclientFactory,
		serverService,
		vmGroupsFactory,
		restoreForAgents,
		blrUIFactory,
		BLR_CONSTANTS
	) {
		this.$scope = $scope;
		this.$state = $state;
		this.$dialogs = $dialogs;
		this.AppTypes = AppTypes;
		this.cvLoc = cvLoc;
		this.cvUtil = cvUtil;
		this.replicationGroupsService = replicationGroupsService;
		this.collectionFactory = collectionFactory;
		this.cvToaster = cvToaster;
		this.$log = $log;
		this.VENDORS = VENDORS;
		this.model = this.model || {};
		//add transport modes
		this.transportModeTypes = collectionFactory.getTransportModeTypes();
		this.$stateParams = $stateParams;
		this.repGrpId = $stateParams['repGrpId'];
		this.enableSimplifiedLiveSync = replicationGroupFactory.enableSimplifiedLiveSync;
		this.supportVendors = this.supportVendors || [VSA];
		this.CONSTANT = CONSTANT;
		this.tabService = tabService;
		this.replicationFactory = replicationFactory;
		this.toggleControlClick = this.toggleControlClick.bind(this);
		this.ctrl = this.ctrl || {};
		this.restoreService = restoreService;
		this.subclientService = subclientService;
		this.subclientFactory = subclientFactory;
		this.replicationGroupFactory = replicationGroupFactory;
		this.serverService = serverService;
		this.vmGroupsFactory = vmGroupsFactory;
		this.restoreForAgents = restoreForAgents;
		this.blrUIFactory = blrUIFactory;
		this.BLR_CONSTANTS = BLR_CONSTANTS;
	}

	$onInit() {
		const { cvLoc, cvUtil } = this;
		this.model.frequencyLabel = cvLoc('label.replicationFrequency');
		this.model.frequencyHelpText = cvLoc('info.help.planDetails.rpo.backupFrequency');
		this.model.replicationDestinationLabel = cvLoc('label.plan.storage');
		this.gridTitle = cvLoc('label.nav.replication.monitor.parent');
		this.isContinuous = false;
		this.isPeriodic = false;
		this.toggleStatus = {
			powerOnVmAfterRestore: disableActivity,
			enable: disableActivity,
			deployVmWhenFailover: disableActivity,
			unconditialOverwrite: disableActivity,
			createVmsDuringFailover: disableActivity,
			passUnconditionalOverride: disableActivity
		};

		this.editMode = {
			powerOnVmAfterRestore: false,
			deployVmWhenFailover: false,
			passUnconditionalOverride: false,
			enableDisable: false,
			transportMode: {
				isLoading: false,
				inEditMode: false
			},
			copyPrecedence: {
				isLoading: false,
				inEditMode: false
			},
			accessNode: {
				isLoading: false,
				inEditMode: false
			},
			replicationCopy: {
				isLoading: false,
				inEditMode: false
			},
			snapShotEngine: {
				isLoading: false,
				inEditMode: false
			},
			nMaxSnapshotsPerDRVM: {
				isLoading: false,
				inEditMode: false,
				errorExisted: false
			}
		};

		if (this.$state.current.name === replicationGroupsDetail) {
			this.selectedTabIndex = 0;
			this.isConfiguration = false;
		} else if (this.$state.current.name === replicationGroupsDetailConfiguration) {
			this.selectedTabIndex = 1;
			this.isConfiguration = true;
		}
		this.getReplicationGroupInfo(this.taskId);
		this.replicationGroupFactory.setSimplifiedLiveSyncUI(true);
		// It is not supported for editing yet.
		this.showEditReplicationPairs = false;
		this.ctrl.saveNewRule = this._saveNewRule.bind(this);
		this.ctrl.onDeleteSubclientContent = this._onDeleteSubclientContent.bind(this);
		this.ctrl.onAddVM = this._onAddVMinRuleEnabledGroup.bind(this);
	}

	maxSnapshotChanged(value) {
		let self = this;
		self.nMaxSnapshotsError = _.isNil(value)
			? self.cvUtil.errMsg(self.cvLoc('error.maximumSnapshot'))
			: self.cvUtil.emptyMsg();
		self.editMode.nMaxSnapshotsPerDRVM.errorExisted = _.isNil(value);
		this._confirmChangeforSAN(value);
	}

	initializeTabs() {
		let self = this;
		let url = '/replicationGroups/' + this.taskId + '?repGrpId=' + self.repGrpId;
		let configUrl = '/replicationGroups/configuration/' + self.taskId + '?repGrpId=' + self.repGrpId;

		const tabs = [
			{ title: self.cvLoc('header.overview'), url },
			{ title: self.cvLoc('header.configuration'), url: configUrl }
		];

		self.tabService.init({
			tabs: tabs,
			hasParent: true
		});
	}

	onToggleSuccess(type, isEnabling) {
		this.toggleStatus[type] = isEnabling ? enableActivity : disableActivity;
	}

	getSelectedTransportMode() {
		let self = this;
		this._getTransportModeTypes();
		self.transportModeTypes.forEach(value => {
			value.selected = [value.name, value.value].includes(self.taskDetail.forUI.transportMode);
		});
	}

	getSelectedCopy() {
		let self = this;
		if (self.storagePolicyCopyList) {
			angular.forEach(self.storagePolicyCopyList, function(copy) {
				copy.selected = copy.copyName === self.taskDetail.forUI.replicationCopy;
			});
		}
	}

	getSelectedSnapShotEngine(updatedSnapshot) {
		let self = this;
		if (updatedSnapshot) {
			self.taskDetail.snapToTapeSelectedEngine = updatedSnapshot;
		}

		if (self.snapEngineList) {
			angular.forEach(self.snapEngineList, function(snap) {
				snap.snapShotEngineId = snap.id;
				snap.snapShotEngineName = snap.name;
				snap.selected = snap.id === self.taskDetail.snapToTapeSelectedEngine.snapShotEngineId;
			});
		}
	}

	updateToolTips(optionId) {
		return this.toggleStatus[optionId] === enableActivity ? this.cvLoc('label.enabled') : this.cvLoc('label.disabled');
	}

	getReplicationGroupInfo() {
		let self = this;
		self.serverMessage = self.cvUtil.emptyMsg();
		self.enableRules = cvConfig.enableRulesForPlan;
		self.slaInfo = {};
		self.replicationGroupsDetailsLoaded = false;
		self.replicationGroupsDetailsFailed = false;
		self.showSummaryGrid = false;
		self.replicationGroupsService
			.getReplicationInfo(this.repGrpId)
			.success(function(data) {
				const type = _.get(data, 'replicationGroupDetails.type', PERIODIC);
				self.isContinuous = type !== PERIODIC;
				self.isPeriodic = type === PERIODIC;
				self.replicationGroupEntity = _.get(data, 'replicationGroupDetails.replicationGroup', {});
				self.replicationInfo = _.get(data, 'replicationInfo', {});
				self.siteInfo = _.get(data, 'replicationGroupDetails.siteInfo', []);
				self.showAdvOptionsContinuous =
					self.isContinuous && self.replicationGroupFactory.enableGuestCredentialsContinuous;
				self.backupSchedule = self.replicationInfo.schedule || '';

				// For Periodic, taskDetails in Replication Target under ReplicationInfo
				self.taskDetail = self.isPeriodic
					? _.get(data, 'replicationInfo.replicationTargets')
					: _.get(data, 'replicationGroupDetails.taskDetail');

				self.blrDetailInfo = {
					entity: {
						subclientId: self.taskDetail.subclientId
					},
					blrRecoveryOpts: _.get(data, 'replicationGroupDetails.blrRecoveryOpts', {})
				};

				self.taskDetail.forUI = {};
				self.taskDetail.blrRecoveryOpts = self.blrDetailInfo.blrRecoveryOpts;
				// For Periodic, taskInfo come with an array, for Continuous, it is an object
				const taskInfo = self.isPeriodic ? _.get(self.taskDetail, 'taskInfo[0]') : self.taskDetail;
				self.processCommonData(taskInfo);
				self.taskInfo = taskInfo;
				if (self.isPeriodic) {
					self.replicationInfo.showSelectCopy = true;
					self.storagePolicyCopyList = [];
					const copies = _.get(self.replicationInfo, 'storage.copy', null);
					if (copies) {
						copies.forEach(copy => {
							const storagePolicyCopy = copy.StoragePolicyCopy;
							storagePolicyCopy.name = storagePolicyCopy.copyName;
							storagePolicyCopy.id = storagePolicyCopy.copyId;
							if (copy.isSnapCopy !== SET_TRUE) {
								self.storagePolicyCopyList.push(storagePolicyCopy);
							}
						});
					}

					self.processPeriodicData();
				} else if (self.isContinuous) {
					self.checkSiteInfosActionsStatus(self.siteInfo);
					self.processContinousData();
					self.replicationGroupsDetailsLoaded = true;
					self._handleGridVisibility();
					self.taskDetail.forUI.vmGuestUserName = self._getVMGuestUserName(data);
				}

				self.taskDetail.forUI.replicationGroupId = _.get(self, 'replicationGroupEntity.replicationGroupId', 0);
				// VSA type = 106
				self.taskDetail.forUI.appTypeId = 106;
				self.taskDetail.id = parseInt(self.taskDetail.forUI.replicationGroupId);
				self.taskDetail.replicationGroupName = self.taskDetail.forUI.replicationGroupName;
				self.taskDetail.name = self.taskDetail.forUI.replicationGroupName;
				self.taskDetail.forUI.replicationType = self.isPeriodic
					? self.cvLoc('label.periodic')
					: self.cvLoc('label.continuous');
				if (self.isPeriodic) {
					self.getLiveSyncInfo();
				} else {
					self.replicationGroupDetailLoaded();
				}
				self.initializeTabs();
			})
			.error(function(err) {
				self.onError(err);
				self.replicationGroupsDetailsLoaded = true;
				self.replicationGroupsDetailsFailed = true;
				self._handleGridVisibility();
				self.serverMessage = self.cvUtil.errMsg(err ? err : self.cvLoc('generic_error'));
			});
	}

	replicationGroupDetailLoaded() {
		let self = this;
		if (self.ctrl && self.ctrl.replicationGroupLoaded) {
			self.ctrl.replicationGroupLoaded(self.taskDetail);
		}
	}

	getLiveSyncInfo() {
		let self = this;

		self.restoreForAgents
			.getLiveSyncInfo({
				appTypeId: self.taskDetail.forUI.appTypeId,
				taskId: self.taskId
			})
			.then(
				data => self._onRetrieveLiveSyncInfo(data),
				response => {
					const error = _.get(response, 'data') || self.cvLoc('generic_error');
					self.onError(error);
					self.replicationGroupDetailLoaded();
				}
			);
	}

	_onRetrieveLiveSyncInfo(data) {
		let self = this;
		const siteInfos = _.get(data, 'data.siteInfo', []);
		self.checkSiteInfosActionsStatus(siteInfos);
		self.replicationGroupDetailLoaded();
	}

	checkSiteInfosActionsStatus(siteInfos) {
		let self = this;
		const { taskDetail, BLR_CONSTANTS, blrUIFactory } = self;
		const { forUI } = taskDetail;

		if (_.isEmpty(siteInfos)) {
			return;
		}

		taskDetail.siteInfoGuid = {};
		siteInfos.forEach(siteInfo => {
			siteInfo.FailoverStatus = siteInfo.failoverStatus;
			if (siteInfo.sourceGuid) {
				taskDetail.siteInfoGuid[siteInfo.sourceGuid.toLowerCase()] = siteInfo;
			}

			if (self.isPeriodic) {
				self.vmGroupsFactory.showActionsAvailable(siteInfo);
			} else {
				if (blrUIFactory.isActionAllowed(BLR_CONSTANTS.ACTIONS.EDIT_RECOVERY_OPTIONS, siteInfo)) {
					siteInfo.showEditRecoveryOptions = true;
				}

				if (blrUIFactory.isActionAllowed(BLR_CONSTANTS.ACTIONS.FAILBACK, siteInfo)) {
					siteInfo.showFailback = true;
				}

				if (blrUIFactory.isActionAllowed(BLR_CONSTANTS.ACTIONS.FAILOVER, siteInfo)) {
					siteInfo.showFailover = true;
				}

				siteInfo.showResume = !!blrUIFactory.isActionAllowed(BLR_CONSTANTS.ACTIONS.RESUME, siteInfo);
			}

			if (siteInfo.showEditRecoveryOptions) {
				forUI.showEditRecoveryOptions = true;
			}

			if (siteInfo.showFailover) {
				forUI.showFailover = true;
			}

			if (siteInfo.showFailback) {
				forUI.showFailback = true;
			}
			forUI.showResume = !!siteInfo.showResume;
		});
	}

	openEditRecoveryOptionsModal() {
		this.blrDetailInfo.replicationGroup = this.replicationGroupEntity;
		this.blrDetailInfo.isRepGroupLevel = true;

		this.blrUIFactory.openEditRecoveryOptionsModal(this.blrDetailInfo);
	}

	getSnapshotList() {
		let self = this;
		self.snapEngineList = [
			{
				name: self.cvLoc('Loading'),
				selected: true
			}
		];

		const { subclientService, cvLoc, cvToaster, taskDetail, editMode, onError } = self;
		editMode.snapShotEngine.isLoading = true;
		subclientService
			.getSnapShotEngines(taskDetail.subclientId)
			.success(data => {
				editMode.snapShotEngine.isLoading = false;
				if (data && data.length) {
					self.snapEngineList = data;

					/* Remove the AWS snap */
					const AWSSnapID = 60;
					let awsndex = data.findIndex(snap => snap.id === AWSSnapID);
					self.snapEngineList.splice(awsndex, 1);

					/* Remove the VSA Agent snap */
					const VirtualServerAgentSnapID = 58;
					let vsaIndex = data.findIndex(snap => snap.id === VirtualServerAgentSnapID);
					self.snapEngineList.splice(vsaIndex, 1);
					self.getSelectedSnapShotEngine();
				} else {
					// When there is no data, show dropdown with selected value
					self.snapEngineList = [taskDetail.snapToTapeSelectedEngine];
					return;
				}
			})
			.error(err => {
				editMode.snapShotEngine.isLoading = false;
				// When there is error, show dropdown with selected value
				self.snapEngineList = [taskDetail.snapToTapeSelectedEngine];
				onError(err);
			});
	}

	getSubclientInfo() {
		let self = this;
		const { subclientService, cvLoc, cvToaster, taskDetail, onError } = self;
		subclientService
			.getSubClientDetails(taskDetail.subclientId)
			.success(data => {
				if (!data) {
					return;
				}
				self.subClientInfo = data;
				const snapCopyInfo = _.get(data, 'commonProperties.snapCopyInfo', null);

				if (!snapCopyInfo) {
					taskDetail.forUI.showSnapshotEngine = false;
					return;
				}

				const snapToTapeSelectedEngine = _.get(snapCopyInfo, 'snapToTapeSelectedEngine', {});
				snapToTapeSelectedEngine.name = snapToTapeSelectedEngine.snapShotEngineName;
				snapToTapeSelectedEngine.id = snapToTapeSelectedEngine.snapShotEngineId;
				snapToTapeSelectedEngine.selected = true;
				taskDetail.snapToTapeSelectedEngine = snapToTapeSelectedEngine;
				taskDetail.forUI.showSnapshotEngine = snapCopyInfo.isSnapBackupEnabled;
				taskDetail.forUI.snapShotEngine = _.get(snapToTapeSelectedEngine, 'snapShotEngineName', '');
				/*
				 * DO not allow editing of snapshot engines and adding secondary copy for snap enabled
				 * periodic groups
				 */
				self.showSnapEdit = !taskDetail.forUI.showSnapshotEngine;
				self.showEdit = !taskDetail.forUI.showSnapshotEngine;

				/*
				 * Snapshot to retain count is applicable only for destination VMWare Periodic group that do
				 * not have snap engines(Direct groups)
				 */
				if (
					self.isPeriodic &&
					self._isVMWareVendor(taskDetail.destinationVendor) &&
					!snapCopyInfo.isSnapBackupEnabled
				) {
					taskDetail.forUI.showMaxSnapshot = true;
					const maxSnapshotDRVM = _.get(taskDetail.forUI.advancedRstOptions, '[0].nMaxSnapshotsPerDRVM', 0);
					taskDetail.forUI.nMaxSnapshotsPerDRVM = maxSnapshotDRVM;
					taskDetail.nMaxSnapshotsPerDRVM = maxSnapshotDRVM;
				}
				self.selectedContent = {
					content: _.get(data, 'vmContent.children')
				};
				self.collectionFactory.setCollectionContent(self.selectedContent);
				self.collectionFactory.setInstanceId(self.taskDetail.instanceId);
				self.collectionFactory.setApplicationId(106);
				self.collectionFactory.setBackupSetId(0);
				this._handleGridVisibility();
			})
			.error(err => {
				onError(err);
			});
	}

	getRPO() {
		let self = this;
		const { summary } = self.replicationInfo;
		self.profileId = _.get(summary, 'plan.planId', 0);
		self.planSubType = summary.subtype;
		self.planType = summary.type;
		self.showEdit = true;
		self.slaInfo = {};
		if (summary.rpoInMinutes || summary.rpoInMinutes === 0) {
			self.slaInfo.rpoHours =
				summary.rpoInMinutes % 60 == 0 ? summary.rpoInMinutes / 60 - ((summary.rpoInMinutes / 60) % 1) : null;
			self.slaInfo.rpoInMinutes = summary.rpoInMinutes;
		}
	}

	getDefaultUIValue() {
		let self = this;
		const { forUI } = self.taskDetail;
		const NOT_SET = self.cvLoc('label.notSet');

		forUI.clientName = NOT_SET;
		forUI.proxy = NOT_SET;
		forUI.vendorType = NOT_SET;
		forUI.destinationName = NOT_SET;
		forUI.destinationId = null;
		forUI.destinationLabel = null;
	}

	processCommonData(taskInfo) {
		let self = this;
		self.getDefaultUIValue();

		if (!taskInfo) {
			return;
		}

		const { taskDetail, cvLoc, cvUtil } = self;
		const { forUI } = taskDetail;
		const associations = _.get(taskInfo, 'associations[0]', null);
		const proxy = _.get(taskInfo, 'subTasks[0].options.restoreOptions.destination.destClient', null);
		const vsRstOptions = _.get(taskInfo, 'subTasks[0].options.restoreOptions.virtualServerRstOption', null);
		taskDetail.forUI.replicationGroupName = _.get(
			self,
			'replicationGroupEntity.replicationGroupName',
			self.cvLoc('label.none')
		);

		self.taskId = _.get(taskInfo, 'task.task.taskId', _.get(self.$stateParams, 'id', 0));
		forUI.showRepGrpStatus = self.isPeriodic;
		self.getRPO();
		if (associations) {
			var applicationName = associations.appName;
			forUI.appName = applicationName;

			// Source Hypervisor
			taskDetail.forUI.clientName = associations.clientName;
			taskDetail.forUI.clientId = associations.clientId;
			taskDetail.subclientId = associations.subclientId;
			taskDetail.sourceVendor = associations.instanceName;
			taskDetail.instanceId = associations.instanceId;

			self.replicationGroupFactory.setSubclientEntity({
				subclientId: associations.subclientId,
				subclientName: associations.subclientName
			});
		}

		if (proxy) {
			forUI.proxy = proxy.clientName;
			forUI.accessNode = taskDetail.forUI.proxy;
			forUI.selectedProxy = [proxy]; // For isteven select proxy
		}

		if (vsRstOptions) {
			const vmAllocPolicyId = _.get(vsRstOptions, 'allocationPolicy.vmAllocPolicyId', null);
			const vCenterClientName = _.get(vsRstOptions, 'vCenterInstance.clientName', null);
			if (vmAllocPolicyId) {
				self.isRecoveryTarget = true; //permission error shown for end user only if destination is recovery target
				taskDetail.forUI.destinationName = _.get(vsRstOptions, 'allocationPolicy.vmAllocPolicyName', null);
				taskDetail.forUI.destinationId = vmAllocPolicyId;
				taskDetail.forUI.destinationUrl = '#/lifeCyclePolicyDetails/' + taskDetail.forUI.destinationId;
				taskDetail.forUI.destinationLabel = cvLoc('label.replicationTarget');
			} else if (vCenterClientName) {
				self.isRecoveryTarget = false;
				taskDetail.forUI.destinationName = vCenterClientName;
				taskDetail.forUI.destinationId = _.get(vsRstOptions, 'vCenterInstance.clientId', null);
				taskDetail.forUI.destinationUrl = '#/clientDetails/' + taskDetail.forUI.destinationId;
			}

			let policyType = _.get(vsRstOptions, 'allocationPolicy.policyType', null);
			if (policyType == 'VMW_LIVEMOUNT' || policyType == 'VMW_BACKUP_LABTEMPLATE' || policyType == 'VMW_LABTEMPLATE') {
				policyType = 'VMWARE';
			}

			let destinationHyperV = {
				clientId: _.get(taskDetail, 'forUI.destinationId', 0)
			};

			if (self.ctrl && self.ctrl.loadDestinationProxies) {
				self.ctrl.loadDestinationProxies(policyType, destinationHyperV);
			}

			taskDetail.destinationVendor = policyType || _.get(vsRstOptions, 'vCenterInstance.instanceName', '');
			taskDetail.forUI.vendorType = policyType
				? cvUtil.lookupEnumConstant(taskDetail.destinationVendor)
				: taskDetail.destinationVendor;

			if (self.VENDORS['AZURE_V2'].policyTypes.includes(taskDetail.destinationVendor)) {
				taskDetail.forUI.vendorType = self.VENDORS['AZURE_V2'].text;
			}

			if (cv.userRole === ROLE_MSP_USER || cv.userRole === ROLE_TENANT_USER) {
				self.isEndUser = true;
			}

			taskDetail.forUI.advancedRstOptions = _.get(vsRstOptions, 'diskLevelVMRestoreOption.advancedRestoreOptions', []);
			self._handleGridVisibility();
		}
	}

	processContinousData() {
		let self = this;
		const { forUI } = self.taskDetail;
		forUI.isDisabled = !!_.get(self.siteInfo, '[0].flags');

		if (!_.isEmpty(self.blrDetailInfo.blrRecoveryOpts)) {
			if (!self.blrDetailInfo.additionalInfo) {
				self.blrDetailInfo.additionalInfo = self.blrUIFactory.setDefaultBlrAdditionalInfo();
			}

			self.blrDetailInfo = self.blrUIFactory.setBlrInfo(self.blrDetailInfo);
		}

		forUI.advancedRstOptions = forUI.advancedRstOptions.filter(option => {
			const siteInfoCopy = self.taskDetail.siteInfoGuid[option.guid];
			return !_.get(siteInfoCopy, 'isMarkedForDeletion');
		});
		self._handleGridVisibility();
	}

	processPeriodicData() {
		let self = this;
		const { taskDetail, cvLoc } = self;
		const { forUI } = taskDetail;
		const taskInfo = _.get(taskDetail, 'taskInfo[0]', null);
		const associations = _.get(taskInfo, 'associations[0]', null);
		const proxy = _.get(taskInfo, 'subTasks[0].options.restoreOptions.destination.destClient', null);
		const vsRstOptions = _.get(taskInfo, 'subTasks[0].options.restoreOptions.virtualServerRstOption', null);
		const storagePolicy = _.get(taskInfo, 'subTasks[0].options.restoreOptions.storagePolicy', null);
		taskDetail.taskInfo[0].replicationGroupName = taskDetail.forUI.replicationGroupName;
		taskDetail.forUI.replicationSchedule = _.get(taskInfo, 'subTasks[0].pattern.description', '');
		forUI.isDisabled = _.get(taskInfo, 'task.taskFlags.disabled', null);

		self.getRPO();
		self.getSubclientInfo();
		if (vsRstOptions) {
			if (!!vsRstOptions.diskLevelVMRestoreOption) {
				taskDetail.forUI.showPassUnconditionalOverwrite = angular.isDefined(
					vsRstOptions.diskLevelVMRestoreOption.passUnconditionalOverride
				);
				self.onToggleSuccess(
					passUnconditionalOverride,
					vsRstOptions.diskLevelVMRestoreOption.passUnconditionalOverride
				);

				let showDeployVmWhenFailoverSupportedVendors = [
					this.VENDORS.AZURE_V2.policyType,
					this.VENDORS.GOOGLE_CLOUD.policyType,
					this.VENDORS.AZURE_STACK.policyType
				];

				taskDetail.forUI.powerOnVmAfterRestore = vsRstOptions.diskLevelVMRestoreOption.powerOnVmAfterRestore;
				self.onToggleSuccess(powerOnVmAfterRestore, vsRstOptions.diskLevelVMRestoreOption.powerOnVmAfterRestore);
				const policyType = _.get(vsRstOptions, 'allocationPolicy.policyType', null);
				if (policyType) {
					self.showPowerOnOption =
						policyType !== 'AZURE_RESOURCE_MANAGER' &&
						policyType !== 'ORACLE_CLOUD_INFRASTRUCTURE' &&
						policyType !== 'GOOGLE_CLOUD' &&
						policyType !== 'OPENSTACK' &&
						policyType !== 'AZURE_STACK';

					self.showDeployVmWhenFailover = showDeployVmWhenFailoverSupportedVendors.includes(policyType);
				} else {
					self.showPowerOnOption = !!vsRstOptions.diskLevelVMRestoreOption.powerOnVmAfterRestore;
					self.showDeployVmWhenFailover = !!vsRstOptions.diskLevelVMRestoreOption.deployVmWhenFailover;
				}
				//deployVmWhenFailover supported only for azurerm, stack
				var deployVmWhenFailoverState = vsRstOptions.diskLevelVMRestoreOption.deployVmWhenFailover;

				taskDetail.forUI.deployVmWhenFailover = deployVmWhenFailoverState;
				self.onToggleSuccess(deployVmWhenFailover, deployVmWhenFailoverState);
				taskDetail.forUI.diskOption = vsRstOptions.diskLevelVMRestoreOption.diskOption;
				taskDetail.forUI.transportMode = vsRstOptions.diskLevelVMRestoreOption.transportMode;

				self.getSelectedTransportMode();
			}
		}
		self.getSelectedCopy();
		if (storagePolicy) {
			forUI.replicationCopy = storagePolicy.copyName;
		}

		taskDetail.taskInfo[0].subTasks.copyPrecedence = _.get(
			taskInfo,
			'subTasks[0].options.restoreOptions.browseOption.mediaOption.copyPrecedence.copyPrecedenceApplicable',
			null
		);
		taskDetail.useAllocationPolicyForRestore = _.get(
			taskInfo,
			'subTasks[0].options.restoreOptions.useAllocationPolicyForRestore',
			false
		);

		/* Guest credentials should be shown for Non AWS - AWS replication and if the proxy is Amazon */
		if (
			!self._isAmazonVendor(self.taskDetail.sourceVendor) &&
			self._isAmazonVendor(self.taskDetail.destinationVendor)
		) {
			self._getAmazonProxies().then(function(proxies) {
				self.replicationGroupsDetailsLoaded = true;
				self.taskDetail.taskInfo[0].isAmazonProxy = self._isAmazonProxy(proxies, proxy);
				self.taskDetail.taskInfo[0].showGuestCredentials = self.taskDetail.taskInfo[0].isAmazonProxy;
				self._handleGridVisibility();
			});
		} else {
			self.replicationGroupsDetailsLoaded = true;
			self._handleGridVisibility();
		}
	}

	// when update is failed or cancel, revert the data to the original one;
	getOriginalDiskLevelVMOption() {
		return angular.copy(
			_.get(
				this.taskDetail,
				'taskInfo[0].subTasks[0].options.restoreOptions.virtualServerRstOption.diskLevelVMRestoreOption',
				{}
			)
		);
	}

	// Click the toggle
	toggleControlClick(optionId) {
		let self = this;
		let taskInfo = angular.copy(self.taskDetail.taskInfo[0]);
		let diskLevelVMRestoreOption = self.getOriginalDiskLevelVMOption();

		if (!optionId || !diskLevelVMRestoreOption) {
			return;
		}

		self.editMode[optionId] = true; // Disable the click to prevent multiple clicks
		const successCB = () => {
			this.updateAdvanceOptionSuccess(diskLevelVMRestoreOption);
			this.onToggleSuccess(optionId, diskLevelVMRestoreOption[optionId]);
			this.editMode[optionId] = false;
		};

		const failureCB = () => (self.editMode[optionId] = false);

		switch (optionId) {
			case powerOnVmAfterRestore:
			case deployVmWhenFailover:
			case passUnconditionalOverride:
				diskLevelVMRestoreOption[optionId] = !diskLevelVMRestoreOption[optionId];
				break;
			default:
				return;
				break;
		}

		taskInfo.subTasks[0].options.restoreOptions.virtualServerRstOption.diskLevelVMRestoreOption = diskLevelVMRestoreOption;
		self.saveUpdate(taskInfo, successCB, failureCB);
	}

	turnOffEdit(optionId) {
		this.editMode[optionId].isLoading = false;
		this.editMode[optionId].inEditMode = false;
	}

	resetError() {
		this.nMaxSnapshotsError = this.cvUtil.emptyMsg();
	}

	updateSelectedValues(selectedValues, optionId, revertData, isCancel) {
		let self = this;
		const diskLevelVMRestoreOption = self.getOriginalDiskLevelVMOption();
		let taskInfo = angular.copy(self.taskDetail.taskInfo[0]);
		const restoreOption = _.get(self.taskDetail, 'taskInfo[0].subTasks[0].options.restoreOptions');
		let preSeletedSnap = null;
		const successCB = () => {
			self.updateAdvanceOptionSuccess(diskLevelVMRestoreOption);
			self.turnOffEdit(optionId);
			self.taskDetail.forUI[optionId] = selectedValueName || selectedValue; // update UI value

			/*
			 * If snapshots is greater than 0, then reset the value of transport mode to 'Auto' if 'SAN' was
			 * the earlier selected value
			 */
			if (optionId === nMaxSnapshotsPerDRVM && selectedValues > 0 && self._isSANTransportModeSelectedinUI()) {
				self.taskDetail.forUI.transportMode = self.collectionFactory.getAutoTransportModeType().name;
			}
			self.onSuccess(self.cvLoc('msg.updated'));
		};

		const failureCB = () => {
			if (preSeletedSnap) {
				self.taskDetail.snapToTapeSelectedEngine = preSeletedSnap;
			}
			self.editMode[optionId].isLoading = false;
		};

		// When there is error , display error and not do any thing when click confirm button
		if (!optionId || (self.editMode[optionId].errorExisted && !isCancel)) {
			return;
		}

		self.editMode[optionId].isLoading = true;
		const selectedValue = _.get(selectedValues, '[0]', selectedValues); // single mod, take first item
		const selectedValueName = _.get(selectedValue, 'name', null);
		if (
			_.isNil(selectedValue) ||
			(selectedValueName && self.taskDetail.forUI[optionId] === selectedValueName) ||
			revertData === selectedValue
		) {
			// Cancel clicked
			if (!_.isNil(revertData)) {
				self.taskDetail.forUI[optionId] = revertData;
			}
			self.editMode[optionId].errorExisted = false; // reset error status
			self.resetError();

			self.turnOffEdit(optionId);
			return;
		}

		switch (optionId) {
			case transportMode:
				diskLevelVMRestoreOption.transportMode = parseInt(
					self.cvUtil.lookupEnumConstant(_.get(selectedValue, 'value', '').toUpperCase(), 'value')
				);

				break;
			case accessNode:
				restoreOption.destination.destClient.clientName = selectedValue.clientName;
				restoreOption.destination.destClient.clientId = selectedValue.clientId;
				selectedValue.name = selectedValue.clientName; //update for istEvenUI
				break;
			case replicationCopy:
				restoreOption.storagePolicy = selectedValue;
				break;
			case snapShotEngine:
				preSeletedSnap = _.cloneDeep(self.taskDetail.snapToTapeSelectedEngine);
				self.taskDetail.snapToTapeSelectedEngine = selectedValue;
				self.subClientInfo.commonProperties.snapCopyInfo.snapToTapeSelectedEngine =
					self.taskDetail.snapToTapeSelectedEngine;
				self.saveSnapEngine(successCB, failureCB);
				return;
				break;
			case nMaxSnapshotsPerDRVM:
				const advancedRstOptions = _.get(diskLevelVMRestoreOption, 'advancedRestoreOptions', []);
				advancedRstOptions.forEach(option => {
					option.nMaxSnapshotsPerDRVM = selectedValues;
				});
				/*
				 * If snapshots is greater than 0, then reset the value of transport mode to 'Auto' if 'SAN'
				 * was the earlier selected value
				 */
				if (selectedValues > 0 && this._isSANTransportModeSelected()) {
					diskLevelVMRestoreOption.transportMode = 0;
				}
				break;
			default:
				break;
		}

		taskInfo.subTasks[0].options.restoreOptions = restoreOption;
		taskInfo.subTasks[0].options.restoreOptions.virtualServerRstOption.diskLevelVMRestoreOption = diskLevelVMRestoreOption;
		self.saveUpdate(taskInfo, successCB, failureCB);
	}

	saveSnapEngine(successCB, failureCB) {
		let self = this;
		const subClientInfo = {};
		const snapCopyInfo = _.get(self.subClientInfo, 'commonProperties.snapCopyInfo', {});
		const subClientEntity = _.get(self.subClientInfo, 'subClientEntity', {});
		subClientInfo.commonProperties = {
			snapCopyInfo
		};

		subClientInfo.vsaSubclientProp = {
			proxyESXServer: {
				snapFailoverESXHosts: [snapCopyInfo.proxyESXHost],
				vsInstance: {
					clientId: subClientEntity.clientId,
					instanceId: subClientEntity.instanceId,
					applicationId: 106,
					appName: 'Virtual Server'
				}
			}
		};

		self.subclientFactory.updateSubclient(subClientEntity, subClientInfo).then(
			data => {
				if (successCB) {
					successCB();
				}
			},
			err => {
				const errMsg = _.get(err, 'data', '');
				self.onError(errMsg);
				if (failureCB) {
					failureCB();
				}
			}
		);
	}
	// when it succeed, update the data, no need to reload page;
	updateAdvanceOptionSuccess(diskLevelVMRestoreOption) {
		let self = this;
		self.taskDetail.taskInfo[0].subTasks[0].options.restoreOptions.virtualServerRstOption.diskLevelVMRestoreOption = diskLevelVMRestoreOption;
	}

	saveUpdate(taskInfo, successCallBack, failCallback) {
		if (!taskInfo) {
			return;
		}
		let self = this;
		self.restoreService
			.updateAdvanceOptionsRepGrp({ taskInfo })
			.success(data => {
				if (successCallBack) {
					successCallBack();
				}
			})
			.error(err => {
				if (failCallback) {
					failCallback();
				}
				self.onError(err);
			});
	}

	toggleEdit(type) {
		if (!type) {
			return;
		}
		let self = this;
		self.resetError();

		// When toggle Edit, reset the selected value to the correct value from backend
		switch (type) {
			case transportMode:
				self.getSelectedTransportMode();
				break;
			case replicationCopy:
				self.getSelectedCopy();
				break;
			case snapShotEngine:
				self.getSnapshotList();
				break;
			default:
				break;
		}

		self.editMode[type].inEditMode = true;
		self.editMode[type].errorExisted = false;
	}

	enableDisableReplicationGroup() {
		let self = this;
		let skipCallback = true;
		this.editMode.enableDisable = true;
		this.replicationGroupFactory.enableDisableReplicationGroups(this.taskDetail, skipCallback).then(
			function(data) {
				self.editMode.enableDisable = false;
				if (data) {
					self.taskDetail.forUI.isDisabled = !self.taskDetail.forUI.isDisabled;
					let successMessage = self.taskDetail.forUI.isDisabled
						? self.cvLoc('notification.replication.VSAREP_DISABLED')
						: self.cvLoc('notification.replication.VSAREP_ENABLED');
					self.onSuccess(successMessage);
				}
			},
			function(data) {
				self.editMode.enableDisable = false;
			}
		);
	}

	onError(e) {
		let self = this;
		self.cvToaster.showErrorMessage({
			ttl: '10000', //10 sec
			message: e ? e : self.cvLoc('generic_error')
		});
	}

	onSuccess(msg) {
		let self = this;
		if (!msg) {
			return;
		}

		self.cvToaster.showSuccessMessage({
			ttl: '10000', //10 sec
			message: msg
		});
	}

	/*
	 * For PIT replications, SAN transport mode needs backend fix and the fix is not straight forward.
	 * Removing it from the UI, till the backend can handle it.
	 */
	_getTransportModeTypes() {
		const nMaxSnapshotsPerDRVMValue = _.get(this, 'taskDetail.forUI.nMaxSnapshotsPerDRVM', null);
		if (nMaxSnapshotsPerDRVMValue > 0) {
			let sanTransportMode = this.collectionFactory.getSANTransportModeType() || {};
			this.collectionFactory.splice(this.transportModeTypes, sanTransportMode.id);
		} else {
			this.transportModeTypes = this.collectionFactory.getTransportModeTypes();
		}
	}

	/*
	 * If the value of the snapshots is greater than 0 and the tranport mode is 'SAN', confirm the change from
	 * the user
	 */
	_confirmChangeforSAN(nMaxSnapshotsPerDRVM) {
		if (nMaxSnapshotsPerDRVM > 0) {
			if (this._isSANTransportModeSelected()) {
				this.$dialogs.confirm(this.cvLoc('label.confirmAction'), this.cvLoc('prompt.confirmSANModeChange'), {
					noFunction: () => {
						this.taskDetail.forUI.nMaxSnapshotsPerDRVM = this.taskDetail.nMaxSnapshotsPerDRVM;
					},
					yesFunction: () => {}
				});
			}
		}
	}

	/*
	 * Checks if the currently selected transport mode is SAN in the taskDetail
	 */
	_isSANTransportModeSelected() {
		const sanTransportMode = this.collectionFactory.getSANTransportModeType() || {};
		const transportModeValue = _.get(
			this,
			'taskDetail.taskInfo[0].subTasks[0].options.restoreOptions.virtualServerRstOption.diskLevelVMRestoreOption.transportMode',
			''
		);
		return transportModeValue == sanTransportMode.id || transportModeValue == sanTransportMode.name;
	}

	/*
	 * Checks if the currently selected transport mode is SAN in forUI object
	 */
	_isSANTransportModeSelectedinUI() {
		const sanTransportMode = this.collectionFactory.getSANTransportModeType() || {};
		const transportModeValue = _.get(this, 'taskDetail.forUI.transportMode', '');
		return transportModeValue == sanTransportMode.id || transportModeValue == sanTransportMode.name;
	}

	_getAmazonProxies() {
		return this.serverService.getProxyServers(this.VENDORS.AMAZON.type).then(
			function(data) {
				return _.get(data, 'data', []);
			},
			function(err) {
				return [];
			}
		);
	}

	_isAmazonProxy(amazonProxyServersList, proxyServer) {
		if (_.isEmpty(amazonProxyServersList) || _.isEmpty(proxyServer)) {
			return false;
		}
		return amazonProxyServersList.findIndex(proxy => proxy.clientId === proxyServer.clientId) >= 0;
	}

	_isAmazonVendor(vendor) {
		return vendor && this.VENDORS.AMAZON.type.toLowerCase() === vendor.toLowerCase();
	}

	_isVMWareVendor(vendor) {
		let VMW = this.VENDORS.VMW;
		return (vendor && VMW.type === vendor) || VMW.text.toLowerCase() === vendor.toLowerCase();
	}

	_handleGridVisibility() {
		let selectedContent = _.get(this, 'selectedContent.content', []);
		this.showRulesGrid = selectedContent.findIndex(content => content.type != 'VM') >= 0;
		this.showSummaryGrid =
			this.replicationGroupsDetailsLoaded && this.taskDetail.forUI.advancedRstOptions && !this.showRulesGrid;
	}

	_saveNewRule() {
		return this._saveSubClientContent();
	}

	_onDeleteSubclientContent() {
		return this._saveSubClientContent();
	}

	_onAddVMinRuleEnabledGroup() {
		return this._saveSubClientContent();
	}

	_saveSubClientContent() {
		let self = this;
		return this.serverService
			.modifyCollectionContent({
				name: _.get(this, 'subClientInfo.subClientEntity.subclientName'),
				collectionId: _.get(this, 'subClientInfo.subClientEntity.subclientId'),
				collectionContentJson: JSON.stringify(this.collectionFactory.getCollectionContent())
			})
			.success(function(wr) {
				self.onSuccess(self.cvLoc('msg.updated'));
				return true;
			})
			.error(function(data, status) {
				self.onError(data);
				return false;
			});
	}

	_getVMGuestUserName(data) {
		let vmGuestCredentials = _.get(data, 'replicationGroupDetails.vmGuestCredentials', []);
		return _.get(
			vmGuestCredentials,
			'credentialEntity.credentialName',
			_.get(vmGuestCredentials, 'userPassword.userName', '')
		);
	}
}

ReplicationGroupVSADirectiveController.$inject = [
	'$scope',
	'$state',
	'$dialogs',
	'AppTypes',
	'cvLoc',
	'cvUtil',
	'$log',
	'replicationGroupsService',
	'collectionFactory',
	'cvToaster',
	'VENDORS',
	'restoreOptionFactory',
	'replicationGroupFactory',
	'$stateParams',
	'tabService',
	'replicationFactory',
	'restoreService',
	'subclientService',
	'subclientFactory',
	'serverService',
	'vmGroupsFactory',
	'restoreForAgents',
	'blrUIFactory',
	'BLR_CONSTANTS'
];

vsaAppServerModule.directive('cvRepGroupVsa', function() {
	return {
		restrict: 'E',
		templateUrl: appUtil.appRoot + 'modules/disasterRecovery/repGrp/partials/vsa/repGrpVSA.jsp',
		scope: {},
		bindToController: {
			ctrl: '='
		},
		controllerAs: 'repGroupDetail',
		controller: ReplicationGroupVSADirectiveController
	};
});

export default vsaAppServerModule;
