import 'modules/disasterRecovery/js/directives/dr-machines.directive.js';
import 'modules/disasterRecovery/js/dr.constants.js';
import 'modules/disasterRecovery/js/services/failoverGroup/failoverGroup.svc.js';
import * as machinesColumnsTemplate from 'modules/disasterRecovery/js/columns/drArrayReplicationMachines.column.template.js';

import { drAppFailoverModule } from 'common/js/modules';

/**
 * A factory used for snap array replication
 */

export const arrayReplicationFactory = drAppFailoverModule.factory('arrayReplicationFactory', [
	'cvLoc',
	'cvUtil',
	'$log',
	'failoverService',
	'FailoverConstants',
	'$uibModal',
	function(cvLoc, cvUtil, $log, failoverService, FailoverConstants, $modal) {
		var factory = {};

		factory.getDevtestColumnsTemplate = function() {
			return machinesColumnsTemplate.getDevtestColumns(cvLoc);
		};

		factory.getFailoverColumns = function() {
			return machinesColumnsTemplate.getFailoverColumns(cvLoc);
		};

		factory.getFailoverDetailsColumns = function() {
			return machinesColumnsTemplate.getFailoverDetailsColumns(cvLoc);
		};

		factory.getTestFailoverDetailsColumns = function() {
			return machinesColumnsTemplate.getTestFailoverDetailsColumns(cvLoc);
		};

		factory.getSelectMachinesColumns = function() {
			return machinesColumnsTemplate.getSelectMachinesColumns(cvLoc);
		};

		factory.getArrayMachines = function(subclientId, copyId, clientId) {
			return failoverService.getArrayEntities(subclientId, copyId, clientId).then(
				function(machines) {
					machines = machines.data || [];
					let datastoreMap = _.get(machines, 'datastoreMap', []);
					machines = processMachinesData(datastoreMap);
					return machines;
				},
				function(err) {
					$log.error('Error fetching the machines data');
					return _.get(err, 'data');
				}
			);
		};

		/*
		 * If clientId is sent in the request of ArrayEntities, the response contains the copy, machines and datatores.
		 */
		factory.getFailoverCopies = function(subclientId) {
			return failoverService.getArrayEntities(subclientId, 0, 0).then(
				function(machines) {
					let datastoreMap = _.get(machines, 'data.datastoreMap', []);
					let copyMap = new Map();
					copyMap = convertDSMaptoCopyMap(datastoreMap);
					copyMap.forEach((value, key, map) => {
						map.set(key, processMachinesData(value));
					});
					return copyMap;
				},
				function(err) {
					$log.error('Error fetching the machines data');
					return _.get(err, 'data');
				}
			);
		};

		/*
		 * If clientId is sent in the request of ArrayEntities, the response contains the copy, machines and datatores.
		 */
		factory.getTestFailoverCopies = function(clientId) {
			return failoverService.getArrayEntities(0, 0, clientId).then(
				function(machines) {
					let datastoreMap = _.get(machines, 'data.datastoreMap', []);
					let copyMap = new Map();
					copyMap = convertDSMaptoCopyMap(datastoreMap);
					copyMap.forEach((value, key, map) => {
						map.set(key, processMachinesData(value));
					});
					return copyMap;
				},
				function(err) {
					$log.error('Error fetching the machines data');
					return _.get(err, 'data');
				}
			);
		};

		factory.getArrayDatastores = function(subclientId, copyId) {
			return failoverService.getArrayEntities(subclientId, copyId).then(
				function(entities) {
					entities = _.get(entities, 'data.datastoreMap', []);
					return removeDuplicateMachines(entities);
				},
				function(err) {
					$log.error('Error fetching the array datastores data');
					return _.get(err, 'data');
				}
			);
		};

		factory.getSnapDRReplicationStatus = function(vAppId) {
			return failoverService.getSnapDRReplicationStatus(vAppId).then(
				function(siteInfo) {
					siteInfo = _.get(siteInfo, 'data.siteInfo', []);
					siteInfo.forEach(siteInfo => {
						getDatastoreLabelForSiteInfo(siteInfo);
						siteInfo.failoverStatusText =
							siteInfo.failoverStatus === 'VSAREP_NONE' ? '' : cvUtil.lookupEnumConstant(siteInfo.failoverStatus);
						siteInfo.testFailoverStatusText =
							siteInfo.testFailoverStatus === FailoverConstants.DRTestFailoverStatus.DR_TEST_FAILOVER_NONE
								? ''
								: cvUtil.lookupEnumConstant(siteInfo.testFailoverStatus);
					});
					siteInfo = _.sortBy(siteInfo, [site => site.sourceName]);
					return siteInfo;
				},
				function(err) {
					$log.error('Error fetching the snap dr replication status data');
					return _.get(err, 'data');
				}
			);
		};

		/*
		 * This function adds the following values to each machine for the edit operation
		 * 	- ReplicationId
		 * 	- Priority
		 * 	- IP
		 * 	- Destination computer name
		 */
		factory.addValuesForEdit = function(machinesList, vApp) {
			// Move the priority to each vmSequnce from vmGroup
			if (vApp.config && vApp.config.vmGroups) {
				vApp.config.vmGroups.forEach(vmGroup => {
					vmGroup.vmSequence.forEach(sequence => {
						sequence.priority = vmGroup.groupId;
					});
				});
			}

			// create a map of machineslist with the machine guid.
			// Put the replication id of the matching machines to the map.
			let vmSequences = _.flatMap(vApp.config.vmGroups, vmGroup => vmGroup.vmSequence);
			let machinesMap = new Map();
			vmSequences.forEach(vmSequence => {
				machinesMap.set(vmSequence.vmInfo.vmGUID, vmSequence);
			});
			machinesList.forEach(machine => {
				if (machinesMap.has(machine.clientGUID)) {
					let m = machinesMap.get(machine.clientGUID);
					machine.replicationId = m.replicationId;
					machine.priority = m.priority;
					machine.ip = _.get(m, 'vmStaticIPAddressOptions');
					machine.ipText =
						_.get(machine, 'ip', []).length > 0 ? cvLoc('label.configured') : cvLoc('label.notConfigured');
					machine.destComputerName = m.destComputerName;
				}
			});
		};

		factory.assignPriority = function(initialPriority) {
			return $modal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/drPriority.jsp',
				backdrop: 'static',
				controller: 'drPriorityController',
				controllerAs: 'drPriorityCtrl',
				resolve: {
					priority: () => initialPriority
				}
			}).result;
		};

		factory.configureIPAddress = function(selectedRow) {
			return $modal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/drConfigureArrayMachines.jsp',
				backdrop: 'static',
				controller: 'configureArrayMachinesController',
				controllerAs: 'configureMachinesCtrl',
				resolve: {
					machine: () => _.cloneDeep(selectedRow)
				}
			}).result;
		};

		/*
		 * If a machine belongs to multiple datastores, it should be shown under only one datastore for the user.
		 * Maintain a map to remove duplicate machines.
		 */
		function removeDuplicateMachines(datastores) {
			let result = [];
			let uniqueMachines = new Map();
			datastores.forEach(ds => {
				let datastoreToAdd = {
					datastore: ds.datastore,
					vm: []
				};
				ds.vm.forEach(vm => {
					if (!uniqueMachines.get(vm.clientId)) {
						uniqueMachines.set(vm.clientId, vm.clientName);
						vm.priority = 1;
						datastoreToAdd.vm.push(vm);
					}
				});
				result.push(datastoreToAdd);
			});

			return result;
		}

		/*
		 * Appends datastore and priority to each machine and removes any duplicate entries.
		 * If the machine exists in multiple datastores, appends the datastore name to the label.
		 */
		function processMachinesData(datastoreMap) {
			let uniqueMachines = new Map();
			datastoreMap.forEach(dsMap => {
				dsMap.vm.forEach(vm => {
					vm.datastore = dsMap.datastore;
					vm.copy = dsMap.copy || {};
					vm.datastoreLabel = dsMap.datastore.strDatastorename;
					vm.priority = 1;
					vm.ipText = cvLoc('label.notConfigured');

					let machine = uniqueMachines.get(vm.clientId);
					if (machine) {
						machine.datastoreLabel = `${machine.datastoreLabel}, ${vm.datastoreLabel}`;
					} else {
						machine = vm;
					}
					machine.hostName = vm.hostName; // hostname contains the ESX server the machine belongs to
					uniqueMachines.set(machine.clientId, machine);
				});
			});

			// convert the map to an array and return
			return [...uniqueMachines.values()];
		}

		/*
		 * Changes the datastoreMap to a Map structure with copyid as the key and all it's machines as it's values
		 */
		function convertDSMaptoCopyMap(datastoreMap) {
			let copyMap = new Map();
			datastoreMap.forEach(dsMap => {
				let vms = [];
				if (copyMap.has(dsMap.copy.copyId)) {
					vms = copyMap.get(dsMap.copy.copyId);
				}
				vms.push(dsMap);
				copyMap.set(dsMap.copy.copyId, vms);
			});

			return copyMap;
		}

		/*
		 * The siteInfo response contains the LUN's which might result in the same datastore being shown multiple times.
		 * Hence, a unique check has been performed to filter out the duplicate ones.
		 */
		function getDatastoreLabelForSiteInfo(siteInfo) {
			let uniqueDatastores = _.uniqWith(siteInfo.datastore, datastoreComparator);
			uniqueDatastores = _.sortBy(uniqueDatastores, ds => ds.source.strDatastorename);

			siteInfo.datastoreLabel = '';
			uniqueDatastores.forEach(ds => {
				siteInfo.datastoreLabel += `${ds.source.strDatastorename}, `;
			});
			siteInfo.datastoreLabel = siteInfo.datastoreLabel.substring(0, siteInfo.datastoreLabel.length - 2);
		}

		/*
		 * Compares 2 datastores for equality using UUID
		 */
		function datastoreComparator(ds1, ds2) {
			return ds1.source.strDatastoreUUID === ds2.source.strDatastoreUUID;
		}

		return factory;
	}
]);

export default arrayReplicationFactory;
