import 'adminConsole/js/services/clientGroups.svc.js';
import 'modules/disasterRecovery/js/dr.constants.js';
import 'modules/disasterRecovery/js/services/failoverGroup/failoverGroup.svc.js';
import 'modules/servers/js/services/servers.svc.js';
import 'users/js/services/users.svc.js';
import 'vsa/js/services/restore.svc.js';
import 'modules/vmManagement/js/constants/vmLifeCyclePolicy.constants.js';
import 'vsa/js/controllers/configureReplicationGroup.ctrl.js';
import 'vsa/js/factories/replicationGroups.factory.js';

import { drAppFailoverModule } from 'common/js/modules';

/**
 * Contains the library of functions used by the DR app
 */

var failoverMod = drAppFailoverModule;

failoverMod.factory('drOrchestrationFactory', [
	'$uibModal',
	'$log',
	'cvToaster',
	'cvLoc',
	'clientGroupService',
	'serverService',
	'restoreService',
	'userService',
	'failoverService',
	'replicationGroupFactory',
	'FailoverConstants',
	'FailoverEnum',
	'POLICY_APP_TYPE',
	'VENDORS',
	function drOrchestrationFactory(
		$modal,
		$log,
		cvToaster,
		cvLoc,
		clientGroupService,
		serverService,
		restoreService,
		userService,
		failoverService,
		replicationGroupFactory,
		FailoverConstants,
		FailoverEnum,
		POLICY_APP_TYPE,
		VENDORS
	) {
		var factory = {
			// public functions
			addGridColumn: addGridColumn,
			addGridColumns: addGridColumns,
			areVendorsSame: areVendorsSame,
			checkTargetType: checkTargetType,
			createBackupMachinesGridColumns: createBackupMachinesGridColumns,
			createReplicatedMachinesGridColumns: createReplicatedMachinesGridColumns,
			filterReplicationTargetsByVendor: filterReplicationTargetsByVendor,
			getAzureRMTargets: getAzureRMTargets,
			getClientGroups: getClientGroups,
			getClients: getClients,
			getHyperVisors: getHyperVisors,
			getLifeCyclePolicies: getLifeCyclePolicies,
			getLocalLang: getLocalLang,
			getMachinesList: getMachinesList,
			getProxyServers: getProxyServers,
			getReplicationTargets: getReplicationTargets,
			getUsers: getUsers,
			getVirtualLabAndLiveMountTargets: getVirtualLabAndLiveMountTargets,
			isAzureVendor: isAzureVendor,
			isAzureRMVendor: isAzureRMVendor,
			isLiveMountTarget: isLiveMountTarget,
			isSupportedDestHypervisor: isSupportedDestHypervisor,
			isSupportedHypervisor: isSupportedHypervisor,
			isVirtualLabTarget: isVirtualLabTarget,
			isVMWareVendor: isVMWareVendor,
			openDevtestCreateModal: openDevtestCreateModal,
			openFailoverCreateModal: openFailoverCreateModal,
			removeAllGridColumns: removeAllGridColumns,
			removeGridColumns: removeGridColumns,

			// public variables
			azureRMTargets: [],
			clientGroups: [],
			clients: [],
			failoverEligibleMachines: [],
			hypervisors: [],
			proxyServers: [],
			replicationTargets: [],
			users: [],
			virtualLabAndLiveMountTargets: []
		};

		// private variables
		const toasterTTL = 10000;

		// The vendors supported in replication group will be also be supported in Failover groups
		const supportedHypervisors = replicationGroupFactory.supportedSourceVendors();

		return factory;

		function getMachinesList(entityID, entityType, vAppSource) {
			return failoverService
				.getMachinesList(entityID, entityType, vAppSource)
				.success(function(data) {
					factory.failoverEligibleMachines = data;
				})
				.error(function(e) {
					cvToaster.showErrorMessage({
						ttl: toasterTTL,
						message: e ? e : cvLoc('generic_error')
					});
				});
		}

		/**
		 * Fetches the list of all the clients groups and adds the type 'client' to each client
		 */
		function getClients() {
			return clientGroupService
				.getClientsForClientGroup(0)
				.success(function(data) {
					if (data) {
						factory.clients = [];
						data.forEach(function(item, index) {
							var client = item;
							client.id = item.client.clientEntity.clientId;
							client.name = item.client.clientEntity.clientName;
							client.type = 'client';

							factory.clients.push(client);
						});
						factory.clients.sort(comparename);
					}
				})
				.error(function(data) {
					$log.error('Error loading clients');
				});
		}

		/**
		 * Fetches the list of all the clients groups and adds the type 'clientGroup' to each group
		 */
		function getClientGroups() {
			return clientGroupService
				.getClientGroups()
				.success(function(data) {
					if (data) {
						factory.clientGroups = [];
						data.forEach(function(item, index) {
							var clientGroup = item;
							clientGroup.id = item.Id;
							clientGroup.name = item.name;
							clientGroup.isClientGroup = true;

							factory.clientGroups.push(clientGroup);
						});
						factory.clientGroups.sort(comparename);
					}
				})
				.error(function(data) {
					$log.error('Error loading client groups');
				});
		}

		/**
		 * Fetches the list of all the hypervisors and adds a type = 'client'
		 */
		function getHyperVisors() {
			return serverService
				.getServers()
				.success(function(data) {
					if (data) {
						factory.hypervisors = [];
						data.forEach(function(item, index) {
							if (item.type && isSupportedHypervisor(item.type)) {
								var hypervisor = item;
								hypervisor.id = item.clientId;
								hypervisor.name = item.name;
								hypervisor.isClientGroup = false;

								factory.hypervisors.push(hypervisor);
							}
						});
						factory.hypervisors.sort(comparename);
					}
				})
				.error(function(e) {
					$log.error(e && e.errMsg ? e.errMsg : 'Error loading hypervisors');
					factory.hypervisors = {};
				});
		}

		function getUsers() {
			return userService.getUsers().then(function(data) {
				if (angular.isUndefined(data.data) || angular.isUndefined((factory.users = data.data.usersList))) {
					factory.users = {};
				}
				factory.users.forEach(function(user) {
					if (!user.fullName) {
						user.fullName = user.userEntity.userName;
					}
				});
			});
		}

		function getProxyServers(vendor) {
			return failoverService.getProxyServers(vendor).then(function(data) {
				if (data) {
					factory.proxyServers = [];
					data.data.forEach(function(item) {
						var proxyServer = item;
						proxyServer.id = item.clientId;
						proxyServer.name = item.clientName;
						proxyServer.locationType = 0; // Check Api.x -> AutofailoverLocation
						factory.proxyServers.push(proxyServer);
					});
					factory.proxyServers.sort(comparename);
				}
			});
		}

		function getReplicationTargets(appType) {
			return restoreService.getReplicationTargets(appType).then(function(data) {
				if (angular.isUndefined(data.data) || angular.isUndefined((factory.replicationTargets = data.data))) {
					factory.replicationTargets = {};
				}
				factory.replicationTargets.forEach(function(target) {
					if (target && target.entity) {
						target.name = target.entity.vmAllocPolicyName;
					}
				});
			});
		}

		function getLifeCyclePolicies() {
			let vmPolicyAppType = POLICY_APP_TYPE.REPLICATION; //replication life cycle policy app type
			return restoreService.getLifeCyclePolicies(vmPolicyAppType).then(
				function(data) {
					factory.lifeCyclePolicies = data && data.data ? data.data : [];
					factory.lifeCyclePolicies.forEach(function(policy) {
						policy.lifeCyclePolicyName = policy.entity.vmAllocPolicyName;
					});
				},
				function(e) {
					$log.error(e && e.errMsg ? e.errMsg : 'Error loading recovery targets');
				}
			);
		}

		function isSupportedHypervisor(vendor) {
			return vendor && supportedHypervisors.indexOf(vendor) >= 0;
		}

		function isSupportedDestHypervisor(srcVendor, destVendor) {
			return VENDORS[srcVendor] && VENDORS[srcVendor].repGroupSupport.indexOf(destVendor) >= 0;
		}

		function getVirtualLabAndLiveMountTargets(appType) {
			return factory.getReplicationTargets(appType).then(function() {
				factory.virtualLabAndLiveMountTargets = [];
				factory.virtualLabAndLiveMountTargets = factory
					.filterReplicationTargetsByVendor(FailoverConstants.VMAllocationPolicyType['VMW_LIVEMOUNT'])
					.concat(
						factory.filterReplicationTargetsByVendor(FailoverConstants.VMAllocationPolicyType['VMW_BACKUP_LABTEMPLATE'])
					);
			});
		}

		function getAzureRMTargets() {
			return factory.getReplicationTargets().then(function() {
				factory.azureRMTargets = [];
				factory.azureRMTargets = factory.filterReplicationTargetsByVendor(
					FailoverConstants.VMAllocationPolicyType['AZURE_RESOURCE_MANAGER']
				);
			});
		}

		function filterReplicationTargetsByVendor(vendor) {
			var targets = [];
			factory.replicationTargets.forEach(function(target) {
				var policyType;
				if (
					target.entity &&
					(policyType = target.entity.policyType) &&
					policyType === FailoverConstants.VMAllocationPolicyType[vendor]
				) {
					targets.push(target);
				}
			});

			return targets;
		}

		function isLiveMountTarget(target) {
			return factory.checkTargetType(target, FailoverConstants.VMAllocationPolicyType['VMW_LIVEMOUNT']);
		}

		function isVirtualLabTarget(target) {
			return factory.checkTargetType(target, FailoverConstants.VMAllocationPolicyType['VMW_BACKUP_LABTEMPLATE']);
		}

		function checkTargetType(target, policyType) {
			if (
				angular.isDefined(target) &&
				angular.isDefined(target.entity) &&
				angular.isDefined(target.entity.policyType) &&
				target.entity.policyType == policyType
			) {
				return true;
			}

			return false;
		}

		function areVendorsSame(vendor1, vendor2) {
			return vendor1 && vendor2 && vendor1 === vendor2;
		}

		function isAzureVendor(vendor) {
			return factory.areVendorsSame(FailoverConstants.Vendors['AZURE'], vendor);
		}

		function isAzureRMVendor(vendor) {
			return factory.areVendorsSame(FailoverConstants.Vendors['AZURE_V2'], vendor);
		}

		function isVMWareVendor(vendor) {
			return factory.areVendorsSame(FailoverConstants.Vendors['VMW'], vendor);
		}

		function createBackupMachinesGridColumns() {
			return [
				{
					field: 'client.clientName',
					sort: {
						direction: 'asc'
					},
					displayName: cvLoc('label.name'),
					cellTemplate: '<span class="crop">{{row.entity.client.clientName}}</span>'
				},
				{
					field: 'destClient.clientName',
					displayName: cvLoc('label.destinationName'),
					cellTemplate: '<span class="crop">{{row.entity.destClient.clientName}}</span>'
				},
				{
					field: 'lastbackuptime.time',
					displayName: cvLoc('label.lastbackuptime'),
					cellTemplate: '<span class="crop"><div class="time" val="{{row.entity.lastbackuptime.time}}"></div></span>'
				},
				{
					field: 'entity.groupId',
					displayName: cvLoc('label.priority'),
					cellClass: 'topPaddingZero',
					cellTemplate: '<input type="number" ng-model=row.entity.groupId ng-min="1" ng-required >'
				},
				{
					field: 'actions',
					displayName: cvLoc('label.actions'),
					cellTemplate:
						'<div class="btn-group" uib-dropdown><a href class="uib-dropdown-toggle" uib-dropdown-toggle ng-disabled="disabled"><span class="grid-action-icon"></span></a><ul uib-dropdown-menu class="dropdown-menu" role="menu"><li><a href="" data-ng-click="grid.appScope.addPreScript(row.entity)">' +
						cvLoc('label.addPreScript') +
						'</a></li><li><a href data-ng-click="grid.appScope.addPostScript(row.entity)">' +
						cvLoc('label.addPostScript') +
						'</a></li></ul></div>'
				}
			];
		}

		function createReplicatedMachinesGridColumns() {
			return [
				{
					field: 'client.clientName',
					sort: {
						direction: 'asc'
					},
					displayName: cvLoc('label.source'),
					cellTemplate: '<span class="crop">{{row.entity.client.clientName}}</span>'
				},
				{
					field: 'destClient.clientName',
					displayName: cvLoc('label.destinationName'),
					cellTemplate: '<span class="crop">{{row.entity.destClient.clientName}}</span>'
				},
				{
					field: 'entity.groupId',
					displayName: cvLoc('label.priority'),
					cellClass: 'topPaddingZero',
					cellTemplate: '<input type="number" ng-model=row.entity.groupId ng-min="1" ng-required >'
				},
				{
					field: 'actions',
					displayName: cvLoc('label.actions'),
					cellTemplate:
						'<div class="btn-group" uib-dropdown><a href class="uib-dropdown-toggle" uib-dropdown-toggle ng-disabled="disabled"><span class="grid-action-icon"></span></a><ul uib-dropdown-menu class="dropdown-menu" role="menu"><li><a href="" data-ng-click="grid.appScope.addPreScript(row.entity)">' +
						cvLoc('label.addPreScript') +
						'</a></li><li><a href data-ng-click="grid.appScope.addPostScript(row.entity)">' +
						cvLoc('label.addPostScript') +
						'</a></li></ul></div>'
				}
			];
		}

		function removeAllGridColumns(columnDefs) {
			columnDefs = new Array();
		}

		function removeGridColumns(columnsToRemove, columnDefs) {
			if (columnsToRemove && columnDefs) {
				for (var i = 0; i < columnsToRemove.length; i++) {
					for (var idx = 0; idx < columnDefs.length; idx++) {
						if (columnDefs[idx].field === columnsToRemove[i]) {
							columnDefs.splice(idx, 1);
							break;
						}
					}
				}
			}
		}

		function addGridColumns(columnsToAdd, columnDefs) {
			if (columnsToAdd && columnDefs) {
				columnsToAdd.forEach(function(columnToAdd) {
					addGridColumn(columnToAdd, columnDefs);
				});
			}
		}

		function addGridColumn(columnToAdd, columnDefs) {
			if (columnToAdd && columnDefs) {
				columnDefs.push(columnToAdd);
			}
		}

		function getLocalLang(textForNothingSelected) {
			return {
				selectAll: cvLoc('label.selectAll'),
				selectNone: cvLoc('label.selectNone'),
				reset: cvLoc('label.reset'),
				search: cvLoc('label.search'),
				nothingSelected: textForNothingSelected
			};
		}

		function openDevtestCreateModal(vApp) {
			return $modal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/devtestGroup/devtestGroupCreate.jsp',
				backdrop: 'static',
				keyboard: false,
				controller: 'devtestGroupCreateController',
				controllerAs: 'dgCreateCtrl',
				resolve: {
					vApp: () => _.cloneDeep(vApp)
				}
			});
		}

		function openFailoverCreateModal(vAppSource, vApp) {
			var getParams = function() {
				return {
					vAppSource: vAppSource,
					vApp: _.cloneDeep(vApp)
				};
			};

			return $modal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/failoverGroup/failoverGroupCreate.jsp',
				backdrop: 'static',
				keyboard: false,
				controller: 'failoverGroupCreateController',
				controllerAs: 'fgCreateController',
				resolve: {
					controllerParams: getParams
				}
			});
		}

		/* ================ HELPER FUNCTIONS ================= */

		function comparename(a, b) {
			return a.name.toLowerCase().localeCompare(b.name.toLowerCase());
		}

		function isRegkeyEnabled(regKeyName) {
			return !!(
				cv &&
				cv.additionalSettings &&
				cv.additionalSettings.AdminConsole &&
				cv.additionalSettings.AdminConsole[regKeyName]
			);
		}
	}
]);

export default failoverMod;
