import 'modules/disasterRecovery/js/dr.constants.js';
import 'modules/disasterRecovery/js/factory/dr.factory.js';
import 'modules/disasterRecovery/js/factory/drCRUD.factory.js';
import 'modules/disasterRecovery/js/factory/drOperations.factory.js';
import ContextMenu from 'common/js/contextMenu/index.js';
import * as failoverColumnsTemplate from './failoverGroup.column.template.js';
import { drAppFailoverModule } from 'common/js/modules';

/**
 * Shows the list of Failover groups and performs failover operations on each group
 */

export class FailoverGroupListController {
	constructor(
		$scope,
		$state,
		$stateParams,
		$modal,
		cvUtil,
		cvLoc,
		drOrchestrationFactory,
		drCRUDFactory,
		drOperationsFactory,
		tabService,
		FailoverConstants,
		FailoverEnum
	) {
		this.$scope = $scope;
		this.$state = $state;
		this.$stateParams = $stateParams;
		this.$modal = $modal;
		this.cvUtil = cvUtil;
		this.cvLoc = cvLoc;
		this.drOrchestrationFactory = drOrchestrationFactory;
		this.drCRUDFactory = drCRUDFactory;
		this.drOperationsFactory = drOperationsFactory;
		this.tabService = tabService;
		this.FailoverConstants = FailoverConstants;
		this.FailoverEnum = FailoverEnum;
		this._init();
	}
	_init() {
		this.initTabService();
		this._setupGridOptions();
		this.$scope.$on('$destroy', this._destroy.bind(this));
	}
	initTabService() {
		let defaultTabInfo = {
			tabs: [
				{
					title: this.cvLoc('label.nav.replication.groups'),
					state: 'replicationGroups'
				},
				{
					title: this.cvLoc('label.nav.recovery'),
					state: 'failover'
				},
				{
					title: this.cvLoc('label.nav.storageTargets.rpStores'),
					state: 'rpStores'
				}
			],
			actions: [
				{
					value: 'AddFailoverGroup',
					title: this.cvLoc('label.addfailoverGroup'),
					dropdown: true
				}
			],
			onActionItemSelect: this._onContextMenuSelect.bind(this)
		};

		this._createAddGroupContextMenu();
		this.tabService.init({ ...defaultTabInfo });
	}
	onActionsClick(event) {
		event.preventDefault();
	}
	_initColumnDefs() {
		return failoverColumnsTemplate.getColumns(this.onActionsClick.bind(this), {
			cvLoc: this.cvLoc,
			cvUtil: this.cvUtil
		});
	}
	_setupGridOptions() {
		const self = this;
		this.gridOptions = {};
		this.$scope.gridOptions = this.gridOptions;
		this.gridOptions.sortDirection = { field: 'vAppEntity.vAppName', dir: 'asc' };
		this.gridOptions.columns = this._initColumnDefs();
		this.gridOptions.gridTitle = this.cvLoc('label.nav.recovery');
		this.gridOptions.tableName = 'failoverGroupTable';
		this.gridOptions.idField = 'vAppEntity.vAppName';
		this.gridOptions.beforeGridInitialize = ({ grid }) => {
			self.grid = grid;
		};
		this.gridOptions.url = this.getFailoverGroups.bind(this);
		this.gridOptions.usePageToolbar = true;
		this.gridOptions.gridToolbarMenu = [];
		this.gridOptions.actionMenu = this._getActionsList();
		this.gridOptions.onActionMenuOpen = this._showValidActions.bind(this);
		this.gridOptions.pageSize = 15;
		this.gridOptions.pageSizes = [10, 15, 25, 50, 100, 500, 1000];
	}
	deleteGroup(row) {
		let rowValue = this._getRowValue(row);
		let appId = rowValue.vAppEntity.vAppId;
		let appName = rowValue.vAppEntity.vAppName;
		this.drCRUDFactory.deleteGroup(appId, appName);
	}
	plannedFailover(row) {
		let rowValue = this._getRowValue(row);
		this.drOperationsFactory.plannedFailover(rowValue);
	}
	undoFailover(row) {
		let rowValue = this._getRowValue(row);
		this.drOperationsFactory.undoFailover(rowValue);
	}
	unPlannedFailover(row) {
		let rowValue = this._getRowValue(row);
		let bodyText = rowValue.isSourceArrayFailover ? this.cvLoc('label.confirmUnplannedFailoverArray') : '';
		let additionalConfirm = true;
		this.drOperationsFactory.unPlannedFailover(rowValue, bodyText, additionalConfirm);
	}
	testBoot(row) {
		let rowValue = this._getRowValue(row);
		this.drOperationsFactory.testBoot(rowValue);
	}

	testFailover(row) {
		let rowValue = this._getRowValue(row);
		this.drOperationsFactory.testFailover(rowValue);
	}
	undoTestFailover(row) {
		let rowValue = this._getRowValue(row);
		this.drOperationsFactory.undoTestFailover(rowValue);
	}

	getFailoverGroups(options) {
		let drCRUDFactory = this.drCRUDFactory;
		let self = this;
		self.recoveryGroups = [];
		drCRUDFactory
			.getFailoverGroups()
			.then(function(data) {
				self.recoveryGroups = self.recoveryGroups.concat(data.data);
				return _.get(cv, 'additionalSettings.AdminConsole.enableDevtestGroup', false)
					? drCRUDFactory.getDevtestGroups()
					: Promise.resolve('');
			})
			.then(function(data) {
				if (_.has(data, 'data')) {
					self.recoveryGroups = self.recoveryGroups.concat(data.data || []);
				}
			})
			.finally(function() {
				self.isDataLoading = false;
				options.success(self.recoveryGroups);
			});
	}

	/* vAppSource should contain value from FailoverEnum.VAppSource */
	createFailoverURL(vAppSource) {
		let self = this;
		this.drOrchestrationFactory.openFailoverCreateModal(vAppSource).result.then(function() {
			self.$state.reload();
		});
	}

	_showValidActions(row) {
		let rowValue = this._getRowValue(row);
		let drType = this.FailoverConstants.DROrchestrationType;
		let actionsList = ['Delete'];
		if (rowValue.isSourceArrayFailover) {
			actionsList = actionsList.concat([drType.PLANNED_FAILOVER, drType.UNPLANNED_FAILOVER, drType.UNDO_FAILOVER]);
		} else if (rowValue.isSourceArrayTestFailover) {
			actionsList = actionsList.concat([drType.TEST_FAILOVER, drType.UNDO_TEST_FAILOVER]);
		} else if (!rowValue.isFailoverGroup) {
			actionsList = actionsList.concat([drType.TEST_FAILOVER, drType.UNDO_TEST_FAILOVER]);
		} else {
			actionsList = actionsList.concat([drType.PLANNED_FAILOVER, drType.UNPLANNED_FAILOVER]);
		}
		this._showActions(row, actionsList);
	}

	_getActionsList() {
		let drType = this.FailoverConstants.DROrchestrationType;
		return [
			{
				id: 'Delete',
				label: this.cvLoc('Delete'),
				onSelect: this.deleteGroup.bind(this),
				contextMenuOnly: true
			},
			{
				id: drType.PLANNED_FAILOVER,
				label: this.cvLoc('label.plannedFailover'),
				onSelect: this.plannedFailover.bind(this),
				contextMenuOnly: true
			},
			{
				id: drType.TESTBOOT,
				label: this.cvLoc('label.testBoot'),
				onSelect: this.testBoot.bind(this),
				contextMenuOnly: true
			},
			{
				id: drType.UNDO_FAILOVER,
				label: this.cvLoc('label.undoFailover'),
				onSelect: this.undoFailover.bind(this),
				contextMenuOnly: true
			},
			{
				id: drType.UNPLANNED_FAILOVER,
				label: this.cvLoc('label.unplannedFailover'),
				onSelect: this.unPlannedFailover.bind(this),
				contextMenuOnly: true
			},
			{
				id: drType.TEST_FAILOVER,
				label: this.cvLoc('label.testFailover'),
				onSelect: this.testFailover.bind(this),
				contextMenuOnly: true
			},
			{
				id: drType.UNDO_TEST_FAILOVER,
				label: this.cvLoc('label.undoTestFailover'),
				onSelect: this.undoTestFailover.bind(this),
				contextMenuOnly: true
			}
		];
	}

	_showActions(row, actionsList) {
		this._getActionsList().forEach(action => row.grid.hideActionMenuOption(action.id));
		actionsList.forEach(action => row.grid.showActionMenuOption(action));
	}

	_createAddGroupContextMenu() {
		var self = this;
		const options = {};
		options.target = '.cv-main-bar-action.menu:nth-child(1)';
		options.actionMenu = this._getContextMenuOptions();
		options.onContextMenuSelect = this._onContextMenuSelect.bind(this);
		const element = angular.element('#addFailoverGroup');

		this.timer = setTimeout(() => {
			const target = angular.element('.cv-main-bar-action.menu:nth-child(1)');
			if (target.length > 0) {
				self.addGroupContextMenu = new ContextMenu(element, options);
				self.addGroupContextMenu.build();
				clearInterval(this.timer);
				return;
			}
		}, 1000);
	}

	_getContextMenuOptions() {
		var self = this;
		let cvLoc = self.cvLoc;
		const menu = [
			{
				id: this.FailoverEnum.VAppSource.REPLICATION,
				value: this.FailoverEnum.VAppSource.REPLICATION,
				label: this.cvLoc('label.source.replication')
			},
			{
				id: this.FailoverEnum.VAppSource.ARRAYREPLICATION,
				value: this.FailoverEnum.VAppSource.ARRAYREPLICATION,
				label: this.cvLoc('label.source.arrayreplication')
			}
		];

		return menu;
	}

	_onContextMenuSelect(value) {
		switch (value) {
			case this.FailoverEnum.VAppSource.REPLICATION:
				this.createFailoverURL(value);
				break;
			case this.FailoverEnum.VAppSource.ARRAYREPLICATION:
				this.createFailoverURL(value);
				break;
		}
	}

	_destroy() {
		//Clean context menu on scope destroy
		if (this.addGroupContextMenu) {
			this.addGroupContextMenu.destroy();
		}
		clearInterval(this.timer);
	}

	_getRowValue(row) {
		return _.get(row, 'selectedRowValues[0]', row.rowValue);
	}
}
FailoverGroupListController.$inject = [
	'$scope',
	'$state',
	'$stateParams',
	'$uibModal',
	'cvUtil',
	'cvLoc',
	'drOrchestrationFactory',
	'drCRUDFactory',
	'drOperationsFactory',
	'tabService',
	'FailoverConstants',
	'FailoverEnum'
];
drAppFailoverModule.controller('failoverGroupListController', FailoverGroupListController);
export default drAppFailoverModule;
