import 'modules/commcellRegistration/js/services/commcellRegistration.svc.js';
import 'adminConsole/js/services/security.svc.js';
import 'modules/commcellRegistration/js/factory/serviceCommcell.constant.js';

import { acAppSecurityModule } from 'common/js/modules';

var commcellRegistrationMod = acAppSecurityModule;

class EditSecurityAssociationController {
	constructor(
		cvLoc,
		securityService,
		commcellList,
		$q,
		commcellRegistrationService,
		commcellId,
		$modalInstance,
		cvToaster,
		cvUtil,
		ENTITY_TYPES,
		serviceCommcellConstants,
		commcellAssociationsList,
		commcellName,
		$dialogs,
		commcellRegistrationFactory
	) {
		this.cvLoc = cvLoc;
		this.securityService = securityService;
		this.commcellList = [];
		if (commcellList) {
			if (cvConfig.showRouterForServiceCommcell) {
				// Remove IdP commcell listed in global service commcell page
				this.commcellList = commcellList.filter(commcell => !commcell.isIdp);
			} else {
				// Remove commcells which aren't configured for IdP routing and cloud services
				this.commcellList = commcellList.filter(
					commcell =>
						commcell.isGlobalIDPCommcell || commcell.multiCommcellType === serviceCommcellConstants.CLOUDSERVICECOMMCELL
				);
			}
		}
		this.$q = $q;
		this.commcellRegistrationService = commcellRegistrationService;
		this.$modalInstance = $modalInstance;
		this.cvToaster = cvToaster;
		this.cvUtil = cvUtil;
		this.ENTITY_TYPES = ENTITY_TYPES;
		this.serviceCommcellConstants = serviceCommcellConstants;
		this.commcellId = commcellId;
		this.commcellAssociationsList = commcellAssociationsList;
		this.commcellName = commcellName;
		this.$dialogs = $dialogs;
		this.commcellRegistrationFactory = commcellRegistrationFactory;

		this.selectedUser = null;
		this.selectedCommcells = null;
		this.selectedRule = null;
		this.serverMessage = this.cvUtil.infoMsgLoc('Loading');
		this.associationMap = {};
		this.associationsList = [];
		this.dataLoaded = false;
		this.showCommcellDropdown = !(this.commcellAssociationsList && this.commcellId);

		this.errorMessages = {
			showCommcellNotSelected: _.isEmpty(this.commcellList),
			roleNotSelected: false,
			associationEntityNotSelected: false
		};

		this.labelConfigCommcell = {
			nothingSelected: this.cvLoc('label.commcell')
		};

		this.labelConfigRole = {
			nothingSelected: this.cvLoc('label.selectRole')
		};

		this._initializeModal();
	}

	_initializeModal() {
		this._loadAsyncData();
		this._setupSelect2Config();
		this._setupCommcellDropdown();
	}

	_buildMaps(commcellAssociations) {
		let hashMap = {},
			associationsList = [],
			self = this;

		_.forEach(commcellAssociations, association => {
			let isUser = _.has(association, 'userOrGroup.userId');
			let isDomain = _.has(association, 'userOrGroup.providerId');
			let selector;

			if (isUser) selector = `u${_.get(association, 'userOrGroup.userId')}`;
			else if (isDomain) selector = `d${_.get(association, 'userOrGroup.providerId')}`;
			else selector = `ug${_.get(association, 'userOrGroup.userGroupId')}`;

			if (hashMap[selector]) {
				hashMap[selector].commcells.push({
					commCell: { commCellId: association.entity.entityId, commCellName: association.entity.entityName }
				});
				hashMap[
					selector
				].commcellFormattedList = `${hashMap[selector].commcellFormattedList}, ${association.entity.entityName}`;
			} else {
				let name = isUser
					? association.userOrGroup.userName
					: isDomain
					? association.userOrGroup.providerDomainName
					: association.userOrGroup.userGroupName;

				let associationObject = {
					associatedEntity: {
						loginName: name,
						name: association.userOrGroup.displayName,
						userInfo: {
							id: isUser
								? association.userOrGroup.userId
								: isDomain
								? association.userOrGroup.providerId
								: association.userOrGroup.userGroupId,
							name,
							userType: isUser
								? self.serviceCommcellConstants.USER
								: isDomain
								? self.serviceCommcellConstants.PROVIDER
								: self.serviceCommcellConstants.COMMCELL_GROUP
						},
						entityType: self._getEntityTypeName(association),
						providerType: association.providerType
					},
					commcells: [
						{
							commCell: {
								commCellId: association.entity.entityId,
								commCellName: association.entity.entityName
							}
						}
					],
					commcellFormattedList: association.entity.entityName
				};

				hashMap[selector] = associationObject;
				associationsList.push(associationObject);
			}
		});
		this.associationMap = hashMap;
		this.associationsList = associationsList;
		this._initializeGrid();
	}

	_loadAsyncData() {
		let self = this;
		let fetchArray = [this.securityService.getRoles()];

		if (this.showCommcellDropdown) fetchArray.push(this.commcellRegistrationService.getCommcellAssociations());

		this.$q.all(fetchArray).then(
			function(resolvedArray) {
				self.dataLoaded = true;
				self.serverMessage = self.cvUtil.emptyMsg();
				self.roles = _.get(resolvedArray, '[0].data', []);
				let commcellAssociations;
				if (self.showCommcellDropdown) commcellAssociations = _.get(resolvedArray, '[1].data.associations', []);
				else commcellAssociations = self.commcellAssociationsList;
				self._buildMaps(commcellAssociations);
			},
			function(err) {
				self.serverMessage = {
					message: err.errorMessage || err,
					type: 'error'
				};
			}
		);
	}

	_setupCommcellDropdown() {
		this.commcells = [
			{
				commCell: {
					commCellName: this.cvLoc('label.commcell')
				},
				'commCell.commCellName': this.cvLoc('label.commcell'),
				commcellGroup: true
			},
			...this.commcellList,
			{
				commcellGroup: false
			}
		];
	}

	_setupSelect2Config() {
		var self = this;
		this.searchConfig = {
			placeholder: 'placeholder.multicommcellTextbox',
			httpConfig: {
				method: 'GET',
				url: 'getUserSuggestionsListPlan.do',
				params: {
					returnDomain: true,
					pageSize: 10
				}
			},
			pagination: true,
			pageParam: 'pageSize',
			payloadPath: 'params.term',
			iconClass: this._getIconClass.bind(this),
			enableSVGIcons: true
		};
	}

	_getIconClass(entity) {
		switch (entity.userInfo.userType) {
			case this.serviceCommcellConstants.EXTERNAL_GROUP:
			case this.serviceCommcellConstants.COMMCELL_GROUP:
				return '<i class="k-i-myspace"></i>';
			case this.serviceCommcellConstants.USER:
				return '<i class="k-i-user"></i>';
			case this.serviceCommcellConstants.PROVIDER:
				return this.commcellRegistrationFactory.getIconsForProviderType(entity.providerType);
			default:
				return '<i class="k-i-email"></i>';
		}
	}

	_initializeGrid() {
		let self = this;
		this.gridOptions = {
			gridTitle: this.cvLoc('header.associationsList'),
			tableName: 'multiCommcellAssociationList',
			enableFiltering: false,
			hasViews: false,
			enableColumnMenu: false,
			enablePaging: true,
			pageSize: 10,
			enableMultiSelect: true,
			enableCheckBoxColumn: true,
			cvOnGridEmpty: this.cvLoc('info.noSecurityAssociationsDefined'),
			columns: this._getColumnDefinitions(),
			data: this.associationsList,
			beforeGridInitialize: ({ grid }) => {
				self.grid = grid;
			},
			actionMenu: [
				{
					id: 'DELETE_ASSOCIATION',
					label: this.cvLoc('action.delete'),
					onSelect: event => {
						self.removeAssociation(event);
					}
				}
			]
		};
	}

	_clearInputs() {
		this.selectedUser = this.selectedRole = this.selectedCommcells = null;
		_.forEach(this.commcells, commcell => {
			commcell.selected = false;
		});
	}

	_validateInputValues(checkForDanglingInput) {
		if (!checkForDanglingInput) {
			this.errorMessages.showAssociationEntityNotSelected = !this.selectedUser;
			if (this.showCommcellDropdown) this.errorMessages.showCommcellNotSelected = _.isEmpty(this.selectedCommcells);
		}

		return this.selectedUser && (!_.isEmpty(this.selectedCommcells) || !this.showCommcellDropdown);
	}

	_completeInputsBeforeSubmit() {
		if (this._validateInputValues(true)) {
			this.addAssociation();
			return true;
		} else if (!this.selectedUser && _.isEmpty(this.selectedCommcells && this.showCommcellDropdown)) {
			return true;
		}

		return this._validateInputValues();
	}

	addAssociation() {
		let self = this;

		if (!this._validateInputValues()) return;

		let mapSelector = this.getMapSelectorFromParsedAssociationArray(this.selectedUser);

		// Pre populating commcell details in case when associations is opened from service commcell details page
		if (!this.showCommcellDropdown) {
			this.selectedCommcells = [
				{
					commCell: {
						commCellId: this.commcellId,
						commCellName: this.commcellName
					}
				}
			];
		}

		if (this.associationMap[mapSelector]) {
			_.forEach(this.selectedCommcells, commcell => {
				let isNewCommcell =
					_.findIndex(
						self.associationMap[mapSelector].commcells,
						iter => iter.commCell.commCellId === commcell.commCell.commCellId
					) === -1;

				if (isNewCommcell) {
					self.associationMap[mapSelector].commcells = self.associationMap[mapSelector].commcells.concat(commcell);
					self.associationMap[
						mapSelector
					].commcellFormattedList = `${self.associationMap[mapSelector].commcellFormattedList},${commcell.commCell.commCellName}`;
				}
			});
		} else {
			let associationObject = {
				associatedEntity: angular.copy(self.selectedUser),
				commcells: angular.copy(self.selectedCommcells),
				commcellFormattedList: self.selectedCommcells.reduce((commcellString, commcell) => {
					let commcellName = _.get(commcell, 'commCell.commCellName');
					if (!commcellString.length) return commcellName;
					return commcellString.concat(`,${commcellName}`);
				}, '')
			};

			associationObject.associatedEntity.entityType = self.commcellRegistrationFactory.getEntityTypeFromUserSuggestion(
				self.selectedUser
			);
			associationObject.associatedEntity.providerType = self.selectedUser.providerType;

			self.associationMap[mapSelector] = associationObject;
			self.associationsList.push(associationObject);
		}

		this._clearInputs();
		this.grid.refreshData();
	}

	removeAssociation(e) {
		let self = this;
		let indicesToBedeleted = [];
		if (!e.selectedRowValues || !e.selectedRowValues.length) return;
		e.selectedRowValues.forEach(associationToDelete => {
			let mapSelector = self.getMapSelectorFromParsedAssociationArray(associationToDelete.associatedEntity);
			let associationObject = self.associationMap[mapSelector];
			indicesToBedeleted.push(self.associationsList.indexOf(associationObject));
			delete self.associationMap[mapSelector];
		});
		_.remove(this.associationsList, (association, index) => indicesToBedeleted.includes(index));
		this.grid.refreshData();
	}

	getMapSelectorFromParsedAssociationArray(userObject) {
		let associationEntityId = userObject.userInfo.id;
		// let roleId = roleObject[0].roleId;
		switch (userObject.userInfo.userType) {
			case this.serviceCommcellConstants.USER:
				return `u${associationEntityId}`;
			case this.serviceCommcellConstants.PROVIDER:
				return `d${associationEntityId}`;
			default:
				return `ug${associationEntityId}`;
		}
	}

	closeModal() {
		this.$modalInstance.dismiss();
	}

	expandAssociationsList() {
		let self = this;
		let expandedAssociationsList = [];
		_.forEach(this.associationsList, association => {
			_.forEach(association.commcells, commcell => {
				let associationObject = {
					entity: {
						entityName: commcell.commCell.commCellName,
						entityId: commcell.commCell.commCellId,
						entityType: this.serviceCommcellConstants[this.ENTITY_TYPES.SERVICE_COMMCELL_ENTITY],
						_type_: self.serviceCommcellConstants.COMMCELL_TYPE
					}
				};
				if (association.associatedEntity.userInfo.userType === self.serviceCommcellConstants.USER)
					associationObject.userOrGroup = {
						userId: association.associatedEntity.userInfo.id,
						userName: association.associatedEntity.loginName,
						_type_: self.serviceCommcellConstants.USER_TYPE
					};
				else if (association.associatedEntity.userInfo.userType === self.serviceCommcellConstants.PROVIDER)
					associationObject.userOrGroup = {
						providerId: association.associatedEntity.userInfo.id,
						providerDomainName: association.associatedEntity.loginName,
						_type_: self.serviceCommcellConstants.PROVIDER_TYPE
					};
				else
					associationObject.userOrGroup = {
						userGroupId: association.associatedEntity.userInfo.id,
						userGroupName: association.associatedEntity.loginName,
						displayName: association.associatedEntity.name,
						_type_: self.serviceCommcellConstants.USER_GROUP_TYPE
					};

				associationObject.providerType = association.associatedEntity.providerType;
				expandedAssociationsList.push(associationObject);
			});
		});

		return expandedAssociationsList;
	}

	saveAssociations() {
		let isInputComplete = this._completeInputsBeforeSubmit();

		if (!isInputComplete) return;

		let self = this,
			serviceCommcellId;
		let expandedAssociations = this.expandAssociationsList();

		if (!this.showCommcellDropdown) serviceCommcellId = this.commcellId;

		this.serverMessage = this.cvUtil.emptyMsg();

		this.commcellRegistrationService
			.setCommcellAssociations(expandedAssociations, serviceCommcellId)
			.success(response => {
				if (_.get(response, 'errorCode') && _.get(response, 'errorMessage')) {
					self.serverMessage = self.cvUtil.errMsg(response.warningMessage);
					return;
				} else if (_.get(response, 'warningCode') && _.get(response, 'warningMessage')) {
					self.$dialogs.notify(self.cvLoc('heading.warning'), response.warningMessage).result.finally(() => {
						self.$modalInstance.close(expandedAssociations);
					});
				} else {
					self.cvToaster.showSuccessMessage({
						message: self.cvLoc('info.associationCompleted'),
						ttl: '5000'
					});
					self.$modalInstance.close(expandedAssociations);
				}
			})
			.error(err => {
				self.serverMessage = {
					message: err.errorMessage || err,
					type: 'error'
				};
			});
	}

	_getColumnDefinitions() {
		let self = this;
		let userColumn = {
			'associatedEntity.loginName': {
				title: this.cvLoc('label.providerUserGroup'),
				type: 'string',
				width: this.showCommcellDropdown ? '30%' : '50%',
				template: rowItem => {
					let iconClass = self._getIconClass(rowItem.associatedEntity);
					return `<span class="crop">
								<div class="k-icon font-primary-color margin-right-5 alert-detail-icon">${iconClass}</div>
								<span>${rowItem.associatedEntity.loginName}</span>
							</span>`;
				}
			}
		};

		let commcellsColumn = {
			commcellFormattedList: {
				title: this.cvLoc('label.serviceCommcells'),
				type: 'string',
				width: '40%',
				hidden: !this.showCommcellDropdown, // when its commcell details page, commcell name is implied
				template: `<span class="crop">#:commcellFormattedList#</span>`
			}
		};

		let providerTypeColumn = {
			'associatedEntity.entityType': {
				title: this.cvLoc('label.type'),
				type: 'string',
				width: '*',
				searchable: false
			}
		};

		return {
			...userColumn,
			...commcellsColumn,
			...providerTypeColumn
		};
	}

	_preselectRole(selectedEntity) {
		if (selectedEntity.userInfo.userType !== this.serviceCommcellConstants.PROVIDER) return;
		_.forEach(this.roles, role => {
			role.isItemDisabled = true;
			if (role.roleId === this.serviceCommcellConstants.VIEW_ROLE) {
				role.isItemDisabled = false;
				role.selected = true;
			}
		});
		this.disableRoleSelection = true;
	}

	_getEntityTypeName(association) {
		switch (association.userOrGroup._type_) {
			case this.serviceCommcellConstants.USER_GROUP_TYPE:
				return this.cvLoc('label.userGroup');
			case this.serviceCommcellConstants.USER_TYPE:
				return this.cvLoc('label.user');
			default:
				return this.commcellRegistrationFactory.getProviderEntityTypeName(association.providerType);
		}
	}
}

EditSecurityAssociationController.$inject = [
	'cvLoc',
	'securityService',
	'commcellList',
	'$q',
	'commcellRegistrationService',
	'commcellId',
	'$uibModalInstance',
	'cvToaster',
	'cvUtil',
	'ENTITY_TYPES',
	'serviceCommcellConstants',
	'commcellAssociationsList',
	'commcellName',
	'$dialogs',
	'commcellRegistrationFactory'
];

commcellRegistrationMod.controller('editSecurityAssociationController', EditSecurityAssociationController);

export default commcellRegistrationMod;
