(function() {
	'use strict';

	var cvCommon = angular.module('cvCommon');

	cvCommon.factory('globalCacheFactory', [
		'GLOBAL_CACHE_FIELDS',
		'cvUtil',
		'$httpParamSerializer',
		'cvMultiCommcell',
		'cvGridFactory',
		function(GLOBAL_CACHE_FIELDS, cvUtil, $httpParamSerializer, cvMultiCommcell, cvGridFactory) {
			var factory = {};
			//method to form fqs for date field - used by server side caching
			factory.getDateFq = function(colName, colFilterValue) {
				//convert last 1hour, last 24hours, last 1week to actual time
				let start = cvGridFactory.convertTimeDuration(colFilterValue);
				let end = new Date();
				let colFilters = [];
				let startTS = null;
				let endTS = null;
				if (colFilterValue === 'never') {
					colFilters.push(colName + ':eq:null');
				} else {
					if (start && end) {
						if (start === 'custom') {
							//set time range if it's custom date pick
							let range = JSON.parse(colFilterValue);
							start = range.from;
							end = range.to;
							startTS = parseInt(start);
							endTS = parseInt(end);
						} else {
							startTS = parseInt(start.getTime() / 1000);
							endTS = parseInt(end.getTime() / 1000);
						}
					}
					colFilters.push(colName + ':gt:' + startTS);
					colFilters.push(colName + ':lt:' + endTS);
				}
				return colFilters;
			};
			factory.getAndResetCacheRefreshFlag = function() {
				// the flag is set when the user presses ctrl f5 or f5 (set in apputil.js)
				let crFlag = null;
				if (localStorage) {
					crFlag = localStorage.getItem('cacheRefreshFlag');
					localStorage.removeItem('cacheRefreshFlag'); //hardRefresh should occur just once so reset the flag
				}
				return crFlag;
			};
			//method to form fqs for size field with range values - used by server side caching
			factory.getSizeFq = function(colName, colFilterValue) {
				let colFilters = [];
				if (colFilterValue) {
					let values = colFilterValue.split(',');
					if (values && values.length > 1) {
						colFilters.push(colName + ':gteq:' + values[0]);
						colFilters.push(colName + ':lteq:' + values[1]);
					}
				}
				return colFilters;
			};

			//method to get global search filter
			factory.getGlobalSearchFilter = function(options, commaSeperatedFieldNames) {
				let globalFilter = null;
				//filters from grid header search bar
				if (commaSeperatedFieldNames && _.get(options, 'data.filter.filters', []).length > 0) {
					options.data.filter.filters.forEach(function(filter) {
						if (
							filter.filters &&
							filter.filters.length > 0 &&
							filter.filters[0].filters &&
							filter.filters[0].filters.length > 0
						) {
							//if it is a search from search bar
							const ele = filter.filters[0];
							const searchValue = ele.filters[0].value;
							globalFilter = {
								searchText: searchValue,
								supportedColumns: commaSeperatedFieldNames
							};
						}
					});
				}
				return globalFilter;
			};

			/**
			 * Converts the filter condition in kendo grid to the corresponding condition in mongo cache API
			 *
			 * @param {String} kendoFilterCondition
			 * @param {Boolean} isMultiValueFilter True if the filter value is a list
			 */
			factory.mapFilterConditionToMongoApi = function(kendoFilterCondition, isMultiValueFilter) {
				if (_.isUndefined(isMultiValueFilter)) {
					isMultiValueFilter = false;
				}
				let apiFilterCondition = kendoFilterCondition || 'contains';
				switch (kendoFilterCondition) {
					case 'doesNotContain':
						apiFilterCondition = 'notContains';
						break;
					case 'equals':
						apiFilterCondition = isMultiValueFilter ? 'in' : 'eq';
						break;
					case 'doesNotEqual':
						apiFilterCondition = isMultiValueFilter ? 'nin' : 'neq';
						break;
				}
				return apiFilterCondition;
			};

			//common url to use CRE api
			factory.commonCreUrl =
				cv.contextPath + '/proxy/cr/reportsplusengine/datasets/System.HTTPWrapper/data?parameter.api=';
			//default device parameters
			factory.defaultParams = {
				limit: 20,
				offset: 0,
				format: 'raw',
				cacheId: localStorage.creCacheId ? localStorage.creCacheId : cvUtil.generateUUID()
			};
			localStorage.creCacheId = factory.defaultParams.cacheId;
			localStorage.jobCacheId = cvUtil.generateUUID();

			factory.refreshCacheId = function() {
				localStorage.creCacheId = cvUtil.generateUUID();
				factory.defaultParams.cacheId = localStorage.creCacheId;
			};
			/*
			 *function to get cre api urls
			 *type: 'device', 'client'
			 *urlOptions: api and all the params
			 *
			 * */
			factory.getUrl = function(urlOptions) {
				// $httpParamSerializer will sort the request parameters alphabetically.
				var url =
					factory.commonCreUrl + encodeURIComponent(urlOptions.api) + '&' + $httpParamSerializer(urlOptions.params);
				//adding headers
				for (var key in urlOptions.headers) {
					if (angular.isDefined(urlOptions.headers[key])) {
						url = url + '&parameter.headers=' + key + ':' + urlOptions.headers[key];
					}
				}
				//add row expression parameters
				if (urlOptions.rowExpression) {
					url = url + '&parameter.rowExpression=' + urlOptions.rowExpression;
				}
				//append data source to url
				if (urlOptions.datasource && urlOptions.datasource.length) {
					for (var i = 0; i < urlOptions.datasource.length; i++) {
						url = url + '&datasource=' + urlOptions.datasource[i];
					}
				}
				return url;
			};

			//function to convert global search filters to match database format
			factory.getGlobalFilters = function(globalFilter) {
				var filters = '';
				var columnsLength = globalFilter.supportedColumns.length;
				for (var i = 0; i < columnsLength; i++) {
					filters =
						filters +
						globalFilter.supportedColumns[i] +
						" like '%" +
						globalFilter.searchText +
						"%'" +
						(i === columnsLength - 1 ? '' : ' OR ');
				}
				return filters;
			};

			//function to convert column filters to match database format
			factory.getColumnFilters = function(colFilters) {
				var filters = '';
				var index = 0;
				var numOfFilters = Object.keys(colFilters).length;
				for (var key in colFilters) {
					//filter a range, like backup size, backup time..
					if (Array.isArray(colFilters[key])) {
						filters =
							filters +
							key +
							' BETWEEN ' +
							colFilters[key][0] +
							' AND ' +
							colFilters[key][1] +
							(index === numOfFilters - 1 ? '' : ' AND ');
					} else if (typeof colFilters[key] === 'number') {
						//if filter on a number value, like configured or not
						filters = filters + key + '=' + colFilters[key] + (index === numOfFilters - 1 ? '' : ' AND ');
					} else if (
						typeof colFilters[key] === 'boolean' ||
						colFilters[key] === 'true' ||
						colFilters[key] === 'false'
					) {
						//if filter on a boolean value, like isClientDelete or not
						//since SQL has true, false, null 3 types, need to check both false and null
						if (colFilters[key] === 'true' || colFilters[key] === 'false') {
							colFilters[key] = colFilters[key] === 'true';
						}
						filters =
							filters +
							(colFilters[key] ? '' : '(') +
							key +
							(colFilters[key] ? '=TRUE' : '=FALSE' + ' OR ' + key + ' IS NULL' + (colFilters[key] ? '' : ')')) +
							(index === numOfFilters - 1 ? '' : ' AND ');
					} else {
						//if filter on text input strings, like device name, plan name..
						//handling multiple filters for same type, like create a view with multiple client name
						var filtersInOneRule = colFilters[key].split(',');
						for (var i = 0; i < filtersInOneRule.length; i++) {
							filters =
								filters +
								key +
								" like '%" +
								filtersInOneRule[i] +
								"%'" +
								(i === filtersInOneRule.length - 1 ? '' : ' OR ');
						}
						filters = filters + (index === numOfFilters - 1 ? '' : ' AND ');
					}
					index += 1;
				}
				return filters;
			};

			//function to get selected datassource
			factory.getSelectedDataSource = function(dataSource) {
				var selectedDataSource = [];
				//if stateparams has _cn, read datasource from stateparams, otherwise from commcell selection dropdown
				if (dataSource) {
					selectedDataSource.push(dataSource);
				} else {
					//check if selected commcells equals all the commcells, remove datasource since we don't need to pass it
					selectedDataSource = cvMultiCommcell.getSelectedCommcell();
					var allCommcells = cv.sessionContext.dataSourceList;
					if (selectedDataSource && selectedDataSource.length === allCommcells.length) {
						selectedDataSource = null;
					}
				}
				return selectedDataSource;
			};

			factory.getUrlParams = function(filterOption, columnFields, cacheIdString) {
				//remove dot(.) if sort field has it
				var colName = filterOption.sortOptions.fields;

				if (colName.indexOf('.') !== -1) {
					colName = colName.substring(colName.lastIndexOf('.') + 1);
				}
				var pageInfo = {
					limit: filterOption.pageInfo ? filterOption.pageInfo.limit : undefined,
					offset: filterOption.pageInfo ? (filterOption.pageInfo.page - 1) * filterOption.pageInfo.limit : 0,
					sortField: colName,
					sortDirection: filterOption.sortOptions.directions
				};
				var urlParams = {
					limit: pageInfo.limit,
					offset: pageInfo.offset,
					format: 'raw',
					fields_def: JSON.stringify(columnFields),
					cacheId: cacheIdString + '_' + factory.defaultParams.cacheId,
					orderby: pageInfo.sortField + ' ' + pageInfo.sortDirection,
					where: factory.getFilters(filterOption)
				};
				return urlParams;
			};

			factory.getFilters = function(filterOptions) {
				var filters = null;
				//if global search present, apply global search for all the applicable columns
				if (filterOptions.globalFilter) {
					filters = factory.getGlobalFilters(filterOptions.globalFilter);
					//if have both column filter and global filter, eg: search under view
					if (!_.isEmpty(filterOptions.filters)) {
						var columnFilters = factory.getColumnFilters(filterOptions.filters);
						//have to obey both column filter and global filters
						filters = '(' + columnFilters + ')' + ' AND ' + '(' + filters + ')';
					}
				} else if (!_.isEmpty(filterOptions.filters)) {
					filters = factory.getColumnFilters(filterOptions.filters);
				}
				return filters;
			};

			//function to get filter which raw data
			/*
			value: the search text
			col: the column name needs to filter
			type: which type to filter,
			forceOperator: force to use this operator
			*/
			factory.getNewFilter = function(value, col, type, forceOperator) {
				var operator = type && type.toLowerCase() === 'string' ? 'contains' : 'equals';
				if (forceOperator) {
					operator = forceOperator;
				}
				const condition = '=';
				col = '[' + col + ']';
				var funcExpression = '_exprString';
				if (type && type.toLowerCase() === 'string') {
					funcExpression = '_exprString';
					value = value.replace(/"/g, '""');
				} else if (
					type &&
					['double', 'float', 'integer', 'long', 'short', 'range'].indexOf(type.toLowerCase()) !== -1
				) {
					funcExpression = '_exprInt';
				} else if (type && ['timestamp', 'date'].indexOf(type.toLowerCase()) !== -1) {
					funcExpression = '_exprDateTime';
				} else {
					funcExpression = '_exprString';
				}
				let expression;
				if (type === 'range') {
					const arr = value.split(',');
					expression = col + ' ' + condition + ' ' + funcExpression + ' (">' + arr[0] + ' && <' + arr[1] + '")';
				} else {
					expression = col + ' ' + condition + ' ' + funcExpression + ' ("' + value + '","' + operator + '")';
				}
				return expression;
			};
			return factory;
		}
	]);
})();
