import ContextMenu from '../kendocontextmenu';
import AppendedMenu from '../menu';
import * as utils from '../componentUtils/utils';

export default class ToolbarMenu extends AppendedMenu {
	constructor(args) {
		super(args);
	}

	initialize(args) {
		super.initialize(args);
		this.maxSize = utils.getArg(args, 'maxSize', 0);
		this.moreItemTemplate = args.moreItemTemplate;

		this.visibilityMap = {};
		_.forEach(args.options, option => {
			this.visibilityMap[option.id] = !option.hidden;
		});
	}

	build() {
		this._contextMenu = new ContextMenu({
			filter: this.getSelector(),
			cssClass: 'toolbar-menu-context-menu',
			showOn: '',
			name: `${this.name}_moreContextMenu`,
			options: this._formatContextMenuOptions(this._menuOptions)
		});

		this.dataSource.push({
			encoded: false,
			cssClass: 'toolbar-menu-item toolbar-menu-more-item',
			attr: {
				id: `${this.name}_moreItem`
			},
			text:
				this.moreItemTemplate ||
				'<span data-menu-more="">More actions <span class="k-icon k-i-arrow-60-down"></span></span></span>'
		});
		super.build();
	}

	_onOptionSelect(event) {
		if ($(event.item).find('[data-menu-more]').length) {
			// This is handled by the context menu
			this._contextMenu.open(event.item);
			return;
		}
		super._onOptionSelect(event);
	}

	_formatContextMenuOptions(options) {
		const self = this;
		return _.map(options, option => ({
			...option,
			onSelect: event => {
				if (_.isFunction(option.onSelect)) {
					event.menu = self;
					delete event.contextMenu;
					option.onSelect.call(self, event);
				}
			}
		}));
	}

	_getMoreOptionsElement() {
		return this.element
			.find('[data-menu-more]')
			.parents('li')
			.first();
	}

	enable(id) {
		super.enable(id);
		this._contextMenu.enable(id);
	}

	disable(id) {
		super.disable(id);
		this._contextMenu.disable(id);
	}

	refreshDisplay() {
		const moreOptionsElement = this._getMoreOptionsElement();
		if (!this.maxSize || this.maxSize < 0) {
			for (let i = 0; i < this._menuOptions.length; i++) {
				this._setInContextMenu(this._menuOptions[i].id, false);
			}
			moreOptionsElement.hide();
			return;
		}
		// Only the first shown elements before maxSize should be displayed in
		// the menu, the rest should be displayed in the context menu
		let visibleCount = 0;
		for (var i = 0; i < this._menuOptions.length; i++) {
			const id = this._menuOptions[i].id;
			this._setInContextMenu(id, visibleCount >= this.maxSize);
			if (this.visibilityMap[id]) {
				visibleCount += 1;
			}
		}
		this._contextMenu.refreshDisplay();
		if (visibleCount > this.maxSize) {
			moreOptionsElement.show();
		} else {
			moreOptionsElement.hide();
		}
	}

	_setInContextMenu(id, inContextMenu) {
		const li = this._getOptionElement(id);
		const contextMenuLi = this._contextMenu._getOptionElement(id);
		if (inContextMenu) {
			li.addClass('toolbar-menu-context-menu-item');
			contextMenuLi.addClass('toolbar-menu-context-menu-item');
			li.removeClass('toolbar-menu-item');
			contextMenuLi.removeClass('toolbar-menu-item');
		} else {
			li.removeClass('toolbar-menu-context-menu-item');
			contextMenuLi.removeClass('toolbar-menu-context-menu-item');
			li.addClass('toolbar-menu-item');
			contextMenuLi.addClass('toolbar-menu-item');
		}
	}

	setVisibility(id, visible) {
		this.visibilityMap[id] = visible;
		this._contextMenu.setVisibility(id, visible);
		super.setVisibility(id, visible);
	}

	isVisible(id) {
		let isVisible = super.isVisible(id);
		if (!isVisible) {
			// ID is not visible in the main menu. Check if it is visiblie in
			// the more menu
			isVisible = this._contextMenu.isVisible(id);
		}
		return isVisible;
	}

	add(options) {
		if (!_.isArray(options)) {
			options = [options];
		}

		let menuItems = [];
		_.forEach(options, option => {
			let item = this._buildItem(option);
			this.visibilityMap[option.id] = true;
			menuItems.push(item);
		});
		this.kendoMenu.insertBefore(menuItems, 'li:last-child');

		this._buildOptionsMap(this.optionsMap, options);
		this._menuOptions.push(...options);
		this._contextMenu.add(this._formatContextMenuOptions(options));

		this.queueRefreshDisplay();
	}

	removeAll() {
		this._remove('li:not([data-menu-more])');
		this.optionsMap = {};
		this._menuOptions.length = 0;
		this.visibilityMap = {};
		this._contextMenu.removeAll();
	}

	remove(id) {
		super.remove(id);
		delete this.visibilityMap[id];
		this._contextMenu.remove(id);
	}

	destroy() {
		super.destroy();
		this._contextMenu.destroy();
	}
}
