/**
 * Usage:
 * 
 * Note: Error callback will be used when client timeout, onerror, reconnect and transport failure. So having
 * a good error_callback function is recommended.
 * 
 * pushService.subsribe(success_callback, subscriptionInfo, error_callback);
 * 
 * pushService.unsubscribe(subscriptionInfo);
 */
var pushService = (function() {
	'use strict';
	var socket = null;
	var messageListener = {};
	var errorCallbacks = {};
	var logged = false;
	var contextPath = cv.contextPath + '/ws';
	function init() {
		var reqHeaders = {};
		var localeParam = cvUtil.readCookie('locale');
		var request = {
			url : contextPath,
			contentType : 'application/json',
			transport : 'websocket',
			reconnectInterval : 60000,
			trackMessageLength : true, // ensure message recieved is whole.
			enableXDR : true, // cross origin requests.
			timeout : 360000
		};
		if (localeParam) {
			reqHeaders['Accept-Language'] = localeParam;
			request.headers = reqHeaders;
		}

		request.onMessage = function(response) {
			if (response.responseBody === '') {
				console.debug('Response Body for is empty.');
				return;
			}
			var responseBody = response.responseBody;
			var jobsResp = JSON.parse(responseBody);
			for ( var callback in messageListener) {
				messageListener[callback].forEach(function(listener) {
					listener(jobsResp);
				});
			}
		}

		request.onClose = function(response) {
			console.debug('Closing connection.');
			logged = false;
		}

		request.onOpen = function(response) {
			console.debug('Connected using the ' + request.transport);
			logged = true;
		}

		request.onError = function(response) {
			console.debug('Socket or Server is down');
			handleError(response);
		}

		var handleError = function(response) {
			if (response.responseBody === '') {
				console.debug('Response Body for is empty.');
				return;
			}
			var responseBody = response.responseBody;
			if (responseBody) {
				var resp = JSON.parse(responseBody);

				for ( var errIdx in errorCallbacks) {
					errorCallbacks[errIdx](resp);
				}
			}
		}

		// on web-socket failure- Atmosphere provides additional fallback transport
		// mechanisms. Here we try to connect through long-polling
		// transport mechanism.
		request.onTransportFailure = function(response) {
			console.debug('Connection failed. Trying Fallback option');
			request.fallbackTransport = 'long-polling';
			handleError(response);
		}

		//
		request.onReconnect = function(response) {
			console.debug('Connection lost, trying to reconnect in ' + request.reconnectInterval);
			logged = false;
			handleError(response);
		}

		// on timeout try to reopen the connection
		request.onClientTimeout = function(response) {
			console.debug('Client closed the connection. Reconnecting in: ' + request.reconnectInterval);
			socket = atmosphere.subscribe(request);
			handleError(response);
			logged = false;
		}

		request.onReopen = function(response) {
			console.debug('Re-opening the connection using ' + request.transport);
		}
		socket = atmosphere.subscribe(request);

	}

	if (!socket || (((((socket == null)))))) {
		init();
	}

	// return an object with subscribe and unsubscribe methods.
	// Subscribe method sends a subscription info with the following
	// attributes. 'msg' and 'type' attributes are mandatory.
	// 'subscription' is optional.
	// unsubscribe method doesn't receive a callback.
	// {'msg':'subscribe', 'type':'JOB', ['subscription':{}]}
	// Register the callback recieved from jobs, events and alerts
	// inside messageListener.
	// NOTE: Data sent inside 'type' should follow the SubscriptionType
	// enums standards for code readability.
	return {
		subscribe : function(subsriptionInfo, _callback, _errorCallback) {
			if (!('type' in subsriptionInfo)) {
				console.debug('Subscription Type expected');
				return;
			}
			if (!('msg' in subsriptionInfo)) {
				console.debug('Subscription Message expected');
				return;
			}
			messageListener[subsriptionInfo.type] = messageListener[subsriptionInfo.type] || [];
			messageListener[subsriptionInfo.type].push(_callback);
			errorCallbacks[subsriptionInfo.type] = _errorCallback;
			if (logged) {
				socket.push(JSON.stringify(subsriptionInfo));
			} else {
				if (!socket || (((((socket == null)))))) {
					init();
				}
				setTimeout(function() {
					socket.push(JSON.stringify(subsriptionInfo))
				}, 1000);
			}
			return function unsubscribe() {
				if (messageListener[subsriptionInfo.type]) {
					messageListener[subsriptionInfo.type].splice(messageListener[subsriptionInfo.type]
							.indexOf(_callback), 1);
				}
				if ('msg' in subsriptionInfo) {
					subsriptionInfo.msg = "unsubscribe";
				}
				if (logged) {

					socket.push(JSON.stringify(subsriptionInfo));
				} else {
					if (!socket || (((((socket == null)))))) {
						init();
					}
					setTimeout(function() {
						socket.push(JSON.stringify(subsriptionInfo))
					}, 1000);
				}
			};
		}
	};

})();