/**
 * Get path as an array of a deep property of an object
 * @param {object} obj 
 * @param {string} prop -  property/key inside an object 
 * @param {function(any)} extraValidation -  to perform an extra check when a deep property is found 
 * @returns {array} Array of path to deep property
 */

export const getDeepPropertyPathArray = (obj, prop, extraValidation) => {

    if (obj && _.isObjectLike(obj) && !_.isArray(obj)) {
        if (obj.hasOwnProperty(prop)) {
            if (_.isFunction(extraValidation) && !extraValidation(obj[prop])) {
                return;
            }
            return [];
        }
        for (let key in obj) {
            let pathArray = [];
            const deepPropertyPathArray = getDeepPropertyPathArray(obj[key], prop, extraValidation);
            if (deepPropertyPathArray) {
                pathArray = _.concat(pathArray, key, deepPropertyPathArray);
                return pathArray;
            }
        }
    }
    return false;
};


/**
 * Get path of a deep property of an object
 * @param {object} obj 
 * @param {string} prop -  property/key inside an object 
 * @param {boolean} includePropsInPath -  to include provided prop at the end of returned path
 * @param {function(any)} extraValidation -  to perform an extra check when a deep property is found 
 * @returns {string}  path to deep property of an object
 */

export const getDeepPropertyPath = (obj, prop, includePropsInPath, extraValidation) => {

    const deepPropertyPathArray = getDeepPropertyPathArray(obj, prop, extraValidation);
    if (deepPropertyPathArray) {
        const path = deepPropertyPathArray.join('.');
        const propsInPath = deepPropertyPathArray.length ? '.' + prop : prop;
        return includePropsInPath ? path + propsInPath : path;
    }
    return false;
};

/**
 * get path and required array from Object which have arrays of objects.
 * 
 * @param {object} obj -  Object which have arrays of objects
 * @param {string} prop -  property/key inside an object 
 * @param {function(any)} extraValidation -  to perform an extra check when a deep property is found
 * @returns {object}  {path: path to deep property of  object, array: required array whose object's deep property has to set to root level}
 */

export const getDeepPropertyPathFromArrayInObject = (obj, prop, extraValidation) => {

    for (let key in obj) {
        if (_.isArray(obj[key]) && !_.isEmpty(obj[key])) {
            const path = getDeepPropertyPath(obj[key][0], prop, true, extraValidation);
            if (path) {
                return {
                    path,
                    array: obj[key]
                };
            }
        }
    }
    return {};
};

/**
 * set a deep property of array of Object to its root level by  providing its path.
 * 
 * @param {object[]} arr -  Array of objects whose property has to set to the root level
 * @param {string} path -  path to the deep property
 * @param {string} prop -  property/key inside an object which have to set to the root level
 * @param {function(any)} callback -  invokes callback fn  for each element
 */

export const setPropertyToRootLevelByPath = (arr, path, prop, callback) => {
    _.forEach(arr, ele => {
        ele[prop] = _.get(ele, path);
        _.isFunction(callback) && ele[prop] && callback(ele[prop]);
    });
};


/**
 * set a deep property of an Object to its root level without providing a its path.
 * 
 * @param {object| object[]} obj -  Object which have arrays of objects - set property to the root level of thee objects of the required array
 *                                 | array of object - set property to the root level of thee objects of the required array
 *                                 | *single object : set property the root level of the object
 * @param {string} prop -  property/key inside an object which have to set to the root level
 * @param {function(any)} extraValidation -  to perform an extra check when a deep property is found, deep property will be passed as first arg
 * @param {boolean} hasOnlySingleEntity -  whether *single object or not
 */

const setPropertyToRootLevel =  (obj, prop, extraValidation, hasOnlySingleEntity) => {
    if (!obj || !_.isObjectLike(obj)) {
        return;
    }

    if (hasOnlySingleEntity) {                                                      //single object
        const path = getDeepPropertyPath(obj, prop, true, extraValidation);
        if (path) {
            obj[prop] = _.get(obj, path);
        }
    }

    if (Array.isArray(obj) && obj.length) {                                         //array of object
        const path = getDeepPropertyPath(obj[0], prop, true, extraValidation);
        path && setPropertyToRootLevelByPath(obj, path, prop);
    } else if (!Array.isArray(obj)) {                                              //Object which have arrays of objects
        const pathAndArray = getDeepPropertyPathFromArrayInObject(obj, prop, extraValidation);
        if (!pathAndArray || !pathAndArray.path) {
            return;
        }
        setPropertyToRootLevelByPath(pathAndArray.array, pathAndArray.path, prop);
    }
};


 export default  setPropertyToRootLevel;