/**
 * @author ssubash // This is to isolate the kendo global variable. // We can now inject a kendoTest variable
 *         for // testing
 */
import { IdleQueue } from './idlize/IdleQueue';
import Grid from './grid';
import AppendedMenu from '../appendedmenu';
import ContextMenu from '../kendocontextmenu';
import ToolbarMenu from '../toolbarmenu';
import CallOut from '../call-out';
import * as GridConstants from './grid.constants';
import CsvKendoGridExporter from './grid.exporter.csv';
import Tooltip from '../tooltip';
import DragDrop from '../dragdrop';
import encodeText from '../adminconsole-utils/utils/encodeText';

const PUSH_UPDATE_MIN_TASK_TIME = 10; // In milliseconds
const TIME_BETWEEN_PUSH_REFRESHES = 2000; // In milliseconds

const rowProcessedSymbol = Symbol('Row processed');

export default class KendoGrid extends Grid {
	constructor(options, gridElement, angularLibs) {
		super(options, gridElement, kendo, angularLibs);
		this.queue = new IdleQueue({
			defaultMinTaskTime: PUSH_UPDATE_MIN_TASK_TIME
		});
	}

	build() {
		if (this.buildFinished) {
			return;
		}

		const self = this;

		this._selectionMapCache = {}; // Used to persist selection data across pages
		this.builtRowDetails = [];
		this.columnMenuElements = [];

		if (this.enableCsvExport) {
			this.csvExporter = new CsvKendoGridExporter();
		}

		let pageable = false;
		if (this.enablePaging) {
			pageable = {
				pageSize: this.options.pageSize || 15,
				alwaysVisible: !!this.options.alwaysShowPaging
			};
			if (_.isUndefined(this.options.pageSizes) || _.isNull(this.options.pageSizes)) {
				if (this.enableServerLoading) {
					//for server side pagination,do not show "All" option and limit to 500
					pageable.pageSizes = [10, 20, 50, 100, 200, 500];
				} else {
					// Use kendo default page sizes
					pageable.pageSizes = true;
				}
			} else {
				pageable.pageSizes = this.options.pageSizes;
			}
		} else if (this.enableVirtualization) {
			pageable = {
				numeric: false,
				previousNext: false,
				alwaysVisible: this.enableServerLoading
			};
		}

		this.eventEmitter.on(GridConstants.GRID_COLUMN_FILTER_SET, e => {
			this.callbacks.onColumnFilterSet(e);
		});
		this.eventEmitter.on(GridConstants.GRID_COLUMN_FILTER_CLEAR, e => {
			this.callbacks.onColumnFilterClear(e);
		});
		this.eventEmitter.on(GridConstants.GRID_CLEAR_ALL_COLUMN_FILTERS, e => {
			this.callbacks.onColumnFilterClear({
				clearAll: true
			});
		});

		this.eventEmitter.on(GridConstants.VIEW_ADDED, data => {
			this.changeView(data.selectedViewId);
			this._requestRefreshViewsMenu();
		});

		this.callbacks.beforeGridInitialize.call(this, { grid: this });

		return this.getViewSvc().loaded.finally(() => {
			this.eventEmitter.on(GridConstants.GRID_DATA_SRC_CHANGE, this._onDataChange, this);

			if (!this.enableHorizontalScroll) {
				this.gridElement.addClass('fixed-width');
			}

			this.gridElement.css('height', this.height ? this.height : '');
			this.gridElement.css('max-height', this.height ? this.height : '');

			this.kendoGrid = this.gridElement.kendoGrid({
				dataSource: this.getDataSrc().getDataSource(),
				height: this.height,
				sortable: this.enableSorting ? { allowUnsort: this.allowUnsort } : false,
				columnMenu: this.enableColumnMenu,
				filterable: this.enableFiltering,
				pageable,
				toolbar: kendo.template($(`#toolbar-template`).html())({
					html: this.options.toolbarTemplate || '',
					useClassicToolbar: this.useClassicToolbar,
					usePageToolbar: this.usePageToolbar
				}),
				columns: this.gridColumns.getColumns(),
				scrollable: this.scrollable,
				columnShow: e => {
					this._onToggleColumnVisibility(e.column.field, true);
					this.callbacks.onColumnShow.call(self, { field: e.column.field });
				},
				columnHide: e => {
					this._onToggleColumnVisibility(e.column.field, false);
					this.callbacks.onColumnHide.call(self, { field: e.column.field });
				},
				columnResize: e => {
					this._onColumnResize(e.column.field, e.newWidth);
				},
				sort: e => {
					this._onSortColumn(e.sort.field, e.sort.dir);
				},
				persistSelection: this.persistSelection,
				resizable: this.enableColumnResizing,
				change: e => {
					e.preventDefault();
					self._refreshSelection();
				},
				groupable: this.grouping,
				selectable: !this.enableCheckBoxColumn ? 'row' : !this.enableMultiSelect ? 'row' : 'multiple, row',
				noRecords: {
					template: `#=this.gridEmptyMessage#`
				},
				pageSizes: this.pageSizes,
				detailTemplate: this.rowDetailTemplate,
				detailInit:
					self.detailInit ||
					(!this.callbacks.onRowDetailInit
						? null
						: e => {
								const rowDetailObj = {
									row: e.masterRow,
									detailRow: e.detailRow,
									value: e.data
								};
								this.builtRowDetails.push(rowDetailObj);
								this.callbacks.onRowDetailInit.call(this, rowDetailObj);
								self.requestRefreshScrollbar();
						  }),
				detailExpand: e => {
					self.requestRefreshScrollbar();
				},
				detailCollapse: e => {
					self.requestRefreshScrollbar();
				},
				dataBinding: e => {
					if (this.preventDataBinding) {
						e.preventDefault();
					}
				},
				dataBound(e) {
					// If options.data is used instead of data.url, this dataBound
					// function will be called before the build function continues,
					// so we finish the build here instead.
					self._finishBuild();
					self._fitGridHeight();

					if (self.builtRowDetails.length > 0) {
						self.callbacks.onRowDetailsDestroy.call(self, {
							details: self.builtRowDetails
						});
					}
					self.builtRowDetails = [];

					self.requestRefreshScrollbar();
					self._requestRefreshCheckboxes();
					const rows = e.sender.tbody.children();
					if (self.hideGridOnEmpty && !self.gridRendered) {
						self.gridRendered = true;
						self.gridEmptyElement.removeClass('hidden');
						self._showGridEmptyElement();
					}
					if (rows.length > 0) {
						if (self.hideGridOnEmpty && !self.dataExists) {
							self.dataExists = true;
							// Reset grid empty message to 'No results found' if the grid has data
							self.gridElement.getKendoGrid().gridEmptyMessage = self.noResultsMessage;
							self._hideGridEmptyElement();
						}
						for (let j = 0; j < rows.length; j++) {
							const row = $(rows[j]);
							const dataItem = e.sender.dataItem(row);

							self._processDataItem(dataItem);

							let id;
							if (self.idField) {
								id = _.get(dataItem, self.idField);
							} else {
								// Use the row uid as the id
								id = row.attr('data-uid');
							}
							if (id) {
								// Add row id:
								row.attr('id', `cv-k-grid-tr-${encodeURIComponent(id)}`);
							}
							if (id in self._selectionMapCache) {
								// Update the selection cache
								self._selectionMapCache[id] = dataItem;
							}

							const checkboxLabel = row.find('.k-checkbox-label');
							// checkboxLabel.off();
							checkboxLabel.on('click', event => {
								event.stopPropagation();
								checkboxLabel.parent().trigger($.Event('click', event));
							});
							if (self.enableCheckBoxColumn) {
								//  If dataItem has hideCheckBox, then hide the checkbox on that row
								if (_.get(dataItem, 'hideCheckBox', false)) {
									row
										.find('td')
										.eq(0)
										.empty();
								}
							}
							if (self.disableField && _.get(dataItem, self.disableField)) {
								row.addClass('cv-k-disabled');
							}
							if (self.options.onGridDataBound) {
								self.options.onGridDataBound(dataItem, row);
							}
						}
					}
					const grid = e.sender;
					const gridRows = grid.tbody.find("[role='row']");

					// gridRows.unbind("click");
					gridRows.on('contextmenu', event => {
						self._closeActiveActionButton();
						// On right-click, select the row (if it is not disabled):
						const row = $(event.target).closest('tr');
						const dataItem = self.getDataByRowElement(row);
						if (self.disableField && _.get(dataItem, self.disableField)) {
							// Do not select disabled rows
							return;
						}
						if (self.enableMultiSelect) {
							row
								.addClass('k-state-selected')
								.find('.k-checkbox')
								.attr('aria-checked', 'true');
							if (self.idField) {
								self.grid._selectedIds[_.get(dataItem, self.idField)] = true;
							}
						} else {
							self.grid.select(row);
						}
						self._refreshSelection();
					});

					if (!self.lastRefreshGridTime) {
						// Execute only the first time dataBound is triggered
						if (self.usePageToolbar) {
							self.gridHeaderElement.find('#searchInput').trigger("focus");
						}
					}
					self.lastRefreshGridTime = Date.now();
				},
				columnMenuInit(e) {
					self.columnMenuElements.push(e.container); // Store column menu element to destroy later
					// Add classes for the column menu and children for automation (ids cannot be used because
					// kendo grid is already using ids for some of these elements)
					const columnMenuElement = e.container.find('ul.k-menu');
					const columnMenuAcId = `${self.tableName}_${e.field}_columnMenu`;
					columnMenuElement.addClass(columnMenuAcId);
					columnMenuElement.children('.k-sort-asc').addClass(`${columnMenuAcId}_sortAsc`);
					columnMenuElement.children('.k-sort-desc').addClass(`${columnMenuAcId}_sortDesc`);
					columnMenuElement.children('.k-separator').each((index, li) => {
						$(li).addClass(`${columnMenuAcId}_separator${index}`);
					});
					columnMenuElement.children('.k-filter-item').addClass(`${columnMenuAcId}_filterItem`);
					// Set ac id for the column visibility menu and children:
					const columnsItemElement = columnMenuElement.children('.k-columns-item');
					const columnsItemAcId = `${columnMenuAcId}_columnsItem`;
					columnsItemElement.addClass(columnsItemAcId);
					columnsItemElement.find('li').each((index, li) => {
						const columnName = $(li)
							.find('[data-field]')
							.attr('data-field');
						$(li).addClass(`${columnsItemAcId}_${columnName}`);
					});

					self.gridColumns.onColumnMenuInit(e);
				},
				columnMenuOpen(e) {
					self.openColumn = self.gridColumns.onColumnMenuOpen(e);
					self.initialFilterInputValue =
						self.openColumn && self.openColumn.columnFilter ? self.openColumn.columnFilter.getInputValue() : '';
				},
				filterMenuOpen(e) {
					self.gridColumns.onFilterMenuOpen(e);	
				}
			});

			this._finishBuild();
		});
	}

	_finishBuild() {
		if (this.buildFinished) {
			return;
		}
		this.buildStarted = true;
		const self = this;
		requestAnimationFrame(() => {
			self.tooltip = new Tooltip(self.gridElement);
			self.tooltip.build();
			self.callOut = new CallOut(self.gridElement, self.options.callOutActions);
	        self.callOut.build();
		}, 300);

		if (this.options.enableHeaderCheckbox) {
			$('thead tr th[data-index=0]').css('visibility', 'hidden');
		}

		this.grid = this.gridElement.getKendoGrid();
		this.grid.gridEmptyMessage = this.gridEmptyMessage;
		this.gridHeaderElement = this.gridElement.children('.k-header');

		if (this.enableHorizontalScroll) {
			// Restore the saved table width set by kendo last visit
			this.grid.table.add(this.grid.thead.parent()).css('width', this.gridViewSvc.getTableWidth() || '');
		}

		if (this.enableCheckBoxColumn) {
			// If checkbox selection is enabled, disable selection on row click

			// This is a hacky workaround to remove the click handler bound by
			// kendoGrid that handles row selection without removing the click
			// handlers for expanding/collapsing rows
			const clickHandlers = $._data(this.grid.table[0], 'events')?.click;
			if (clickHandlers) {
				clickHandlers.forEach(clickHandler => {
					if (clickHandler?.selector
						&& clickHandler.selector.indexOf('.k-hierarchy-cell') < 0
						&& clickHandler.selector.indexOf('.k-grouping-row') < 0
					) {
						this.grid.table.off('click', clickHandler.selector, clickHandler.handler);
					}
				});
			}
		}
		// Kendo grid allows drag-to-select on touchscreens, but this overrides
		// touch scroll behavior. Remove drag-to-select functionality to allow
		// touch scrolling:
		this.grid.table.unbind("touchstart");
		this.grid.table.unbind("mousedown");

		if (this.idField && this.enableVirtualScroll && !this.enableServerLoading) {
			// Kendo select-all checkbox will only select visible rows. For grids with
			// virtual scroll, we must manually perform a select all. This requires idField to be set.
			const selectAllCell = this.grid.thead.find('[data-cv-multi-select=true] .k-checkbox-label');
			selectAllCell.on('click', e => {
				e.stopPropagation();
				e.preventDefault();
				if (!this.areAllVisibleSelected()) {
					// If not all selected, select all
					this.selectAllVisible();
				} else {
					// If all rows are selected, deselect all
					this.deselectAllVisible();
				}
			});
		}

		if (this.hideGridOnEmpty) {
			// Grid is hidden by default, shown only once it has some data
			this._updateGridEmptyElement();
			this._showGridEmptyElement();
			// Hide the grid empty element initially, as the kendo grid will
			// display a loading spinner first
			this.gridEmptyElement.addClass('hidden');
		}

		this.setTitle(this.gridTitle);
		this._setupContextMenus(this.options);
		this._setupRowReordering();
		this._setupSearchInput();
		if (this.options.hideToolbar) {
			this.gridHeaderElement.addClass('hidden');
		}

		this._setupCustomSelectAll();
		this._onDataChange();

		this.buildFinished = true;
	}

	_setupCustomSelectAll(){
		if(this.customSelectAll){
			let dataSrc = [
				{ id: "deselectAll",text: this.angularLibs.cvLoc('label.customSelectAll.deselectAll'), icon:"fa fa-square-o", isDisabled:false },
				{ id: "selectAll",text:this.angularLibs.cvLoc('label.customSelectAll.selectAll'), icon:"fa fa-check-square", isDisabled:false }
			];
			this.customSelectAll.data.unshift(...dataSrc);
			//this will tell if its new custom state, to make sure it dont get override by _refreshCheckboxes 
			this.customSelectAll.retainCustomState = false;
			this.gridHeaderCustomSelectID = 'deselectAll';

			$("#customSelectAllElemId").kendoDropDownList({
				dataTextField: "text",
				dataValueField: "id",
				dataSource: this.customSelectAll.data,
				valueTemplate: `<i class="#: data.icon #" style="font-size:18px; text-indent: -0.3em; line-height:1.5em;">`,
				template: `<span class="padding-left-15 padding-right-15 #: isDisabled ? 'k-state-disabled': ''# ">#: data.text #</span>`,
				autoWidth: true,
				change:this._onCustomSelectAllChange.bind(this),
				select:this._onCustomSelectAllSelect.bind(this)
			});
			this.customSelectAll.customSelectAllEle = $("#customSelectAllElemId").data("kendoDropDownList");
		}
	}

	_onCustomSelectAllChange(e){
		if('onChange' in this.customSelectAll){
			this.customSelectAll.onChange(e);
		}
	}

	_onCustomSelectAllSelect(e){
		this.gridHeaderCustomSelectID = _.get(e, 'dataItem.id','deselectAll');
		if(_.get(e.dataItem,'isDisabled',false)){
			e.preventDefault();
			return;
		}
		switch (_.get(e, 'dataItem.id','')) {
			case 'deselectAll':
				this.customSelectAll.retainCustomState = false;
				this.deselectAll();
				this._refreshSelection(); // deselectAll not trigger the refresh grid selection. Manully refresh to update action menu.
				break;
			case 'selectAll':
				this.customSelectAll.retainCustomState = false;
				this.selectAll();
				break;
			default:
				break;
		}
		if('onSelect' in this.customSelectAll){
			this.customSelectAll.onSelect(e);
		}
	}

	setCustomSelectAllDisabled(id,state){
		if (_.has(this, 'customSelectAll.customSelectAllEle')) {
			this.customSelectAll.data.forEach((dataItem,index)=>{
				if(dataItem.id === id){
					this.customSelectAll.customSelectAllEle.dataItem(index).set('isDisabled', state);
				}
			})			
		}
	}


	_closeActiveActionButton() {
		if (this.activeActionButton) {
			this.activeActionButton.removeClass('action-btn-open');
			delete this.activeActionButton;
		}
	}
	
	_openActiveActionButton() {
		if (this.activeActionButton) {
			this.activeActionButton.addClass('action-btn-open');
		}
	}

	_hideScrollbar() {
		this.gridElement.children('.k-grid-header').addClass('no-scroll');
		this.gridElement.children('.k-grid-content').css('width', `calc(100% + ${kendo.support.scrollbar()}px)`);
	}

	_showScrollbar() {
		this.gridElement.children('.k-grid-header').removeClass('no-scroll');
		this.gridElement.children('.k-grid-content').css('width', '100%');
	}

	refreshScrollbar() {
		const gridContentDiv = this.gridElement.children('.k-grid-content');
		if (this.grid.table) {
			if (this.grid.table.height() > gridContentDiv.height()) {
				this._showScrollbar();
			} else {
				this._hideScrollbar();
			}
		}
		this.refreshScrollbarRequested = false;
	}

	requestRefreshScrollbar() {
		if (this.refreshScrollbarRequested) {
			return;
		}
		requestAnimationFrame(() => {
			this.refreshScrollbar();
		});
		this.refreshScrollbarRequested = true;
	}

	_showGridEmptyElement() {
		this.gridElement.addClass('grid-empty');
		this.gridEmptyElement.addClass('grid-empty');
		this.gridEmptyElementShown = true;
	}

	_hideGridEmptyElement() {
		this.gridEmptyElement.removeClass('grid-empty');
		this.gridElement.removeClass('grid-empty');
		this.gridEmptyElementShown = false;
	}

	_onToggleColumnVisibility(field, visible) {
		const column = this.gridColumns.getColumn(field);
		if (column) {
			column.hidden = !visible;
			const columnPref = this.getViewSvc().getColumnPref(field);
			if (!columnPref || columnPref.isHidden != column.hidden) {
				this.getViewSvc().toggleColumnVisibility(field);
			}
		}
	}

	_onColumnResize(field, width) {
		const column = this.gridColumns.getColumn(field);
		if (column) {
			if (typeof width === 'number') {
				if (this.enableHorizontalScroll) {
					width = `${width}px`;
				} else {
					// Convert number to a percentage:
					const tableWidth = this.gridElement
						.children('.k-grid-header')
						.find('table')
						.width();
					width = `${(width * 100) / tableWidth}%`;
				}
			}
			let tableWidth = '';
			if (this.enableHorizontalScroll) {
				// Save the table width set by kendo
				tableWidth = this.grid.table[0].style.getPropertyValue('width');
			}
			const widthObj = {};
			widthObj[field] = width;
			this.getViewSvc().saveColumnWidths(widthObj, tableWidth);
		}
	}

	_onSortColumn(field, dir) {
		const column = this.gridColumns.getColumn(field);
		if (column) {
			this.getViewSvc().saveColumnSort(column, dir);
		}
	}

	getDataByRowElement(row) {
		return this.grid.dataItem(row);
	}

	refreshData(hardRefresh) {
		this.getDataSrc()
			.getDataSource()
			.read({
				hardRefresh: !!hardRefresh
			});
	}

	/**
	 * Redraws the grid table
	 */
	refreshGrid() {
		if (this.grid) {
			this._executeDataTaskCallback(null, this.grid.refresh.bind(this.grid));
		}
	}

	_requestAnimationFrameForRefreshGrid() {
		delete this.currentWaitTimeFromLastRefresh;

		if (!this.refreshGridFrameRequested) {
			this.refreshGridFrameRequested = true;
			requestAnimationFrame(() => {
				this.refreshGridFrameRequested = false;
				this.refreshGrid();
			});
		}
	}

	requestRefreshGrid(waitTimeFromLastRefresh) {
		clearTimeout(this.refreshGridTimeoutId); // Cancel the existing request
		if (!waitTimeFromLastRefresh || waitTimeFromLastRefresh <= 0) {
			// Immediately request the refresh if no waitTime was defined
			this._requestAnimationFrameForRefreshGrid();
		} else {
			// If a refresh was already requested sooner than the new waitTime, use the smaller waitTime:
			if (this.currentWaitTimeFromLastRefresh && this.currentWaitTimeFromLastRefresh < waitTimeFromLastRefresh) {
				waitTimeFromLastRefresh = this.currentWaitTimeFromLastRefresh;
			} else {
				this.currentWaitTimeFromLastRefresh = waitTimeFromLastRefresh;
			}
			// Calculate the remaining wait time:
			const timeSinceLastRefreshGrid = Date.now() - this.lastRefreshGridTime;
			const remainingWaitTime = waitTimeFromLastRefresh - timeSinceLastRefreshGrid - 1;
			if (remainingWaitTime <= 0) {
				// Wait time is up, request the refresh immediately
				this._requestAnimationFrameForRefreshGrid();
			} else {
				this.refreshGridTimeoutId = setTimeout(() => {
					this._requestAnimationFrameForRefreshGrid();
				}, remainingWaitTime);
			}
		}
	}

	/**
	 * Subtracts the height of the pager, toolbar, and header from the sourceHeight
	 * to calculate the height of the grid content.
	 * If sourceHeight is not specified, the gridElement's height will be used
	 */
	_calculateGridContentHeight(sourceHeight) {
		let pagerHeight = this.gridElement.children('.k-grid-pager').outerHeight(true);
		if (this.enableVirtualization && !this.enableServerLoading) {
			// Paging is hidden for virtualization that is not server loading
			this.gridElement.children('.k-grid-pager').addClass('hidden');
			pagerHeight = 0;
		}
		const toolbarHeight = this.gridElement.children('.k-grid-toolbar').outerHeight(true);
		const headerHeight = this.gridElement.children('.k-grid-header').outerHeight(true);
		const totalHeight = sourceHeight ? sourceHeight : this.gridElement.height();
		const gridHeight = totalHeight - (toolbarHeight + headerHeight + pagerHeight);
		return gridHeight;
	}

	_fitGridHeight() {
		if (this.gridHeightFitted) {
			return;
		}
		if (this.parentModal.length > 0 && this.useFixedHeight) {
			// If height is unset and the grid is inside a modal, calculate height
			// automatically
			const self = this;
			const MIN_HEIGHT = 250;
			requestAnimationFrame(() => {
				const gridContentElement = self.gridElement.children('.k-grid-content').first();

				// Have the grid fill the modal:
				const modalHeaderHeight = self.parentModal.find('.modal-header, .setup-title').first().outerHeight(true);
				const modalFooterHeight = self.parentModal.find('.modal-footer, .button-container').last().outerHeight(true);
				const modalHeight = self.parentModal.height();
				let newGridContentHeight = self._calculateGridContentHeight(modalHeight - modalHeaderHeight - modalFooterHeight);
				gridContentElement.css('max-height', newGridContentHeight);
				gridContentElement.css('height', newGridContentHeight);

				// If too big, shrink the modal
				const modalContentHeight = self.parentModal.find('.modal-content').first().outerHeight(true);
				if (modalContentHeight > modalHeight) {
					let diff = modalContentHeight - modalHeight;
					newGridContentHeight -= diff;
					if (newGridContentHeight < MIN_HEIGHT) {
						newGridContentHeight = MIN_HEIGHT;
					}
					gridContentElement.css('max-height', newGridContentHeight);
				}
				gridContentElement.css('height', null);
			});
		} else if (this.height) {
			const self = this;
			requestAnimationFrame(() => {
				const gridHeight = self._calculateGridContentHeight();
				if (gridHeight > 0) {
					this.gridElement.children('.k-grid-content').css('max-height', gridHeight);
				}
			});
		}
		this.gridHeightFitted = true;
	}

	isBuildFinished() {
		return this.buildFinished;
	}

	isBuildStarted() {
		return this.buildStarted;
	}

	setTitle(title) {
		super.setTitle(title);
		if (this.gridHeaderElement) {
			const headerElement = this.gridHeaderElement.find('#grid-title-text');
			if (title) {
				headerElement.removeClass('hidden');
			} else {
				headerElement.addClass('hidden');
			}
			headerElement.html(title);
		}
	}

	_setupRowReordering() {
		const self = this;
		if (this.options.enableRowReordering) {
			const options = {
				filter: '>tbody >tr',
				placeholder(element) {
					return $('<tr colspan="4" class="placeholder"></tr>');
				},
				cursor: 'move',
				change(event) {
					const skip = self.grid.dataSource.skip();
					const oldIndex = event.oldIndex + skip;
					const newIndex = event.newIndex + skip;
					const data = self.grid.dataSource.data();
					const dataItem = self.grid.dataSource.getByUid(event.item.data('uid'));

					self.grid.dataSource.remove(dataItem);
					self.grid.dataSource.insert(newIndex, dataItem);
				},
				hint(element) {
					const style = 'width: ${self.width}px;';
					const table = $(`<table style=${style} class="k-grid k-widget"></table>`);
					table.append(element.clone());
					table.css('opacity', 0.8);
					return table;
				}
			};
			const dragDrop = new DragDrop(this.grid.table, options);
			dragDrop.build();
		}
	}

	_hideSearchInput() {
		if (!this.searchHidden) {
			this.gridHeaderElement.find('#search-input-container').addClass('hidden');
			this.searchHidden = true;
		}
	}

	_showSearchInput() {
		if (this.searchHidden) {
			this.gridHeaderElement.find('#search-input-container').removeClass('hidden');
			this.searchHidden = false;
		}
	}

	_setupSearchInput() {
		const inputElement = this.gridHeaderElement.find('#searchInput');
		const clearElement = this.gridHeaderElement.find('.k-i-close');
		if (!this.options.searchable) {
			this._hideSearchInput();
		}
		if (this.useClassicToolbar) {
			clearElement.addClass('hidden');
		}
		const self = this;
		inputElement.on(
			'input',
			_.debounce(
				function(e) {
					self.search($(this).val());
				},
				self.enableServerLoading ? 1200 : 200
			)
		);
		inputElement.on('focus', e => {
			this.gridHeaderElement.find('.cv-k-grid-toolbar-primary').addClass('cv-k-grid-search-focus');
		});
		inputElement.on('blur', e => {
			if (!this._preventSearchInputBlur) {
				this.gridHeaderElement.find('.cv-k-grid-toolbar-primary').removeClass('cv-k-grid-search-focus');
			}
			this._preventSearchInputBlur = false;
		});

		this.gridHeaderElement.find('#search-input-container label').on('focus', e => {
			inputElement.trigger("focus");
		});

		clearElement.on('mousedown touchstart', e => {
			this._preventSearchInputBlur = true;
		});
		clearElement.on('click', e => {
			inputElement.val('');
			this.search('');
			inputElement.trigger("focus");
		});
	}

	_setupContextMenus(options) {
		const self = this;

		// Setup grid context menu:
		this.gridContextMenu = new ContextMenu({
			target: 'body',
			filter: `${this.getSelector()} > .k-header .k-pager-refresh`,
			showOn: 'click',
			cssClass: 'grid-header-context-menu',
			name: `${this.tableName}_gridContextMenu`,
			attrs: {
				[Tooltip.ATTRS.ALLOW_TOOLTIPS]: 'true'
			},
			options: [
				{
					id: 'clearFilter',
					label: this.angularLibs.cvLoc('gridMenu.clearAllFilters'),
					onSelect: event => {
						self._onClearFilter();
					}
				},
				{
					id: 'editView',
					label: this.angularLibs.cvLoc('label.editView'),
					onSelect: event => {
						self.editView();
					}
				},
				{
					id: 'createView',
					label: this.angularLibs.cvLoc('label.createView'),
					hidden: this.disableViewCreation,
					onSelect: event => {
						self.createView();
					}
				},
				{
					id: 'deleteView',
					label: this.angularLibs.cvLoc('label.deleteView'),
					onSelect: event => {
						self.deleteView();
					}
				},
				{
					id: 'reorderViews',
					label: this.angularLibs.cvLoc('label.reorderViews'),
					onSelect: event => {
						self.reorderViews();
						// TODO angular digest?
					}
				},
				{
					id: 'resetColumns',
					label: this.angularLibs.cvLoc('label.resetColumnSize'),
					onSelect: event => {
						self.resetColumnWidth();
					}
				},
				{
					id: 'resetColumnVisibility',
					label: this.angularLibs.cvLoc('label.resetColumnVisibility'),
					onSelect: event => {
						self.resetColumnVisibilities();
					},
				},
				{
					id: 'exportToCsv',
					label: this.angularLibs.cvLoc('label.exportToCsv'),
					hidden: !self.enableCsvExport,
					onSelect: event => {
						self.exportToCsv();
					}
				}
			],
			onOpen: event => {
				if (options.hasViews === false) {
					event.contextMenu.hide('deleteView');
					event.contextMenu.hide('editView');
					event.contextMenu.hide('createView');
					event.contextMenu.hide('reorderViews');
				} else {
					const view = self.gridViewSvc.getCurrentView();
					// TODO deleteView should only be hidden if there are no views that can be deleted
					if (view && view.canBeDeleted) {
						event.contextMenu.show('deleteView');
						event.contextMenu.show('editView');
					} else {
						event.contextMenu.hide('deleteView');
						event.contextMenu.hide('editView');
					}
					const visibleViews = self.gridViewSvc.getVisibleViews();
					if (visibleViews.length > 1) {
						event.contextMenu.show('reorderViews');
					} else {
						event.contextMenu.hide('reorderViews');
					}
				}
			}
		});
		_.forEach(options.gridContextMenu, menuOption => {
			this.gridContextMenu.add(menuOption);
		});
		// Hide the gridContextMenu:
		if (
			options.hideGridContextMenu ||
			(options.hasViews === false && (!_.isArray(options.gridContextMenu) || _.isEmpty(options.gridContextMenu)))
		) {
			this.gridHeaderElement.find('#grid-context-menu-container').addClass('hidden');
		}

		// Setup views dropdown:
		this.viewsMenu = new ContextMenu({
			target: 'body',
			filter: `${this.getSelector()} > .k-header .view-selection .dropdown-toggle`,
			cssClass: 'views-menu',
			showOn: 'click',
			name: `${this.tableName}_viewsMenu`,
			attrs: {
				[Tooltip.ATTRS.ALLOW_TOOLTIPS]: 'true'
			},
			options: [],
			onSelect: event => {
				if (event.optionId === 'systemViews' || event.optionId === 'customViews') {
					return;
				}
				this.changeView(event.optionId);
			}
		});
		this._refreshViewsMenu();

		// Setup grid toolbar and action menus:
		const _buildMenuEvent = function(event) {
			const selected = self.getSelectedRows();
			let currentRowValue;
			if (event.rowUid) {
				currentRowValue = self.getDataSrc().getDataByUid(event.rowUid);
			}
			return {
				optionId: event.optionId,
				grid: self,
				rowValue: currentRowValue,
				selectedRowValues: selected.values,
				selectedRows: selected.rows
			};
		};

		this.gridToolbarMenuElement = this.gridHeaderElement.find('#toolbar-menu');
		this.gridToolbarMenuContainerElement = this.gridToolbarMenuElement.parent();
		if (options.gridToolbarMenu && options.gridToolbarMenu.length > 0) {
			this.gridToolbarMenu = new AppendedMenu({
				element: this.gridToolbarMenuElement,
				name: `${this.tableName}_gridToolbarMenu`,
				options: options.gridToolbarMenu.map(option => ({
					...option,
					cssClass: `${option.cssClass || ''} ${option.hideOnGridEmpty ? 'hidden-on-grid-empty' : ''}`,
					onSelect: event => {
						if (_.isFunction(option.onSelect)) {
							option.onSelect.call(self, _buildMenuEvent(event));
						}
					}
				}))
			});
			this._refreshToolbarMenuWidth();
		} else {
			this.gridToolbarMenuContainerElement.addClass('hidden');
		}

		if (this.hideGridOnEmpty) {
			if (options.gridEmptyMenu && options.gridEmptyMenu.length > 0) {
				this.gridEmptyMenu = new AppendedMenu({
					element: this.gridEmptyElement.find('#grid-empty-menu'),
					name: `${this.tableName}_gridEmptyMenu`,
					options: options.gridEmptyMenu.map(option => ({
						...option,
						onSelect: event => {
							if (_.isFunction(option.onSelect)) {
								option.onSelect.call(self, _buildMenuEvent(event));
							}
						}
					}))
				});
				this.gridEmptyElement.find('#grid-empty-menu-container').removeClass('hidden');
			} else {
				this.gridEmptyElement.find('#grid-empty-menu-container').addClass('hidden');
			}
		}

		this.batchActionMenuElement = this.gridHeaderElement.find('#batch-action-menu');
		this.batchActionMenuContainerElement = this.batchActionMenuElement.parent();
		if (options.actionMenu && options.actionMenu.length > 0) {
			this.optionsToDisableOnDeselect = [];
			const _getOptionsToDisableOnDeselect = (options, result) => {
				if (!_.isArray(options)) {
					return;
				}
				options.forEach(option => {
					if (!('disableOnDeselect' in option) || option.disableOnDeselect) {
						result.push(option.id);
					}
					_getOptionsToDisableOnDeselect(option.subOptions, result);
				});
			};
			_getOptionsToDisableOnDeselect(options.actionMenu, this.optionsToDisableOnDeselect);
			const _buildMenuOptions = menuOptions => {
				if (!_.isArray(menuOptions) || _.isEmpty(menuOptions)) {
					return null;
				}
				return menuOptions.map(option => ({
					...option,
					subOptions: _buildMenuOptions(option.subOptions),
					disabled: 'disableOnDeselect' in option ? option.disableOnDeselect : true,
					onSelect: event => {
						if (_.isFunction(option.onSelect)) {
							option.onSelect.call(self, _buildMenuEvent(event));
						}
					}
				}));
			};

			const batchActionMenuList = options.actionMenu.filter(option => !option.contextMenuOnly);
			if (batchActionMenuList.length > 0) {
				this.batchActionMenu = new ToolbarMenu({
					element: this.batchActionMenuElement,
					name: `${this.tableName}_batchActionMenu`,
					options: _buildMenuOptions(batchActionMenuList),
					maxSize: 2,
					moreItemTemplate: this.useClassicToolbar
						? '<span class="k-icon k-i-more-vertical" data-menu-more=""></span>'
						: options.moreItemTemplate? options.moreItemTemplate: undefined
				});
				this._refreshBatchActionMenuWidth();
				this._refreshBatchActionMenuVisibility();
			} else {
				this.batchActionMenuContainerElement.addClass('hidden');
			}
			const actionContextMenuList = options.actionMenu.filter(option => !option.batchMenuOnly);
			if (actionContextMenuList.length > 0) {
				this.actionContextMenu = new ContextMenu({
					alignToAnchor: false,
					target: 'body',
					name: `${this.tableName}_actionContextMenu`,
					attrs: {
						[Tooltip.ATTRS.ALLOW_TOOLTIPS]: 'true'
					},
					options: _buildMenuOptions(actionContextMenuList),
					filter: `${this.getSelector()} > .k-grid-content > table > tbody > tr, ${this.getSelector()} > .k-grid-content > .k-virtual-scrollable-wrap > table > tbody > tr`,
					onOpen: event => {
						self._openActiveActionButton();
						if (_.isFunction(options.onActionMenuOpen)) {
							options.onActionMenuOpen.call(self, _buildMenuEvent(event));
						}
					},
					onClose: event => {
						self._closeActiveActionButton();
					}
				});
				this.grid.tbody.on('click', '> tr:not(.k-detail-row) .action-btn', event => {
					self._closeActiveActionButton();
					self.activeActionButton = $(event.currentTarget);

					// Select the row (if it is not disabled):
					const row = $(event.target).closest('tr');
					const dataItem = self.getDataByRowElement(row);
					if (!self.disableField || !_.get(dataItem, self.disableField)) {
						// Do not select disabled rows
						self.grid.clearSelection();
						self.grid.select(row);
						self._refreshSelection();
					}

					setTimeout(() => {
						// setTimeout used so that the selection change event can be
						// triggered first
						self.actionContextMenu.open(event.currentTarget);
					});
				});
			}
		} else {
			this.batchActionMenuContainerElement.addClass('hidden');
		}
	}

	changeView(viewId) {
		if (this.gridViewSvc.isViewHidden(viewId)) {
			// Do nothing if the target view is hidden
			return;
		}
		const previousViewId = this.gridViewSvc.getCurrentViewId();
		let defaultPrevented = false;
		const beforeViewChangeEvent = {
			grid: this.kendoGrid,
			preventDefault: () => {
				defaultPrevented = true;
			},
			previousViewId,
			previousView: this.gridViewSvc.getView(previousViewId),
			viewId, // TODO rename viewId/view to currentViewId/currentView
			view: this.gridViewSvc.getView(viewId)
		};
		Promise.resolve(this.callbacks.beforeViewChange.call(this, beforeViewChangeEvent)).then(() => {
			if (defaultPrevented) return;
			this.angularLibs.$location.search('view', viewId);
			this._refreshViewsMenuLink();
			this.gridViewSvc.applyFilters();

			const afterViewChangeEvent = {
				grid: this.kendoGrid,
				previousViewId,
				previousView: this.gridViewSvc.getView(previousViewId),
				viewId, // TODO rename viewId/view to currentViewId/currentView
				view: this.gridViewSvc.getView(viewId)
			};
			this.callbacks.afterViewChange.call(this, afterViewChangeEvent);
		});
	}

	hideView(viewId) {
		super.hideView(viewId);
		if (viewId === this.gridViewSvc.getCurrentViewId()) {
			// If hiding the current view, change the view to the default
			this.changeView(this.gridViewSvc.getValidDefaultView());
		}
		this._requestRefreshViewsMenu();
	}

	showView(viewId) {
		super.showView(viewId);
		this._requestRefreshViewsMenu();
	}

	_showViewsMenuLink() {
		this.gridHeaderElement.find('.view-selection').removeClass('hidden');
	}

	_hideViewsMenuLink() {
		this.gridHeaderElement.find('.view-selection').addClass('hidden');
	}

	_refreshViewsMenuLink() {
		if (this.options.hasViews) {
			this._showViewsMenuLink();
		} else {
			this._hideViewsMenuLink();
			return;
		}
		const currentViewId = this.gridViewSvc.getCurrentViewId();
		const views = this.gridViewSvc.getAllViews();
		if (!views || views.filter(view => view.id === currentViewId || !this.gridViewSvc.isViewHidden(view.id)).length <= 1) {
			this._hideViewsMenuLink();
			return;
		}
		const view = this.gridViewSvc.getCurrentView();
		if (view) {
			let name = decodeURIComponent(view.name);
			if (!this.useClassicToolbar && !this.usePageToolbar && name === this.gridViewSvc.allViewName) {
				// The "All" view should just display "View" in the new toolbar
				name = this.angularLibs.cvLoc('label.view');
			}
			this.gridHeaderElement.find('.cv-k-view-dropdown-label').text(name);
			if (view.description) {
				this.gridHeaderElement.find('.view-selection').attr('title', view.description);
			} else {
				this.gridHeaderElement.find('.view-selection').attr('title', null);
			}
		} else {
			this.gridHeaderElement.find('.cv-k-view-dropdown-label').html('&nbsp;');
		}
	}

	_requestRefreshViewsMenu() {
		if (this.viewsMenu && !this.refreshViewsMenuRequested) {
			const self = this;
			requestAnimationFrame(() => {
				self._refreshViewsMenu();
			});
			this.refreshViewsMenuRequested = true;
		}
	}

	_refreshViewsMenu() {
		const views = this.gridViewSvc.getAllViews();
		const currentViewId = this.gridViewSvc.getCurrentViewId();
		this.viewsMenu.removeAll();
		const encoderElem = document.createElement('div');
		_.forEach(views, view => {
			if (view) {
				let cssClass = '';
				if (view.id === currentViewId) {
					cssClass = 'selected-view';
				}
				const attrs = {};
				if (view.description) {
					attrs.title = view.description;
				}
				const name = decodeURIComponent(view.name);
				this.viewsMenu.add({
					id: view.id,
					label: encodeText(name, encoderElem),
					cssClass,
					attrs,
					hidden: this.gridViewSvc.isViewHidden(view.id) && view.id !== currentViewId
				});
			}
		});
		if (currentViewId) {
			this.viewsMenu._getOptionElement(currentViewId).addClass('selected-view');
		}
		this._refreshViewsMenuLink();
		this.refreshViewsMenuRequested = false;
	}

	_onDataChange() {
		//if grid uses gridDataSrc.data instead of url, onChange event will be triggered before _finishBuild. In this case, we need to build grid first.
		if (!this.isBuildFinished() && !this.isBuildStarted()) {
			this._finishBuild();
		}

		if (this.enablePaging) {
			const currentPage = this.getPage();
			const totalPages = this.getTotalPages();
			if (currentPage >= totalPages) {
				// If past the last page, go to the last page
				this.setPage(totalPages - 1);
			}
		}
		// removed condition to check if its not server side loading, 
		//search box should always appear if we only have >=10 items in the list
		if (this.options.searchable) {
			// Refresh the search input visibility. 
			this._refreshSearchInput();
		}
	}

	_refreshSearchInput() {
        const { searchFilter } = this.getDataSrc();
        if (searchFilter) {
            // If the user has a search input a search, keep the
            // search input visible
            this._showSearchInput();
            return;
        }
        if (this.getDataSrc().getDataSource().total() < 10 && !this.showSearchOnFewItems) {
            this._hideSearchInput();
        } else {
            this._showSearchInput();
        }
	}
	
	getTotalCount(){
		return this.getDataSrc().getDataSource().total();
	}

	synckendoGrid(){
		this.getDataSrc().getDataSource().sync();

	}

	setOptions(options){
		setTimeout(() => {
			this.grid.setOptions(options);
		});
	}

	_onClearFilter() {
		// this.gridElement.find("#searchInput").val('');
		// this.gridDataSrc.setSearch('');
		this.gridColumns.clearColumnFilters(false);
		this.gridDataSrc.clearUserFilters(true);
	}

	_updateGridEmptyElement() {
		const newGridEmptyElement = $(
			this.kendo.template(
				$(`#grid-empty-template`)
					.first()
					.html()
			)(this)
		);
		if (this.gridEmptyElement) {
			if (this.gridEmptyMenu) {
				this.gridEmptyMenu.element.detach();
			}
			this.gridEmptyElement.replaceWith(newGridEmptyElement);
			if (this.gridEmptyMenu) {
				newGridEmptyElement.find('#grid-empty-menu').replaceWith(this.gridEmptyMenu.element);
			} else {
				newGridEmptyElement.find('#grid-empty-menu-container').addClass('hidden');
			}
		} else {
			this.gridElement.children('.k-grid-toolbar').after(newGridEmptyElement);
		}
		this.gridEmptyElement = newGridEmptyElement;
	}

	_requestRefreshCheckboxes() {
		if (!this.refreshCheckboxesRequested) {
			const self = this;
			requestAnimationFrame(() => {
				self._refreshCheckboxes();
			});
			this.refreshCheckboxesRequested = true;
		}
	}

	_refreshCheckboxes() {
		let allSelected = true;
		if (this.grid.tbody) {
			this.grid.tbody
				.children()
				.not('.k-detail-row')
				.each((index, element) => {
					const row = $(element);
					if(_.get(this, 'customSelectAll.retainCustomState', false)){
						row.find('.k-checkbox').attr('aria-checked', true);
					} else {
						const selected = row.hasClass('k-state-selected');
						row.find('.k-checkbox').attr('aria-checked', selected);
						if (!selected) {
							allSelected = false;
						}
					}
				});
		}
		if (this.grid.thead) {
			if (allSelected && this.idField && this.enableVirtualScroll && !this.enableServerLoading) {
				// Only visible rows have been checked so far. For grids with virtual scroll, we must
				// manually check the current data. This requires idField to be set.
				allSelected = this.areAllVisibleSelected();
			}
			this.grid.thead.find('.k-checkbox').attr('aria-checked', allSelected);
			if(this.customSelectAll){
				if(this.customSelectAll.retainCustomState){
					this.customSelectAll.customSelectAllEle.value(this.gridHeaderCustomSelectID);
				} else if(allSelected){
					this.customSelectAll.customSelectAllEle.value("selectAll");
				} else {
					//change icon to delselect
					this.customSelectAll.customSelectAllEle.value("deselectAll");
				}
			}
		}
		this.refreshCheckboxesRequested = false;
	}

	_requestRefreshSelection() {
		if (!this.refreshSelectionRafId) {
			this.refreshSelectionRafId = requestAnimationFrame(() => {
				this._refreshSelection();
			});
		}
	}

	/**
	 * Removes deselected rows from the selection cache and updates the cache
	 * with the latest data objects avilable in the data source
	 */
	_updateSelectionCache() {
		if (!this.idField) {
			// Do nothing if idField is not set
			return;
		}
		// Remove deselected rows from the cache:
		_.forEach(Object.keys(this._selectionMapCache), (dataObj, id) => {
			if (!this.grid._selectedIds[id]) {
				delete this._selectionMapCache[id];
			}
		});
		// Add the current selection to the cache:
		_.forEach(this.grid._selectedIds, (selected, id) => {
			if (selected) {
				const dataObj = this.getData(id);
				if (dataObj) {
					// When enableServerLoading is true, only data currently
					// visible will be available. We only update the cache for
					// data that is currently available
					this._selectionMapCache[id] = dataObj;
				}
			}
		});
	}

	_refreshSelection() {
		if (this.refreshInProgress) {
			// Do not process selection changes while the grid is refreshing
			this._requestRefreshSelection();
			return;
		}
		if (this.refreshSelectionRafId) {
			cancelAnimationFrame(this.refreshSelectionRafId);
			delete this.refreshSelectionRafId;
		}

		this._requestRefreshCheckboxes();

		let selected = {
			rows: [],
			values: []
		};

		const newSelectionIds = new Set();
		const newSelection = this.grid.select();
		newSelection.each((index, elem) => {
			const dataItem = this.grid.dataItem(elem);
			const id = _.get(dataItem, this.idField);
			if (this.disableField && dataItem && _.get(dataItem, this.disableField)) {
				// Deselect all rows that are disabled
				this.deselect(id, true);
				$(elem).removeClass('k-state-selected');
			} else {
				selected.rows.push($(elem));
				selected.values.push(dataItem);
				newSelectionIds.add(String(id));
			}
		});

		this._updateSelectionCache();
		if (this.enableMultiSelect) {
			// Get the full selection
			selected = this.getSelectedRows();
		} else {
			// Multiselect disabled, remove all other previously selected rows
			Object.keys(this.grid._selectedIds).forEach(id => {
				if (!newSelectionIds.has(id)) {
					this.deselect(id, true);
				}
			});
		}

		if (this.optionsToDisableOnDeselect) {
			if (selected.rows.length > 0) {
				this.optionsToDisableOnDeselect.forEach(id => {
					this.enableActionMenuOption(id);
				});
			} else {
				this.optionsToDisableOnDeselect.forEach(id => {
					this.disableActionMenuOption(id);
				});
			}
		}

		this.callbacks.onGridSelectionChange.call(this, {
			grid: this,
			values: selected.values,
			rows: selected.rows
		});

		if(this.customSelectAll && !this.customSelectAll.retainCustomState){
			// if user change grid manully then call reset.
			if('reset' in this.customSelectAll){
				this.customSelectAll.reset();
			}
		}
	}

	/**
	 * Sets the width of the batch action menu container to the width of the batch action menu. Needed for css
	 * transitions to work correctly.
	 */
	_refreshBatchActionMenuWidth() {
		if (this._refreshBatchActionMenuWidthRequested) {
			return;
		}
		this._refreshBatchActionMenuWidthRequested = true;
		requestAnimationFrame(() => {
			this._refreshBatchActionMenuWidthRequested = false;
			if (!this.useClassicToolbar) {
				this.batchActionMenuContainerElement.css('width', this.batchActionMenuElement.width());
			}
		});
	}

	_refreshBatchActionMenuVisibility() {
		if (!this.batchActionMenu || this._refreshBatchActionMenuVisibilityRequested) {
			return;
		}
		this._refreshBatchActionMenuVisibilityRequested = true;
		requestAnimationFrame(() => {
			this._refreshBatchActionMenuVisibilityRequested = false;
			if (this.batchActionMenu.hasVisibleChildren()) {
				this.batchActionMenuContainerElement.removeClass('hidden');
			} else {
				this.batchActionMenuContainerElement.addClass('hidden');
			}
		});
	}

	/**
	 * Sets the width of the toolbar menu container to the width of the toolbar menu. Needed for css
	 * transitions to work correctly.
	 */
	_refreshToolbarMenuWidth() {
		requestAnimationFrame(() => {
			if (!this.useClassicToolbar && !this.usePageToolbar) {
				this.gridToolbarMenuContainerElement.css('width', this.gridToolbarMenuElement.width());
			}
		});
	}

	enableActionMenuOption(id) {
		if (this.batchActionMenu) {
			this.batchActionMenu.enable(id);
		}
		if (this.actionContextMenu) {
			this.actionContextMenu.enable(id);
		}
	}

	disableActionMenuOption(id) {
		if (this.batchActionMenu) {
			this.batchActionMenu.disable(id);
		}
		if (this.actionContextMenu) {
			this.actionContextMenu.disable(id);
		}
	}

	hideActionMenuOption(id) {
		if (this.batchActionMenu) {
			this.batchActionMenu.hide(id);
			this._refreshBatchActionMenuWidth();
			this._refreshBatchActionMenuVisibility();
		}
		if (this.actionContextMenu) {
			this.actionContextMenu.hide(id);
		}
	}

	showActionMenuOption(id) {
		if (this.batchActionMenu) {
			this.batchActionMenu.show(id);
			this._refreshBatchActionMenuWidth();
			this._refreshBatchActionMenuVisibility();
		}
		if (this.actionContextMenu) {
			this.actionContextMenu.show(id);
		}
	}

	enableToolbarMenuOption(id) {
		if (this.gridToolbarMenu) {
			this.gridToolbarMenu.enable(id);
		}
	}

	disableToolbarMenuOption(id) {
		if (this.gridToolbarMenu) {
			this.gridToolbarMenu.disable(id);
		}
	}

	hideToolbarMenuOption(id) {
		if (this.gridToolbarMenu) {
			this.gridToolbarMenu.hide(id);
			this._refreshToolbarMenuWidth();
		}
	}

	showToolbarMenuOption(id) {
		if (this.gridToolbarMenu) {
			this.gridToolbarMenu.show(id);
			this._refreshToolbarMenuWidth();
		}
	}

	enableGridEmptyMenuOption(id) {
		if (this.gridEmptyMenu) {
			this.gridEmptyMenu.enable(id);
		}
	}

	disableGridEmptyMenuOption(id) {
		if (this.gridEmptyMenu) {
			this.gridEmptyMenu.disable(id);
		}
	}

	hideGridEmptyMenuOption(id) {
		if (this.gridEmptyMenu) {
			this.gridEmptyMenu.hide(id);
		}
	}

	showGridEmptyMenuOption(id) {
		if (this.gridEmptyMenu) {
			this.gridEmptyMenu.show(id);
		}
	}

	hideGridContextMenuOption(id) {
		if (this.gridContextMenu) {
			this.gridContextMenu.hide(id);
		}
	}

	showGridContextMenuOption(id) {
		if (this.gridContextMenu) {
			this.gridContextMenu.show(id);
		}
	}

	_getRowValues(rows) {
		const values = [];
		rows.each((index, row) => {
			const rowUid = $(row).attr('data-uid');
			values.push(
				this.gridDataSrc
					.getDataSource()
					.getByUid(rowUid)
					.toJSON()
			);
		});

		return {
			rows,
			values
		};
	}

	getDisplayedRows() {
		return this._getRowValues(this.grid.items());
	}

	getSelectedRows() {
		if (this.idField) {
			// If idField is set, use _selectedIds as it will persist selection
			const selection = {
				rows: [],
				values: []
			};
			_.forEach(this.grid._selectedIds, (selected, id) => {
				if (selected) {
					let dataItem = this.getData(id);
					if (!dataItem) {
						// Data item is not available on the current page, check the cache
						dataItem = this._selectionMapCache[id];
					}
					let dataRow;
					if (dataItem) {
						dataRow = this.grid.tbody.children(`[data-uid="${dataItem.uid}"]`);
						selection.values.push(dataItem);
						selection.rows.push(dataRow);
					}
				}
			});
			return selection;
		}
		// Without idField, _selectedIds cannot be populated so only the
		// selection on the current page can be returned:
		return this._getRowValues(this.grid.select());
	}

	getSelectedRowsLength() {
		const selectedRows = this.getSelectedRows();
		if (selectedRows) {
			return this.getSelectedRows().rows.length;
		}
		return 0;
	}

	hideColumn(colName) {
		this.grid.hideColumn(colName);
	}

	showColumn(colName) {
		this.grid.showColumn(colName);
	}

	resetColumnVisibilities() {
		const columnVisibilities = {};
		Object.keys(this.options.columns).forEach(colName => {
			columnVisibilities[colName] = !!this.options.columns[colName].hidden;
			if (columnVisibilities[colName]) {
				this.grid.hideColumn(colName);
			} else {
				this.grid.showColumn(colName);
			}
		});
		this.gridViewSvc.saveColumnVisibilities(columnVisibilities);
	}

	resetColumnWidth() {
		const cols = this.grid.columns.filter(col => !col.hidden);
		const tables = this.grid.thead.parent().add(this.grid.table);
		tables.css('width', '');
		const colElements = tables.children('colgroup').children('col:not(.k-hierarchy-col)');

		const { length } = colElements;
		const self = this;
		$.each(colElements, (index, col) => {
			const colId = cols[index % (length / 2)].id;
			const actualColDef = self.gridColumns.columns.filter(col => col.id === colId);
			let width = '';
			if (actualColDef.length) {
				width = actualColDef[0].width;
			}
			$(col).css('width', width);
		});
		// Update table prefs:
		let tableWidth = '';
		if (this.enableHorizontalScroll) {
			// Save the table width set by kendo
			tableWidth = this.grid.table[0].style.getPropertyValue('width');
		}
		const colWidthsObj = {};
		_.forEach(this.gridColumns.columns, col => {
			if (col.field && this.gridViewSvc.getColumnPref(col.field)) {
				colWidthsObj[col.field] = null;
			}
		});
		this.gridViewSvc.saveColumnWidths(colWidthsObj, tableWidth);
	}

	/**
	 * Processes data objects for use in kendo grid
	 * 
	 * @param {Object}
	 *            dataItem
	 */
	_processDataItem(dataItem) {
		if (dataItem && !dataItem[rowProcessedSymbol]) {
			if (this.idField && !(this.idField in dataItem)) {
				dataItem[this.idField] = _.get(dataItem, this.idField);
			}
			dataItem[rowProcessedSymbol] = true;
		}
	}

	_executeDataTaskCallback(data, callback) {
		this.refreshInProgress = true;
		// Updating the kendo DataSource will reset the grid filter inputs and
		// selection, so we must save the input values and set them back
		// afterwards:
		if (this.openColumn && this.openColumn.columnFilter) {
			const inputValue = this.openColumn.columnFilter.getInputValueForRestore();
			callback(data);
			this.openColumn.columnFilter.restoreInputValue(inputValue);
		} else {
			callback(data);
		}
		this.refreshInProgress = false;
	}

	/**
	 * If addToQueue is true, each data object will be added to the queue as a separate task. Otherwise, the
	 * callback will be called with the data immediately. After processing the data, a grid refresh will be
	 * requested.
	 * 
	 * @param {(Object|Object[])}
	 *            data
	 * @param {boolean}
	 *            addToQueue
	 * @param {function}
	 *            callback
	 * @return {Promise} Promise that resolves once all data has been pushed
	 */
	_handleDataTask(data, addToQueue, callback) {
		return new Promise((resolve, reject) => {
			if (this.destroyed) {
				// Do not process push updates if the grid has been destroyed
				reject();
			} else if (!data) {
				resolve();
				return;
			} else if (!_.isArray(data)) {
				data = [data];
			}
			if (this.idField) {
				// kendo needs the idField to be part of the root of the dataObj.
				// kendo normally handles this when the data is part of the dataSource,
				// but when pushing data it needs to be handled before the push:
				data.forEach(dataObj => {
					this._processDataItem(dataObj);
				});
			}
			if (!addToQueue) {
				if (data.length > 0) {
					this.preventDataBinding = true; // Prevent immediate grid refresh on push update
					_.forEach(data, dataObj => {
						this._executeDataTaskCallback(dataObj, callback);
					});
					this.preventDataBinding = false;
				}
				this.requestRefreshGrid();
				resolve();
			} else if (data.length > 0) {
				_.forEach(data, (dataObj, index) => {
					const task = () => {
						if (this.openColumn && this.openColumn.columnFilter && this.openColumn.columnFilter.isVisible()) {
							// _executeDataTaskCallback will interrupt the user's input.
							// Pause execution while the columnFilter is visible.
							this.queue.unshiftTask(task);
						} else {
							this.preventDataBinding = true; // Prevent immediate grid refresh on push update
							this._executeDataTaskCallback(dataObj, callback);
							this.preventDataBinding = false;
	
							this.requestRefreshGrid(TIME_BETWEEN_PUSH_REFRESHES);
	
							if (index + 1 === data.length) {
								// Resolve the promise once we've reached the end
								// of the data array.
								resolve();
							}
						}
					};
					this.queue.pushTask(task);
				});
			} else {
				resolve();
			}
		});
	}

	/**
	 * Adds data to the grid.
	 * 
	 * @param {object[]}
	 *            data - The array of data objects to add
	 * @param {boolean}
	 *            addToQueue - If true, each data object will be added whenever the browser has idle time. The
	 *            grid will refresh every 2 seconds while the queue is being processed or when the queue is
	 *            empty. This is recommended if there is a large amount of data to be added or if a grid
	 *            refresh is not immediately needed. Otherwise, the grid data will be immediately updated and
	 *            the grid will be refreshed at the next animation frame.
	 * @returns {Promise} Resolves once the grid has been fully updated. Rejects if this function is called
	 *          after the grid has been destroyed.
	 */
	addData(data, addToQueue) {
		return this._handleDataTask(data, addToQueue, dataObj => {
			this.gridDataSrc.getDataSource().add(dataObj);
		});
	}

	/**
	 * Adds data to the grid or updates matching data in the grid.
	 * 
	 * @param {object[]}
	 *            data - The array of data objects to add or update
	 * @param {boolean}
	 *            addToQueue - If true, each data object will be added/updated whenever the browser has idle
	 *            time. The grid will refresh every 2 seconds while the queue is being processed or when the
	 *            queue is empty. This is recommended if there is a large amount of data to be pushed or if a
	 *            grid refresh is not immediately needed. Otherwise, the grid data will be immediately updated
	 *            and the grid will be refreshed at the next animation frame.
	 * @returns {Promise} Resolves once the grid has been fully updated. Rejects if this function is called
	 *          after the grid has been destroyed.
	 */
	pushData(data, addToQueue) {
		return this._handleDataTask(data, addToQueue, dataObj => {
			this.gridDataSrc.getDataSource().pushUpdate(dataObj);
		});
	}

	/**
	 * Removes matching data in the grid.
	 * 
	 * @param {object[]}
	 *            data - The array of data objects to remove
	 * @param {boolean}
	 *            addToQueue - If true, each data object will be removed whenever the browser has idle time.
	 *            The grid will refresh every 2 seconds while the queue is being processed or when the queue
	 *            is empty. This is recommended if there is a large amount of data to be removed or if a grid
	 *            refresh is not immediately needed. Otherwise, the grid data will be immediately updated and
	 *            the grid will be refreshed at the next animation frame.
	 * @returns {Promise} Resolves once the grid has been fully updated. Rejects if this function is called
	 *          after the grid has been destroyed.
	 */
	removeData(data, addToQueue) {
		return this._handleDataTask(data, addToQueue, dataObj => {
			this.gridDataSrc.getDataSource().pushDestroy(dataObj);
		});
	}

	/**
	 * Removes all data from the grid. Any queued adds, pushes, and removes are cancelled.
	 */
	removeAllData() {
		this.queue.clearPendingTasks();
		this._executeDataTaskCallback([], data => {
			this.gridDataSrc.getDataSource().data([]);
			this.requestRefreshGrid();
		});
	}

	/**
	 * This function should be used to perform multiple data source modifications at once to minimize the
	 * number of grid refreshes needed.
	 * 
	 * @param {Object}
	 *            dataObj
	 * @param {Object[]}
	 *            dataObj.add
	 * @param {Object[]}
	 *            dataObj.push
	 * @param {Object[]}
	 *            dataObj.remove
	 * @returns {Promise} Resolves once the grid has been fully updated. Rejects if this function is called
	 *          after the grid has been destroyed.
	 */
	modifyData(dataObj, addToQueue) {
		const promises = [];
		if (dataObj.add) {
			promises.push(this.addData(dataObj.add, addToQueue));
		}
		if (dataObj.push) {
			promises.push(this.pushData(dataObj.push, addToQueue));
		}
		if (dataObj.remove) {
			promises.push(this.removeData(dataObj.remove, addToQueue));
		}
		return Promise.all(promises);
	}

	/**
	 * Removes all data from the grid. Any queued adds, pushes, and removes are cancelled.
	 */
	removeAllData() {
		this.queue.clearPendingTasks();
		this._executeDataTaskCallback([], data => {
			this.gridDataSrc.getDataSource().data([]);
			this.requestRefreshGrid();
		});
	}

	/**
	 * Replaces data in the grid with the new data. Any queued adds, pushes, and removes are cancelled.
	 * 
	 * @param {Object[]}
	 *            data The new data array.
	 */
	setData(data) {
		this.queue.clearPendingTasks();
		this._executeDataTaskCallback(data, data => {
			this.gridDataSrc.getDataSource().data(data);
			this.requestRefreshGrid();
		});
	}

	getData(id) {
		return this.gridDataSrc.getData(id);
	}

	/**
	 * this function is used to select rows based on ids defined via idField
	 * 
	 */

	select(ids, preventRefresh) {
		if (!_.isArray(ids)) {
			ids = [ids];
		}
		_.forEach(ids, id => {
			const dataItem = this.getData(id);
			if (this.disableField) {
				// If disableField is defined, only data that exists in the data
				// source and are not disabled will be selected
				if (dataItem && !_.get(dataItem, this.disableField)) {
					this.grid._selectedIds[id] = true;
					this._selectionMapCache[id] = dataItem;
				}
			} else {
				this.grid._selectedIds[id] = true;
				this._selectionMapCache[id] = dataItem;
			}
		});
		if (!preventRefresh) {
			this.requestRefreshGrid();
		}
	}

	/**
	 * Selects all rows. Requires idField to be set.
	 */
	selectAll(preventRefresh) {
		const allData = this.gridDataSrc.getAllData();
		const allDataIds = [];
		_.forEach(allData, dataItem => {
			const id = _.get(dataItem, this.idField);
			allDataIds.push(id);
		});
		this.select(allDataIds, preventRefresh);
	}

	/**
	 * Selects all visible rows. Requires idField to be set.
	 */
	selectAllVisible(preventRefresh) {
		const allData = this.gridDataSrc.getFilteredData();
		const allDataIds = [];
		_.forEach(allData, dataItem => {
			const id = _.get(dataItem, this.idField);
			allDataIds.push(id);
		});
		this.select(allDataIds, preventRefresh);
	}

	/**
	 * @returns {boolean}
	 */
	areAllSelected() {
		const allDataLength = this.gridDataSrc.getAllDataLength();
		if (allDataLength > 0) {
			const numSelected = this.getSelectedRowsLength();
			if (numSelected >= allDataLength) {
				return true;
			}
		}
		return false;
	}

	areAllVisibleSelected() {
		const filteredData = this.gridDataSrc.getFilteredData();
		if (filteredData.length > 0) {
			for (let i = 0; i < filteredData.length; i++) {
				const id = _.get(filteredData[i], this.idField);
				if (!this.grid._selectedIds[id]) {
					return false;
				}
			}
			return true;
		}
		return false;
	}

	/**
	 * Deselect rows. Requires idField to be set
	 * 
	 * @param {string|string[]}
	 *            ids The id or ids to deselect
	 * @param {boolean}
	 *            [preventRefresh] If true, the grid will not be redrawn
	 */
	deselect(ids, preventRefresh) {
		if (!_.isArray(ids)) {
			delete this.grid._selectedIds[ids];
			delete this._selectionMapCache[ids];
		} else {
			_.forEach(ids, id => {
				delete this.grid._selectedIds[id];
				delete this._selectionMapCache[id];
			});
		}
		if (!preventRefresh) {
			this.requestRefreshGrid();
		}
	}

	/**
	 * Deselects all rows. Requires the idField to be set.
	 * If idField is not set, only visible selected rows will be deselected.
	 */
	deselectAll(preventRefresh) {
		if (this.idField) {
			this.deselect(Object.keys(this.grid._selectedIds), preventRefresh);
		} else {
			this.grid.clearSelection();
		}
	}

	/**
	 * Deselects all rows. Requires the idField to be set.
	 * If idField is not set, only visible selected rows will be deselected.
	 */
	deselectAllVisible(preventRefresh) {
		if (this.idField) {
			const allData = this.gridDataSrc.getFilteredData();
			const allDataIds = [];
			_.forEach(allData, dataItem => {
				const id = _.get(dataItem, this.idField);
				allDataIds.push(id);
			});
			this.deselect(allDataIds, preventRefresh);
		} else {
			this.grid.clearSelection();
		}
	}

	getOptions() {
		return this.grid.getOptions();
	}

	search(term) {
		if (this.destroyed || term === this.gridDataSrc.getSearchTerm()) {
			// Prevent the search from executing if the grid is destroyed or the search term is unchanged
			return;
		}
		this.gridDataSrc.setSearch(term);
		if (term === '') {
			this.gridHeaderElement.find('.cv-k-grid-toolbar-primary').removeClass('cv-k-grid-search-active');
			if (this.useClassicToolbar) {
				this.gridHeaderElement.find('.k-i-close').addClass('hidden');
			}
		} else {
			this.gridHeaderElement.find('.cv-k-grid-toolbar-primary').addClass('cv-k-grid-search-active');
			if (this.useClassicToolbar) {
				this.gridHeaderElement.find('.k-i-close').removeClass('hidden');
			}
		}
	}

	getColumns() {
		return this.gridColumns.getColumnsWithCheckBox();
	}

	exportToCsv() {
		return this.csvExporter.export(this);
	}

	/**
	 * @param {number}
	 *            page The zero-based page index
	 */
	setPage(page) {
		this.grid.pager.page(page + 1);
	}

	getPage() {
		return this.grid.pager.page() - 1;
	}

	getTotalPages() {
		return this.grid.pager.totalPages();
	}

	/**
	 * @deprecated Use deselectAll() instead
	 */
	clearAllGridSelections(){
		this.deselectAll();
	}

	destroy() {
		if (this.builtRowDetails.length > 0) {
			this.callbacks.onRowDetailsDestroy.call(this, {
				details: this.builtRowDetails
			});
		}
		this.builtRowDetails = [];

		super.destroy();
		if (this.tooltip) {
			this.tooltip.destroy();
		}
		delete cv.gridCompanyId;
		if (this.callOut) {
			this.callOut.destroy();
		}

		this.grid.destroy();
		
		if (this.viewsMenu) {
			this.viewsMenu.destroy();
		}
		if (this.actionContextMenu) {
			this.actionContextMenu.destroy();
		}
		if (this.batchActionMenu) {
			this.batchActionMenu.destroy();
		}
		if (this.gridToolbarMenu) {
			this.gridToolbarMenu.destroy();
		}
		if (this.gridEmptyMenu) {
			this.gridEmptyMenu.destroy();
		}
		if (this.gridContextMenu) {
			this.gridContextMenu.destroy();
		}
		this.columnMenuElements.forEach(elem => {
			const popup = elem.getKendoPopup();
			if (popup) {
				popup.destroy();
			}
		});
		this.queue.clearPendingTasks();
		// TODO ensure all elements are destroyed
	}
}
