/**
 * @author ssubash
 */
import * as GridConstants from './grid.constants';

export default class GridViewSvc {
	constructor(grid, options, angularLibs) {
		this.grid = grid;
		this.hasViews = options.hasViews !== false;
		this.hasDefaultView = options.hasDefaultView !== false;
		this.tableName = options.tableName;
		this.location = angularLibs.$location;
		this.cvLoc = angularLibs.cvLoc;
		this.userPrefService = angularLibs.userPrefService;
		this.allViewName = angularLibs.cvLoc("viewname.all");
		this.loaded = this.loadTablePreferences();
		// this._updateUrlIfDefaultView();
	}

	getCurrentViewId() {
		if (!this.hasViews) {
			return this.tablePrefs.defaultView;
		}
		let currentView = this.location.search().view;
		if (!currentView || !this.getView(currentView)) {
			currentView = this.tablePrefs.defaultView;
		}
		return currentView;
	}

	getDefaultViewId() {
		return this.tablePrefs.defaultView;
	}

	getAdditionalPref(key) {
		if (!this.tablePrefs.additionalPrefs) {
			this.tablePrefs.additionalPrefs = {};
		}
		return this.tablePrefs.additionalPrefs[key];
	}

	putAdditionalPref(key, value) {
		if (!this.tablePrefs.additionalPrefs) {
			this.tablePrefs.additionalPrefs = {};
		}
		this.tablePrefs.additionalPrefs[key] = value;
	}

	loadUserPreferences() {
		return this.userPrefService.getLoadedUserPrefs();
	}

	loadTablePreferences() {
		const self = this;
		return this.loadUserPreferences().finally(() => {
			return new Promise((resolve, reject) => {
				// Load current table prefs:
				if (self.tableName && cv.userPref[self.tableName]) {
					self.tablePrefs = JSON.parse(cv.userPref[self.tableName]);
					if (!self.tablePrefs.name) {
						// data in old style. discard it
						self.tablePrefs = undefined;
					}

					// data version is old. update it
					if (self.tablePrefs && !self.tablePrefs.version) {
						return self.userPrefService.setViewAsDefault(self.tablePrefs.name, self.tablePrefs.defaultView).then(
							() => {
								reject();
							},
							() => {
								self.tablePrefs = undefined;
								resolve();
							}
						);
					}
				}
				resolve();
			}).then(
				() => {
					// tablePrefs is valid or undefined
					// Initialize new table preferences
					self.tablePrefs = self.tablePrefs || {
						name: self.tableName,
						views: {},
						columnPrefs: {},
						additionalPrefs: {},
					};

					if (!self.tablePrefs.columnPrefs) {
						self.tablePrefs.columnPrefs = {};
					}

					if (!self.tablePrefs.additionalPrefs) {
						self.tablePrefs.additionalPrefs = {};
					}

					Object.keys(self.tablePrefs.views).forEach(viewId => {
						if (!self.tablePrefs.views[viewId]) {
							// Remove null/undefined views
							delete self.tablePrefs.views[viewId];
						}
					});

					// Default any invalid column widths to ''
					_.forEach(self.tablePrefs.columnPrefs, columnPref => {
						if ('width' in columnPref) {
							const width = columnPref.width;
							if (typeof width === 'string') {
								const validWidthRegex = /^\d+(\.\d+)?(%|px|em|rem|vw|vh|vmin|vmax|pt|pc|ex|ch|in|cm|mm|q)?$/; // Matches a number optionally followed by a css unit
								if (!validWidthRegex.test(width)) {
									columnPref.width = '';
								}
							} else if (typeof width !== 'number') {
								columnPref.width = '';
							}
						}
					});

					// Remove invalid rules from user views:
					// (Invalid rules can occur when column definitions are changed while
					// the user has views already created)
					const emptyViewIds = new Set();
					const validColumnFields = new Set();
					self.grid.getColumns().forEach(column => {
						if (column.field && !column.option.disableViewFiltering && column.filterable) {
							validColumnFields.add(column.field);
						}
					});
					_.forEach(self.tablePrefs.views, (view, viewId) => {
						if (view.filters) {
							for (let i = 0; i < view.filters.length; i++) {
								if (view.canBeDeleted && !validColumnFields.has(view.filters[i].columnName)) {
									// Remove non system filters that do not apply to valid columns
									view.filters.splice(i, 1);
									i--;
								}
							}
							if (view.filters.length === 0) {
								emptyViewIds.add(viewId);
							}
						}
					});
					emptyViewIds.forEach(viewId => {
						// Remove views if they have no valid rules
						delete self.tablePrefs.views[viewId];
					});

					self.tablePrefs.viewOrder = self.tablePrefs.viewOrder || [];
					if (!self.hasViews || self.hasDefaultView) {
						// add default view to tableprefs
						self.tablePrefs.views[self.allViewName] = {
							name: self.allViewName,
							id: self.allViewName,
							canBeDeleted: false, // System views cannot be deleted
							filters: [],
							advancedFilters: {}
						};

						// Add it to the start of view order only if it is not present
						if (self.tablePrefs.viewOrder.indexOf(self.allViewName) === -1) {
							self.tablePrefs.viewOrder.unshift(self.allViewName);
						}

						// if no valid view is set to default set default view
						if (!self.tablePrefs.defaultView || !self.getView(self.tablePrefs.defaultView)) {
							self.tablePrefs.defaultView = self.allViewName;
						}
					} else if (!self.tablePrefs.defaultView || !self.getView(self.tablePrefs.defaultView)) {
						// if no valid view is set to default set default view
						self.tablePrefs.defaultView = self.getValidDefaultView();
					}
					// If a view exists but is not a part of the viewOrder, append it:
					_.forEach(self.tablePrefs.views, (view, viewId) => {
						if (self.tablePrefs.viewOrder.indexOf(viewId) === -1) {
							self.tablePrefs.viewOrder.append(viewId);
						}
					});

					// Refresh the views arrays:
					self.sortedViews = self.sortedViews || [];
					self.sortedViews.length = 0;
					self.systemViews = self.systemViews || [];
					self.customViews = self.customViews || [];
					_.forEach(self.tablePrefs.viewOrder, (viewId, index) => {
						const view = self.tablePrefs.views[viewId];
						if (view) {
							self.sortedViews.push(view);
						} else {
							self.tablePrefs.viewOrder.splice(index, 1);
						}
					});

					if (!self.hiddenViewIds) {
						// Hidden views are only populated on initial load, since only
						// system views can be hidden
						self.hiddenViewIds = new Set();
						_.forEach(self.tablePrefs.views, (view, viewId) => {
							if (view.hiddenByDefault) {
								self.hiddenViewIds.add(viewId);
							}
						});
						// If the current view is hidden, change it:
						if (this.isViewHidden(this.getCurrentViewId())) {
							let visibleView = this.getValidDefaultView();
							this.location.search('view', visibleView);
						}
					}
				},
				() => {
					// data version is old, update the tablePrefs object
					return self.updateTablePrefs(true, null);
				}
			);
		});
	}

	setViewHidden(id, hidden) {
		if (hidden) {
			this.hiddenViewIds.add(id);
		} else {
			this.hiddenViewIds.delete(id);
		}
	}

	isViewHidden(id) {
		return this.hiddenViewIds.has(id);
	}

	getValidDefaultView() {
		const viewObj = this.getView(this.tablePrefs.defaultView);
		if (viewObj && !this.isViewHidden(viewObj.id)) {
			return viewObj.id;
		}
		// No visible default is set, return the first visible view, or the first view if none are visible:
		if (this.tablePrefs.viewOrder.length) {
			return this._getFirstVisibleView() || this.tablePrefs.viewOrder[0];
		}
		// There are no visible views, return the default view:
		if (viewObj && viewObj.id) {
			return viewObj.id;
		}
		return Object.keys(this.tablePrefs.views)[0];
	}

	_getFirstVisibleView() {
		return this.tablePrefs.viewOrder.find(viewId => !this.isViewHidden(viewId));
	}

	getVisibleViews() {
		const visibleViews = [];
		_.forEach(this.tablePrefs.views, view => {
			if (!this.isViewHidden(view.id)) {
				visibleViews.push(view);
			}
		});
		return visibleViews;
	}

	getView(viewId) {
		return this.tablePrefs.views[viewId];
	}

	// _updateUrlIfDefaultView() {
	// 	const id = this.options.selectedViewId || "All";
	// 	const viewFilters = this.getViewFilters(id);
	// 	// TODO update location
	// 	eventEmitter.emit(GridConstants.GRID_FILTER, viewFilters);
	// }

	/**
	 * A view information will be persisted in the backend userPreferences.
	 * Once successful, we update the table preferences and update the grid.
	 * TODO : Make all of them PURE.
	 */
	saveView(viewObj, cvToaster) {
		if (!this.tableName) {
			return Promise.resolve();
		}
		const setAsDefault = !!viewObj.isDefault;
		delete viewObj.isDefault;
		const self = this;
		this.userPrefService
			.saveTableView(this.tableName, viewObj, setAsDefault)
			.success(function(viewId) {
				self.updateTablePrefs(true, viewId);
			})
			.error(function(e) {
				cvToaster.showErrorMessage({
					ttl: '3000', //3 sec
					message: e + '. ' + self.cvLoc('error.saveViewFailed') + ' ' + viewObj.name
				});
			});
	}

	editView(viewObj, cvToaster) {
		if (!this.tableName) {
			return Promise.resolve();
		}
		const self = this;
		const setAsDefault = !!viewObj.isDefault;
		delete viewObj.isDefault;
		this.userPrefService
			.updateTableView(this.tableName, viewObj, setAsDefault)
			.success(function(viewId) {
				self.updateTablePrefs(true, viewId);
			})
			.error(function(e) {
				cvToaster.showErrorMessage({
					ttl: '3000', //3 sec
					message: e + '. ' + self.cvLoc('error.saveViewFailed') + ' ' + viewObj.name
				});
			});
	}

	toggleColumnVisibility(field) {
		if (!this.tableName) {
			return Promise.resolve();
		}
		return this.userPrefService.toggleColumnVisibility(this.tableName, field).then(() => this.updateTablePrefs(true));
	}

	saveColumnVisibilities(columnVisibilities) {
		if (!this.tableName) {
			return Promise.resolve();
		}
		return this.userPrefService
			.setColumnVisibilities(this.tableName, columnVisibilities || {})
			.then(() => this.updateTablePrefs(true));
	}

	saveColumnSort(column, dir) {
		if (!this.tableName) {
			return Promise.resolve();
		}
		const sortObj = {};
		if (dir) {
			sortObj[column.field] = {
				name: column.field,
				isHidden: !!column.hidden,
				sortDirection: dir
			};
		}
		return this.userPrefService.sortColumns(this.tableName, sortObj).then(() => this.updateTablePrefs(true));
	}

	saveColumnWidths(columnWidths, tableWidth) {
		if (!this.tableName) {
			return Promise.resolve();
		}
		return this.userPrefService
			.setColumnWidths(this.tableName, columnWidths || {}, tableWidth)
			.then(() => this.updateTablePrefs(true));
	}

	/**
	 * Update table preferences
	 *
	 * TODO: Do we need callback ? After updating the table preferences.
	 * TODO : Make all of them PURE.
	 */
	updateTablePrefs(updateFromServer, viewId) {
		let promise;
		if (updateFromServer) {
			promise = this.userPrefService.updateUserPrefs().then(response => {});
		} else if (this.tableName) {
			// update local object
			cv.userPref[this.tableName] = angular.toJson(this.getTablePrefs());
		}
		const self = this;
		return Promise.resolve(promise)
			.then(() => {
				return self.loadTablePreferences();
			})
			.then(() => {
				if (arguments.length > 1) {
					// If viewId is specified, send event for views updated
					// TODO rename VIEW_ADDED event to VIEWS_UPDATED
					this.grid.eventEmitter.emit(GridConstants.VIEW_ADDED, {
						name: self.tablePrefs.name,
						views: self.getAllViews(),
						selectedViewId: viewId || self.tablePrefs.defaultView
					});
				}
			});
	}

	getNumberOfViews() {
		return this.tablePrefs.views ? Object.keys(this.tablePrefs.views).length : 0;
	}

	getAllViews() {
		return this.sortedViews;
	}

	getView(viewId) {
		return this.tablePrefs.views[viewId];
	}

	getCurrentView() {
		return this.getView(this.getCurrentViewId());
	}

	getViewFilters(id) {
		const view = this.getView(id);
		if (view) {
			return view.filters;
		}
		return null;
	}

	getViewAdvancedFilters(id) {
		const view = this.getView(id);
		if (view) {
			return view.advancedFilters;
		}
		return null;
	}

	hasView(viewId) {
		return !_.isEmpty(this.tablePrefs.views[viewId].name);
	}

	getTablePrefs() {
		return this.tablePrefs;
	}

	getColumnPrefs() {
		return this.tablePrefs.columnPrefs;
	}

	getColumnPref(field) {
		return this.tablePrefs.columnPrefs[field];
	}

	getTableWidth() {
		return this.tablePrefs.tableWidth;
	}

	applyFilters() {
		this.grid.eventEmitter.emit(GridConstants.GRID_FILTER);
	}

	// getAdvancedFilter(advancedFilters) {
	// 	if (advancedFilters) {
	// 		return advancedFilters.reduce(function(map, filter) {
	// 				var filterKey, filterValue;
	// 				if (filter.key) {
	// 					filterKey = filter.key;
	// 					filterValue = gridApi.grid.appScope[filterKey] === undefined ? filter.defaultValue
	// 							: gridApi.grid.appScope[filterKey];
	// 				} else {
	// 					filterKey = filter;
	// 					filterValue = gridApi.grid.appScope[filterKey];
	// 				}
	// 				map[filterKey] = filterValue;
	// 				return map;
	// 			},{})
	// 	}
	// 	return {};
	// };
}
