import ComponentRegistry from '../componentRegistry';
import * as GridConstants from './grid.constants';
import {setPropertyToRootLevelByPath} from '../propertyToRootLevel';
import EntityTagsCalloutTemplate from 'adminConsole/js/controllers/columnTemplates/entityTagsCallout.template.js';

/**
 * @author ssubash.
 *
 * This represents a Grid Data source.
 */
export default class GridDataSrc {
	constructor({ gridViewSvc, gridEventEmitter, options, kendo, initialUserFilters, companyId, angularLibs }) {
		this.gridViewSvc = gridViewSvc;
		this.gridEventEmitter = gridEventEmitter;
		this.currentSearch = '';
		this.companyId = companyId;
		this.angularLibs = angularLibs;
		this._ensureKendoAvailable(kendo);
		this.options = options;
		this.entityTagsCalloutTemplates = new EntityTagsCalloutTemplate(this.angularLibs.cvLoc);
		this.userFilters = {};
		this._putUserFilters(initialUserFilters, true);
		this.extraFilters = [];
		if (_.isArray(options.extraFilters)) {
			Array.prototype.push.apply(this.extraFilters, options.extraFilters);
		}
		this.dataSrcConfig = this.buildDataSourceOptions(options);
		this.dataSource = new kendo.data.DataSource(this.dataSrcConfig);
		if (this.options.processAdvancedFilters) {
			this.processAdvancedFilters = this.options.processAdvancedFilters;
		}
		this.gridFilterListener = this._putUserFilters.bind(this);
		this.gridEventEmitter.on(GridConstants.GRID_FILTER, this.gridFilterListener);
	}

	buildDataSourceOptions(options) {
		const config = {};
		config.change = event => {
			this.gridEventEmitter.emit(GridConstants.GRID_DATA_SRC_CHANGE, event);
			if (_.isFunction(options.change)) {
				options.change.call(this, event);
			}
		};
		config.sort = options.sortDirection;
		if (options.enableVirtualization) {
			config.batch = true;
		}
		config.serverPaging = options.enableServerLoading;
		config.serverFiltering = options.enableServerLoading;
		config.serverSorting = options.enableServerLoading;
		this._configurePageSize(config, options);
		this._configureSchema(config, options);
		this._configureTransport(config, options);
		this._configureGroup(config, options);
		config.filter = this.buildFilters();
		return config;
	}

	_configurePageSize(config, options) {
		config.pageSize = options.pageSize || 100;
	}

	_configureSchema(config, options) {
		config.schema = {};
		config.schema.model = {};
		config.schema.model.id = options.idField || '';
		config.schema.model.fields = this._getFields(options);
		if (this.options.enableServerLoading) {
			// If enableServerLoading is true, the 'total' property of the
			// array will be used as the total.
			config.schema.total = 'total';
		} else {
			// If enableServerLoading is false, the 'length' property of the
			// array will be used as the total.
			config.schema.total = 'length';
		}
	}

	_setEntityInfo(data, gridOptions) {
	
		if (_.isString(gridOptions.entityInfoPath)) {
			setPropertyToRootLevelByPath(data, gridOptions.entityInfoPath, 'entityInfo', entity => {
				if (!_.get(entity, 'companyId', 0)) {
					entity.companyName = this.angularLibs.cvLoc('label.company.commcell');
				}
			});
		}
	}

	_setTagsInfo(data,gridOptions) {
		const self = this;
		if (_.isString(gridOptions.tagsInfoPath)) {
			setPropertyToRootLevelByPath(data, gridOptions.tagsInfoPath, 'tags', null);
			_.forEach(data, ele => {
				ele["tagsCalloutTemplate"] = self.entityTagsCalloutTemplates.getTagsTemplate(ele.tags)
			});
		}
	}
	
	_customURLHandler(gridOptions, readOptions) {
		cv.gridCompanyId =  this.companyId;
		const successFn = data => {
			this._setEntityInfo(data, gridOptions);
			this._setTagsInfo(data, gridOptions);
			if (gridOptions.enableVirtualization && !gridOptions.enableServerLoading) {
				//Kendo-Grid is appending new data to exiting if refreshed after an infinite scrolling attempt.
				//So clearing existing data 
				this.getDataSource().data([]);
			} 
			readOptions.success(data);
		};
		gridOptions.url({
			...readOptions,
			success: successFn
		}, this.companyId);
	}

	_configureTransport(config, options) {
		if (options.url) {
			const self = this;
			config.transport = {};
			config.transport.read = {};
			if (options.showCompanyColumnAndFilter) {
				config.transport.read = this._customURLHandler.bind(this, options);
			} else {
				config.transport.read = options.url;
			}
		} else {
			if (options.showCompanyColumnAndFilter) {
				this._setEntityInfo(data, options);
			} 
			if(options.showTags) {
				this._setTagsInfo(data, options);
			}
			
			config.data = options.data;
		}
	}

	_configureGroup(config, options) {
		if (options.group) {
			config.group = options.group;
		}
	}

	getDataSource() {
		return this.dataSource;
	}

	getData(id) {
		return this.getDataSource().get(id);
	}

	getAllData() {
		return this.getDataSource().data();
	}

	getAllDataLength() {
		return this.getDataSource().total();
	}

	getDataByUid(rowUid) {
		return this.getDataSource().getByUid(rowUid);
	}

	_ensureKendoAvailable(kendo) {
		if (!kendo) {
			throw 'Kendo not available';
		}
	}

	_getFields(options) {
		const fields = _.mapValues(options.columns, value => {
			return {
				type: value.type || 'string',
				editable: value.editable || false,
				nullable: value.nullable || false
			};
		});
		if (options.idField && !(options.idField in fields)) {
			fields[options.idField] = {
				type: 'string',
				editable: false,
				nullable: false
			};
		}
		return fields;
	}

	/**
	 * @param {Object}
	 *            filter
	 * @param {string}
	 *            filter.columnName
	 * @param {*}
	 *            filter.filterValue
	 */
	addFilter(filter) {
		this.extraFilters.push(filter);
	}

	/**
	 * @param {number|Object}
	 *            filter The index or the filter object to remove
	 */
	removeFilter(filter) {
		if (typeof filter === 'number') {
			this.extraFilters.splice(filter, 1);
		} else {
			for (let i = 0; i < this.extraFilters.length; i++) {
				if (this.extraFilters[i] === filter) {
					this.extraFilters.splice(i, 1);
					i -= 1;
				}
			}
		}
	}

	clearFilters() {
		this.extraFilters.length = 0;
	}

	getFilterList() {
		const filterList = [];
		const viewFilters = this.gridViewSvc.getViewFilters(this.gridViewSvc.getCurrentViewId()) || [];
		_.forEach(viewFilters, filter => {
			filterList.push(filter);
		});
		this.extraFilters.forEach(filter => {
			filterList.push(filter);
		});
		Object.keys(this.userFilters).forEach(columnName => {
			filterList.push(this.userFilters[columnName]);
		});
		return filterList;
	}

	/**
	 * @param {string}
	 *            columnName
	 * @param {*}
	 *            filterValue
	 */
	buildKendoFilter(filter) {
		const columnOptions = this.options.columns[filter.columnName];
		let filterClass;
		if (columnOptions) {
			filterClass = ComponentRegistry.instance.get(columnOptions.filterType);
			if (_.isFunction(columnOptions.buildFilter)) {
				const kendoFilter = columnOptions.buildFilter(filter.filterValue, filter.columnName, filter.filterCondition);
				return kendoFilter;
			} else if (columnOptions.filterType === 'datetimedropdown' && !filter.isSearchFilter) {
				// Datetime dropdown is a special case, has its own buildFilter function
				const kendoFilter = filterClass.buildFilter(filter.filterValue, filter.columnName, columnOptions.data);
				return kendoFilter;
			}
		}

		// No custom filter builder:

		let value = filter.filterValue;
		if (!_.isArray(value)) {
			if (columnOptions && columnOptions.filterType === 'checkboxmultiselect') {
				if (_.isString(value)) {
					value = value.split(',');
				} else {
					value = [value];
				}
			} else {
				value = [value];
			}
		}

		let condition = filter.filterCondition;
		// If condition is undefined or invalid, use default condition:
		if (
			!condition ||
			// Check if condition is a possible filterCondition:
			!Object.values(GridConstants.FILTER_CONDITIONS).some(filterCondition => filterCondition.value === condition)
		) {
			if (columnOptions && !_.isEmpty(columnOptions.filterConditions)) {
				// If columnOptions defines filterConditions, the first is used
				// as the default filter condition
				condition = columnOptions.filterConditions[0].value;
			} else if (filterClass) {
				// Use the filter type's default filter condition
				condition = filterClass.DEFAULT_FILTER_CONDITION.value;
			} else {
				// If no filter type specified or the columnName does not have
				// a definition, use "contains" as default
				condition = GridConstants.FILTER_CONDITIONS.CONTAINS.value;
			}
		}

		// TODO use native kendo operators if applicable
		// Build filter based on condition:
		return {
			logic:
				condition === GridConstants.FILTER_CONDITIONS.DOES_NOT_EQUAL.value ||
				condition === GridConstants.FILTER_CONDITIONS.DOES_NOT_CONTAIN.value
					? 'and'
					: 'or',
			filters: value.map((filterValue, index) => {
				let operator;
				// If the filter already has an operator, use it. Do not build one:
				if ('operator' in filter) {
					if (_.isArray(filter.operator)) {
						operator = filter.operator[index];
					} else {
						operator = filter.operator;
					}
				}
				// If no operator present, build an operator using condition:
				if (!_.isFunction(operator)) {
					const operatorValue = String(filterValue).toUpperCase();
					const numericOperatorValue = Number(filterValue);
					if (condition === GridConstants.FILTER_CONDITIONS.EQUALS.value) {
						operator = rowValue => String(rowValue).toUpperCase() === operatorValue;
					} else if (condition === GridConstants.FILTER_CONDITIONS.LESS_THAN.value) {
						operator = rowValue => Number(rowValue) < numericOperatorValue;
					} else if (condition === GridConstants.FILTER_CONDITIONS.GREATER_THAN.value) {
						operator = rowValue => Number(rowValue) > numericOperatorValue;
					} else if (condition === GridConstants.FILTER_CONDITIONS.DOES_NOT_EQUAL.value) {
						operator = rowValue => String(rowValue).toUpperCase() !== operatorValue;
					} else if (condition === GridConstants.FILTER_CONDITIONS.DOES_NOT_CONTAIN.value) {
						operator = rowValue =>
							!String(rowValue)
								.toUpperCase()
								.includes(operatorValue);
					} else {
						// Default condition is 'contains'
						operator = rowValue =>
							String(rowValue)
								.toUpperCase()
								.includes(operatorValue);
					}
				}
				return {
					field: filter.columnName,
					value: filterValue,
					// type: type,
					operator: operator
				};
			})
		};
	}

	_putUserFilters(filters, preventApplyFilter) {
		const self = this;
		let forceRefresh = false;
		_.forEach(filters, filter => {
			if (_.isUndefined(filter.filterValue) || _.isNull(filter.filterValue) || filter.filterValue === '') {
				delete self.userFilters[filter.columnName];
				// When removing a filter, kendo grid will not refresh so we
				// must force a refresh:
				forceRefresh = self.options.enableServerLoading;
			} else {
				self.userFilters[filter.columnName] = filter;
			}
		});
		if (!preventApplyFilter) {
			this.applyFilters(forceRefresh);
		}
	}

	getUserFilter(columnName) {
		return this.userFilters[columnName];
	}

	clearUserFilters(refreshGridFilter) {
		_.forEach(Object.keys(this.userFilters), columnName => {
			delete this.userFilters[columnName];
		});
		if (refreshGridFilter) {
			this.applyFilters();
		}
	}

	buildFilters() {
		const filterList = this.getFilterList();
		const gridFilter = {
			logic: 'and',
			filters: []
		};
		filterList.forEach(filter => {
			const kendoFilter = this.buildKendoFilter(filter);
			if (kendoFilter) {
				gridFilter.filters.push(kendoFilter);
			}
		});
		if (this.processAdvancedFilters) {
			const advancedFilters = this.gridViewSvc.getViewAdvancedFilters(this.gridViewSvc.getCurrentViewId());
			const processedAdvancedFilters = this.processAdvancedFilters(advancedFilters);
			if (_.isArray(processedAdvancedFilters)) {
				Array.prototype.push.apply(gridFilter.filters, processedAdvancedFilters);
			}
		}
		// TODO advancedfilter
		if (this.searchFilter) {
			gridFilter.filters.push(this.searchFilter);
		}
		return gridFilter;
	}

	applyFilters(forceRefresh) {
		const filter = this.buildFilters();
		this.dataSource.filter(filter);
		if (forceRefresh || (this.options.enableServerLoading && filter.filters.length === 0)) {
			// Kendo does not reload data on filter clear, force reload:
			this.dataSource.read();
		}
	}

	getSearchTerm() {
		return this.currentSearch;
	}

	/**
	 * @param {string}
	 *            term
	 */
	setSearch(term) {
		this.currentSearch = term;
		const searchFilters = [];
		if (!_.isUndefined(term) && !_.isNull(term) && term !== '') {
			// // Split search terms and filter out empty strings:
			// const splitTerms = term.split(" ").filter(part => part);
			const splitTerms = [term.trim()];
			Object.keys(this.options.columns).forEach(columnName => {
				const column = this.options.columns[columnName];
				if (column.searchable !== false) {
					const kendoFilters = [];
					splitTerms.forEach(termPart => {
						const kendoFilter = this.buildKendoFilter({
							columnName: column.searchField || columnName,
							filterValue: termPart,
							filterCondition: GridConstants.FILTER_CONDITIONS.CONTAINS.value,
							isSearchFilter: true
						});
						if (kendoFilter) {
							kendoFilters.push(kendoFilter);
						}
					});
					if (kendoFilters.length > 1) {
						searchFilters.push({
							logic: 'and',
							filters: kendoFilters
						});
					} else if (kendoFilters.length > 0) {
						searchFilters.push(kendoFilters[0]);
					}
				}
			});
		}
		if (searchFilters.length > 0) {
			this.searchFilter = { logic: 'or', filters: searchFilters };
		} else {
			delete this.searchFilter;
		}
		this.applyFilters();
	}

	/**
	 * @returns {Object[]} All the data that matches the current filters
	 */
	getFilteredData() {
		return this.filterData(this.getAllData());
	}

	/**
	 * @returns {Object[]} The data that matches the current filters and page
	 */
	getVisibleData() {
		return this.dataSource.view();
	}

	/**
	 * Filters a set of data according to the specified filters
	 * @param {Object[]} data The data array to filter
	 * @param {*} [kendoFilters] The filter object. If undefined, this GridDataSrc's
	 * current filters will be used
	 */
	filterData(data, kendoFilters) {
		if (_.isUndefined(kendoFilters)) {
			kendoFilters = this.buildFilters();
		}
		return GridDataSrc.filterData(data, kendoFilters);
	}

	static filterData(data, kendoFilters) {
		const dataSource = new kendo.data.DataSource();
		dataSource.data(data);
		dataSource.filter(kendoFilters);
		return dataSource.view();
	}
}
