// import gridSort from './gridSort';
// import gridFilter from './gridFilter';
import EventEmitter from 'eventemitter3';
import GridDataSrc from './gridDataSrc';
import GridColumns from './gridColumns';
import GridViewSvc from './gridview.svc';
import * as GridDeleteView from './grid.delete.view';
import * as GridView from './gridview';
import * as GridReorderViews from './grid.reoder.view';

/**
 * @author ssubash
 */
export default class Grid {
	constructor(options, gridElement, _vendorGrid, angularLibs) {
		this.options = options;
		this.tableName = options.tableName;
		this.exportName = options.exportName || this.tableName;
		this._setDefaultOptions(this.options);
		//Company Column and filter will be hidden if there is no company
		this.options.showCompanyColumnAndFilter =
			!_.isEmpty([...cv.providers, ...cv.deactivatedProviders]) && (cv.isMspUser || cv.isUserOnlyMspAdmin) && this.options.showCompanyColumnAndFilter;
		this.options.showTags = options.showTags;
		this.gridElement = gridElement;
		this.gridElement.attr('id', this.tableName);
		this.gridElement.data('cvKendoGrid', this);
		this.eventEmitter = new EventEmitter();
		this.kendo = _vendorGrid;
		if (this.options.enableFiltering !== false) {
			this.enableFiltering = {
				extra: false,
				messages: {
					title: '',
					value: '',
				},
				operators: {
					string: {
						startsWith: 'Starts with'
					},
					number: {
						gte: 'Greater than equal to',
						lte: 'Less than equal to'
					},
					date: {
						lte: 'Before or equal to',
						gte: 'After or equal to'
					}
				}
			};
		}
		this.angularLibs = angularLibs;
		this.location = angularLibs.$location;
		this.disableViewCreation = !!this.options.disableViewCreation;
		this.enableSorting = this.options.enableSorting !== false;
		this.allowUnsort = !!this.options.allowUnsort;
		this.disableDefaultSort = !!this.options.disableDefaultSort;
		if ('useClassicToolbar' in options) {
			this.useClassicToolbar = !!this.options.useClassicToolbar;
		} else {
			// Default classic toolbar to true
			this.useClassicToolbar = true;
		}
		this.usePageToolbar = !!this.options.usePageToolbar;
		this.enablePaging = this.options.enablePaging !== false;
		this.enableVirtualization = this.options.enableVirtualization || !this.enablePaging; // If paging is disabled, use virtualization
		if (this.enableVirtualization) {
			// Virtualization overrides paging
			this.enablePaging = false;
		}
		this.enableVirtualScroll = !!this.options.enableVirtualScroll;
		this.parentModal = this.gridElement.parents('.modal');
		// Use a fixed height if no height is set and useFixedHeight is true or grid is using virtualization
		if ('height' in options) {
			// If height is set, do not use fixed height
			this.useFixedHeight = false;
		} else if ('useFixedHeight' in options) {
			this.useFixedHeight = options.useFixedHeight;
		} else if (this.enableVirtualization && !this.parentModal.length) {
			// Default useFixedHeight to true when using virtualization
			this.useFixedHeight = true;
		}
		if (this.useFixedHeight) {
			this.height = $(window).height() - 150;
		} else {
			this.height = options.height;
		}
		this.enableColumnMenu = this.options.enableColumnMenu !== false;
		this.enableColumnResizing = this.options.enableColumnResizing !== false;
		this.showSearchOnFewItems = this.options.showSearchOnFewItems || false;
		this.enableServerLoading = this.options.enableServerLoading;
		this.enableCheckBoxColumn = this.options.enableCheckBoxColumn;
		if (!this.enableCheckBoxColumn) {
			// Multiselect requires checkboxes
			this.enableMultiSelect = false;
		} else if ('enableMultiSelect' in this.options) {
			this.enableMultiSelect = this.options.enableMultiSelect;
		} else {
			// options.enableMultiSelect is not defined and checkboxes are enabled
			this.enableMultiSelect = true;
		}
		this.customSelectAll = this.options.customSelectAll;
		this.enableCsvExport = this.options.enableCsvExport !== false;
		this.exportAllPages = this.options.exportAllPages;
		this.hideGridOnEmpty = this.options.hideGridOnEmpty !== false;
		this.persistSelection = this.options.persistSelection !== false;
		if (this.options.showCompanyColumnAndFilter && !this.options.hideCompanyFilter) {
			this.options.toolbarTemplate = `<div class="cv-entity-filter"></div>` + _.get(this.options, 'toolbarTemplate', '');
		}
		this.gridEmptyMessage = this.options.gridEmptyMessage || angularLibs.cvLoc('label.noDataAvailable');
		this.noResultsMessage = this.options.noResultsMessage || angularLibs.cvLoc('label.noResultsFound');
		this.gridEmptySvg = this.options.gridEmptySvg;
		if (this.options.rowDetailTemplateId) {
			this.rowDetailTemplate = $(`#${this.options.rowDetailTemplateId}`).html();
		}
		this.persistenceStrategy = this.options.persistenceStrategy;
		this.gridTitle = this.options.gridTitle;
		this.pageSizes = this.options.pageSizes;
		this.detailInit = this.options.detailInit;
		this.enableHorizontalScroll = this.options.enableHorizontalScroll;
		if (this.rowDetailTemplate || this.detailInit || this.options.onRowDetailInit) {
			// Horizontal scroll is not supported with detail row
			this.enableHorizontalScroll = false;
		}
		this.idField = this.options.idField;
		this.disableField = this.options.disableField;
		this.gridColumns = new GridColumns(
			gridElement,
			this.eventEmitter,
			{
				enableCheckBoxColumn: this.enableCheckBoxColumn,
				enableMultiSelect: this.enableMultiSelect,
				enableHorizontalScroll: this.enableHorizontalScroll,
				enableServerLoading: this.enableServerLoading,
				customSelectAll: this.customSelectAll,
				idField: this.idField,
				columns: this.options.columns,
				showCompanyColumnAndFilter: this.options.showCompanyColumnAndFilter,
				showOnlyCompanyFilter: this.options.showOnlyCompanyFilter,
				showTags: this.options.showTags
			},
			angularLibs
		);
		this._setupEventListeners(this.options);
		if (this.enableVirtualization) {
			this.scrollable = {
				endless: true
			};
			if (this.enableVirtualScroll) {
				this.scrollable.virtual = true;
			}
		}

		if (options.enableGrouping) {
			this.grouping = {
				messages: {
					empty: 'Drop column headers here for grouping'
				}
			};
		}

		this.gridViewSvc = new GridViewSvc(this, this.options, angularLibs);
		this.gridViewSvc.loaded.finally(() => {
			// Apply column preferences:
			const columnPrefs = this.gridViewSvc.getColumnPrefs();
			_.forEach(columnPrefs, (columnPref, field) => {
				const column = this.gridColumns.getColumn(field);
				if (column) {
					if (column.allowHiding) {
						column.hidden = !!columnPref.isHidden;
					}
					if (column.sortable && columnPref.sortDirection) {
						this.options.sortDirection = {
							field: field,
							dir: columnPref.sortDirection
						};
					}
					if (columnPref.width) {
						column.width = columnPref.width;
					}
				}
			});

			if (!('sortDirection' in this.options) && !this.disableDefaultSort) {
				// No sort direction set by controller or by column preference,
				// set default sort
				const sortColumn = this.gridColumns.getFirstColumn();
				if (sortColumn.sortable) {
					this.options.sortDirection = {
						field: sortColumn.field,
						dir: "asc",
					};
				}
			}
			if (this.options.sortDirection && !this.options.sortDirection.compare) {
				// If no sort comparator is specified, inherit the column definition's sort comparator
				const sortColumn = this.gridColumns.getColumn(this.options.sortDirection.field);
				const sortComparator = _.get(sortColumn, 'sortable.compare');
				if (sortComparator) {
					this.options.sortDirection.compare = sortComparator;
				}
			}

			let companyId = -1;
			const queryParams = this.location.search();
			if (_.get(queryParams, 'companyId') >= 0 && !this.options.hideCompanyFilter) {
				companyId = _.toNumber(queryParams.companyId);
			}
			this.gridDataSrc = new GridDataSrc({
				gridViewSvc: this.gridViewSvc,
				gridEventEmitter: this.eventEmitter,
				options: this.options,
				kendo: _vendorGrid,
				initialUserFilters: this.gridColumns.getFilterValues(),
				companyId,
				angularLibs
			});
			this.gridColumns.build();
			this.gridColumns.initializeFilterData(this.gridDataSrc);
		});
	}

	_setDefaultOption(options, optionName, defaultValue) {
		if (!(optionName in options)) {
			options[optionName] = defaultValue;
		}
	}

	_setDefaultOptions(options) {
		this._setDefaultOption(options, 'searchable', true);
		this._setDefaultOption(options, 'hasViews', true);
		this._setDefaultOption(options, 'hasDefaultView', true);
		this._setDefaultOption(options, 'gridToolbarMenu', []);
		this._setDefaultOption(options, 'gridContextMenu', []);
		this._setDefaultOption(options, 'change', () => {});
		this._setDefaultOption(options, 'customSelectAll', false);
		return options;
	}

	setHeight(height) {
		this.height = height;
	}

	setTitle(title) {
		this.gridTitle = title;
	}

	registerCallbacks(key, fnCallback) {
		this.callbacks[key] = fnCallback || function() {};
	}

	getColumns() {
		return this.gridColumns.getColumns();
	}

	getColumnNames() {
		return this.gridColumns.getColumnNames();
	}

	_setupEventListeners(options) {
		this.callbacks = {};
		this.registerCallbacks('dataBound', options.gridDataInitialized);
		this.registerCallbacks('onColumnShow', options.onColumnShow);
		this.registerCallbacks('onColumnHide', options.onColumnHide);
		this.registerCallbacks('onGridSelectionChange', options.onGridSelectionChange);
		// this.registerCallbacks('addSelectedItems', options.addSelectedItems);
		this.registerCallbacks('beforeGridInitialize', options.beforeGridInitialize);
		this.registerCallbacks('afterGridInitialize', options.afterGridInitialize);
		this.registerCallbacks('beforeViewChange', options.beforeViewChange);
		this.registerCallbacks('afterViewChange', options.afterViewChange);
		if (options.onRowDetailInit) {
			this.registerCallbacks('onRowDetailInit', options.onRowDetailInit);
		}
		this.registerCallbacks('onRowDetailsDestroy', options.onRowDetailsDestroy);
		this.registerCallbacks('onColumnFilterSet', options.onColumnFilterSet);
		this.registerCallbacks('onColumnFilterClear', options.onColumnFilterClear);
	}

	getSelector() {
		return '#' + this.gridElement.attr('id');
	}

	getDataSrc() {
		return this.gridDataSrc;
	}

	getData(id) {
		return this.gridDataSrc.getData(id);
	}

	getDataByUid(uid) {
		return this.gridDataSrc.getDataByUid(uid);
	}

	getViewSvc() {
		return this.gridViewSvc;
	}

	addFilters(filters, preventApplyFilters) {
		if (!_.isArray(filters)) {
			filters = [filters];
		}
		_.each(filters, filter => {
			if (filter) {
				this.getDataSrc().addFilter(filter);
			}
		});
		if (!preventApplyFilters) {
			this.getDataSrc().applyFilters();
		}
	}

	removeFilters(filters, preventApplyFilters) {
		if (!_.isArray(filters)) {
			filters = [filters];
		}
		_.each(filters, filter => {
			if (filter) {
				this.getDataSrc().removeFilter(filter);
			}
		});
		if (!preventApplyFilters) {
			this.getDataSrc().applyFilters();
		}
	}

	modifyFilters(filtersObj, preventApplyFilters) {
		if (filtersObj.remove) {
			this.removeFilters(filtersObj.remove, true);
		}
		if (filtersObj.add) {
			this.addFilters(filtersObj.add, true);
		}
		if (!preventApplyFilters) {
			this.getDataSrc().applyFilters();
		}
	}

	createView(viewId) {
		return GridView.createView(this.angularLibs.$modal, viewId || null, this, this.angularLibs.cvToaster);
	}

	deleteView() {
		return GridDeleteView.deleteView(
			this.angularLibs.$modal,
			this.angularLibs.userPrefService,
			this.gridViewSvc.getCurrentViewId(),
			this.angularLibs.cvToaster,
			this.angularLibs.cvLoc,
			this.options,
			this
		);
	}

	reorderViews() {
		return GridReorderViews.reorderViews(this.angularLibs.$modal, this.angularLibs.userPrefService, this);
	}

	hideView(viewId) {
		this.gridViewSvc.setViewHidden(viewId, true);
	}

	showView(viewId) {
		this.gridViewSvc.setViewHidden(viewId, false);
	}

	editView() {
		return this.createView(this.gridViewSvc.getCurrentViewId());
	}

	destroy() {
		this.destroyed = true;
		this.eventEmitter.removeAllListeners();
	}
}
