import * as GridConstants from './grid.constants';

const DEFAULT_FILTER_CONDITION_OPTIONS = [
	GridConstants.FILTER_CONDITIONS.CONTAINS,
	GridConstants.FILTER_CONDITIONS.DOES_NOT_CONTAIN
];

export default class ColumnFilter {
	static get FILTER_CONDITION_OPTIONS() {
		return DEFAULT_FILTER_CONDITION_OPTIONS;
	}

	static get DEFAULT_FILTER_CONDITION() {
		return this.FILTER_CONDITION_OPTIONS[0];
	}

	constructor(options, angularLibs) {
		this.dataTextField = 'label';
		this.dataValueField = 'value';
		this.angularLibs = angularLibs;
		this.initialize(options);
		this.loadUrlParam();
	}

	get textInput() {
		return this.element;
	}

	get wrapper() {
		return this.element;
	}

	initialize(options) {
		this.dataSrc = {
			data: options.data
			// sort: {
			// 	field: this.dataTextField,
			// 	dir: "desc"
			// }
		};
		// this.dataSrc = options.data;
		if (options.filterUrl) {
			this.dataSrc = new ColumnFilter.ColumnFilterDataSource(options, this);
		}
		this.index = options.index;
		this.enableFilterConditionDropdown = options.enableFilterConditionDropdown;
		this._onFilterInputChange = options.onFilterInputChange || (() => {});
		this._onDropdownFilterSelect = options.onDropdownFilterSelect || (() => {});
		this._parseViewValue = options.parseViewValue || (val => val);
		this._toViewValue = options.toViewValue || (val => val || '');
		this.onFilterSubmit = options.onFilterSubmit || (() => {});
		this.value = options.filterValue || '';
		this.itemTemplate = options.itemTemplate;
		this.meta = options.key;
		this.filterQueryParam = options.filterQueryParam;
		this.filterConditionQueryParam = options.filterConditionQueryParam;
		this.filterLabelQueryParam = options.filterLabelQueryParam;
		this.filterConditionOptions = options.filterConditions || this.constructor.FILTER_CONDITION_OPTIONS;
		this.setFilterCondition(options.filterCondition || this.getDefaultFilterCondition().value);
	}

	loadUrlParam() {
		if (this.angularLibs && this.angularLibs.$location) {
			const searchParams = this.angularLibs.$location.search();
			if (this.filterQueryParam && this.filterQueryParam in searchParams) {
				this.fromViewValue(searchParams[this.filterQueryParam]);
			}
			if (this.filterConditionQueryParam && this.filterConditionQueryParam in searchParams) {
				this.fromViewFilterCondition(searchParams[this.filterConditionQueryParam]);
			}
		}
	}

	updateUrlParam() {
		if (this.angularLibs && this.angularLibs.$location && this.angularLibs.$scope) {
			if (this.filterQueryParam) {
				let value = this.getValue();
				if (!_.isEmpty(value) && !_.isUndefined(value) && !_.isNull(value)) {
					value = this.toViewValue();
				} else {
					value = null;
				}
				this.angularLibs.$scope.$applyAsync(() => {
					this.angularLibs.$location.search(this.filterQueryParam, value).replace();
				});
			}
			if (this.filterConditionQueryParam) {
				let fc = this.getFilterCondition();
				this.angularLibs.$scope.$applyAsync(() => {
					this.angularLibs.$location.search(this.filterConditionQueryParam, fc).replace();
				});
			}
		}
	}

	build(element, initialFilterData) {
		this.container = element.parent();
		// Build the filter condition dropdown:
		this.filterConditionElement = $(`<input />`);
		this.filterConditionElement.attr('id', `${element.attr('id')}_filterConditionDropdown`);
		if (!this.enableFilterConditionDropdown) {
			this.filterConditionElement.addClass('hidden');
		}
		element.before(this.filterConditionElement);
		this.filterConditionElement.kendoDropDownList({
			dataSource: this.filterConditionOptions.map(condition => ({
				value: condition.value,
				label: this.angularLibs.cvLoc(condition.label),
			})),
			dataValueField: 'value',
			dataTextField: 'label',
		});
		this.filterConditionDropdown = this.filterConditionElement.getKendoDropDownList();

		this.element = element;
		if (initialFilterData) {
			this.dataSrc.data = initialFilterData;
		}
		this.buildComplete = true;
	}

	isBuilt() {
		return this.buildComplete;
	}

	isVisible() {
		return this.container.is(":visible");
	}

	setGridColumn(gridColumn) {
		this.gridColumn = gridColumn;
	}

	getGridColumn() {
		// Starts uninitialized. Should be set if needed
		return this.gridColumn;
	}

	onColumnMenuOpen() {
		this.applyValue();
		this.applyFilterCondition();
		this.refreshInputTooltip();
	}

		/**
	 * Fired when the filter menu inside the column menu is opened
	 */
	onFilterMenuOpen() {}

	fromViewValue(val) {
		this.setValue(this._parseViewValue(val));
		this.applyValue();
		return this.getValue();
	}

	toViewValue() {
		return this._toViewValue(this.getValue());
	}

	fromViewFilterCondition(fc) {
		this.setFilterCondition(fc);
		this.applyFilterCondition();
		return this.getFilterCondition();
	}

	clearFilter() {
		this.setValue('');
		this.setInputValue('');
		this.setFilterCondition(this.getDefaultFilterCondition().value);
		this.setInputFilterCondition(this.getDefaultFilterCondition().value);
	}

	refreshInputTooltip() {
		if (this.isBuilt()) {
			const title = this.getInputTooltipText() || null;
			this.element.attr('title', title);
			this.textInput.attr('title', title);
			this.wrapper.attr('title', title);
		}
	}

	onFilterInputChange(...args) {
		this.refreshInputTooltip();
		return this._onFilterInputChange(...args);
	}

	onDropdownFilterSelect(...args) {
		this.refreshInputTooltip();
		return this._onDropdownFilterSelect(...args);
	}

	setValue(val) {
		this.value = val;
		this.updateUrlParam();
	}

	getValue() {
		return this.value;
	}

	setInputValue(val) {
		if (this.isBuilt()) {
			this.element.val(val);
			this.refreshInputTooltip();
		}
	}

	getInputValue() {
		if (this.isBuilt()) {
			return this.element.val();
		}
		return null;
	}

	restoreInputValue(val) {
		this.setInputValue(val);
	}

	getInputValueForRestore() {
		return this.getInputValue();
	}

	getInputTooltipText() {
		return this.getInputValue();
	}

	isFilterConditionValid(filterCondition) {
		return this.filterConditionOptions.some(condition => condition.value === filterCondition);
	}

	setFilterCondition(filterCondition) {
		if (!this.isFilterConditionValid(filterCondition)) {
			// If filterCondition is invalid, set the default filter condition
			filterCondition = this.getDefaultFilterCondition().value;
		}
		this.filterCondition = filterCondition;
		this.updateUrlParam();
	}

	getFilterCondition() {
		return this.filterCondition;
	}

	setInputFilterCondition(filterCondition) {
		if (!this.isFilterConditionValid(filterCondition)) {
			// If filterCondition is invalid, set the default filter condition
			filterCondition = this.getDefaultFilterCondition().value;
		}
		if (this.isBuilt()) {
			this.filterConditionDropdown.value(filterCondition);
		}
	}

	getInputFilterCondition() {
		if (this.isBuilt()) {
			return this.filterConditionDropdown.value();
		}
		return null;
	}

	getFilterConditionOptions() {
		return this.filterConditionOptions;
	}

	getDefaultFilterCondition() {
		return this.filterConditionOptions[0];
	}

	applyInputValue() {
		this.setValue(this.getInputValue());
		return this.getValue();
	}

	applyValue() {
		this.setInputValue(this.getValue());
		return this.getValue();
	}

	applyInputFilterCondition() {
		this.setFilterCondition(this.getInputFilterCondition());
		return this.getFilterCondition();
	}

	applyFilterCondition() {
		this.setInputFilterCondition(this.getFilterCondition());
		return this.getFilterCondition();
	}

	destroy() {
		if (this.isBuilt()) {
			this.filterConditionDropdown.destroy();
		}
	}
}

class ColumnFilterDataSource {
	constructor(options, columnFilter) {
		this.transport = {};
		this.transport.read = options.filterUrl;
		this.columnFilter = columnFilter;
		if (_.isFunction(options.filterUrl)) {
			this.serverFiltering = !!options.filterServerLoading;
			const readFunction = options.filterUrl;
			this.transport.read = function(options) {
				readFunction.call(columnFilter, options, columnFilter);
			};
		}
	}
}
ColumnFilter.ColumnFilterDataSource = ColumnFilterDataSource;

class GenericColumnFilter extends ColumnFilter {
	build(element, initialFilterData) {
		this.onFilterInputChange = _.debounce(super.onFilterInputChange.bind(this), 150);
		element.on('input', this.onFilterInputChange);
		element.on('change', this.onFilterInputChange);
		super.build(element, initialFilterData);
	}
}
ColumnFilter.GenericColumnFilter = GenericColumnFilter;
