import * as utils from '../componentUtils/utils';

const ATTRS = {
	MENU_ID: 'data-cv-menu-id',
	MENU_ITEM_ID: 'data-cv-menu-item-id',
	MENU_OPTION: 'data-menu-option'
};

export default class Menu {
	constructor(args) {
		this.id = kendo.guid();
		this.name = args.name || '';
		this.initialize(args);
		this.build();
	}

	_buildOptionText(option) {
		return `<a id="${option.id}" ${ATTRS.MENU_OPTION}="" ${option.url ? `href="${option.url}"` : ''}>${
			option.label
		}</a>`;
	}

	_buildItem(option) {
		let subItems = [];
		if (_.isArray(option.subOptions)) {
			subItems = _.map(option.subOptions, subOption => this._buildItem(subOption));
		}
		const item = {
			content: option.content,
			cssClass: `${option.cssClass || ''} ${option.hidden ? 'hidden' : ''} ${option.disabled ? 'disabled' : ''}`,
			attr: {
				id: `${this.name}_${option.id}`,
				...option.attrs
			},
			encoded: false,
			items: subItems.length > 0 ? subItems : null,
			text: this._buildOptionText(option)
		};
		item.attr[ATTRS.MENU_ITEM_ID] = option.id;
		return item;
	}

	_buildOptionsMap(optionsMap, options) {
		if (!_.isArray(options) || options.length === 0) {
			return optionsMap;
		}
		_.forEach(options, option => {
			optionsMap[option.id] = option;
			this._buildOptionsMap(optionsMap, option.subOptions);
		});
		return optionsMap;
	}

	initialize(args) {
		this.element = args.element;
		for (let key in args.attrs) {
			this.element.attr(key, args.attrs[key]);
		}
		const originalElementId = this.element.attr('id');
		if (originalElementId) {
			if (this.name !== originalElementId) {
				// Use the element's id as its name for automation
				this.name = originalElementId;
			} else {
				// Element already has an id, add the name as a class instead
				this.element.addClass(this.name);
			}
		} else if (this.name) {
			// Add the name as the element's id for automation
			this.element.attr('id', this.name);
		}
		this.element.attr(ATTRS.MENU_ID, this.id);
		this.onOptionVisibilityChange = utils.getArg(args, 'onOptionVisibilityChange', () => {});
		this.optionsMap = this._buildOptionsMap({}, args.options);
		this.onSelect = args.onSelect;
		this._menuOptions = args.options;
		this.dataSource = [];
		_.forEach(args.options, option => {
			const item = this._buildItem(option);
			this.dataSource.push(item);
		});
	}

	build() {
		this.element.kendoMenu({
			dataSource: this.dataSource,
			select: event => {
				this._onOptionSelect(event);
			}
		});
		this.kendoMenu = this.element.getKendoMenu();

		_.forEach(this._menuOptions, option => {
			if (option.disabled) {
				this.disable(option.id);
			}
		});

		this.queueRefreshDisplay();
	}

	getSelector() {
		return `[${ATTRS.MENU_ID}=${this.id}]`;
	}

	_onOptionSelect(event) {
		const element = $(event.item).find(`[${ATTRS.MENU_OPTION}]`);
		const id = element.attr('id');
		if (_.isFunction(this.onSelect) && !this.isDisabled(id)) {
			this.onSelect.call(this, {
				menu: this,
				optionId: id
			});
		}
		const option = this.optionsMap[id];
		if (option && _.isFunction(option.onSelect) && !this.isDisabled(id)) {
			option.onSelect.call(this, {
				menu: this,
				optionId: id
			});
		}
	}

	_getOptionElement(id) {
		return this.element
			.find(`#${id}`)
			.parents('li')
			.first();
	}

	isDisabled(id) {
		return this._getOptionElement(id).hasClass('disabled');
	}

	enable(id) {
		this._getOptionElement(id).removeClass('disabled');
	}

	disable(id) {
		this._getOptionElement(id).addClass('disabled');
	}

	queueRefreshDisplay() {
		if (this.refreshCancelId) {
			// Do nothing if refresh display already queued
			return;
		}
		this.refreshCancelId = requestAnimationFrame(() => {
			this.refreshCancelId = 0;
			this.refreshDisplay();
		});
	}

	cancelRefreshDisplay() {
		if (this.refreshCancelId) {
			cancelAnimationFrame(this.refreshCancelId);
		}
	}

	refreshDisplay() {}

	setVisibility(id, visible) {
		this.queueRefreshDisplay();
		const li = this._getOptionElement(id);
		if (visible) {
			li.removeClass('hidden');
		} else {
			li.addClass('hidden');
		}
		this.onOptionVisibilityChange({
			menu: this,
			optionId: id,
			visible: visible
		});
	}

	hasVisibleChildren() {
		return _.some(this.optionsMap, option => this.isVisible(option.id));
	}

	isVisible(id) {
		const li = this._getOptionElement(id);
		return li.length && !li.hasClass('hidden');
	}

	isHidden(id) {
		return !this.isVisible(id);
	}

	hide(id) {
		this.setVisibility(id, false);
	}

	show(id) {
		this.setVisibility(id, true);
	}

	_insert(options, parentId, callback) {
		this.queueRefreshDisplay();

		if (!_.isArray(options)) {
			options = [options];
		}
		this._buildOptionsMap(this.optionsMap, options);
		let parent = null;
		if (parentId) {
			parent = this._getOptionElement(parentId);
		}
		callback.call(this, _.map(options, option => this._buildItem(option)), parent);
	}

	add(options, parentId) {
		this._insert(options, parentId, (items, parent) => this.kendoMenu.append(items, parent));
	}

	insertAfter(options, parentId) {
		this._insert(options, parentId, (items, parent) => this.kendoMenu.insertAfter(items, parent));
	}

	insertBefore(options, parentId) {
		this._insert(options, parentId, (items, parent) => this.kendoMenu.insertBefore(items, parent));
	}

	_remove(selector) {
		this.queueRefreshDisplay();
		this.kendoMenu.remove(selector);
	}

	removeAll() {
		this._remove('li');
		this.optionsMap = {};
	}

	remove(id) {
		this._remove(this._getOptionElement(id));
		delete this.optionsMap[id];
	}

	destroy() {
		this.cancelRefreshDisplay();
		this.kendoMenu.destroy();
	}
}
Menu.ATTRS = ATTRS;
