/*
 * Directive for Kendo Editor
 * Usage:
 * 	<div class="cv-editor" id="editor"></div>
 *
 * 	editorParams = {
 *		getTokens: <function>, // (optional, for using tokens) takes an id as input, returns a promise containing data about tokens
 *		processTokens: <function>, // (optional, for using tokens) takes the output from getTokens and returns an array in the format shown below
 *		editorContent: <string>, // the html text to display in the editor when it first loads. Can be overwritten by setText().
 *		tools: <array>, // list of Kendo Editor tools to use. Available tools can be found here: https://demos.telerik.com/kendo-ui/editor/all-tools except insertImage. Three custom features, "tokens", "uploadImage", and "specialCharacters", are also available.
 *		maxFileSize: <integer> // maximum file size for image upload, in bytes.
 *		textareaHeight: <string> // height of the text area within the editor. Defaults to "400px".
 *	};
 *
 *	processTokens returned array: [
 *		{
 *			name: <string>, // the string that displays in the token dropdown menu
 *			value: <string>, // the string that gets inserted into the editor
 *			title: <string> // (optional) If given, it sets the title attribute of the item in the dropdown list that the user can view when hovering over the item.
 *		},
 *		...
 *	];
 *
 *	// to use the custom tools mentioned above, you must import them:
 *	import * as uploadImage from "../../../../common/js/directives/editor-uploadImage.js";
 *	import * as tokens from "../../../../common/js/directives/editor-tokens.js";
 *	import * as specialCharacters from "../../../../common/js/directives/editor-characters.js";
 *
 *  //constructor:
 *  var editor = new Editor(params, $("#editor"), template, cvLoc, {uploadImage: uploadImage, tokens: tokens, specialCharacters: specialCharacters}); // template must first be retrieved from common/partials/editor.jsp using $http.get
 *
 *  //methods:
 *  editor.getText() // returns the content of the text editor
 *  editor.setText(<string>, <boolean>) // sets the content of the text editor. If the boolean is true, it records the string as the original one, which you can later use to check if the content has been modified.
 *  editor.getAndProcessTokens(data) // calls getTokens(data) then processTokens() on the result, then sets the tokens to the data returned by processTokens
 *
 *  //properties:
 *  editor.originalText // the original string after it is loaded into the editor (may undergo minor non-visible changes that make it slightly different from the input text string)
 *
 *
 */
'use strict';



export default class Editor {
	constructor(editorParams, element, template, cvLoc, customTools) {
		if(!editorParams) {
			return "Error: editorParams is not defined";
		} // error statement
		if (!element[0]) {
			return "Error: no element found";
		}
		this.editorParams = editorParams;
		this.container = element;
		this.cvLoc = cvLoc;
		this.customTools = customTools;

		this.template = template;
		this._build();

	}

	// sets the text of the kendo editor. If the second parameter is true, it signifies that this is the original text (helpful later for determining if the user has edited the body)
	setText(text, original) {
		var editor = this.container.find("#editor").data("kendoEditor");
		if (editor) {
			editor.value(text);
		}
		if (original) {
			// stores the original text to detect if the body has been modified
			this.originalText = editor.value() || "";
		}
	}
	// check anchor tags
	areAnchorTagsClean(html) {
		var result = true;
		var entireHtml = $.parseHTML(html);

		var anchorTags = $('<div>').append(entireHtml).find("a");
		$.each(anchorTags, function(index, aTag){
			//check protocol against valid white list
			var protocol = aTag.protocol.toUpperCase();
			if(protocol === 'HTTP:' || protocol === 'HTTPS:' || protocol === 'MAILTO:' || protocol === 'FTP:' || protocol === 'FTPS:'){
			}else{
				console.error('Protocol is not valid for href : ' + aTag.href);
				result = false;
				return false;//only breaks the loop
			}
		});
		return result;
	}

	//set rel="noopener noreferrer" for all anchor tags. https://www.owasp.org/index.php/HTML5_Security_Cheat_Sheet#Tabnabbing
	_applyTabnabbingFilter(html){
		html = html.replace(/<a[^>]*/, function(a) {
			  // first remove existing rel attribute if any
			  a = a.replace(/rel\s*=\s*['"][^"']*['"]/, '');

			  // then add rel="noopener noreferrer" attribute
			  a += ' rel="noopener noreferrer" ';

			  return a;
			});
		return html;
	}
	// compare the current text to originalText
	isModified() {
		var editor = this.container.find("#editor").data("kendoEditor");
		if (editor) {
			return $.trim(editor.value()) != this.originalText;
		}
		return false;
	}
	//Before getting editor data using getTExt() for saving, always check by calling isTextSecure that the data is safe to save.
	isTextSecure(){
		var editor = this.container.find("#editor").data("kendoEditor");
		if (editor) {
			var html = $.trim(editor.value());
			return this.areAnchorTagsClean(html);
		}
		return true;
	}
	// get the content of the kendo editor
	getText() {
		var editor = this.container.find("#editor").data("kendoEditor");
		if (editor) {
			var html = this._applyTabnabbingFilter($.trim(editor.value()));
			//to ensure unsecure value is never saved.
			if(this.isTextSecure(html) === false){
				console.error("Editor text is not secure. Resetting it to empty string");
				return "";
			}
			return html;
		}
	}

	// calls the getTokens function defined in editorParams with the given parameter. Sets the resulting tokens in the Token dropdown tool
	getAndProcessTokens(data) {
		if (this.customTools && this.customTools.tokens) {
			this.customTools.tokens.getAndProcessTokens(data, this.editorParams.getTokens, this.editorParams.processTokens);
		}
	}
	// sets the tokens manually to the provided data, in the same format as processTokens data
	setTokens(data) {
		if (this.customTools && this.customTools.tokens) {
			this.customTools.tokens.setTokens(data);
		}
	}
	_build() {
		// default lodash template settings interfere with JSP tags, so this defines new settings.
		_.templateSettings = {
			interpolate : /<@=(.+?)@>/gim,
			evaluate : /<@(.+?)@>/gim,
			escape : /<@-(.+?)@>/gim
		};
		var editorParams = this.editorParams;
		var templateFunc = _.template(this.template); // read the template JSP for use by the Kendo editor and its custom tools
		var container = this.container;
		var self = this;

		container.html(templateFunc({
			"content" : editorParams.editorContent || "",
			"height" : editorParams.textareaHeight || "400px"
		}));

		// check editorParams for any of the custom tools
		if (editorParams.tools) {
			var uploadImage = editorParams.tools.indexOf("uploadImage");
			if (uploadImage > -1 && self.customTools && self.customTools.uploadImage) {
				editorParams.tools[uploadImage] = {
					name: "uploadImage",
					tooltip: self.cvLoc('editor.insertImage.label.insertImage'),
					exec: _onUploadToolClick // opens the popup to insert an image
				};
			}

			var tokens = editorParams.tools.indexOf("tokens");
			if (tokens > -1 && self.customTools && self.customTools.tokens) {
				editorParams.tools[tokens] = {
					name: "token",
					template: $("#token-template").html()
				};
				var editor = container.find("#editor").data("kendoEditor");
				self.customTools.tokens.createTokenDropDown($("#tokenTool"), self.cvLoc, editor);
			}

			var specialCharsTool = editorParams.tools.indexOf("specialCharacters");
			if (specialCharsTool > -1 && self.customTools && self.customTools.specialCharacters) { // if special characters tool
				editorParams.tools[specialCharsTool] = {
					name: "specialCharacters",
					tooltip: self.cvLoc('editor.specialChars'),
					exec: onSpecialCharClick // opens the popup to insert special characters
				};
			}

			// prevent insertImage because of Content Security Policy
			_.remove(editorParams.tools, function(tool) {
				return tool == "insertImage"
			})
		}

		function onSpecialCharClick(e) {
			self.customTools.specialCharacters.onSpecialCharClick(e, $(this).data("kendoEditor"));
		}


		var defaultTools = [ // tools to use if editorParams.tools is not provided
			"bold",
			"italic",
			"underline",
			"justifyLeft",
			"justifyCenter",
			"justifyRight",
			"justifyFull",
			"insertUnorderedList",
			"insertOrderedList",
			"indent",
			"outdent",
			"createLink",
			"unlink" ];

		container.find("#editor").kendoEditor({ // initialize the Kendo Editor
			tools : editorParams.tools || defaultTools,
			/*
			 * deserialization: { custom: function(html) { return html.replace(/(style=)([^"].+?)[;>]/g,
			 * '$1"$2;"'); // handles forgotten quotes around style attributes } },
			 */
			serialization: {
				custom: function(html) {
					return html.replace(/<br \/>/g, "<br/>"); // Kendo converts <br/> to <br /> so this reverts that conversion. Future email templates should not rely on <br> tags anyway.
				}
			},
			resizable: {content:true} //to allow users to resize the editor
		});

		if (tokens > -1 && self.customTools && self.customTools.tokens) {
			var editor = container.find("#editor").data("kendoEditor");
			self.customTools.tokens.createTokenDropDown($("#tokenTool"), self.cvLoc, editor); // creates the token tool
		}

		function _onUploadToolClick(e) { // open the popup to insert images
			var popupHtml = $("#window").html().replace(/makeThis_/g,''); // Since the popup copies html from the page, it is necessary to change the ID so that the elements in the popup can be properly found by the JS code.
			var editor = $(this).data("kendoEditor");
			self.customTools.uploadImage.onUploadToolClick(e, popupHtml, editor, self.cvLoc, self.editorParams.maxFileSize);
		}

	}
}
