import { cvCommonModule } from "common/js/modules";
import {FILE_UPLOAD_PARAMS, FILE_UPLOAD_STATUS, FILE_UPLOAD_CONSTANTS} from "adminConsole/js/constants/cvFileUpload.constant.js";

class CvFileUploadDirective {
    constructor() {
        this.restrict = 'E';
        this.templateUrl = appUtil.appRoot + 'adminConsole/partials/cvFileUpload.jsp';
        this.scope = {
            enableMultiSelect: '@cvEnableMultiSelect',
            allowedFileTypes: '@cvAllowedFileTypes',
            uploadedFiles: '@cvUploadedFiles',
            uploadLimitMB: '@cvUploadLimitMb',
            onComplete: '&cvOnComplete',
            uploadInfo: '=cvUploadInfo',
            isUploading: '=cvIsUploading',
            disableUpload: '=cvDisableUpload'
        };
        this.controllerAs = 'fileUploadCtrl';
		this.controller = CvFileUploadController;
        this.bindToController = true;
    }
}

class CvFileUploadController {
    constructor($scope, $http, cvLoc, cvUtil, cvToaster) {
        this.$scope = $scope;
        this.$http = $http;
        this.cvLoc = cvLoc;
        this.cvUtil = cvUtil;
        this.cvToaster = cvToaster;

        this.fileUploadMap = new Map();
        this.completedUploads = 0;
    }

    $onInit() {
        this.enableMultiSelect = (!_.isUndefined(this.enableMultiSelect) && this.enableMultiSelect !== "false");
        if(this.enableMultiSelect) {
            $('#cv-file-upload').attr("multiple", "true");
        }
        this.isUploading = false;
        if(!_.isEmpty(this.uploadedFiles)) {
            this.uploadedFilesList = _.split(this.uploadedFiles, ',');
        } else {
            this.uploadedFilesList = [];
        }

        this.uploadLimit = (_.parseInt(this.uploadLimitMB) || FILE_UPLOAD_CONSTANTS.DEFAULT_UPLOAD_LIMIT_MB) * 1024 * 1024; // reducing to bytes
    }

    uploadClicked = () => {
        $('#cv-file-upload').trigger('click');
    }

    _validateUploadLimit = (files) => {
        let fileSize = 0;
        _.each(files, file => {
            fileSize += file.size;
        });
        if(fileSize > this.uploadLimit) {
            this.cvToaster.showErrorMessage({
                message: this.cvLoc('error.uploadSizeLimitExceeded', `<b>${this.uploadLimit / (1024 * 1024)} MB</b>`)
            });
            return false;
        }
        return true;
    }

    uploadFiles = () => {
        if(!_.isUndefined(_.head(this.selectedFiles))) {
            if(this._validateUploadLimit(this.selectedFiles)) {
                this.isUploading = true;
                const uploadParams = this._getUploadParams();
                _.each(this.selectedFiles, (file) => {
                    this._startFileUpload(file, uploadParams);
                });
            }
        }
    }

    _onUploadFinished = () => {
        this.completedUploads++;
        if(this.fileUploadMap.size !== this.completedUploads) {
            return;
        }

        const callbackArr = [];
        const failedUploads = [];
        for(const [key, value] of this.fileUploadMap) {
            callbackArr.push({
                file: value.file,
                status: value.fileStatus
            });

            if(value.fileStatus === FILE_UPLOAD_STATUS.COMPLETED) {
                this.uploadedFilesList.push(value.file.name);
            } else if (value.fileStatus === FILE_UPLOAD_STATUS.FAILED) {
                failedUploads.push(value.file.name);
            }
        }
        this.onComplete({filesInfo: callbackArr});
        if(_.size(failedUploads) > 0) {
            this.cvToaster.showErrorMessage({
                message: this.cvLoc('error.fileUploadFailed', `<b>${failedUploads.toString()}</b>`)
            });
        } else {
            this.cvToaster.showSuccessMessage({
                message: this.cvLoc('info.uploadFinished')
            });
        }
        
        this.isUploading = false;
        this.fileUploadMap.clear();
        this.completedUploads = 0;
    }

    _getUploadParams = () => {
        const uploadParams = [];
        if(!_.isUndefined(this.uploadInfo)) {
            _.forOwn(this.uploadInfo.params, (value, key) => {
                uploadParams.push(this.cvUtil.createSingleParamString(key, value));
            });
        }
        uploadParams.push(this.cvUtil.createSingleParamString(FILE_UPLOAD_PARAMS.STREAMING_PROTOCOL, true));
        uploadParams.push(this.cvUtil.createSingleParamString(FILE_UPLOAD_PARAMS.IS_FILE, true));

        return uploadParams;
    }

    _getModifiedTimeInSec = (file) => {
        const dateObj = new Date(file.lastModified);
        return (dateObj.getTime() / 1000).toFixed(0);
    }

    _startFileUpload = (file, uploadParams) => {
        const uploadInfoForFile = [...uploadParams];
        uploadInfoForFile.push(this.cvUtil.createSingleParamString(FILE_UPLOAD_PARAMS.FILE_NAME, file.name));
        uploadInfoForFile.push(this.cvUtil.createSingleParamString(FILE_UPLOAD_PARAMS.FILE_SIZE, file.size));
        uploadInfoForFile.push(this.cvUtil.createSingleParamString(FILE_UPLOAD_PARAMS.FILE_LAST_MODIFIED, this._getModifiedTimeInSec(file)));
        
        const fileInfo = new CvFileUploadInfo(file, this.cvUtil);
        fileInfo.fileStatus = FILE_UPLOAD_STATUS.RUNNING;
        this.fileUploadMap.set(fileInfo.fileId, fileInfo);

        this.$http({
            url: this.uploadInfo.url,
            method: 'POST',
            data: _.join(uploadInfoForFile, '&'),
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
            }
        }).then((successResp) => {
            this._processHeaderResponse(fileInfo, successResp);
        }, (errorResp) => {
            fileInfo.fileStatus = FILE_UPLOAD_STATUS.FAILED;
            this._onUploadFinished();
        });
    }

    _processHeaderResponse = (fileInfo, response) => {
        const returnCode = _.get(response, "data.returnCode");
        if (returnCode === "Success" || returnCode === "FileExists") {
            fileInfo.requestId = _.get(response, "data.requestId");
            fileInfo.currentOffset = _.get(response, "data.chunkOffset");
            this._uploadFileData(fileInfo);
        } else if (returnCode == "Error") {
            fileInfo.fileStatus = FILE_UPLOAD_STATUS.FAILED;
            this._onUploadFinished();
        }
    }

    _uploadFileData = (fileInfo) => {
        if(fileInfo.currentOffset >= fileInfo.file.size) {
            fileInfo.fileStatus = FILE_UPLOAD_STATUS.COMPLETED;
            this._onUploadFinished();
            return;
        }

        const rangeTo = Math.min(fileInfo.file.size, fileInfo.currentOffset + FILE_UPLOAD_CONSTANTS.CHUNK_SIZE);
        const blob = fileInfo.file.slice(fileInfo.currentOffset, rangeTo, fileInfo.file.type);

        this.$http.post(this.uploadInfo.url, blob, {
            headers: {
                'Content-Type': 'application/offset+octet-stream;'
            },
            transformRequest: angular.identity,
            params: {
                requestId: fileInfo.requestId
            }
        }).then((successResp) => {
            const returnCode = _.get(successResp, "data.returnCode");
            if (returnCode == "Success") {
                fileInfo.currentOffset = _.get(successResp, "data.chunkOffset");
                this._uploadFileData(fileInfo);
            } else if (returnCode == "Error") {
                fileInfo.fileStatus = FILE_UPLOAD_STATUS.FAILED;
                this._onUploadFinished();
            }
        }, (errorResp) => {
            fileInfo.fileStatus = FILE_UPLOAD_STATUS.FAILED;
            this._onUploadFinished();
        });
    }
}

class CvFileUploadInfo {
    constructor (file, cvUtil) {
        this.fileId = cvUtil.generateUUID();
        this.file = file;
        this.fileStatus = FILE_UPLOAD_STATUS.IDLE;
        this.percentComplete = 0;
        this.currentOffset = 0;
        this.requestId = "";
    }
}

CvFileUploadController.$inject = [
    '$scope',
    '$http',
    'cvLoc',
    'cvUtil',
    'cvToaster'
];

cvCommonModule.directive('cvFileUpload', () => new CvFileUploadDirective());
export default cvCommonModule;