import CvWizardEvent from 'common/js/directives/cv-wizard/cvWizardEvent.js';
import EventEmitter from 'eventemitter3';
import CvWizardStep from './cvWizardStep';

/**
 * This class represents the data structure and logic of a wizard.
 */
export default class CvWizard extends EventEmitter {
    /**
     * 
     * @param {Object} config Configuration object. The properties of this `CvWizard` object
     * will be kept in sync with the `config` object to support angular's 2-way binding.
     * @param {Object} config.model The model object
     * @param {boolean} config.enableSkipping Whether or not skipping is enabled for the wizard
     */
    constructor(config) {
        super();
        this._id = _.uniqueId('CvWizard');
        this._config = config;
        this._stepMap = {};
        this._stepOrder = [];
        this._activeStepIndex = -1;
    }
    
    /*
    _model and _enableSkipping are defined by getter/setters so that they are 2-way bound to
    the config object
    */
    
    get _model() {
        return this._config.model;
    }
    
    set _model(value) {
        this._config.model = value;
    }
    
    get _enableSkipping() {
        return this._config.enableSkipping;
    }
    
    set _enableSkipping(value) {
        this._config.enableSkipping = value;
    }
    
    /**
     * @return {string} The wizard's unique id
     */
    getId() {
        return this._id;
    }
    
    /**
     * @param {Object} model The new model object
     */
    setModel(model) {
        this._model = model;
    }
    
    /**
     * @return {Object} The wizard's model object
     */
    getModel() {
        return this._model;
    }
    
    /**
     * @param {boolean} enabled Whether skipping should be enabled
     */
    setSkippingEnabled(enabled) {
        this._enableSkipping = enabled;
    }
    
    /**
     * @return {boolean} Whether skipping is enabled
     */
    isSkippingEnabled() {
        return this._enableSkipping;
    }
    
    /**
     * @return {boolean} Whether the active step is the first step (index is 0)
     */
    isFirstStep() {
        return this._activeStepIndex === 0;
    }
    
    /**
     * @return {boolean} Whether the active step is the last step
     */
    isLastStep() {
        return this._activeStepIndex === (this._stepOrder.length - 1);
    }
    
    /**
     * @return {number} The number of steps in the wizard
     */
    getStepCount() {
        return this._stepOrder.length;
    }
    
    /**
     * @return {string[]} The array of step ids in order
     */
    getStepOrder() {
        return this._stepOrder;
    }
    
    /**
     * @param {string[]} stepOrder The array of step ids in their new order.
     * All step ids must be valid and all step ids must be present.
     */
    setStepOrder(stepOrder) {
        this._stepOrder = stepOrder;
    }
    
    /**
     * @param {number} index The new active step index
     */
    setActiveStepIndex(index) {
        this._activeStepIndex = index;
    }
    
    /**
     * @return {number} The index of the active step
     */
    getActiveStepIndex() {
        return this._activeStepIndex;
    }
    
    /**
     * Sets the active step index to the index of the step that has the specified
     * id. If not found, the active step index will be set to -1
     * @param {string} id The id of the step
     */
    setActiveStepIndexById(id) {
        this._activeStepIndex = this._stepOrder.indexOf(id);
    }
    
    /**
     * @return {string} The id of the active step
     */
    getActiveStepId() {
        return this._stepOrder[this._activeStepIndex];
    }
    
    /**
     * @return {CvWizardStep} The active step object
     */
    getActiveStep() {
        return this.getStep(this._activeStepIndex);
    }
    
    /**
     * @param {number} index The index of the step to return
     * @return {CvWizardStep} The step object at the specified index, or `null`
     * if the index is out of bounds
     */
    getStep(index) {
        if (index < 0 || index >= this._stepOrder.length) {
            return null;
        }
        return this.getStepById(this._stepOrder[index]);
    }
    
    /**
     * @param {string} id The id of the step to return
     * @return {CvWizardStep} The step object with the specified id, or `null`
     * if no step exists with the specified id.
     */
    getStepById(id) {
        return this._stepMap[id];
    }
    
    /**
     * @param {CvWizardStep} step The step to look up
     * @return {number} The index of the step, or `-1` if the step is not found
     */
    indexOf(step) {
        return this._stepOrder.indexOf(step.id);
    }
    
    /**
     * @return {CvWizardStep} The step object of the first step
     */
    getFirstStep() {
        return this.getStep(0);
    }
    
    /**
     * @return {CvWizardStep} The step object of the last step
     */
    getLastStep() {
        return this.getStep(this._stepOrder.length - 1);
    }
    
    /**
     * @param {CvWizardStep} step The step to add. It will be appended to the
     * end of the step list
     */
    addStep(step) {
        this._stepMap[step.id] = step;
        this._stepOrder.push(step.id);
    }
    
    /**
     * Removes the step by id. If the active step is removed, the next step (or
     * the previous step if no next step) will be entered
     * @param {string} id The id of the step to remove
     * @param {boolean} [preventStepActivation] If true, the step list and active
     * step index will update with activating / deactivating the active step
     */
    removeStepById(id, preventStepActivation) {
        const stepIndex = this._stepOrder.indexOf(id);
        if (stepIndex >= 0) {
            // The step exists
            if (stepIndex === this.getActiveStepIndex()) {
                // The step being removed is the active step. Go to the next
                // step or the previous if possible.
                if (this.getStepCount() > 1) {
                    let newStepIndex = stepIndex;
                    if (stepIndex < this.getStepCount() - 1) {
                        newStepIndex++;
                    } else if (stepIndex > 0) {
                        newStepIndex--;
                    }
                    this._navigateToStep(newStepIndex, preventStepActivation);
                } else if (!preventStepActivation) {
                    // No other steps, just leave the active step
                    this.getActiveStep().leave();
                }
            } else if (stepIndex < this.getActiveStepIndex()) {
                // The active step is past the removed step. Update the active
                // step index
                this._activeStepIndex--;
            }
            delete this._stepMap[id];
            this._stepOrder.splice(stepIndex, 1);
        }
    }
    
    /**
     * Emits an event with the previous and new step ids. If the event completes
     * successfully, the wizard will advance to the specified step.
     * @param {string} type The type of event.
     * @param {number} newIndex The index of the step to change to.
     * @param {boolean} [stepCompleteAfterChange] Whether to mark the current
     * step as true or false before changing steps. If unset, it will not be updated.
     * @return {Promise} A promise that, when resolved, returns whether or not
     * the step was changed successfully
     */
    _doStepChangeEvent(type, newIndex, stepCompleteAfterChange=null) {
        if (this.getActiveStepIndex() === newIndex) {
            // Do nothing if the new step is already active
            return false;
        }
        let _stepCompleteUpdatePrevented = false;
        const _promisesToResolve = [];
        const eventDetails = {
            model: this.getModel(),
            oldStepId: this.getActiveStepId(),
            newStepId: this.getStep(newIndex).getId(),
            /**
             * @return {boolean} Whether or not the step completion update has
             * been prevented by an event listener
             */
            isStepCompleteUpdatePrevented() {
                return _stepCompleteUpdatePrevented;
            },
            /**
             * If called by an event listener, the step's completion status will
             * not be updated
             */
            preventStepCompleteUpdate() {
                _stepCompleteUpdatePrevented = true;
            },
            /**
             * Adds a promise that will be resolved before the step change event continues
             * @param {Promise} promise 
             */
            resolve(promise) {
                _promisesToResolve.push(promise);
            }
        };
        const event = new CvWizardEvent(type, eventDetails, true);
        this.emit(type, event);
        return Promise.all(_promisesToResolve).catch((err) => {
            // If any promise is rejected, prevent the step change event
            console.log(err);
            event.preventDefault();
        }).finally(() => {
            if (!event.isDefaultPrevented()) {
                if (stepCompleteAfterChange !== null && !_stepCompleteUpdatePrevented) {
                    this.getActiveStep().setCompleted(!!stepCompleteAfterChange);
                }
                this._navigateToStep(newIndex);
            }
            return !event.isDefaultPrevented();
        });
    }
    
    /**
     * Leaves the current step and enters the specified step
     * @param {number} index The index of the step to navigate to. If the index
     * is out of bounds, the closest bound will be used instead
     * @param {boolean} [preventStepActivation] If true, the active step index
     * will update with activating / deactivating the active step
     */
    _navigateToStep(index, preventStepActivation) {
        // Ensure index is within bounds:
        if (index >= this._stepOrder.length) {
            index = this._stepOrder.length - 1;
        } else if (index < 0) {
            index = 0;
        }
        if (index === this._activeStepIndex) {
            // Do nothing if the step is not changing
            return;
        }
        
        let oldStepId = null;
        const oldStep = this.getActiveStep();
        if (oldStep) {
            oldStepId = oldStep.getId();
        }
        let newStepId = null;
        let newStep = this.getStep(index);
        if (newStep) {
            newStepId = newStep.getId();
        }
        
        // Switch steps:
        if (oldStep && !preventStepActivation) {
            oldStep.leave(newStepId);
        }
        this._activeStepIndex = index;
        if (newStep && !preventStepActivation) {
            newStep.enter(oldStepId);
        }
    }
    
    /**
     * Changes the active step to the specified index. The current step will not
     * be marked as completed.
     */
    goToStep(index) {
        return this._doStepChangeEvent(CvWizard.EVENT.GOTO, index);
    }
    
    /**
     * Sets the active step to the next index. The current step will be marked as complete.
     */
    next() {
        return this._doStepChangeEvent(CvWizard.EVENT.NEXT, this.getActiveStepIndex() + 1, true);
    }
    
    /**
     * Sets the active step to the next index. The current step will not be marked as complete.
     */
    skip() {
        return this._doStepChangeEvent(CvWizard.EVENT.NEXT, this.getActiveStepIndex() + 1);
    }
    
    /**
     * Sets the active step to the previous index. The current step will be marked as incomplete.
     */
    previous() {
        return this._doStepChangeEvent(CvWizard.EVENT.PREVIOUS, this.getActiveStepIndex() - 1, false);
    }
    
    /**
     * Emits an event with the active step id
     * @param {string} type The event type
     * @return {boolean} Whether or not the event completed successfully
     */
    _doWizardExitEvent(type) {
        const event = new CvWizardEvent(type, {
            model: this.getModel(),
            activeStepId: this.getActiveStepId()
        }, true
        );
        this.emit(type, event);
        return !event.isDefaultPrevented();
    }
    
    /**
     * Emits a finish event. If the finish event completes, a lateFinish event
     * will also be emitted.
     */
    finish() {
        const finished = this._doWizardExitEvent(CvWizard.EVENT.FINISH);
        if (finished) {
            this._doWizardExitEvent(CvWizard.EVENT.LATE_FINISH);
        }
    }
    
    /**
     * Emits a cancel event.
     */
    cancel() {
        this._doWizardExitEvent(CvWizard.EVENT.CANCEL);
    }
};
Object.defineProperty(CvWizard, 'EVENT', {
    writable: false,
    value: Object.freeze({
        NEXT: 'next',
        PREVIOUS: 'previous',
        GOTO: 'goto',
        FINISH: 'finish',
        LATE_FINISH: 'lateFinish',
        CANCEL: 'cancel',
    }),
});