const ATTRS = {
	ACTION_ID: 'data-action-id'
};

const CONTENT_TEMPLATE = kendo.template(`
	# if (title) { #
		<div class="modal-header">
			<h4 class="modal-title">#= title #</h4>
		</div>
	# } #
	<div class="modal-body">#= content #</div>
	# if (actions && actions.length > 0) { #
		<div class="modal-footer">
			# for (var i = 0; i < actions.length; i++) { #
				<button type="button" ${ATTRS.ACTION_ID}="#= actions[i].id #"
						class="btn # if (actions[i].primary) { # btn-primary # } else { # btn-default # } #">
					#= actions[i].title #
				</button>
			# } #
		</div>
	# } #
	<a class="modal__close-btn"></a>
`);
const DIALOG_WIDTH = 500;

export default class Dialog {
	/**
	 *
	 * @param {Object} options
	 * @param {boolean} [options.backdrop=true]
	 * @param {string} [options.title]
	 * @param {Object[]} [options.actions]
	 * @param {string} options.actions[].id
	 * @param {string} options.actions[].title
	 * @param {function} options.actions[].onClick
	 * @param {boolean} [options.actions[].primary=false]
	 * @param {boolean} [options.visible=true]
	 * @param {function} [options.onClose]
	 * @param {function} [options.onHide]
	 */
	constructor(options = {}) {
		this.backdrop = options.backdrop !== false;
		this.title = options.title;
		this.content = options.content;
		this.dialogClass = options.dialogClass;
		this.visible = options.visible !== false;
		this.onClose = options.onClose;
		this.onHide = options.onHide;
		this.actions = options.actions;
		this.actionsMap = {};
		if (this.actions && this.actions.length > 0) {
			this.actions.forEach(action => {
				this.actionsMap[action.id] = action;
			});
		}

		this.build();
	}

	build() {
		this.element = $(`<div></div>`);
		this.element.appendTo(document.body);
		this.element.kendoDialog({
			modal: this.backdrop,
			closable: false,
			visibile: this.visible,
			width: DIALOG_WIDTH,
			title: false,
			content: CONTENT_TEMPLATE(this),
			close: this.onClose,
			hide: this.onHide
		});
		this.dialog = this.element.data('kendoDialog');
		this.dialog.element.addClass('modal-content');
		this.dialog.wrapper.addClass('cv-kendo-dialog modal-dialog');
		this.dialog.element.on('click', '.modal__close-btn', e => {
			this.close();
		});
		this.dialog.element.on('click', `[${ATTRS.ACTION_ID}]`, e => {
			const id = e.target.getAttribute(ATTRS.ACTION_ID);
			const action = this.actionsMap[id];
			if (typeof action.onClick === 'function') {
				action.onClick.call(this);
			}
		});
	}

	open() {
		this.dialog.open();
	}

	close() {
		this.dialog.close();
	}

	destroy() {
		this.dialog.destroy();
		this.element.remove();
	}

	static confirm(header, message, options = {}) {
		return new Promise((resolve, reject) => {
			let resolved = false;
			const dialog = new Dialog({
				title: `
					<span class="glyphicon glyphicon-check"></span>
					${header}
				`,
				content: message,
				actions: [
					{
						id: 'no',
						title: cvUtil.cvLocalize('no'),
						onClick: () => {
							dialog.close();
						}
					},
					{
						id: 'yes',
						title: cvUtil.cvLocalize('yes'),
						primary: true,
						onClick: () => {
							resolved = true;
							dialog.close();
							let promise;
							if (options && typeof options.yesFunction === 'function') {
								promise = Promise.resolve(options.yesFunction());
							} else {
								promise = Promise.resolve();
							}
							promise.finally(resolve);
						}
					}
				],
				onClose: e => {
					if (!resolved) {
						let promise;
						if (options && typeof options.noFunction === 'function') {
							promise = Promise.resolve(options.noFunction());
						} else {
							promise = Promise.resolve();
						}
						promise.finally(reject);
					}
				},
				onHide: () => {
					dialog.destroy();
				}
			});
		});
	}

	static notify(header, message, options = {}) {
		return new Promise((resolve, reject) => {
			const dialog = new Dialog({
				title: `
					<span class="glyphicon glyphicon-info-sign"></span>
					${header}
				`,
				content: message,
				actions: [
					{
						id: 'close',
						title: cvUtil.cvLocalize('label.close'),
						onClick: () => {
							dialog.close();
						}
					}
				],
				onClose: () => {
					resolve();
				},
				onHide: () => {
					dialog.destroy();
				}
			});
		});
	}

	static error(header, message, options = {}) {
		return new Promise((resolve, reject) => {
			const dialog = new Dialog({
				title: `
					<div class="text-danger">
						<span class="glyphicon glyphicon-check"></span>
						<span>${header}</span>
					</div>
				`,
				content: `
					<div class="text-danger">${message}</div>
				`,
				actions: [
					{
						id: 'close',
						title: cvUtil.cvLocalize('label.close'),
						onClick: () => {
							dialog.close();
						}
					}
				],
				onClose: () => {
					resolve();
				},
				onHide: () => {
					dialog.destroy();
				}
			});
		});
	}
}
