/*
 * Formatters for search components
 * Use closure instead of prototype as there will be only one instance of this class.
 */

var cvFormatters = (function() {

	return {
		Constants : {
			oneKb : 1024,
			MAX_SIZE : 9007199254740992,
			sizeUnits : [ 'BYTES', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB' ],
			countUnits : [ 'h', 'k', 'M', 'G', 'T', 'P', 'E' ],
			relativeSizing : "relative",
			defaultDateFormat : "MMM D, YYYY hh:mm:ss A",
			durationObject : [ {
				unit : 'y',
				factor : 12
			}, {
				unit : 'M',
				factor : 30
			}, {
				unit : 'd',
				factor : 24
			}, {
				unit : 'h',
				factor : 60
			}, {
				unit : 'm',
				factor : 60
			}, {
				unit : 's',
				factor : 1000
			}, {
				unit : 'ms',
				factor : 1
			} ]
		},

		highlighterStyleClasses : {},

		getDisplayValue : function(value, renderer, dataModel, row) {
			var dispValue = value;
			if (typeof dispValue === "undefined" || dispValue == null) {
				return cvSearchMessages.NotAvailable;
			}
			if (renderer) {
				if (renderer.type == "text" && renderer.valueDispStrMap) {
					dispValue = cvFormatters.formatTextFromMap(renderer.valueDispStrMap, dispValue);
				} else if (renderer.type == "size") {
					dispValue = cvFormatters.formatSize(dispValue, renderer);
				} else if (renderer.type == "number") {
					dispValue = cvFormatters.formatNumber(dispValue);
				} else if (renderer.type == "date") {
					dispValue = cvFormatters.formatDate(dispValue,
							renderer.fmt,
							renderer.srcFmt,
							renderer.customDateFormat,
							renderer.timezone,renderer.srcTimezone);
				} else if (renderer.type == "hyperlink") {
					dispValue = cvFormatters.getHyperlink(dispValue);
				} else if (renderer.type == "boolean") {
					dispValue = cvFormatters.formatBoolean(dispValue, renderer.booleanType);
				} else if (renderer.type == "toggle_boolean") {
					dispValue = cvFormatters.formatToggleBoolean(dispValue, renderer, row, dataModel);
				} else if (renderer.type == "buttonlink") {
					dispValue = cvFormatters.getButtonLink(dispValue, renderer, dataModel);
				} else if (renderer.type == "formatstring") {
					dispValue = cvFormatters.formatString(dispValue, renderer.arguments);
				} else if (renderer.type == "numberToString") {
					dispValue = cvFormatters.formatNumberToString(dispValue);
				} else if (renderer.type == "link") {
					dispValue = cvFormatters.formatLink(dispValue, renderer, row);
				} else if (renderer.type == "unc") {
					dispValue = cvFormatters.formatUNC(dispValue, renderer, row);
				} else if (renderer.type == "highlighter") {
					dispValue = cvFormatters.formatHighlighter(dispValue, renderer, row, dataModel);
				} else if (renderer.type == "cellHighlighter") {
					dispValue = cvFormatters.formatCellHighlighter(dispValue, renderer, row, dataModel);
				} else if (renderer.type == "urlPartsPrefixRemover") {
					dispValue = cvFormatters.urlPartsPrefixRemover(dispValue, renderer);
				} else if (renderer.type == "duration") {
					dispValue = cvFormatters.formatTimeDuration(dispValue, renderer);
				}
			}
			return dispValue;
		},
		/*
		 * given value and count, return value <span class='count'>(count)</span>
		 */
		getFacetDisplayString : function(obj, renderer) {
			if (obj && Object.keys(obj).length === 0) {
				str = cvSearchMessages.NotAvailable;
			}
			if (obj.dispName) {
				str = obj.dispName;
			} else if (obj.isRange) {
				str = this.getDisplayValue(obj.start, renderer) + " - " + this.getDisplayValue(obj.end, renderer);
			} else {
				str = this.getDisplayValue(obj.value, renderer);
			}
			var valHtml = "<span class='value'>" + str + "</span>";
			var countHtml = "<span class='count'> (" + this.formatNumber(obj.count) + ")</span>";
			if (renderer && renderer.hideCount) {
				return valHtml;
			}
			return valHtml + countHtml;
		},
		formatNumber : function(n) {
			if (n === undefined || n === null) {
				return 0;
			}
			var nStr;
			try {
				nStr = n.toLocaleString();
			} catch (e) {
				return n;
			}
			return nStr;
		},
		formatNumberToString : function(n, decimal) {
			if (n === undefined || n === null) {
				return 0;
			}
			var ret = n; // test -- 1000000000000000 -->1000.00 T
			var nStr = new String(n);
			var base = 1000;
			if (decimal == undefined) {
				decimal = 2;
			}
			try {
				if (nStr.length > 3) {
					var number = parseInt(n);
					if (number !== 0) {
						var countGroup = parseInt(this.log10(number) / this.log10(base));
						var countUnitStr = this.Constants.countUnits[countGroup];
						var roundOff = 0;

						if (decimal) {
							roundOff = (number / Math.pow(base, countGroup)).toFixed(decimal);
						} else {
							roundOff = parseInt((number / Math.pow(base, countGroup)));
						}

						ret = roundOff + " " + countUnitStr;
					}
					return ret;
				}
			} catch (err) {
				// console.error(err);
			}
			return ret;
		},
		log10 : function(val) {
			return Math.log(val) / Math.LN10;
		},
		formatTextFromMap : function(map, value) {
			if (map && map.hasOwnProperty(value)) {
				return map[value];
			}
			return value;
		},
		formatSize : function(fileSizeInBytes, renderer) {
			if (fileSizeInBytes === undefined || fileSizeInBytes === null) {
				return 0;
			}

			var source = null, target = null;
			if (renderer === undefined || renderer.source === undefined) {

				renderer = {};
				renderer.target = this.Constants.relativeSizing;
				renderer.source = 'BYTES';
			}

			source = renderer.source.toUpperCase();
			target = renderer.target.toUpperCase();

			if (renderer.target == this.Constants.relativeSizing) {
				var i = this.Constants.sizeUnits.indexOf(source);
				do {
					fileSizeInBytes = fileSizeInBytes / this.Constants.oneKb;
					i++;
				} while (fileSizeInBytes >= this.Constants.oneKb);

				// i+1 because BYTES is the first entry in sizeUnites and we
				// start from KB
				return Math.max(fileSizeInBytes, 0)
						.toFixed(typeof renderer.tofixed != undefined ? renderer.tofixed : 2) +
						" " + this.Constants.sizeUnits[i];
			} else {
				// diff determines how many times we would have to divide or
				// multiply the source by 1024
				var diff = this.Constants.sizeUnits.indexOf(target) - this.Constants.sizeUnits.indexOf(source);
				if (diff < 0) {
					fileSizeInBytes = fileSizeInBytes * Math.pow(this.Constants.oneKb, Math.abs(diff));
				} else {
					fileSizeInBytes = fileSizeInBytes / Math.pow(this.Constants.oneKb, Math.abs(diff));
				}
				return Math.max(fileSizeInBytes, fileSizeInBytes == 0 ? 0 : 0.001)
						.toFixed(typeof renderer.tofixed != "undefined" ? renderer.tofixed : 3) +
						" " + target;
			}

		},
		getSizeInBytes : function(value, unit) {
			var i = this.Constants.sizeUnits.indexOf(unit);
			if (i > -1) {
				return Math.round(value * Math.pow(this.Constants.oneKb, i), 0);
			}
			return value;
		},
		formatFromNow : function(ts) {
			if (ts == 0) {
				return cvSearchMessages.NotAvailable;
			}
			var date = new Date(cvFormatters.getDisplayValue(ts, {
				type : "date",
				srcFmt : "ts"
			}));
			return moment(date).fromNow();

		},

		formatDate : function(d, fmt, srcFmt, customDateFormat, destTimezone, srcTimezone) {
			//Note: If we date is not in the moment js default format then we need to specify source format. otherwise wrong date conversion will happen.
			var timezoneAvailable = false;
			var momentDateObj;
			// check if moment.tz available or not
			if (moment.tz && destTimezone) {
				timezoneAvailable = true;
			}
			if (d === 0 || d === "0") {
				return cvSearchMessages.NotAvailable;
			}
			if (!fmt) {
				fmt = this.Constants.defaultDateFormat;
			}
			if(srcTimezone && srcTimezone.toUpperCase() === "NONE")
				srcTimezone = undefined;
			if (Array.isArray(d)) {
				d = d[0];
			}
			d = String(d);
			if (fmt === 'AutoDateTime') {
				fmt = "MMM DD, YYYY hh:mm:ss A";
			} else if (fmt === 'Auto') {
				fmt = "MMM DD, YYYY";
			} else if (fmt === 'Custom') {
				fmt = customDateFormat;
			}

			if(!srcFmt) {
				momentDateObj =  !srcTimezone ? moment(d) : moment.tz(d,srcTimezone);
				if (momentDateObj.isValid()) {
					return !timezoneAvailable ? momentDateObj.format(fmt) : momentDateObj.tz(destTimezone).format(fmt);
				}
				srcFmt = "ts";
			}

			// ts = timestamp; tsms = time stamp in milliseconds
			if (srcFmt === "ts" || srcFmt === "tsms") {
				if(typeof d === "string" )
					d = parseInt(d);
				if (srcFmt === "ts")
					d *= 1000; // convert to ms	as we need to pass ts in ms for moment object/function
				momentDateObj = !srcTimezone ? moment(d) : moment.tz(d,srcTimezone);
				if (momentDateObj.isValid()) {
					return !timezoneAvailable ? momentDateObj.format(fmt) : momentDateObj.tz(destTimezone).format(fmt);
				}
			}

			momentDateObj = !srcTimezone ? moment(d,srcFmt) : moment.tz(d,srcFmt,srcTimezone);
			return !timezoneAvailable ? momentDateObj.format(fmt) : momentDateObj.tz(destTimezone).format(fmt);
		},
		formatBoolean : function(flag, type) {

			var original = flag;
			if(!type) {
				type = typeof flag;
				if(type === "string" && isNaN(flag)) {
					type = 'truefalse';
				} else
					type = 'number';
			}
			if (type == 'truefalse' && typeof flag != "boolean") {
				try {
					flag = flag.toLowerCase();
				} catch (e) {
					console.error(e);
					return original;
				}
				if (flag === 'false') {
					flag = false;
				} else if (flag === 'true') {
					flag = true;
				} else {
					flag = null;
				}
			} else if (type == 'number' && typeof flag != "boolean") {

				try {
					flag = parseInt(flag);
				} catch (e) {
					console.error(e);
					return original;
				}

				if (flag === 0) {
					flag = false;
				} else {
					flag = true;
				}
			}
			if (flag === true) {
				return "&#10003;";
			} else if (flag === false) {
				return "&#10007;";
			} else {
				return original;
			}

		},

		formatToggleBoolean : function(flag, type, row, dataModel) {
			var attribute = dataModel.settings.aoColumns[dataModel.col].data;
			if (flag === true) {
				return "<span class='glyphicon glyphicon-ok field-attribute-status true' data-value=" + flag +
						" data-attribute='" + attribute + "'></span>";
			} else if (flag === false) {
				return "<span class='glyphicon glyphicon-ok field-attribute-status false' data-value=" + flag +
						" data-attribute='" + attribute + "'></span>";
			}
		},

		getHyperlink : function(href, text, target) {
			if (typeof (href) !== "undefined") {
				if (typeof (text) === "undefined") {
					text = href;
				}
				if (typeof (target) === "undefined") {
					target = "_blank";
				}

				return $("<a/>").attr("href", href).attr("target", href).text(text)[0].outerHTML;
			}
			return "";
		},
		getButtonLink : function(dispValue, renderer, dataModel) {
			var key = renderer.identifier;
			var btn = $("<button type='button' class='btn btn-link'>" + dispValue + "</button>");
			if (key) {
				btn.addClass(key);
				btn.attr("data-" + key, dataModel[key]);
			}
			return btn[0].outerHTML;
		},
		formatString : function(strMsg, args) {
			if (strMsg) {
				if(Array.isArray(args)) {
					for (var i = 0; i < args.length; i++) {
						var regEx = new RegExp("\\{" + i + "\\}", "gm");
						strMsg = strMsg.replace(regEx, args[i]);
					}
				} else if(typeof args === 'object' && !$.isEmptyObject(args)) {
					var keys = Object.keys(args);
					keys.map(function(key) {
						var regEx = new RegExp("\\{" + key + "\\}", "gm");
						strMsg = strMsg.replace(regEx, args[key]);
					});
				}
			}
			return strMsg;
		},
		getNamedArguments : function(strMsg, keyValuePairs, left, right) {
			if (left === undefined && right === undefined) {
				left = "{";
				right = "}";
			}

			// get everything between 'left' and 'right'
			var selectedPairs = {};
			var regEx = new RegExp("\\" + left + "(\\w)+" + "\\" + right, "gm");
			var matches = strMsg.match(regEx);
			if (matches == null) {
				return selectedPairs;
			}
			var keys = $.map(matches, function(val, i) {
				return val.replace(left, "").replace(right, "");
			});

			for (var i = 0; i < keys.length; i++) {
				var pair = keyValuePairs[keys[i]];

				if (pair) {
					selectedPairs[keys[i]] = pair;
				}
			}

			return selectedPairs;
			// var args = {};
			// for(var key in keyValuePairs){
			// 	if(keyValuePairs.hasOwnProperty(key)){
			// 		var regEx = new RegExp("\\" + left + key + "\\" + right, "gm");
			// 		if(regEx.test(strMsg))
			// 			args[key] = keyValuePairs[key];
			// 	}
			// }
			// return args;
		},
		formatNamedArguments : function(strMsg, keyValuePairs, _index, left, right) {
			if (left === undefined && right === undefined) {
				left = "{";
				right = "}";
			}
			var index = _index;
			if (keyValuePairs && Object.keys(keyValuePairs).length > 0) {
				for ( var key in keyValuePairs) {
					var value = keyValuePairs[key];
					var regEx = new RegExp("\\" + left + key + "\\" + right, "gm");
					if (regEx.test(strMsg)) {
						if (Array.isArray(value) && index != null) {
							strMsg = strMsg.replace(regEx, value[index]);
						} else {
							strMsg = strMsg.replace(regEx, value);
						}
					}
				}
			}
			return strMsg;
		},
		formatDurationToString : function(n) {
			var duration = Number(n);
			var calcTime = 1;
			for (var i = 0; i < this.Constants.durationObject.length; i++) {
				calcTime = calcTime * this.Constants.durationObject[i].factor;
			}
			try {
				for (var i = 0; i < this.Constants.durationObject.length; i++) {
					if (duration >= calcTime) {
						if (i + 1 != this.Constants.durationObject.length) {
							return (duration / calcTime).toFixed(2) + " " + this.Constants.durationObject[i].unit;
						} else {
							return Math.round(duration / calcTime) + " " + this.Constants.durationObject[i].unit;
						}
					} else {
						calcTime = calcTime / this.Constants.durationObject[i].factor;
					}
				}
			} catch (err) {
				// console.error(err);
			}
			return duration;
		},
		ellipsis : function(str, len) {
			var substr = str;
			if (typeof (str) === "string" && str.length > len) {
				substr = str.substring(0, len) + "...";
			}
			return substr;
		},

		prependZeroForSingleDigit : function(val) {
			if (typeof (val) !== "undefined" && val != null && val < 10) {
				val = "0" + val;
			}
			return val;
		},
		formatLink : function(val, renderer, row) {
			if (renderer.linkString) {
				if (!renderer.targetWindow) {
					renderer.targetWindow = "_blank";
				}
				// var renderedLink =
				// this.formatNamedArguments(renderer.linkString, row);
				var renderedLink = "";
				var args = this.getNamedArguments(renderer.linkString, row);
				// if multivalued column
				if (Array.isArray(val)) {
					for (var i = 0; i < val.length; i++) {
						renderedLink += "<a href='" + this.formatNamedArguments(renderer.linkString, args, i) +
								"' target='" + renderer.targetWindow + "'>" + val[i] + " </a>";
					}
					return renderedLink;
				} else {
					renderedLink = this.formatNamedArguments(renderer.linkString, args);
					return "<a href='" + renderedLink + "' target='" + renderer.targetWindow + "'>" + val + "</a>";
				}
			} else {
				return val;
			}
		},

		formatTimeDuration : function(time, renderer) {
			var hours = Math.floor(time / 3600);
			var minutes = Math.floor((time % 3600) / 60);
			var seconds = time % 60;
			if (typeof renderer !== "undefined") {
				if (renderer.source === "milliseconds") {
					hours = Math.floor(time / 3600000);
					minutes = Math.floor((time % 3600000) / 60000);
					seconds = Math.floor((time % 60000) / 1000);
				} else if (renderer.source === "minutes") {
					hours = Math.floor(time / 60);
					minutes = time % 60;
					seconds = 0;
				}
			}
			var timeString = "";
			if (hours > 0) {
				timeString = this.prependZeroForSingleDigit(hours) + "h "
			}
			if (minutes > 0) {
				timeString = timeString + this.prependZeroForSingleDigit(minutes) + "m "
			}
			return timeString + this.prependZeroForSingleDigit(seconds) + "s";
		},

		formatUNC : function(val, renderer, row) {
			if (renderer.uncString) {
				var renderedLink = "";
				var args = this.getNamedArguments(renderer.uncString);
				// if multivalued column
				if (Array.isArray(val)) {
					for (var i = 0; i < val.length; i++) {
						renderedLink += "<a href='" + this.formatNamedArguments(renderer.uncString, args, i) +
								"' target='_blank'>" + val[i] + " </a>";
					}
					return renderedLink;
				} else {
					renderedLink = this.formatNamedArguments(renderer.uncString, args);
					return "<a href='" + renderedLink + "' target='_blank'>" + val + "</a>";
				}
			} else {
				return val;
			}
		},

		createHighlightStyleClass : function(currentRule, type) {

			/*
			 * We replace all special characters in the threshold value. Then we add strings based on whether
			 * its a cell or row highlighter, and based on the operator. eg: 490.23 on a row highlighter when
			 * checking equality would result in the css class highlighter-490_23_eq_row
			 */
			var styleClass = "highlighter-" + currentRule.threshold;
			styleClass = styleClass.replace(/[^\w-]/gi, '_');

			if (currentRule.operator == '>') {
				styleClass += '_gt_';
			} else if (currentRule.operator == '<') {
				styleClass += '_lt_';
			} else if (currentRule.operator == '=') {
				styleClass += '_eq_';
			} else if (currentRule.operator == 'regex') {
				styleClass += '_reg_'
			}

			styleClass += type;

			var style = $("<style />", {
				type : "text/css",
				class : "meta_" + styleClass,
				html : "." + styleClass + "{background-color: " + currentRule.highlightColorBack + "; color: " +
						currentRule.highlightColorFore + ";}"
			});

			$('html > head .meta_' + styleClass).remove();
			$('html > head').append(style);

			return styleClass;
		},

		formatHighlighter : function(val, renderer, row, dataModel) {

			for (var i = 0; i < renderer.highlightRuleList.length; i++) {

				var currentRule = renderer.highlightRuleList[i];
				var styleClass = this.createHighlightStyleClass(currentRule, 'row');

				if (renderer.dataType == 'number') {
					currentRule.threshold = parseFloat(currentRule.threshold);
					val = parseFloat(val);
				} else if (renderer.dataType == "string" && (typeof val == "string")) {
					if (currentRule.operator == 'wildcard') {
						// var re =
						// this.parseRegExpInput(currentRule.threshold);
						var re = this.wildcardToRegex(currentRule.threshold);
						if (val.match(re)) {
							dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
						}
					} else if (currentRule.operator == 'contains') {
						if (val.toLowerCase().indexOf(currentRule.threshold.toLowerCase()) > -1) {
							dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
						}
					} else if (currentRule.operator == 'notcontains') {
						if (val.toLowerCase().indexOf(currentRule.threshold.toLowerCase()) == -1) {
							dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
						}
					} else if (currentRule.operator == 'regex') {
						var re = null;

						// If the user has already encapsulated the regex
						// between '/' and '/', just parse it
						if (currentRule.threshold[0] == "/") {
							re = this.parseRegExpInput(currentRule.threshold);
						} else {// add the '/' and give a case insensitive flag
							re = this.parseRegExpInput("/" + currentRule.threshold + "/i");
						}
						if (val.match(re)) {
							dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
						}
					}
				}

				if (currentRule.operator == '>') {
					if (currentRule.threshold < val) {
						dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
					}
				} else if (currentRule.operator == '<') {
					if (currentRule.threshold > val) {
						dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
					}
				} else if (currentRule.operator == '=') {
					var threshold = currentRule.threshold;
					var compareVal = val;

					if (typeof currentRule.threshold == "string" && typeof val == "string") {
						threshold = currentRule.threshold.toLowerCase();
						compareVal = val.toLowerCase();
					} else {
						threshold = currentRule.threshold;
						compareVal = val;
					}

					if (threshold == compareVal) {
						dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
					}
				} else if (currentRule.operator == '!=') {
					var threshold = currentRule.threshold;
					var compareVal = val;

					if (typeof currentRule.threshold == "string" && typeof val == "string") {
						threshold = threshold.toLowerCase();
						compareVal = compareVal.toLowerCase();
					}
					if (threshold != compareVal) {
						dataModel.settings.aoData[dataModel.row].nTr.classList.add(styleClass);
					}
				}
			}

			return val;
		},

		urlPartsPrefixRemover : function(val, renderer) {
			if (renderer && renderer.prefix) {
				return val.substr(val.indexOf(renderer.prefix) + renderer.prefix.length + 1);//current only one char is the separator so +1
			}
			return val.substr(1);
		},

		formatCellHighlighter : function(val, renderer, row, dataModel) {

			for (var i = 0; i < renderer.highlightRuleList.length; i++) {

				var currentRule = renderer.highlightRuleList[i];
				var styleClass = this.createHighlightStyleClass(currentRule, 'cell');

				if (renderer.dataType == 'number') {
					currentRule.threshold = parseFloat(currentRule.threshold);
					val = parseFloat(val);
				} else if (renderer.dataType == "string") {
					if (currentRule.operator == 'wildcard') {
						var re = this.wildcardToRegex(currentRule.threshold);
						if (val.match(re)) {
							dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
						}
					} else if (currentRule.operator == 'contains') {
						if (val.toLowerCase().indexOf(currentRule.threshold.toLowerCase()) > -1) {
							dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
						}
					} else if (currentRule.operator == 'notcontains') {
						if (val.toLowerCase().indexOf(currentRule.threshold.toLowerCase()) == -1) {
							dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
						}
					} else if (currentRule.operator == 'regex') {
						var re = null;

						if (currentRule.threshold[0] == "/") {
							re = this.parseRegExpInput(currentRule.threshold);
						} else {
							re = this.parseRegExpInput("/" + currentRule.threshold + "/i");
						}

						if (val.match(re)) {
							dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
						}
					}
				}

				if (currentRule.operator == '>') {
					if (currentRule.threshold < val) {
						dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
					}
				} else if (currentRule.operator == '<') {
					if (currentRule.threshold > val) {
						dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
					}
				} else if (currentRule.operator == '=') {
					var threshold = currentRule.threshold;
					var compareVal = val;

					if (typeof currentRule.threshold == "string" && typeof val == "string") {
						threshold = threshold.toLowerCase();
						compareVal = compareVal.toLowerCase();
					}
					if (threshold == compareVal) {
						dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
					}
				} else if (currentRule.operator == '!=') {
					var threshold = currentRule.threshold;
					;
					var compareVal = val;

					if (typeof currentRule.threshold == "string" && typeof val == "string") {
						threshold = threshold.toLowerCase();
						compareVal = compareVal.toLowerCase();
					}
					if (threshold != compareVal) {
						dataModel.settings.aoData[dataModel.row].anCells[dataModel.col].classList.add(styleClass);
					}
				}
			}
			return val;
		},

		parseRegExpInput : function(exp) {
			var match = exp.match(new RegExp('^/(.*?)/([gimy]*)$'));
			return new RegExp(match[1], match[2]);
		},
		wildcardToRegex : function(str) {
			var escapedString = this.preg_quote(str);
			escapedString = '^' + escapedString + '$';
			regexString = escapedString.replace(/\\\*/g, '.*').replace(/\\\?/g, '.');
			return new RegExp(regexString, 'g');
		},
		preg_quote : function preg_quote(str, delimiter) {
			// +   original by: booeyOH
			// * Refer:
			// http://stackoverflow.com/questions/13818186/converting-shell-wildcards-to-regex
			// *     example 1: preg_quote("$40");
			// *     returns 1: '\$40'
			// *     example 2: preg_quote("*RRRING* Hello?");
			// *     returns 2: '\*RRRING\* Hello\?'
			// *     example 3: preg_quote("\\.+*?[^]$(){}=!<>|:");
			// *     returns 3: '\\\.\+\*\?\[\^\]\$\(\)\{\}\=\!\<\>\|\:'
			return (str + '').replace(new RegExp('[.\\\\+*?\\\^\$\(){}=!<>|:' + (delimiter || '') + ']', 'g'), '\\$&');
		}
	};
}());
