/**
 * @author nsicoco
 */

export default class AsyncBatchedPageList {
	/**
	 * Sparse data structure that allows large lists of data (or batches) to be
	 * retrieved as smaller lists (or pages).
	 *
	 * Batches may be stored as Promises to allow asynchronous retrieval of
	 * pages after the batch promises resolve.
	 *
	 * @param {Object} options
	 * @param {number} options.pageSize The size of a single page.
	 * @param {number} [options.pagesPerBatch=4]
	 *        The number of pages contained by a batch. Defaults to four pages
	 *        in a single batch.
	 * @param {number} [options.maxBatchCount]
	 *        If specified, batches will be removed in FIFO order whenever the
	 *        number of stored batches exceeds this value.
	 */
	constructor(options) {
		this._pageSize = options.pageSize;
		this._pagesPerBatch = options.pagesPerBatch || 4;
		this._batchSize = this._pageSize * this._pagesPerBatch;
		this._maxBatchCount = options.maxBatchCount;
		this.removeAll();
	}

	/**
	 * @param {number} index
	 * @param {Promise.<Object[]>|Object[]} batch
	 *        The list of data or a Promise that returns the list of data contained
	 *        by the batch
	 * @returns {Promise.<Object[]>|Object[]} The previously stored batch at the index
	 */
	setBatch(index, batch) {
		const previousBatch = this._batchList[index];
		this._batchList[index] = batch;
		if (this._maxBatchCount) {
			this._batchHistory.push(index);
		}
		this._removeOldBatches();
		return previousBatch;
	}

	/**
	 * @param {number} index
	 * @returns {Promise.<Object[]>|Object[]}
	 *          The stored batch data or batch Promise, or null if the batch
	 *          does not exist
	 */
	getBatch(index) {
		return this._batchList[index];
	}

	/**
	 * @param {number} pageIndex The index of the page
	 * @returns {number} The index of the batch that contains the specified page
	 */
	getBatchIndexFromPage(pageIndex) {
		return Math.floor(pageIndex / this._pagesPerBatch);
	}

	/**
	 * @param {number} batchIndex The index of the batch
	 * @returns {number} The page index of the first page of the specified batch
	 */
	getPageIndexFromBatch(batchIndex) {
		return batchIndex * this._pagesPerBatch;
	}

	/**
	 * @returns {number} The number of pages contained in a batch
	 */
	getPagesPerBatch() {
		return this._pagesPerBatch;
	}

	/**
	 * @returns {number} The number of data elements stored in a single batch
	 */
	getBatchSize() {
		return this._batchSize;
	}

	/**
	 * Returns the page of data after waiting for the corresponding batch to be
	 * resolved.
	 * @param {number} page Zero-based page index
	 * @returns {Promise.<Object[]>}
	 *          A promise that resolves with the page of data or null if the
	 *          corresponding batch does not exist.
	 */
	getPage(page) {
		const batchIndex = Math.floor(page / this._pagesPerBatch);
		const batch = this._batchList[batchIndex];
		if (batch) {
			return Promise.resolve(batch).then(batchData => {
				if (batchData) {
					let batchPage = page % this._pagesPerBatch;
					let start = batchPage * this._pageSize;
					const pageData = batchData.slice(start, start + this._pageSize);
					pageData.total = batchData.total;
					return pageData;
				}
				return null;
			});
		}

		return null;
	}

	/**
	 * @param {*} index The index of the batch to remove
	 * @returns {Promise.<Object[]>|Object[]} The previously stored batch at the index
	 */
	removeBatch(index) {
		const previousBatch = this._batchList[index];
		delete this._batchList[index];
		if (this._maxBatchCount) {
			this._batchHistory = this._batchHistory.filter(batch => batch !== index);
		}
		return previousBatch;
	}

	removeAll() {
		this._batchList = [];
		this._batchHistory = [];
	}

	/**
	 * Removes stored batches from the earliest stored batch until the number of
	 * stored batches is less than or equal to the max batch count.
	 */
	_removeOldBatches() {
		if (this._maxBatchCount) {
			while (this._batchHistory.length > this._maxBatchCount) {
				let earliestBatch = this._batchHistory.shift();
				delete this._batchList[earliestBatch];
			}
		}
	}
}
