import XmlToJson from "./xmlToJson";

const COLUMN_TYPES = {
	BOOLEAN: "boolean",
	NUMBER: "number",
	STRING: "string",
	OBJECT: "object",
};

const RESPONSE_TYPES = {
    STRING: "String",
    INTEGER: "Integer",
    LONG: "Long"
};


function isEmpty(input) {
    return _.isEmpty(input);
}

function validateSchema(type, value) {
    switch (type) {
        case RESPONSE_TYPES.STRING: return _.isString(value) || isEmpty(value);
        case RESPONSE_TYPES.INTEGER: return _.isNumber(value) || isEmpty(value);
        case RESPONSE_TYPES.LONG: return _.isNumber(value) || isEmpty(value);
    }
}

/**
 * 
 * @param {*}
 *            input.records
 * @param {*}
 *            input.totalRecordCount
 * @param {*}
 *            config.decodeHtmlEntities
 * @param {*}
 *            config.columns[columnName].decodeHtmlEntities
 * @param {*}
 *            config.columns[columnName].type
 */
function CreConverter(input, config={}) {
	const result = [];
	let response = {
		totalRecordCount: input.totalRecordCount,
		data: result,
	};
	if (isEmpty(input)) {
		return response;
	}
	if (!("decodeHtmlEntities" in config)) {
		config.decodeHtmlEntities = true;
	}
	config.columns = config.columns || {};
	const domParser = new DOMParser();
	input.records.forEach(function(record) {
		const item = {};
		record.forEach(function(cell, index) {
            if (input.columns.length != record.length) {
                throw new Error('Record schema mismatch: Number of columns is not equal to number of cells in record')
            }
            if (!validateSchema(input.columns[index].type, cell)) {
                throw new Error(`Schema datatype mismatch 
                    DataType: ${input.columns[index].type}
                    cell value: ${cell}
                    index: ${index}
                    name: ${input.columns[index].name}
                `);
            }
			const key = input.columns[index].name;
			if (input.columns[index].type === RESPONSE_TYPES.STRING && _.isString(cell)) {
				// Decode HTML entities (whole string is assumed to be encoded):
				let decodeHtmlEntities = config.decodeHtmlEntities;
				if (config.columns[key] && "decodeHtmlEntities" in config.columns[key]) {
					decodeHtmlEntities = config.columns[key].decodeHtmlEntities;
				}
				if (decodeHtmlEntities) {
					const parser = domParser.parseFromString(cell, "text/html");
					cell = _.get(parser, 'body.textContent', "")
				}
				const columnType = (config.columns[key] && config.columns[key].type);
				// Convert string to object:
				if (columnType === COLUMN_TYPES.OBJECT) {
					try {
						let value = XmlToJson(cell, domParser);
						if (isEmpty(value)) {
							value = undefined;
						}
						cell = value;
					} catch (e) {
						cell = e;
					}
				}
			}
			item[key] = cell;
		});
		result.push(item);
	});
	return response;
}
CreConverter.COLUMN_TYPES = COLUMN_TYPES;

export default CreConverter;