import { appDesignerModule } from 'common/js/modules';
import 'lodash/zip';
import capitalize from 'lodash/capitalize';

import { objToString } from './../util';

import { APPS_API_ENDPOINT, WORKFLOW_API_ENDPOINT, DEFAULT_BUSINESS_RULE, DEFAULT_HEADERS } from '../constants';

const baseApi = 'proxy/cr/apps/';
const getRuleApi = ruleId => `${baseApi}business_rules/${ruleId}`;
const getTableFormBusinesRule = (tableId, ruleId) => `/tables/${tableId}/business_rules/${ruleId}/form`;
const getTableForm = tableId => `/tables/${tableId}/form`;
const getCreateRuleApi = () => `${baseApi}/tables/${tableId}/business_rules`;

function BusinessRulesService($http, $log, $q) {
	let headers = DEFAULT_HEADERS;
	let transformRequest = objToString;

	/**
	 * BusinessRules class which is returned as the base of the BusinessRulesService service
	 */
	class BusinessRules {
		/**
		 * creates a instance of BusinessRules
		 * @param {string} [baseUrl] - optional base url to use for REST calls
		 */
		constructor(baseUrl) {
			this._baseUrl = baseUrl !== undefined ? baseUrl : `${APPS_API_ENDPOINT}`;
		}

		/**
		 * @type {string}
		 */
		get baseUrl() {
			return this._baseUrl;
		}

		/**
		 * method to get all available tables
		 * @returns {Promise} $http promise
		 */
		getBusinessRules(tableId) {
			let endpoint = this.baseUrl + `/tables/${tableId}/business_rules`;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
					// return instances of BusinessRule
					.then(rules => rules.map(rule => new BusinessRule(rule.sys_id, tableId, { businessRule: rule })))
			);
		}

		/**
		 * get business rule by ID
		 * @param {string} - id of business rule
		 * @returns {Promise} $http promise
		 */
		getBusinessRuleById(id) {
			if (id === undefined) {
				throw new Error('id required');
			}
			return new BusinessRule(id).promise;
		}

		getWorkflows() {
			const endpoint = WORKFLOW_API_ENDPOINT;

			return $http
				.get(endpoint, { headers: Object.assign(headers, { Accept: 'application/json' }) })
				.then(response => response.data)
				.then(data => {
					return data.container.reduce((acc, workflow, i) => {
						const guid = workflow.uniqueGuid;
						const name = workflow.entity.workflowName;
						const value = workflow;
						const key = guid || workflow.entity.workflowId || name;
						acc[key] = value;
						return acc;
					}, {});
				});
		}

		getBusinessRuleFromData(ruleId, tableId, rule) {
			return new BusinessRule(ruleId, tableId, { businessRule: rule });
		}

		getInputForm(ruleId, tableId) {
			const formEntriesUrl = ruleId ? getTableFormBusinesRule(tableId, ruleId) : getTableForm(tableId);

			let endpoint = this.baseUrl + formEntriesUrl;
			const getForm = () => $http.get(endpoint, { headers }).then(response => response.data);
			const getBusinessRule = () => new BusinessRule(ruleId, tableId);

			return Promise.all([getForm()]).then(
				resp => {
					const formEntries = resp[0];
					//					const rule = resp[1];
					//					const rules = formEntries.properties && formEntries.properties.rules;
					// const ruleClass = new BusinessRule(ruleId, tableId, {businessRule: rules});
					//					formEntries.properties.rules = rule.businessRule.rules;
					return formEntries;
				},
				err => {
					return err;
				}
			);
		}
	}

	// The classes below are not directly exposed by the service but are support of the BusinessRules class

	/**
	 * BusinessRule class used for table instances within the BusinessRules service
	 */
	class BusinessRule {
		/**
		 * creates a instance of BusinessRule
		 * @param {string} [id] - optional id for the BusinessRule instance
		 * @param {object} [config] - configuration object containing baseUrl and table properties
		 */
		constructor(id, tableId, config = {}) {
			// destructuring of config objectº
			let { baseUrl, businessRule = DEFAULT_BUSINESS_RULE } = config;

			// promise which can be used
			this._deferred = $q.defer();

			// REST endpoint for businessRule
			this._baseUrl = baseUrl;

			this.id = id;
			this.promise = this._deferred.promise;
			this.resolved = false;
			this.tableId = tableId;

			this.businessRule = businessRule;

			if (this.id && Object.is(businessRule, DEFAULT_BUSINESS_RULE)) {
				this._getBusinessRule();
			} else {
				this._setProperties(businessRule);
				this.resolved = true;
			}
		}

		get baseUrl() {
			let baseUrl =
				this._baseUrl !== undefined ? this._baseUrl + '/business_rules' : `${APPS_API_ENDPOINT}/business_rules`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * table data model for creation and updates
		 * @private
		 */
		get _businessRule() {
			return {
				rules: [
					{
						title: this.title,
						actions: this.actions || [],
						conditionGroups: this.conditionGroups || [],
						allOrAny: this.allOrAny,
						ruleTriggers: this.ruleTriggers,
						enabled: this.enabled,
						executionMode: this.executionMode || 'AFTER',
						applyRuleToAllForms: this.applyRuleToAllForms
					}
				]
			};
		}

		/**
		 * method to setup view model properties
		 * @private
		 */
		_setProperties(businessRule) {
			const { sys_id, rules } = businessRule;

			this.id = sys_id || this.id;
			const rule = rules && rules['rules'] ? rules['rules'][0] : rules[0];

			if (rule) {
				const {
					title,
					conditionGroups,
					actions,
					enabled,
					allOrAny,
					ruleTriggers,
					executionMode,
					applyRuleToAllForms
				} = rule;
				Object.assign(this, {
					title,
					conditionGroups,
					actions,
					enabled,
					allOrAny,
					ruleTriggers,
					executionMode,
					applyRuleToAllForms
				});
				const ruleTriggerMap = {
					onRecordAdded: 'Insert',
					onRecordUpdated: 'Update',
					onRecordDelete: 'Delete'
				};
				this.ruleTriggersFormatted = `${capitalize(executionMode)} ${Object.keys(ruleTriggers)
					.filter(trigger => ruleTriggers[trigger])
					.map(trigger => ruleTriggerMap[trigger])
					.join(', ')}`;
			}

			this.created = { at: businessRule.sys_created_at };
			this.modified = Object.assign({ at: businessRule.sys_created_at }, businessRule.sys_modified);
			this.version = businessRule.sys_version;
			this.domain = businessRule.sys_domainId;

			this.businessRule = businessRule;

			return this;
		}

		/**
		 * method used by constructor to fetch app instance
		 * @private
		 */
		_getBusinessRule() {
			let endpoint = `${APPS_API_ENDPOINT}/tables/${this.tableId}/business_rules/${this.id}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					rule => {
						this._setProperties(rule);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let promise;
			if (!this.id) {
				// create new
				let postEndpoint = `${APPS_API_ENDPOINT}/tables/${this.tableId}/business_rules`;
				promise = $http.post(postEndpoint, this._businessRule, { headers, transformRequest });
			} else {
				// update existing
				let putEndpoint = `${APPS_API_ENDPOINT}/tables/${this.tableId}/business_rules/${this.id}`;
				promise = $http.put(putEndpoint, this._businessRule, { headers, transformRequest });
			}

			return promise.then(onRequestSuccess, onRequestFailure).then(table => this._setProperties(table));
		}

		remove() {
			let endpoint = `${APPS_API_ENDPOINT}/tables/${this.tableId}/business_rules/${this.id}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}
	}

	function onRequestSuccess(res) {
		return res.data;
	}

	function onRequestFailure(err) {
		$log.error(err);
		return $q.reject(err);
	}

	return new BusinessRules();
}

BusinessRulesService.$inject = ['$http', '$log', '$q'];

appDesignerModule.factory('BusinessRulesService', BusinessRulesService);

export default appDesignerModule;
