import { appDesignerModule } from 'common/js/modules';

// App Designer Utility
import {
	cellNameTemplate,
	cellActionTemplate,
	formatSecurityAssociations,
	cellBusinessRuleNameTemplate
} from './../util';

// temporary localization App Designer strings
import './../temp.localization';

// dependencies injected by Angular
import '../services/tables.svc';
import '../services/tableEditModal.svc';
import '../services/columnEditModal.svc';
import '../services/showSecurityModal.svc';
import '../services/business.svc';
import '../services/appForms.svc';
import '../services/views.svc';
import '../services/cloneHelperService.svc';
import '../services/actions.svc';

import 'adminConsole/js/services/security.svc.js';
import 'adminConsole/js/controllers/security.ctrl.js';

import 'adminConsole/js/directives/acDirectives.js';
import { cellBusinessRuleTriggerTemplate, getErrMessage } from '../util';

function TableConfigCtrl(
	$stateParams,
	$log,
	cvLoc,
	$dialogs,
	cvTableOptions,
	TablesService,
	TableEditModalService,
	ColumnEditModalService,
	ShowSecurityModalService,
	BusinessRulesService,
	ViewsService,
	AppFormsService,
	cvToaster,
	$state,
	securityService,
	securityFactory,
	CloneHelperService,
	ActionsService
) {
	let vm = this;
	const { tableId, tab } = $stateParams;

	// security variables
	vm.showEdit = true;
	vm.formattedSecurityAssociations;
	vm.associationsLoaded = false;
	vm.entityType = 179;
	vm.genericEntity = 131;
	vm.hideInheritedAssociations = true;

	const tabs = {
		field: 0,
		businessrule: 1,
		form: 2,
		views: 3,
		actions: 4
	};

	vm.tabActive = tabs[tab] || 0;
	vm.tableId = tableId;
	vm.loading = true;

	// ideally the initial value should be in the route resolver
	vm.table = {};
	vm.columns = null;

	fetchTable();

	vm.editSecurity = function editSecurity() {
		let entity = {
			entityType: 179,
			entityId: vm.table.rowId,
			entityName: vm.table.name,
			enityType: 'Table'
		};
		ShowSecurityModalService.openModal(entity).then(() => fetchSecurityAssociations(vm.table.rowId));
	};

	vm.selectTab = function(tab) {
		const tabName =
			tab === 1 ? 'businessrule' : tab === 2 ? 'form' : tab === 3 ? 'views' : tab === 4 ? 'actions' : 'field';
		vm.tabActive = tab;

		$state.go('tableConfigTab', { tableId, tab: tabName }, { notify: false });
		fetchByTabActive(tab);
	};

	vm.editColumn = function editColumn(column = vm.table.createColumn()) {
		return ColumnEditModalService.openModal(column).then(fetchColumns);
	};

	vm.createColumn = function createColumn() {
		return ColumnEditModalService.openModal(column).then(fetchColumns);
	};

	vm.deleteColumn = function deleteColumn(column) {
		deleteItem(column, column.name, fetchColumns);
	};

	vm.deleteBusinessRule = function deleteBusinessRule(rule) {
		deleteItem(rule, rule.title, fetchBusinessRules);
	};

	vm.deleteView = function deleteView(view) {
		deleteItem(view, view.name, fetchViews);
	};

	vm.deleteForm = function deleteForm(form) {
		$dialogs.confirm('Confirm delete', 'Are you sure you want to delete <b>' + form.name + '</b>', {
			noFunction: function() {},
			yesFunction: function() {
				AppFormsService.delete(tableId, form.sys_rowId).then(
					() => onDeleteSuccess(form.name, fetchTableForms),
					() => onDeleteFailure(form.name)
				);
			}
		});
	};

	vm.deleteAction = function deleteAction(action) {
		deleteItem(action, action.name, fetchActions);
	};

	vm.navigateToColumnConfig = function(columnId) {
		$state.go('tableConfig', { tableId: appId, columnId });
	};

	vm.navigateToBusinessConfig = function(columnId) {
		$state.go('businessRuleConfig', { tableId: tableId, columnId });
	};

	vm.navigateToViewsConfig = function(viewId) {
		$state.go('viewsConfig', { tableId: tableId, viewId: viewId });
	};

	vm.navigateToFormConfig = function(formId) {
		$state.go('appStudioFormCustomize', { tableId: tableId, formId });
	};

	vm.navigateToActionConfig = function(buttonId) {
		$state.go('actionsConfig', { tableId: tableId, buttonId });
	};

	vm.handleTableEdit = function handleTableEdit() {
		TableEditModalService.openModal(vm.table).then(fetchTable);
	};

	vm.openTableImport = function openTableImport() {
		openDataImportPage({ tableId: vm.table.id, table: vm.table.rowId });
	};

	vm.gridOptionsFields = {
		cvGridTitle: cvLoc('appdesigner.label.columns'),
		cvTableName: 'tablesListTable',
		cvSearchFields: ['name', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noColumns'),
			links: [
				{
					label: cvLoc('appdesigner.label.addColumn'),
					onclick: vm.editColumn
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addColumn'),
				onclick: vm.editColumn,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'columns',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						sort: { direction: 'asc', priority: 0 },
						displayName: cvLoc('appdesigner.table.col.name'),
						enableHiding: false,
						width: '20%',
						cellTemplate: cellNameTemplate(
							'row.entity.name',
							'columnConfig({ tableId: row.entity.tableId, columnId: row.entity.id })'
						)
					},
					{
						field: 'displayName',
						displayName: cvLoc('appdesigner.table.col.display_name'),
						enableHiding: false,
						width: '20%'
					},
					{
						field: 'description',
						displayName: cvLoc('appdesigner.table.col.desc'),
						enableHiding: false,
						width: '20%'
					},
					{
						field: 'fieldOptions.defaultValue',
						displayName: cvLoc('appdesigner.table.col.default_value')
					},
					{
						field: 'type',
						displayName: cvLoc('appdesigner.table.col.type'),
						width: '7%'
					},
					{
						field: 'unique',
						displayName: cvLoc('appdesigner.table.col.unique'),
						width: '6%'
					},
					{
						field: 'required',
						displayName: cvLoc('appdesigner.table.col.required'),
						width: '6%'
					},
					{
						name: 'action',
						width: '3%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.edit'),
								sref: 'columnConfig({ tableId: row.entity.tableId, columnId: row.entity.id})'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteColumn(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	vm.gridOptionsBusinessRules = {
		cvGridTitle: cvLoc('appdesigner.label.businessRules'),
		cvTableName: 'businessRulesList',
		cvSearchFields: ['title', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noBusinessRules'),
			links: [
				{
					label: cvLoc('appdesigner.label.addBusinessRules'),
					onclick: vm.navigateToBusinessConfig
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addBusinessRules'),
				onclick: vm.navigateToBusinessConfig,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'businessRules',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'title',
						sort: { direction: 'asc', priority: 0 },
						displayName: cvLoc('appdesigner.rules.col.title'),
						enableHiding: false,
						width: '40%',
						cellTemplate: cellBusinessRuleNameTemplate(
							'row.entity.title',
							'#/tables/{{row.entity.tableId}}/business-rule/{{row.entity.id}}'
						)
					},
					{
						field: 'ruleTriggersFormatted',
						displayName: cvLoc('appdesigner.rules.col.on'),
						enableHiding: false,
						width: '20%'
					},
					{
						field: 'enabled',
						displayName: cvLoc('appdesigner.rules.col.enabled'),
						enableHiding: false,
						width: '10%'
					},
					{
						name: 'action',
						width: '3%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.clone'),
								action: 'grid.appScope.cloneItem(row.entity.id, grid.appScope.navigateToBusinessConfig)'
							},
							{
								label: cvLoc('appdesigner.label.edit'),
								sref: 'businessRuleConfig({ tableId: row.entity.tableId, businessRuleId: row.entity.id })'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteBusinessRule(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	vm.gridOptionsTableForms = {
		cvGridTitle: cvLoc('appdesigner.label.tableForms'),
		cvTableName: 'tableFormsList',
		cvSearchFields: ['name', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noFormRules'),
			links: [
				{
					label: cvLoc('appdesigner.label.addForm'),
					onclick: vm.navigateToFormConfig
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addForm'),
				onclick: vm.navigateToFormConfig,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'tableForms',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						width: '90%',
						displayName: cvLoc('appdesigner.rules.col.title'),
						enableHiding: false,
						cellTemplate: cellNameTemplate(
							'row.entity.name',
							'appStudioFormCustomize({ tableId: row.entity.tableId,formId: row.entity.sys_rowId})'
						)
					},
					{
						name: 'action',
						width: '3%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.clone'),
								action: 'grid.appScope.cloneItem(row.entity.sys_rowId, grid.appScope.navigateToFormConfig)'
							},
							{
								label: cvLoc('appdesigner.label.edit'),
								sref: 'appStudioFormCustomize({ tableId: row.entity.tableId,formId: row.entity.sys_rowId})'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteForm(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	vm.gridOptionsTableViews = {
		cvGridTitle: cvLoc('appdesigner.label.tableViews'),
		cvTableName: 'viewsList',
		cvSearchFields: ['name', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noViews'),
			links: [
				{
					label: cvLoc('appdesigner.label.addView'),
					onclick: vm.navigateToViewsConfig
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addView'),
				onclick: vm.navigateToViewsConfig,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'tableViews',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						width: '90%',
						displayName: cvLoc('appdesigner.rules.col.title'),
						enableHiding: false,
						cellTemplate: cellNameTemplate(
							'row.entity.name',
							'viewsConfig({ tableId: row.entity.tableId, viewId: row.entity.id})'
						)
					},
					{
						name: 'action',
						width: '3%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.clone'),
								action: 'grid.appScope.cloneItem(row.entity.id, grid.appScope.navigateToViewsConfig)'
							},
							{
								label: cvLoc('appdesigner.label.edit'),
								sref: 'viewsConfig({ tableId: row.entity.tableId, viewId: row.entity.id})'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteView(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	vm.buttonTypes = {
		table: 'Table Level',
		row: 'Row Level'
	};

	vm.gridOptionsTableActions = {
		cvGridTitle: cvLoc('appdesigner.label.tableActions'),
		cvTableName: 'tableActionsList',
		cvSearchFields: ['name'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noUIActions'),
			links: [
				{
					label: cvLoc('appdesigner.label.addAction'),
					onclick: vm.navigateToActionConfig
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addAction'),
				onclick: vm.navigateToActionConfig,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'tableActions',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						width: '40%',
						displayName: cvLoc('appdesigner.actions.col.title'),
						enableHiding: false,
						cellTemplate: cellNameTemplate(
							'row.entity.name',
							'actionsConfig({ tableId: row.entity.tableId,buttonId: row.entity.id})'
						)
					},
					{
						field: 'properties.buttonType',
						width: '20%',
						displayName: cvLoc('appdesigner.actions.col.type'),
						enableHiding: false,
						cellTemplate: '<span>{{grid.appScope.buttonTypes[row.entity.properties.buttonType]}}<span>'
					},
					{
						field: 'properties.clickExpression.action',
						width: '20%',
						displayName: cvLoc('appdesigner.actions.col.onClick'),
						enableHiding: false
					},
					{
						name: 'action',
						width: '3%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.edit'),
								sref: 'actionsConfig({ tableId: row.entity.tableId,buttonId: row.entity.id})'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteAction(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	function fetchByTabActive(tab, init) {
		switch (tab) {
			case 0:
				if (!vm.columns || init) fetchColumns();
				break;
			case 1:
				if (!vm.businessRules || init) fetchBusinessRules();
				break;
			case 2:
				if (!vm.tableForms || init) fetchTableForms();
				break;
			case 3:
				if (!vm.tableViews || init) fetchViews();
				break;
			case 4:
				if (!vm.tableActions || init) {
					fetchActions();
				}
				break;
		}
	}

	function fetchSecurityAssociations(entityId) {
		securityFactory.getSecurityAssociation('APPSTUDIO_TABLE', entityId).success(function(data) {
			vm.formattedSecurityAssociations = formatSecurityAssociations(data.securityAssociations);
			vm.associationsLoaded = true;
		});
	}

	function fetchTable() {
		let req = TablesService.getTableById(tableId);
		req.then(onSuccess, onRequestFailure);

		function onSuccess(res) {
			vm.loading = false;
			vm.table = res;
			fetchSecurityAssociations(res.rowId);
			fetchByTabActive(vm.tabActive, true);
		}

		return req;
	}

	function fetchBusinessRules() {
		BusinessRulesService.getBusinessRules(tableId).then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(rules) {
			vm.businessRules = rules;
		}
	}

	function fetchViews() {
		ViewsService.getViews(tableId).then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(views) {
			vm.tableViews = views.sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase());
		}
	}

	function fetchActions() {
		ActionsService.getActions(tableId).then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(actions) {
			vm.tableActions = actions.sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase());
		}
	}

	vm.cloneItem = function(itemId, callback) {
		CloneHelperService.setItemClone(itemId);
		callback();
	};

	function deleteItem(item, title, callback) {
		$dialogs.confirm('Confirm delete', 'Are you sure you want to delete <b>' + title + '</b>', {
			noFunction: function() {},
			yesFunction: function() {
				item.remove().then(() => onDeleteSuccess(title, callback), onDeleteFailure);
			}
		});
	}

	function onDeleteSuccess(title, callback) {
		callback();
		cvToaster.showSuccessMessage({
			message: cvLoc('appdesigner.msg.deleteSuccessful', '<b>' + title + '</b>')
		});
	}

	function onDeleteFailure(err) {
		if (err) {
			const message = getErrMessage(err);
			cvToaster.showErrorMessage({
				message
			});
		}
	}

	function fetchTableForms() {
		AppFormsService.getAppForms(tableId).then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(forms) {
			vm.tableForms = forms.sort((a, b) => a.name > b.name);
		}
	}

	function fetchColumns() {
		let req = vm.table && vm.table.getColumns && vm.table.getColumns();
		req && req.then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(columns) {
			vm.columns = columns;
		}

		return req;
	}

	function onRequestFailure(err) {
		if (err) {
			const message = getErrMessage(err);
			cvToaster.showErrorMessage({
				message
			});
		}
		$log.error(err);
	}
}

TableConfigCtrl.$inject = [
	'$stateParams',
	'$log',
	'cvLoc',
	'$dialogs',
	'cvTableOptions',
	'TablesService',
	'TableEditModalService',
	'ColumnEditModalService',
	'ShowSecurityModalService',
	'BusinessRulesService',
	'ViewsService',
	'AppFormsService',
	'cvToaster',
	'$state',
	'securityService',
	'securityFactory',
	'CloneHelperService',
	'ActionsService'
];

appDesignerModule.controller('TableConfigCtrl', TableConfigCtrl);

export default appDesignerModule;
