import { appDesignerModule } from 'common/js/modules';
import get from 'lodash/get';

// temporary localization App Designer strings
import './../temp.localization';

// dependencies injected/need by Angular
import '../services/apps.svc';
import '../services/reports.svc';
import '../services/pageType.svc';
import { getErrMessage } from '../util';

function PageConfigCtrl(
	$stateParams,
	$log,
	cvLoc,
	$dialogs,
	cvToaster,
	$state,
	AppsService,
	ReportsService,
	PageTypeService
) {
	let vm = this;
	let appId = $stateParams.appId;
	let pageId = $stateParams.pageId;

	vm.pageTypes = AppsService.getPageTypes();
	vm.loading = true;
	vm.page = {};
	vm.selectedType = [];
	vm.viewsList = null;
	vm.list;
	vm.selectedItem;

	vm.pageTypes.forEach(function(pageType) {
		if (vm.page.type === pageType.id) {
			pageType.selected = true;
			vm.selectedType = [pageType];
		}
		vm.appPageType = vm.selectedType[0];
	});

	init();

	vm.handleEditPage = function handlePageEdit() {
		if (vm.selectedView && vm.selectedView.length) {
			vm.page.itemId = `${vm.selectedView[0].id}`;
			vm.page.properties = vm.selectedItem && {
				assocId: vm.selectedItem[0].id,
				subtype: `view`
			};
		}

		vm.page.properties.itemGUID = vm.page.itemId;

		if (vm.page.type == 'table') {
			if (vm.selectedView && vm.selectedView.length > 0) vm.page.itemId = vm.selectedView[0].id;
			else {
				vm.page.itemId = vm.selectedItem[0].rowId;
				delete vm.page.properties.subtype;
			}
		} else if (vm.page.type == 'report') {
			vm.page.itemId = vm.selectedItem[0].id;
		}

		vm.page.save().then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(res) {
			$state.go('appConfig', { appId: appId });
			cvToaster.showSuccessMessage({
				message: cvLoc('appdesigner.msg.saveSuccessful', '<b>' + vm.page.name + '</b>')
			});
		}
	};

	vm.onAssocItemChange = function() {
		fetchItemList();
	};

	vm.onItemChange = function() {
		const item = vm.selectedItem[0];
		if (!item) return;
		vm.page.itemId = getItemId(vm.page.type, item);
		if (vm.appPageType.id === 'table') fetchViews(item.id);
	};

	vm.onPageTypeChange = function onPageTypeChange() {
		if (vm.appPageType && vm.selectedType[0].id === vm.appPageType.id) return;
		vm.page.type = vm.selectedType[0].id;
		vm.appPageType = vm.selectedType[0];
		vm.assocType = vm.appPageType.association ? AppsService.getPageTypeNameById(vm.appPageType.association) : null;
		fetchLists();
	};

	vm.handleDeletePage = function handleDeletePage() {
		let pageName = vm.page.name;

		$dialogs.confirm(
			cvLoc('appdesigner.label.confirmDelete'),
			cvLoc('appdesigner.msg.confirmDelete', `<b>${pageName}</b>`),
			{
				noFunction: function() {},
				yesFunction: function() {
					vm.page.remove().then(onDeleteSuccess, onDeleteFailure);
				}
			}
		);

		function onDeleteSuccess() {
			$state.go('appConfig', { appId: appId });
		}

		function onDeleteFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}
	};

	function fetchItemList() {
		if (!vm.appPageType) return;
		const pageType = vm.page.type;

		vm.appPageType.getList().then(data => {
			vm.list = data;
			data.forEach(function(item) {
				if (pageType === 'report' && vm.page.itemId === item.id) {
					item.selected = true;
					vm.selectedItem = [item];
				}
				if (pageType === 'table') {
					if (vm.page.properties.subtype === 'view') {
						if (vm.page.properties.assocId === item.id) {
							vm.selectedItem = [Object.assign(item, { selected: true })];
						}
					} else {
						if (vm.page.itemId === `${item.rowId}`) {
							vm.selectedItem = [Object.assign(item, { selected: true })];
						}
					}
				}
			});
			if (pageType === 'table' && vm.selectedItem) fetchViews(vm.selectedItem[0].id);
		});
	}

	function fetchViews(tableId) {
		if (!vm.appPageType.views) return;

		vm.appPageType.views.getList(tableId).then(data => {
			vm.viewList = data;
			if (vm.page.properties.subtype !== 'view') return;
			data.forEach(function(item) {
				if (vm.page.itemId === `${item.id}`) {
					vm.selectedView = [Object.assign(item, { selected: true })];
				}
			});
		});
	}

	function fetchLists() {
		vm.list = [];
		fetchItemList();
	}

	function getItemId(type, item) {
		if (vm.page.type === 'report') return item.guid;
		return `${item.id}`;
	}

	function fetchPage() {
		return AppsService.getPageById(appId, pageId).then(onRequestSuccess, onRequestFailure);
		function onRequestSuccess(page) {
			vm.loading = false;
			vm.page = page;
			vm.pageType = PageTypeService.getPageType(page.type);
			vm.pageTypes.forEach(function(pageType) {
				if (vm.page.type === pageType.id) {
					pageType.selected = true;
					vm.selectedType = [pageType];
					vm.appPageType = vm.selectedType[0];
					if (vm.appPageType.association) vm.assocType = AppsService.getPageTypeNameById(vm.appPageType.association);
				}
			});
			fetchLists();
		}

		function onRequestFailure(err) {
			$log.error(err);
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}
	}

	function mockConfigData(page) {
		page.type = 'table';
		page.properties = {};
		return page;
	}

	function init() {
		fetchPage();
	}

	function onRequestFailure(err) {
		if (err) {
			const message = getErrMessage(err);
			cvToaster.showErrorMessage({
				message
			});
		}
	}
}

PageConfigCtrl.$inject = [
	'$stateParams',
	'$log',
	'cvLoc',
	'$dialogs',
	'cvToaster',
	'$state',
	'AppsService',
	'ReportsService',
	'PageTypeService'
];

appDesignerModule.controller('PageConfigCtrl', PageConfigCtrl);

export default appDesignerModule;
