import { appDesignerModule } from 'common/js/modules';

// temporary localization App Designer strings
import './../temp.localization';

// dependencies injected/need by Angular
import '../services/reports.svc';
import '../services/tables.svc';

import '../table-fields/tableField.directive';
import { getErrMessage, pathAccess } from '../util';

function ColumnConfigCtrl($stateParams, $log, cvLoc, $dialogs, cvToaster, $state, ReportsService, TablesService) {
	let vm = this;
	let tableId = $stateParams.tableId;
	let columnId = $stateParams.columnId;

	vm.classes = {
		labelClass: 'col-xs-12 col-lg-3',
		formGroupClass: 'form-group',
		inputContainerClass: 'col-xs-12 col-lg-6'
	};

	vm.loading = true;
	vm.column = null;
	vm.fieldTypes = TablesService.getFieldTypes();

	init();

	vm.isNewColumn = function isNewPage() {
		return columnId === undefined;
	};

	vm.handleMultiselectClose = function handleMultiselectClose() {
		if (vm.selectedColumnType.length > 0) {
			vm.column.type = vm.selectedColumnType[0].id;
		}
	};

	vm.handleEditColumn = function handleColumnEdit() {
		if (vm.column.type === 'Number') {
			if (vm.column.fieldOptions) {
				if (vm.column.fieldOptions.minValue && vm.column.fieldOptions.minValue > vm.column.fieldOptions.maxValue) {
					cvToaster.showErrorMessage({
						message: cvLoc('appDesigner.msg.invalidMinValue')
					});
					return;
				}

				if (vm.column.fieldOptions.maxValue && vm.column.fieldOptions.maxValue < vm.column.fieldOptions.minValue) {
					cvToaster.showErrorMessage({
						message: cvLoc('appDesigner.msg.invalidMaxValue')
					});
					return;
				}

				if (
					(vm.column.fieldOptions.defaultValue &&
						vm.column.fieldOptions.defaultValue < vm.column.fieldOptions.minValue) ||
					vm.column.fieldOptions.defaultValue > vm.column.fieldOptions.maxValue
				) {
					cvToaster.showErrorMessage({
						message: cvLoc('appDesigner.msg.invalidDefaultValue')
					});
					return;
				}
			}
		}

		vm.column.save().then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(res) {
			$state.go('tableConfig', { tableId });
			cvToaster.showSuccessMessage({
				message: cvLoc('appdesigner.msg.saveSuccessful', '<b>' + vm.column.name + '</b>')
			});
		}

		function onRequestFailure(err) {
			const message = getErrMessage(err);
			cvToaster.showErrorMessage({
				message
			});
		}
	};

	vm.checkNeedsChoices = () => {
		return vm.column.type === 'text' || vm.column.type === 'number' || vm.column.type === 'date';
	};

	vm.handleDeleteColumn = function handleDeleteColumn() {
		let columnName = vm.column.name;

		$dialogs.confirm(
			cvLoc('appdesigner.label.confirmDelete'),
			cvLoc('appdesigner.msg.confirmDelete', `<b>${columnName}</b>`),
			{
				noFunction: function() {},
				yesFunction: function() {
					vm.column.remove().then(onDeleteSuccess, onDeleteFailure);
				}
			}
		);

		function onDeleteSuccess() {
			$state.go('tableConfig', { tableId });
		}

		function onDeleteFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}
	};

	vm.handleMultiselectClose = function handleMultiselectClose() {
		if (vm.selectedItem.length > 0) {
			if (vm.column.type === 'report') {
				vm.column.itemId = vm.selectedItem[0].guid;
			}
			if (vm.column.type === 'table') {
				vm.column.itemId = 'table:' + vm.selectedItem[0].id;
			}
		}
	};

	function fetchColumn() {
		return TablesService.getColumnById(tableId, columnId).then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(column) {
			vm.loading = false;
			vm.column = column;
			vm.selectedFieldType = vm.fieldTypes.find(type => type.name === vm.column.fieldType) || null;
		}

		function onRequestFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
			$log.error(err);
		}
	}

	function init() {
		fetchColumn();
	}
}

ColumnConfigCtrl.$inject = [
	'$stateParams',
	'$log',
	'cvLoc',
	'$dialogs',
	'cvToaster',
	'$state',
	'ReportsService',
	'TablesService'
];

appDesignerModule.controller('ColumnConfigCtrl', ColumnConfigCtrl);

export default appDesignerModule;
