SET NOCOUNT ON 
SET QUOTED_IDENTIFIER ON	-- needed for V10
DECLARE @ErrorTag	NVARCHAR(20) = '###ERROR###'
DECLARE @WarningTag	NVARCHAR(20) = '###WARNING###'
DECLARE @AdviceTag	NVARCHAR(20) = '###ADVICE###'
DECLARE @HeaderTag	NVARCHAR(20) = '###HEADER###'
DECLARE @DataTag	NVARCHAR(20) = '###DATA###'
DECLARE @DescTag 	NVARCHAR(20) = '###DESCRIPTION###'
DECLARE @Output		NVARCHAR(MAX) = ''
DECLARE @colSep		NVARCHAR(20) = ':,:'

DECLARE @commServId	INT = 2
DECLARE @ScriptName	NVARCHAR(255) = 'CommServ upgrade configuration check: '


-- Get list of CommServ Clients that must be upgrade at the same time as the CommServer
IF OBJECT_ID('tempdb.dbo.#ClientsToUpgrade') IS NOT NULL
	DROP TABLE #ClientsToUpgrade

CREATE TABLE #ClientsToUpgrade (
	clientId		INT,
	clientName		NVARCHAR(255),
	hostName		NVARCHAR(255),
	softwareVersion	NVARCHAR(255),
	osName			NVARCHAR(255),
	reasonType		INT--,
	--reason			NVARCHAR(255)

	PRIMARY KEY (clientId, reasonType) 
)

-- SP Added in V11
IF EXISTS (SELECT 1 FROM Information_schema.Routines WHERE Specific_schema = 'dbo' AND specific_name = 'AppCheckClientUpgradeAssociations' AND Routine_Type = 'PROCEDURE') 
BEGIN
	INSERT INTO #ClientsToUpgrade
		EXEC AppCheckClientUpgradeAssociations @commServId
END
ELSE
BEGIN
	-- include SP functionality here - remove block in V13?
	/*==============  START of SP BLOCK ==============*/
		-- Local Variables
	DECLARE @ciNodesIntersect		INT = 0
	DECLARE @isClientAServer		INT = 0
	DECLARE @isClientCINode			INT = 0
	DECLARE @isClientPseudoMember	INT = 0
	DECLARE @rowsFound				INT = 0
	DECLARE @passCount				INT = 1
	DECLARE @passCountPrev			INT = 0
	DECLARE @dummyServerPseudo		INT = -1
	DECLARE @dummyCINodePseudo		INT = -2
	DECLARE @dummyClientPseudo		INT = -3
	DECLARE @addedClients			INT = 0

	-- Upgrade Reason Types
	DECLARE @RType0	INT = 0
	DECLARE @RType1	INT = 1
	DECLARE @RType2	INT = 2
	DECLARE @RType3	INT = 3
	DECLARE @RType4	INT = 4
	DECLARE @RType5	INT = 5
	DECLARE @RType6	INT = 6
	DECLARE @RType7	INT = 7
	DECLARE @RType8	INT = 8

	--IF OBJECT_ID('tempdb.dbo.#ReasonTypes') IS NOT NULL
	--	DROP TABLE #ReasonTypes
	--CREATE TABLE #ReasonTypes (
	--	type			INT PRIMARY KEY,
	--	reason			VARCHAR(255)
	--)

	--INSERT INTO #ReasonTypes
	--			SELECT @RType0, 'Client to Upgrade'
	--	UNION	SELECT @RType1, 'Must Upgrade Servers due to Client Upgrade'
	--	UNION	SELECT @RType2, 'Must Upgrade Client due to Server Upgrade'
	--	UNION	SELECT @RType3, 'Must Upgrade Client due to Member Client Upgrade'
	--	UNION	SELECT @RType4, 'Must Upgrade CINodes due to Server upgrade'
	--	UNION	SELECT @RType5, 'Must Upgrade CINodes due to Member Client upgrade'
	--	UNION	SELECT @RType6, 'Must upgrade Member Client due to Client Upgrade'
	--	UNION	SELECT @RType7, 'Must Upgrade CINodes due to Client Upgrade'
	--	UNION	SELECT @RType8, 'Must Upgrade Client due to Release Compatiblity'

	-- Determine All Servers
	IF OBJECT_ID('tempdb.dbo.#ServerClients') IS NOT NULL
		DROP TABLE #ServerClients

	CREATE TABLE #ServerClients (
		clientId		INT,
		clientName		NVARCHAR(255),
		hostName		NVARCHAR(255),
		packageName		NVARCHAR(256),
		packageId		INT,

		PRIMARY KEY(clientId, packageId)
	)

	INSERT INTO #ServerClients
		SELECT	-- Servers to upgrade
			c.id clientId, 
			c.name clientName,
			c.net_hostname hostName,
			sp.DisplayName packageName,
			sp.id packageId
		FROM APP_Client c WITH(NOLOCK)
			INNER JOIN simInstalledPackages sim WITH(NOLOCK) ON 
				c.id = sim.ClientId 
				AND (c.specialClientFlags=0) 
				AND (c.status & 4096) <> 4096
				AND sim.simPackageID IN (20, 23, 726, 727, 252)	-- CS, WorkFlow, WebConsole, MetricServer, WebServer Package Ids
			INNER JOIN simPackage sp WITH(NOLOCK) ON
				sp.id = sim.simPackageID
		UNION ALL
		-- may pull this out a separate query
		SELECT	-- are any MAs, Search install on Servers, need to be upgraded as well
			c.clientId, 
			c.clientName, 
			c.hostName,
			sp.DisplayName,
			sp.id
		FROM 
			(SELECT DISTINCT	-- Servers to upgrade
				c.id clientId, 
				c.name clientName,
				c.net_hostname hostName
			FROM APP_Client c WITH(NOLOCK)
				INNER JOIN simInstalledPackages sim WITH(NOLOCK) ON 
					c.id = sim.ClientId 
					AND (c.specialClientFlags = 0) 
					AND (c.status & 4096) <> 4096
					AND sim.simPackageID IN (20, 23, 726, 727, 252)
			) c
			INNER JOIN simInstalledPackages sim WITH(NOLOCK) ON 
				c.clientId = sim.ClientId 
				AND sim.simPackageID IN (51, 259)		-- Co-resident MA and Search
			INNER JOIN simPackage sp ON
				sp.id = sim.simPackageID
		UNION ALL
		SELECT	-- FireWall Proxies to upgrade
			c.id, 
			c.name, 
			c.net_hostname,
			'FireWall Proxy' ServerRole,
			@dummyServerPseudo
		FROM APP_Client c WITH(NOLOCK)
			INNER JOIN App_FirewallOptions f WITH(NOLOCK) on 
				f.clientId = c.Id 
				AND (c.specialClientFlags = 0) 
				AND c.status & 4096 <> 4096
				AND isDMZ=1

	-- Is the inputted clientId a server?
	SELECT
		@isClientAServer = 1
	FROM #ServerClients sc
	WHERE 
		sc.clientId = @commServId

	
	-- Determine all member clients for Pseudo Clients with AppTypes of OracleRAC(80), Exchange DAG(53), Virtual Server(106), MSSQL(81)
	IF OBJECT_ID('tempdb.dbo.#PseudoClients') IS NOT NULL
		DROP TABLE #PseudoClients

	CREATE TABLE #PseudoClients (
		physClientId		INT,
		pseudoClientId		INT,
		appTypeId			INT,
	
		PRIMARY KEY (physClientId, pseudoClientId, appTypeId)
	)

	INSERT INTO #PseudoClients(appTypeId, pseudoClientId, physClientId)
		SELECT	-- VSA Pseudo Client to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			a.value('@clientId', 'int') physClientId
		FROM
			(SELECT
				a.clientId,
				a.appTypeId,
				CAST(scp.attrVal AS XML) data
			FROM
				APP_Application a WITH(NOLOCK),
				APP_SubClientProp scp WITH(NOLOCK)
			WHERE
				dbo.IsPseudoClient(a.clientId) = 1
				AND a.appTypeId = 106
				AND scp.componentNameId = a.id
				AND scp.cs_attrName = CHECKSUM(N'Virtual Server Proxy Client List')
				AND scp.attrName = N'Virtual Server Proxy Client List'
				AND scp.modified = 0
			) t
			CROSS APPLY t.data.nodes('App_MemberServers/memberServers/client') rc(a)
		WHERE
			a.value('@clientId', 'int') IS NOT NULL
		UNION
		SELECT	-- VSA Pseudo Client to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			c.id physClientId
		FROM
			(SELECT
				a.clientId,
				a.appTypeId,
				CAST(scp.attrVal AS XML) data
			FROM
				APP_Application a WITH(NOLOCK),
				APP_SubClientProp scp WITH(NOLOCK)
			WHERE
				dbo.IsPseudoClient(a.clientId) = 1
				AND a.appTypeId = 106
				AND scp.componentNameId = a.id
				AND scp.cs_attrName = CHECKSUM(N'Virtual Server Proxy Client List')
				AND scp.attrName = N'Virtual Server Proxy Client List'
				AND scp.modified = 0
			) t
			CROSS APPLY t.data.nodes('App_MemberServers/memberServers/client') rc(a)
			INNER JOIN APP_ClientGroupAssoc cga WITH(NOLOCK) ON
				cga.clientGroupId = a.value('@clientGroupId', 'int')
			INNER JOIN APP_Client c WITH(NOLOCK) ON
				c.id = cga.clientId
		UNION
		SELECT	-- VSA Pseudo Client to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			a.value('@clientId', 'int') physClientId
		FROM
			(SELECT 
				a.clientId,
				a.appTypeId,
				CAST(ip.attrVal AS XML) data
			FROM
				APP_Application a WITH(NOLOCK),
				APP_InstanceProp ip WITH(NOLOCK)
			WHERE
				dbo.IsPseudoClient(a.clientId) = 1
				AND a.appTypeId = 106
				AND ip.componentNameId = a.instance
				AND ip.attrName = 'VS Member Servers'
				AND ip.modified = 0
			) t
			CROSS APPLY t.data.nodes('App_MemberServers/memberServers/client') rc(a)
		WHERE
			a.value('@clientId', 'int') IS NOT NULL
		UNION
		SELECT	-- VSA Pseudo Client to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			c.id physClientId
		FROM
			(SELECT 
				a.clientId,
				a.appTypeId,
				CAST(ip.attrVal AS XML) data
			FROM
				APP_Application a WITH(NOLOCK),
				APP_InstanceProp ip WITH(NOLOCK)
			WHERE
				dbo.IsPseudoClient(a.clientId) = 1
				AND a.appTypeId = 106
				AND ip.componentNameId = a.instance
				AND ip.attrName = 'VS Member Servers'
				AND ip.modified = 0
			) t
			CROSS APPLY t.data.nodes('App_MemberServers/memberServers/client') rc(a)
			INNER JOIN APP_ClientGroupAssoc cga WITH(NOLOCK) ON
				cga.clientGroupId = a.value('@clientGroupId', 'int')
			INNER JOIN APP_Client c WITH(NOLOCK) ON
				c.id = cga.clientId
		UNION
		SELECT	-- Exchange DAG Pseudo Clients to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			c.id physClientId
		FROM
			APP_Client c WITH(NOLOCK)
			INNER JOIN (
				SELECT DISTINCT
					a.clientId,
					cp.attrName exchDAGHostname,
					a.appTypeId
				FROM
					APP_Application a WITH(NOLOCK)
					INNER JOIN APP_ClientProp cp WITH(NOLOCK) ON
						a.appTypeId = 53
						AND cp.componentNameId = a.clientId
						AND cp.modified = 0
						AND cp.attrType = 125
				WHERE
					dbo.IsPseudoClient(a.clientId) = 1
			) t ON
				t.exchDAGHostname = c.net_hostname
			INNER JOIN APP_Client pc WITH(NOLOCK) ON
				pc.id = t.clientId
		UNION
		SELECT	-- Oracle RAC Pseudo Clients to Physical Clients
			a.appTypeId,
			a.clientId pseudoClientId,
			rac.clientId physClientId
		FROM
			APP_Application a WITH(NOLOCK) 
			INNER JOIN APP_OracleRacInstance rac WITH(NOLOCK) ON
				a.appTypeId = 80
				AND rac.instaceId = a.instance
		WHERE
			dbo.IsPseudoClient(a.clientId) = 1
		UNION
		SELECT	-- MSSQL Proxy Pseudo Clients to Physical Clients
			a.appTypeId,
			a.clientId pseudoClientId,
			CAST(ip.attrVal AS INT) physClientId
		FROM
			APP_Application a WITH(NOLOCK)
			INNER JOIN APP_InstanceProp ip WITH(NOLOCK) ON
				a.appTypeId = 81
				AND ip.componentNameId = a.instance
				AND ip.attrName = 'Proxy Client Id'
		WHERE
			dbo.IsPseudoClient(a.clientId) = 1
		UNION
		SELECT	-- MSSQL Availability Group Pseudo Clients to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			a.value('@clientId', 'int') physClientId
		FROM
			(
				SELECT
					a.clientId,
					a.appTypeId,
					CAST(ip.attrVal AS XML) data
				FROM
					APP_Application a WITH(NOLOCK)
					INNER JOIN APP_InstanceProp ip WITH(NOLOCK) ON
						a.appTypeId = 81
						AND ip.componentNameId = a.instance
						AND ip.attrName = 'Availability Group'
				WHERE
					dbo.IsPseudoClient(a.clientId) = 1
			) t
			CROSS APPLY t.data.nodes('App_SQLAvailabilityGroup/primaryReplicaClient') rc(a)
		WHERE
			a.value('@clientId', 'int') IS NOT NULL
		UNION
		SELECT	-- MSSQL Availability Replicas Pseudo Clients to Physical Clients
			t.appTypeId,
			t.clientId pseudoClientId,
			a.value('@clientId', 'int') physClientId
		FROM
			(
				SELECT
					a.clientId,
					a.appTypeId,
					CAST(ip.attrVal AS XML) data
				FROM
					APP_Application a WITH(NOLOCK)
					INNER JOIN APP_InstanceProp ip WITH(NOLOCK) ON
						a.appTypeId = 81
						AND ip.componentNameId = a.instance
						AND ip.attrName = 'Availability Replicas'
				WHERE
					dbo.IsPseudoClient(a.clientId) = 1
			) t
			CROSS APPLY t.data.nodes('App_SQLAvailabilityReplicas/SQLAvailabilityReplicasList/replicaClient') rc(a)
		WHERE
			a.value('@clientId', 'int') IS NOT NULL

	-- Is the inputted clientId a member client of a Pseudo Client(s)?
	SELECT TOP 1
		@isClientPseudoMember = 1
	FROM #PseudoClients pc
	WHERE 
		pc.physClientId = @commServId

	-- Determine All CI Node Clients
	IF OBJECT_ID('tempdb.dbo.#CINodes') IS NOT NULL
		DROP TABLE #CINodes

	CREATE TABLE #CINodes (
		clientId		INT PRIMARY KEY,
		clientName		NVARCHAR(255),
		hostName		NVARCHAR(255),
		isMA			SMALLINT DEFAULT 0
	)

	INSERT INTO #CINodes(clientId, clientName, hostName, isMA)
		SELECT --DISTINCT
			C.id clientId,
			C.name clientName,
			C.net_hostname hostName,
			CASE
				WHEN sim.simPackageID IS NOT NULL THEN 1
				ELSE 0
			END isMA
		FROM 
			APP_ClientProp AS CLP WITH(NOLOCK)
			INNER JOIN APP_Client AS C WITH(NOLOCK) ON
				C.id = CLP.componentNameId
			LEFT OUTER JOIN simInstalledPackages sim  WITH(NOLOCK) ON	-- Does the CI Node have an MA installed?
						C.id = sim.ClientId 
						AND sim.simPackageID = 51		-- MA
		WHERE 
			( CLP.attrName = 'FAST_hostname' AND CLP.modified = 0 )
			AND EXISTS(SELECT c1.id
						FROM APP_Client c1 WITH(NOLOCK)
						WHERE c1.id = CLP.componentNameId AND (c1.status & 0x00002) = 0)
			AND
			EXISTS ( SELECT cp1.componentnameId
						FROM APP_ClientProp cp1 WITH(NOLOCK)
						WHERE ( cp1.attrname = 'Content Index Server' OR cp1.attrname in ('PackageStateForId 257', 'PackageStateForId 261','PackageStateForId 260') )
						AND cp1.attrVal = '1'
						AND cp1.componentnameId = CLP.componentnameId)

	-- Is the inputted clientId a CINode?
	SELECT
		@isClientCINode = 1
	FROM #CINodes cn
	WHERE 
		cn.clientId = @commServId

	-- Determine associated MM Hosts MM Host Paths, and VMs for this inputted Client
	-- that should be upgrade at the same time based on release compatibility.
	-- Cannot upgrade a client from V10 to V11 if the MA is at V10.  But can upgrade an MA 
	-- from V10 to V11 and have V10 clients using it.
	IF OBJECT_ID('tempdb.dbo.#AssocClientInfo') IS NOT NULL
		DROP TABLE #AssocClientInfo

	CREATE TABLE #AssocClientInfo (
		clientId	INT,
		archGrpId	INT,
		clientType	VARCHAR(16),
		PRIMARY KEY (clientId, archGrpId, clientType)
	)

	INSERT INTO #AssocClientInfo
		SELECT		-- Find the all media agent associated with current client/MA through a shared copy
			HostClientId,
			- 1,
			'MMHost'
		FROM MMDataPath WITH (NOLOCK)
		WHERE 
			CopyId IN (
				SELECT CopyId
				FROM MMDataPath
				WHERE HostClientId = @commServId
			)
		UNION
		SELECT		-- Find all Subclient Data Archive Groups for the inputted client	
			a.clientId,
			a.dataArchGrpID archGrpId,
			'client' clientType
		FROM APP_Application a WITH (NOLOCK)
		WHERE 
			a.clientId = @commServId
			AND a.dataArchGrpID <> 1
		UNION
		SELECT		-- Find all Subclient Log Archive Groups for the inputted client	
			a.clientId,
			a.logArchGrpID archGrpId,
			'client'
		FROM APP_Application a WITH (NOLOCK)
		WHERE
			a.clientId = @commServId
			AND a.logArchGrpID <> 1
		UNION
		SELECT		-- Find all VM Subclient Data Archive Groups associated with the inputted client (Physical Machine)
			a.clientId,
			a.dataArchGrpID archGrpId,
			'vmClient'
		FROM 
			APP_VMToPMMap v
			INNER JOIN APP_Client c WITH (NOLOCK) ON 
				c.id = v.VMClientId
			INNER JOIN APP_Application a WITH (NOLOCK) ON 
				a.clientId = c.id
		WHERE 
			v.PMClientId = @commServId
			AND a.dataArchGrpID <> 1
		UNION
		SELECT		-- Find all VM Subclient Log Archive Groups associated with the inputted client (Physical Machine)
			a.clientId,
			a.logArchGrpID archGrpId,
			'vmClient'
		FROM 
			APP_VMToPMMap v
			INNER JOIN APP_Client c WITH (NOLOCK) ON 
				c.id = v.VMClientId
			INNER JOIN APP_Application a WITH (NOLOCK) ON 
				a.clientId = c.id
		WHERE 
			v.PMClientId = @commServId
			AND a.logArchGrpID <> 1

	-- Find the all media agent associated with selected clients from the above through copy/data path
	INSERT INTO #AssocClientInfo
		SELECT DISTINCT 
			dp.ClientId,
			-1 archGrpId,
			'MMHostPath'
		FROM #AssocClientInfo C
			INNER JOIN archGroupCopy AGC WITH (NOLOCK) ON 
				AGC.archGroupId = C.archGrpId
			INNER JOIN MMDataPath p WITH (NOLOCK) ON 
				P.CopyId = AGC.id
			INNER JOIN MMDrivePool dp WITH (NOLOCK) ON 
				(p.Flag & 16) = 0
				AND dp.DrivePoolId = p.DrivePoolId
		WHERE 
			dp.ClientId NOT IN (
				SELECT DISTINCT clientId
				FROM #AssocClientInfo
			)
			AND dp.ClientId <> 1

	-- Result Set Output Table
	IF OBJECT_ID('tempdb.dbo.#MustClientsToUpgrade') IS NOT NULL
		DROP TABLE #MustClientsToUpgrade

	CREATE TABLE #MustClientsToUpgrade (
		physClientId		INT,
		pseudoClientId		INT,
		passNumber			INT,
		reasonType			INT

		PRIMARY KEY (physClientId, pseudoClientId, passNumber, reasonType) 
	)

	-- Server clients to upgrade
	INSERT INTO #MustClientsToUpgrade(physClientId, pseudoClientId, passNumber, reasonType)
		SELECT
			c.id,
			@dummyServerPseudo,
			@passCount,
			@RType0
		FROM APP_Client c WITH(NOLOCK)
		WHERE
			c.id = @commServId
		UNION
		SELECT --DISTINCT
			s.clientId,
			@dummyServerPseudo,
			@passCount,
			@RType1
		FROM #ServerClients s
		WHERE
			@isClientAServer = 1
		UNION -- Server / CINodes intersection
		SELECT
			ci.clientId,
			@dummyCINodePseudo,
			@passCount,
			@RType4
		FROM #CINodes ci
		WHERE
			(
				@isClientCINode = 1
				OR @isClientAServer = 1
			)
			AND EXISTS(SELECT 1 FROM #ServerClients sc INNER JOIN #CINodes ci ON sc.clientId = ci.clientId)
		UNION -- Client / CINodes intersection
		SELECT
			ci.clientId,
			@dummyCINodePseudo,
			@passCount,
			@RType7
		FROM #CINodes ci
		WHERE
			@isClientCINode = 1
		UNION -- server / client (pseudo) intersection
		SELECT DISTINCT
			pc.physClientId,
			pc.pseudoClientId,
			@passCount,
			@RType2
		FROM #PseudoClients pc
			INNER JOIN ( -- Any physical server client associated with a pseudo client then all physical clients associated to that pseudo need to upgrade
				SELECT DISTINCT
					s1.clientId serverClientId,
					pc1.pseudoClientId
				FROM #ServerClients s1
					INNER JOIN #PseudoClients pc1 ON
						s1.clientId = pc1.physClientId
			) fnd ON
				@isClientAServer = 1
				AND fnd.pseudoClientId = pc.pseudoClientId
		UNION -- server / client (pseudo) intersection
		SELECT DISTINCT
			pc.physClientId,
			pc.pseudoClientId,
			@passCount,
			@RType6
		FROM #PseudoClients pc
		WHERE
			@isClientPseudoMember = 1
			AND pc.pseudoClientId IN (SELECT c.pseudoClientId FROM #PseudoClients c WHERE c.physClientId = @commServId)
		UNION	-- Associated Clients by release compatibility
		SELECT
			ci.clientId,
			@dummyClientPseudo,
			@passCount,
			@RType8
		FROM #AssocClientInfo ci
	SET @rowsFound = @@ROWCOUNT
	
	-- Continue looping finding all associated member clients until none are found
	SET @passCountPrev = @passCount
	SET @passCount += 1

Clients_Added:

	WHILE (@rowsFound > 0)
	BEGIN
	
		INSERT INTO #MustClientsToUpgrade(physClientId, pseudoClientId, passNumber, reasonType)
			SELECT DISTINCT
				pcNext.physClientId,
				pcNext.pseudoClientId,
				@passCount,
				@RType3
			FROM #MustClientsToUpgrade cu
				INNER JOIN #PseudoClients pc ON
					cu.passNumber = @passCountPrev
					AND cu.physClientId = pc.physClientId
					AND pc.pseudoClientId NOT IN (SELECT u.pseudoClientId FROM #MustClientsToUpgrade u WHERE u.physClientId = pc.physClientId)	-- not already captured
				INNER JOIN #PseudoClients pcNext ON
					pcNext.pseudoClientId = pc.pseudoClientId
			
		SET @rowsFound = @@ROWCOUNT
		SET @passCountPrev = @passCount
		SET @passCount += 1
	END

	-- Double check CINodes against list again if CINodes not part of original processing 
	-- just incase a client is upgraded that is a CINode.
	IF (@ciNodesIntersect = 0
		AND NOT EXISTS (SELECT 1 FROM #MustClientsToUpgrade cu INNER JOIN #CINodes ci ON cu.physClientId = ci.clientId AND cu.reasonType IN (4, 7)))
	BEGIN
		-- Determine if there is any intersection with the CINodes list now
		INSERT INTO #MustClientsToUpgrade(physClientId, pseudoClientId, passNumber, reasonType)
			SELECT DISTINCT
				ci.clientId,
				@dummyCINodePseudo,
				@passCount,
				@RType5
			FROM #CINodes ci
			WHERE
				EXISTS(SELECT 1 FROM #MustClientsToUpgrade cu INNER JOIN #CINodes ci ON cu.physClientId = ci.clientId AND cu.reasonType NOT IN (4, 7))

		SET @rowsFound = @@ROWCOUNT
		SET @ciNodesIntersect = 1

		IF (@rowsFound > 0)
			SET @addedClients = 1
	END

	IF NOT EXISTS (SELECT 1 FROM  #MustClientsToUpgrade cu WHERE cu.pseudoClientId = @dummyServerPseudo)
	BEGIN
		-- Determine if there is any intersection with the Server list now
		INSERT INTO #MustClientsToUpgrade(physClientId, pseudoClientId, passNumber, reasonType)
			SELECT DISTINCT
				sc.clientId,
				@dummyServerPseudo,
				@passCount,
				@RType1
			FROM #ServerClients sc
			WHERE
				EXISTS (SELECT s.clientId FROM #MustClientsToUpgrade cu INNER JOIN #ServerClients s ON s.clientId = cu.physClientId AND cu.pseudoClientId <> @dummyServerPseudo)

		SET @rowsFound = @@ROWCOUNT

		IF (@rowsFound > 0)
			SET @addedClients = 1
	END

	-- Where any new clients added to the upgrade list?
	IF (@addedClients > 0)
	BEGIN
		SET @passCountPrev = @passCount
		SET @passCount += 1
		SET @addedClients = 0
		GOTO Clients_Added
	END

	-- return result set
	INSERT INTO #ClientsToUpgrade
		SELECT DISTINCT
			cu.physClientId clientId,
			c.name clientName,
			c.net_hostname hostName,
			cp.attrVal softwareVersion,
			SUBSTRING(os.attrVal, 1, CHARINDEX(':', os.attrVal) - 1)  operatingSystem,
			cu.reasonType--,
			--rt.reason
		FROM #MustClientsToUpgrade cu
			--INNER JOIN #ReasonTypes rt ON
			--	rt.type = cu.reasonType
			INNER JOIN APP_Client c WITH(NOLOCK) ON
				c.id = cu.physClientId
				AND (c.status & 2) = 0		-- installed
			INNER JOIN APP_ClientProp cp WITH(NOLOCK) ON
				cp.componentNameId = c.id
				AND cp.attrName = 'SP Version Info'
			INNER JOIN APP_ClientProp os WITH(NOLOCK) ON
				os.componentNameId = c.id
				AND os.attrName = 'Sim OS Info'
		WHERE
			cu.physClientId = @commServId
			AND cu.reasonType = 0
		UNION ALL
		SELECT DISTINCT
			cu.physClientId clientId,
			c.name clientName,
			c.net_hostname hostName,
			cp.attrVal softwareVersion,
			SUBSTRING(os.attrVal, 1, CHARINDEX(':', os.attrVal) - 1)  operatingSystem,
			cu.reasonType--,
			--rt.reason
		FROM #MustClientsToUpgrade cu
			--INNER JOIN #ReasonTypes rt ON
			--	rt.type = cu.reasonType
			INNER JOIN APP_Client c WITH(NOLOCK) ON
				c.id = cu.physClientId
				AND (c.status & 2) = 0		-- installed
			INNER JOIN APP_ClientProp cp WITH(NOLOCK) ON
				cp.componentNameId = c.id
				AND cp.attrName = 'SP Version Info'
			INNER JOIN APP_ClientProp os WITH(NOLOCK) ON
				os.componentNameId = c.id
				AND os.attrName = 'Sim OS Info'
		WHERE
			cu.physClientId <> @commServId

	/*==============  END of SP BLOCK ==============*/
END

-- CommServ data
DECLARE @clientId			INT
DECLARE @clientName			NVARCHAR(255)
DECLARE @hostName			NVARCHAR(255)
DECLARE @osName				NVARCHAR(255)
DECLARE @softwareVersion	NVARCHAR(255)
DECLARE @reasonType			INT
DECLARE @reason				NVARCHAR(255)

-- Report column titles
DECLARE @titleCN	NVARCHAR(50) = N'Client Name'
DECLARE @titleHN	NVARCHAR(50) = N'Hostname'
DECLARE @titleOS	NVARCHAR(50) = N'Operating System'
DECLARE @titleSV	NVARCHAR(50) = N'Software Version'
--DECLARE @titleUR	NVARCHAR(50) = N'Upgrade Reason'

-- Get CommServer Record
SELECT
	@clientId = c.clientId--,
	--@clientName = c.clientName,
	--@hostName = c.hostName,
	--@osName = c.osName,
	--@softwareVersion = c.softwareVersion,
	--@reasonType = c.reasonType,
	--@reason = c.reason
FROM #ClientsToUpgrade c
WHERE
	c.clientId = @commServId
	AND c.reasonType = 0	-- CommServ client upgrading

IF (@clientId IS NULL)
BEGIN
	PRINT @ErrorTag + N'CommServ Client not found!'
	PRINT @ScriptName + 'failed'
	SELECT 2 exitCode
	GOTO END_OF_CheckCSUpgrade
END

-- Format Report
PRINT @DescTag + N' Checking CommServ configuration for all clients that MUST be upgraded.'

DECLARE @ClientCnt INT = 0
SELECT @ClientCnt = COUNT(DISTINCT clientId) FROM #ClientsToUpgrade WHERE clientId <> @commServId
IF (@ClientCnt = 0)
BEGIN
	-- nothing to process
	PRINT @ScriptName + 'passed'
	SELECT 0 exitCode
	GOTO END_OF_CheckCSUpgrade
END

PRINT @WarningTag + N' For better performance of your CommCell operations, you must plan to upgrade all of the following clients at the same time.' + NCHAR(13) + NCHAR(10) 
PRINT @HeaderTag 
                + N' ' 
                + @titleCN + @colSep 
                + @titleHN + @colSep 
                + @titleOS + @colSep
                + @titleSV -- + @colSep
                -- + @titleUR 
                + NCHAR(13) + NCHAR(10)

-- insert CommServer Data into report
--PRINT @DataTag
--                + ' ' 
--                + @clientName + @colSep 
--                + @hostName + @colSep 
--                + @osName + @colSep 
--                + @softwareVersion + @colSep 
--                + @reason 
--                + NCHAR(13) + NCHAR(10) 

-- now add clients that need to be upgrade as well
DECLARE @i_clientId INT
DECLARE cur CURSOR FOR SELECT DISTINCT clientId FROM #ClientsToUpgrade WHERE clientId <> @commServId
OPEN cur
FETCH NEXT FROM cur INTO @i_clientId

WHILE @@FETCH_STATUS = 0
BEGIN
	
	SELECT TOP 1
		@Output = @DataTag
					+ ' ' 
					+ c.clientName + @colSep
					+ c.hostName + @colSep 
					+ c.osName + @colSep 
					+ c.softwareVersion -- + @colSep
					-- + c.reason 
					+ NCHAR(13) + NCHAR(10)  
	FROM #ClientsToUpgrade c
	WHERE
		c.clientId = @i_clientId

	PRINT @Output

	FETCH NEXT FROM cur INTO @i_clientId
END

CLOSE cur
DEALLOCATE cur

PRINT @ScriptName + 'passed'
SELECT 1 exitCode

END_OF_CheckCSUpgrade:
