import json
import requests

from CvEEConfigHelper import get_local_logger


class SolrHelper:
    def __init__(self, logger, url, core, default_query=None):
        if logger is None:
            logger = get_local_logger()
        self.logger = logger
        if url.endswith("/"):
            url = url[:-1]
        self.url = url
        self.core = core
        self.default_query = default_query
        self.session = requests.Session()

    def is_alive(self):
        try:
            resp = self.session.get(self.url)
            if resp.status_code == requests.codes.OK:
                return True
            else:
                return False
        except Exception as e:
            self.logger(f"Unable to send request to {self.url}. Exception {e}")
            return False

    def get_merged_params(self, params):
        if "q" in params:
            if self.default_query is not None:
                if "fq" not in params:
                    params["fq"] = params["q"]
                else:
                    params["fq"] = f"{params['fq']}&{params['q']}"
        params["q"] = self.default_query
        return params

    def default_query_resp(self):
        try:
            params = {
                "wt": "json",
            }
            response = self.query(params)
            return response
        except Exception as e:
            raise

    def query(self, params):
        try:
            params = self.get_merged_params(params)
            query_params = ""
            for key, value in params.items():
                query_params += f"{key}={value}&"
            query_url = f"{self.url}/{self.core}/select?{query_params}"
            self.logger.debug(f"Solr query url {query_url}")
            resp = self.session.get(query_url)
            return SolrHelper.process_response(resp, query_url)
        except Exception:
            raise

    def update(self, data):
        try:
            udpate_url = f"{self.url}/{self.core}/update?commit=true"
            self.logger.debug(f"Solr update url {udpate_url}")
            resp = self.session.post(
                udpate_url, data=json.dumps(data), headers={"content-type": "application/json"}
            )
            return SolrHelper.process_response(resp, udpate_url)
        except Exception:
            raise

    def delete(self, query):
        try:
            delete_url = f"{self.url}/{self.core}/update?commit=true"
            # data = f'<delete><query>dataset_id:{dataset_name}</query></delete>'
            data = f"<delete><query>{query}</query></delete>"
            resp = self.session.get(
                delete_url, data=data, headers={"content-type": "application/xml"}
            )
            return SolrHelper.process_response(resp, delete_url)
        except Exception:
            raise

    @staticmethod
    def process_response(resp, url):
        try:
            if resp.status_code == requests.codes.OK:
                return resp
            elif resp.status_code == requests.codes.NOT_FOUND:
                error_msg = f"Either dataset is not created or solr url is not correct {url}"
                raise Exception(error_msg)
            elif resp.status_code == requests.codes.INTERNAL_SERVER_ERROR:
                error_msg = f"Please verify that solr is reachable at {url}"
                raise Exception(error_msg)
            else:
                raise Exception(
                    f"Failed to query {url}. HTTP error code {resp.status_code}. Error message {resp.text}"
                )
        except Exception:
            raise
