import fileinput
import json
import os
import platform
import re
import shutil
import socket
import subprocess
import sys
import time
import traceback
from os.path import abspath, splitext
from pathlib import Path
from sys import modules
from threading import Timer

import requests

from CvEEConfigHelper import (
    MODELS_INSTALLATION_PATH_REG_KEY,
    SPACY_MODELS_MAP,
    SPACY_MODELS_REG_KEY,
    get_cvreg_linux,
    get_local_logger,
    getInstanceName,
    getPythonSitePackagesDir,
    getStompPort,
    is_linux,
    loadRegValue,
    setRegValue,
)
from CvEEMsgQueueHandler import GenericMsgQueueCommunicator
from pathlib import Path
from subprocess import check_output, CalledProcessError

try:
    import win32service
    from winreg import ConnectRegistry, OpenKey, DeleteValue, HKEY_LOCAL_MACHINE, KEY_ALL_ACCESS
except:
    pass

logging = get_local_logger(file_name="InstallPyAnalyticsService.log")


class CommandSender(GenericMsgQueueCommunicator):
    def __init__(self):

        GenericMsgQueueCommunicator.__init__(self)

        # Get host and port from registry

        self.host = self.getMessageQueueHostname()
        self.stompPort = getStompPort()

        self.queue = "PythonCommands"
        self.clients = {self.queue: {"subscribe": False, "client": None}}

    def send(self, command):
        response = self.connectToQueue(retry=False)
        if response != True:
            return response

        message = {"command": command}

        response = self.sendOnConnect(
            self.queue, json.dumps(message), headers={"persistent": "true"}
        )
        if response == True:
            response = "{} command sent to queue successfuly".format(command)

        self.disconnectClients()
        return response

    def getMessageQueueHostname(self):
        key = "messageQueueHostName"
        value = "127.0.0.1"
        return loadRegValue(key, value, str)


def isPkgInstalled(pkgname):
    import importlib

    try:
        i = importlib.import_module(pkgname)
    except ImportError:
        return False
    return True


def unTarAndCopy(file_tar, output_dir):
    import tarfile

    tar = tarfile.open(file_tar, "r:gz")
    tar.extractall(path=output_dir)
    tar.close()


def extract_nltk_models():
    model_path = Path("../ContentAnalyzer/bin/nltk_models/corpora")
    pkgs = {"words": "words.zip", "wordnet": "wordnet.zip", "punkt": "../tokenizers/punkt.zip"}
    import zipfile

    for _, pkg_path in pkgs.items():
        try:
            with zipfile.ZipFile(os.path.join(os.getcwd(), model_path, Path(pkg_path)), "r") as zf:
                zf.extractall(os.path.split(os.path.join(os.getcwd(), model_path, pkg_path))[0])
        except Exception as e:
            raise Exception("Failed to unzip {}. Exception {}.".format(pkg_path, e))


def installDocumentClassifierModels():
    # for now we are packaging all the custom models in one common package cv_custom_models.see
    pkgpath = Path("../ContentAnalyzer/bin/spacy_models/")
    pkgs = {
        "finance": "finance-1.2.3.tar.gz",
        "legal": "legal-1.2.5.tar.gz",
        "technical": "technical-1.2.3.tar.gz",
        "us_medical_forms": "us_medical_forms-1.2.3.tar.gz",
    }
    # extract models in ContenAnalyzer/bin/classifier_models/cv_doc_tag_models
    # TODO: registry controllable path
    model_installation_path = Path("../ContentAnalyzer/bin/classifier_models/cv_doc_tag_models")
    try:
        for pkgname in pkgs:
            try:
                """                
                    Version Format: 1.0.0
                    Third digit (0) - will be increased when we are adding more documents for training
                    Second digit (0) - will be increased when we are changing feature extractor
                    First digit (1) - will be increased in case of train model change
                """
                previous_model_version, current_model_version = get_model_version(
                    model_installation_path / Path("model"), pkgs[pkgname]
                )
                if previous_model_version != current_model_version:
                    logging.info(f"Extracting model {pkgname} to {model_installation_path}")
                    unTarAndCopy(os.path.join(pkgpath, pkgs[pkgname]), model_installation_path)
            except Exception as e:
                raise Exception(f"Could not extract {pkgname} model as there was an error: {e}")

        remove_previous_models()
    except Exception:
        raise


def delete_files(files_, files_path):
    for file_ in files_:
        try:
            file_path = files_path / Path(file_)
            if file_path.exists():
                file_path.unlink()
                logging.info(f"Deleted file {file_}")
        except Exception as e:
            logging.error(f"Could not delete {file_}. Exception {e}")


def remove_previous_models():
    previous_models = {
        "cv_doc_tag_models-1.0.0.tar.gz",
        "cv_doc_tag_models-1.1.1.tar.gz",
        "en_core_web_sm-1.2.0.win-amd64.tar.gz",
        "finance-1.1.2.tar.gz",
        "legal-1.1.2.tar.gz",
        "technical-1.1.2.tar.gz",
        "us_medical_forms-1.1.1.tar.gz",
        "finance-1.2.2.tar.gz",
        "legal-1.2.2.tar.gz",
        "legal-1.2.3.tar.gz",
        "legal-1.2.4.tar.gz",
        "technical-1.2.2.tar.gz",
        "us_medical_forms-1.2.2.tar.gz",
    }
    model_path = Path("../ContentAnalyzer/bin/spacy_models/")
    delete_files(previous_models, model_path)


def installSpacyModel():
    pkgpath = Path("../ContentAnalyzer/bin/spacy_models/")
    large_model_name = "en_core_web_lg"
    pkgs = {
        # 'en_core_web_sm': 'en_core_web_sm-2.0.0.tar.gz', # no need of this package anymore
        "spacy_text_categorizer": "document_classify_v1.1.tar.gz",
        "sklearn_structure_categorizer": "structure_classify_v1.1.tar.gz",
        "en_core_web_lg": "en_core_web_lg-2.0.0.tar.gz",
        "cvterms": "cvterms.tar.gz",
        "cv_person_techterm_classifier": "cv_person_techterm_classifier.tar.gz",
        "cv_address_classifier": "cv_address_classifier.tar.gz",
    }
    # verify if the large model is already extracted
    PYTHON_SITE_PACKAGES_DIR = getPythonSitePackagesDir()
    MODEL_INSTALLATION_PATH = Path(
        loadRegValue(MODELS_INSTALLATION_PATH_REG_KEY, PYTHON_SITE_PACKAGES_DIR, type=str)
    )
    SPACY_MODELS_INSTALL_DIR = Path(
        loadRegValue(
            SPACY_MODELS_REG_KEY,
            os.path.join(PYTHON_SITE_PACKAGES_DIR, Path(SPACY_MODELS_MAP[large_model_name])),
            type=str,
        )
    )
    previous_model_version, current_model_version = get_model_version(
        SPACY_MODELS_INSTALL_DIR, pkgs[large_model_name]
    )

    for pkgname in pkgs:
        try:
            if MODEL_INSTALLATION_PATH not in sys.path:
                sys.path.append(MODEL_INSTALLATION_PATH)
            # ignore if current installed model version is same as previous installed version
            if pkgname == large_model_name and previous_model_version == current_model_version:
                continue

            if not isPkgInstalled(pkgname):
                logging.info("Extracting model {} to {}".format(pkgname, MODEL_INSTALLATION_PATH))
                unTarAndCopy(os.path.join(pkgpath, pkgs[pkgname]), MODEL_INSTALLATION_PATH)
                if pkgname == "spacy_text_categorizer":
                    pkg_path = os.path.join(
                        MODEL_INSTALLATION_PATH, pkgs[pkgname][: pkgs[pkgname].find(".tar.gz")]
                    )
                    if os.path.exists(os.path.join(pkg_path, "vocab")):
                        shutil.rmtree(os.path.join(pkg_path, "vocab"), ignore_errors=True)

        except Exception as e:
            raise Exception(
                "Could not install {} model as there was an error: {}".format(pkgname, e)
            )


def get_model_version(large_model_dir, current_model):
    """
        get previous version from large model directory from meta.json file
        get current version from the package path
        TODO: need to do clean up for old models, will do it in next SP
    """
    previous_version = None
    current_version = None
    try:
        current_version = re.compile(r"\d\.\d\.\d").findall(current_model)
        current_version = current_version[0]
        if os.path.exists(large_model_dir) and os.path.exists(
            os.path.join(large_model_dir, "meta.json")
        ):
            with open(os.path.join(large_model_dir, "meta.json")) as f:
                model_info = json.load(f)
                previous_version = model_info["version"]
    except Exception as e:
        logging.error(
            "Unable to retrieve installed model version information. Exception {}".format(e)
        )
    return current_version, previous_version


def install_pip_packages(pkgs):
    try:
        pkgpath = Path("../ContentAnalyzer/bin/")
        pkg_install_failed = False
        for pkg_name, pkg_full_name in list(pkgs.items()):
            if (isPkgInstalled(pkg_name) == False) or (
                isVersionUpdate(pkg_name, pkg_full_name) == True
            ) or is_repair_install:
                install_args = [
                            "pip",
                            "install",
                            "--no-deps",
                            "--disable-pip-version-check",
                            "-v",
                            "--force-reinstall",
                            str(pkgpath / pkgs[pkg_name]),
                        ]
                try:
                    install_output = subprocess.check_output(install_args, stderr=subprocess.STDOUT)
                except CalledProcessError as e:                    
                    logging.error(f"Failed to install package [{pkg_full_name}]. Process output [{e.output}]")
                    if "PermissionError" in str(e.output):
                        logging.error(f"Permission error. Retrying with --user option")
                        try:
                            install_args.append("--user")
                            install_output = subprocess.check_output(install_args, stderr=subprocess.STDOUT)
                        except CalledProcessError as e:
                            pkg_install_failed = True
                            logging.error(f"Second attempt failed as well. Process output [{e.output}]")
                            continue                 
                    else:
                        pkg_install_failed = True                           
                
                logging.info(f"Successfully installed package [{pkg_full_name}]")
            else:
                logging.debug(f"Package [{pkg_full_name}] is already installed.")
        if pkg_install_failed:
            raise Exception("Failed to install some packages.")
    except Exception:
        raise


def installDependentPackages():
    common_pkgs = {
        "stompest": "stompest-2.3.0.tar.gz",
        "datefinder": "datefinder-0.6.3-py2.py3-none-any.whl",
        "nameparser": "nameparser-0.5.4-py2.py3-none-any.whl",
        "dill": "dill-0.2.7.1.tar.gz",
        "termcolor": "termcolor-1.1.0.tar.gz",
        "expletives": "expletives-0.0.6.tar.gz",
        "ptvsd": "ptvsd-3.2.1-py2.py3-none-any.whl",
        "flashtext": "flashtext-2.7.tar.gz",
        "unidecode": "Unidecode-1.0.22-py2.py3-none-any.whl",
        "portalocker": "portalocker-1.2.1-py2.py3-none-any.whl",
        "ilock": "ilock-1.0.1.tar.gz",
        "wordsegment": "wordsegment-1.3.1-py2.py3-none-any.whl",
        "phonenumbers": "phonenumbers-8.10.6-py2.py3-none-any.whl",
        "simhash": "simhash-1.10.2-py3-none-any.whl",
        "mlflow": "mlflow-1.6.0-py3-none-any.whl",
        "dacite": "dacite-1.5.0-py3-none-any.whl",
        "xmltodict": "xmltodict-0.12.0-py2.py3-none-any.whl",
    }
    window_pkgs = {"re2": "re2-0.2.23-cp37-cp37m-win_amd64.whl"}
    linux_pkgs = {"lxml.etree": "lxml-4.2.5-cp37-cp37m-manylinux1_x86_64.whl"}
    logging.info("Installing packages using pip installer.")
    try:
        install_pip_packages(common_pkgs)
        if is_linux() == True:
            install_pip_packages(linux_pkgs)
        else:
            install_pip_packages(window_pkgs)
    except Exception as e:
        raise Exception(f"Failed to install some pip packages. Exception {e}")


def isVersionUpdate(pkg_name, pkg_full_name):
    """ Check if the current wheel version is different than installed version """
    try:
        installed_version = ""
        try:
            import pkg_resources

            installed_version = pkg_resources.get_distribution(pkg_name).version
        except:
            # if package is already installed and we are not able to get the existing version,
            # then avoid installing it again
            return False
        current_version = re.compile(r"[\d\.]+\d").findall(pkg_full_name)
        logging.info(
            "Installed Version {} Current Version {} library {}".format(
                installed_version, current_version, pkg_name
            )
        )
        if len(current_version) > 0 and current_version[0] != installed_version:
            return True
        return False
    except:
        return True


def updateCondaPackages():
    try:
        logging.info("Installing packages using conda installer.")
        cwd = os.getcwd()
        pkgpath = Path("../ContentAnalyzer/bin/")
        SPACE_PRESENT = False
        pkgs = {}
        # Todo check if this issue is there on linux or not
        if is_linux() == False and len(cwd.split()) > 1:
            SPACE_PRESENT = True
            LOCAL_CV_CONTENTANALYZER_PATH = "%APPDATA%\\CVContentAnalyzer"
            logging.info(
                f"Making a symbolic link between {str(pkgpath)} and {LOCAL_CV_CONTENTANALYZER_PATH}"
            )
            try:
                subprocess.call(["rmdir", LOCAL_CV_CONTENTANALYZER_PATH], shell=True)
            except Exception as e:
                logging.error(f"Couldn't delete existing symbolic link. Exception {e}")
                raise
            try:
                cm = "mklink /J {0} {1}".format(
                    LOCAL_CV_CONTENTANALYZER_PATH, '"' + cwd + "\\" + str(pkgpath) + '"'
                )
                os.system(cm)
            except Exception as e:
                logging.error(
                    "Couldn't create symbolic link to {0} from {1}. Exception {2}".format(
                        '"' + cwd + "\\" + str(pkgpath) + '"', LOCAL_CV_CONTENTANALYZER_PATH, e
                    )
                )
                raise
        if is_linux() == False:
            pkgs = {
                "regex": "regex-2018.11.22-py37hfa6e2cd_1000.tar.bz2",
                "cryptography": "cryptography-vectors-2.3.1-py37_1000.tar.bz2",
                "cymem": "cymem-2.0.2-py37h74a9793_0.tar.bz2",
                "dill": "dill-0.2.8.2-py37_1000.tar.bz2",
                "msgpack_numpy": "msgpack-numpy-0.4.3.2-py_0.tar.bz2",
                "murmurhash": "murmurhash-1.0.0-py37h6538335_0.tar.bz2",
                "OpenSSL": "openssl-1.0.2o-vc14_0.tar.bz2",
                "preshed": "preshed-2.0.1-py37h33f27b4_0.tar.bz2",
                "thinc": "thinc-6.12.0-py37hb854c30_0.tar.bz2",
                "ujson": "ujson-1.35-py37hfa6e2cd_1001.tar.bz2",
                "future": "future-0.17.0-py37_1000.tar.bz2",
                "plac": "plac-1.0.0-py_0.tar.bz2",
                "spacy": "spacy-2.0.16-py37hcce6980_0.tar.bz2",
                # dependency for mlflow
                "docker-py": "docker-py-4.1.0-py37_1.tar.bz2",
                "libprotobuf": "libprotobuf-3.7.1-h7bd577a_0.tar.bz2",
                "protobuf": "protobuf-3.7.1-py37h33f27b4_0.tar.bz2",
                "websocket-client": "websocket-client-0.57.0-py37_0.tar.bz2",
            }
        else:
            pkgs = {
                # linux dependency for mlflow
                "click": "click-6.7-py37_0_linux_64.tar.bz2",
                "itsdangerous": "itsdangerous-0.24-py37_1_linux_64.tar.bz2",
                "pyyaml": "pyyaml-3.13-py37h14c3975_0_linux_64.tar.bz2",
                "cloudpickle": "cloudpickle-0.5.5-py37_0_linux_64.tar.bz2",
                "jinja2": "jinja2-2.10-py37_0_linux_64.tar.bz2",
                "sqlalchemy": "sqlalchemy-1.2.11-py37h7b6447c_0_linux_64.tar.bz2",
                "docker": "docker-py-4.1.0-py37_1_linux_64.tar.bz2",
                "libprotobuf": "libprotobuf-3.7.1-hd408876_0_linux_64.tar.bz2",
                "websocket": "websocket-client-0.57.0-py37_0_linux_64.tar.bz2",
                "entrypoints": "entrypoints-0.2.3-py37_2_linux_64.tar.bz2",
                "markupsafe": "markupsafe-1.0-py37h14c3975_1_linux_64.tar.bz2",
                "werkzeug": "werkzeug-0.14.1-py37_0_linux_64.tar.bz2",
                "flask": "flask-1.0.2-py37_1_linux_64.tar.bz2",
                "gunicorn": "gunicorn-20.0.4-py37_0_linux_64.tar.bz2",
                "protobuf": "protobuf-3.7.1-py37he6710b0_0_linux_64.tar.bz2",
                "_ssl": "openssl-1.1.1-h7b6447c_0_linux_64.tar.bz2",
            }
        common_pkgs = {
            # dependency for mlflow
            "alembic": "alembic-1.3.2-py_0.tar.bz2",  # same for linux
            "databricks-cli": "databricks-cli-0.9.1-py_0.tar.bz2",  # same for linux
            "docker-pycreds": "docker-pycreds-0.4.0-py_0.tar.bz2",  # same for linux
            "mako": "mako-1.1.0-py_0.tar.bz2",  # same for linux, but os versions avaialable
            "querystring_parser": "querystring_parser-1.2.4-py_0.tar.bz2",  # same for linux
            "sqlparse": "sqlparse-0.3.0-py_0.tar.bz2",  # same for linux, but os versions avaialable
            "tabulate": "tabulate-0.8.6-py_0.tar.bz2",  # same for linux, but os versions avaialable
            "waitress": "waitress-1.4.3-py_0.tar.bz2",  # same for linux
        }
        pkgs.update(common_pkgs)
        conda_install_failed = False
        for pkg in pkgs:
            try:
                if not isPkgInstalled(pkg) or is_repair_install:
                    logging.info("Installing package {}".format(pkg))
                    if SPACE_PRESENT:
                        exit_code = os.system(
                            "conda install --offline -f {0}".format(
                                LOCAL_CV_CONTENTANALYZER_PATH + "\\" + pkgs[pkg]
                            )
                        )
                    else:
                        if Path(os.getcwd()) == Path(cwd):
                            os.chdir(pkgpath)
                        exit_code = subprocess.call(
                            ["conda", "install", "--offline", "-f", pkgs[pkg]]
                        )
                    if exit_code != 0:
                        logging.error(f"Failed to install conda package {pkg}")
                        conda_install_failed = True
                else:
                    logging.info(f"Package {pkg} is already installed.")
            except Exception as e:
                raise Exception("Failed to install conda package {pkg}")
        if conda_install_failed:
            raise Exception("Failed to install some conda packages.")
        """
            we have built a separate so file for re2 python, as it is not packaged
            we will be manually copying it to /opt/Anaconda/lib/python3.7/site-packages/
            after copying we need to update the LD_LIBRARY_PATH to /opt/Anaconda/lib
            this will allow pyre2 to get libre2.so which is a c++ library
        """
        if is_linux() == True:
            try:
                logging.info("Checking if re2 installed or not")
                if not isPkgInstalled("re2") or is_repair_install:
                    if Path(os.getcwd()) == Path(cwd):
                        os.chdir(pkgpath)
                    pyre2_so = "re2.cpython-37m-x86_64-linux-gnu.so"
                    anaconda_install_path = loadRegValue(
                        "anacondaInstallPath", "/opt/Anaconda", type=str
                    )
                    dest_dir = Path(anaconda_install_path) / "lib" / "python3.7" / "site-packages"
                    if dest_dir.exists():
                        shutil.copy(pyre2_so, dest_dir)
                    # LD_LIBRARY_PATH is getting set in CvPythonWorker.sh
                    logging.info("Successfully installed re2 library")
                else:
                    logging.info("re2 library is already installed.")
            except Exception as e:
                raise Exception(f"Unable to install re2 package. Exception {e}")
    except Exception:
        raise
    finally:
        os.chdir(Path(cwd))


def setCWD():
    module_path = os.path.dirname(os.path.realpath(__file__))
    os.chdir(module_path)


def uninstallservice(servicename):
    svc_mgr = None
    svc_handle = None

    try:
        svc_mgr = win32service.OpenSCManager(None, None, win32service.SC_MANAGER_ALL_ACCESS)
        svc_handle = win32service.OpenService(svc_mgr, servicename, win32service.SERVICE_ALL_ACCESS)
        win32service.DeleteService(svc_handle)
        logging.info("Deleted service %s" % servicename)
    except:
        logging.error(f"Failed to delete service {servicename}")
    finally:
        if svc_handle != None:
            svc_handle.close()

        if svc_mgr != None:
            svc_mgr.close()


def deleteOldRegKeys(instance):
    # Function to delete old registry keys in Base path
    logging.info("Deleting old registry keys if present")

    try:
        # Get list of vals to delete
        valsToDelete = [
            "sEECheckExtractedEntities",
            "sEEClientRestartTime",
            "sEELogBackupCount",
            "sEELogLevel",
            "sEELogMaxBytes",
            "sEEMaxThreads",
            "sEENumClients",
            "sEETimeout",
            "sEEHardDebug",
            "sEELowPerfMode",
        ]

        # Connect to Registry, and navigate to required key
        aReg = ConnectRegistry(None, HKEY_LOCAL_MACHINE)
        regPath = "SOFTWARE\\Commvault Systems\\Galaxy\\{0}\\Base".format(instance)
        aKey = OpenKey(aReg, regPath, 0, KEY_ALL_ACCESS)

        # Iterate over list, and delete each value if it exists
        for val in valsToDelete:
            try:
                DeleteValue(aKey, val)
            except Exception as e:
                # Value is probably already deleted
                pass
    except Exception as e:
        logging.error("Error while deleting old keys: {}".format(e))


def deleteOldAMQDirectories():
    logging.info("Deleting ActiveMQ Directories if present")
    activemq = "..\\ContentAnalyzer\\apache-activemq"
    activemq_data = "..\\ContentAnalyzer\\activemq-data"
    logging.info(
        "Path to activemq directory is {0} and activemq data directory is {1}".format(
            activemq, activemq_data
        )
    )
    try:
        if "activemq" in activemq and "activemq" in activemq_data:
            shutil.rmtree(activemq, ignore_errors=True)
            shutil.rmtree(activemq_data, ignore_errors=True)
    except Exception as e:
        logging.error(f"Failed to delete old ActiveMQ directories. Exception {e}")


if __name__ == "__main__":

    if len(sys.argv) > 2 and sys.argv[1] == "msgQ":
        command = sys.argv[2]
        setCWD()
        msgQueueClient = CommandSender()
        response = msgQueueClient.send(command)
        logging.info(f"msgQ command response {response}")
    else:
        logging.info("=" * 100)
        logging.info("=" * 100)
        setCWD()
        logging.info("Working directory set to: {}".format(os.getcwd()))
        if is_linux() == True:
            try:
                get_cvreg_linux()
                getInstanceName()
            except Exception as e:
                logging.error(f"Exception occurred. {e}")

        current_library_version = 8
        # get previous library_version
        last_library_version = loadRegValue("lastLibraryVersion", 0, type=int)
        logging.info(
            f"Current library version {current_library_version} | Previous library version {last_library_version}"
        )
        """
            do package installations in case last library version is less than current library version
            we will increase the library version in case new python packages are getting addded
            this is to avoid running these operations for each CE service restart 
        """
        is_repair_install = loadRegValue("repairLibraryInstall", 0, type=int)
        if current_library_version > last_library_version or is_repair_install:
            logging.info("Proceeding with the installation")
            try:
                extract_nltk_models()
                installDependentPackages()
                updateCondaPackages()
                installSpacyModel()
                installDocumentClassifierModels()
                if is_linux() == False:
                    deleteOldAMQDirectories()
                    deleteOldRegKeys(getInstanceName())
                    uninstallservice("CVPyAnalyticsService({})".format(getInstanceName()))

                # update library version key to current version
                setRegValue("lastLibraryVersion", current_library_version, type=int)
                setRegValue("repairLibraryInstall", 0, type=int)
            except Exception as e:
                logging.exception(f"Exception occured. {e}")
                sys.exit(1)
        else:
            logging.info("No new packages to install")
