import re
import os
from CvCAGenericLogger import get_logger_handler
from CvEEConfigHelper import getBaseDir, HELPER_DLL, ROTATING_BACKUP_COUNT, ROTATING_MAX_BYTES

logger_options = {
    "ROTATING_BACKUP_COUNT": ROTATING_BACKUP_COUNT,
    "ROTATING_MAX_BYTES": ROTATING_MAX_BYTES,
}

logger = get_logger_handler(
    os.path.join(getBaseDir(), HELPER_DLL), "ContentAnalyzer", logger_options
)

# common util codes
def luhn_validation_algorithm(num):
    """
        get single digit error with check digit validation
        https://en.wikipedia.org/wiki/Luhn_algorithm
    """
    num = str(num)
    lookup = [0, 2, 4, 6, 8, 1, 3, 5, 7, 9]
    evens = sum(int(i) for i in num[-1::-2])
    odds = sum(lookup[int(i)] for i in num[-2::-2])
    return (evens + odds) % 10 == 0


##############################################################################################

# Entity validations below


def validateGreeceAFM(afm):
    try:
        if afm == "000000000":
            return False

        m = 1
        sum = 0
        for i in range(7, -1, -1):
            m *= 2
            sum += int(afm[i]) * m

        return sum % 11 % 10 == int(afm[8])
    except Exception as e:
        logger.error(f"Failed to validate greece afm {afm}. Exception {e}")
        return True


def validateUKNHS(nhs):
    try:
        nhs = nhs.replace("-", "")
        nhs = nhs.replace(" ", "")
        m = 10
        sum = 0
        for i in range(0, 9):
            sum += int(nhs[i]) * m
            m -= 1
        return 11 - (sum % 11) == int(nhs[9])
    except Exception as e:
        logger.error(f"Failed to validate uk nhs {nhs}. Exception {e}")
        return True


def validateDutchSSN(ssn):
    try:
        ssn = ssn.replace(".", "")
        ssn = ssn.replace(" ", "")
        coefficients = [9, 8, 7, 6, 5, 4, 3, 2, -1]
        if len(ssn) != 9:
            return False
        sum = 0
        for i in range(9):
            try:
                sum += int(ssn[i]) * coefficients[i]
            except:
                return False
        return (sum % 11) == 0
    except Exception as e:
        logger.error(f"Failed to validate dutch ssn {ssn}. Exception {e}")
        return True


def validateDate(date):
    try:
        test_reg = re.compile("\d{4}", re.DOTALL | re.IGNORECASE | re.MULTILINE)
        found_years = [int(d) for d in test_reg.findall(date)]
        for f in found_years:
            if f > 2200 or f < 1200:
                return False
        return True
    except Exception as e:
        logger.error(f"Failed to validate date {date}. Exception {e}")
        return True


def validateIBAN(iban):
    try:
        # change country code to digits
        iban_digits = ""
        for ch in iban:
            if ch.isalpha():
                iban_digits += str((ord(ch) - (65 if ch.isupper() else 97)) + 10)
            elif ch.isdigit():
                iban_digits += ch
        iban_digits = iban_digits[6:] + iban_digits[:6]
        iban_long_value = int(iban_digits)
        if iban_long_value % 97 == 1:
            return True
        return False
    except Exception as e:
        logger.error(f"Failed to validate iban {iban}. Exception {e}")
        return True


def validateCCN(ccn):
    """ credit card validation based on Luhn algorithm
        https://en.wikipedia.org/wiki/Luhn_algorithm
    """
    try:
        # make sure string
        ccn = str(ccn)
        ccn = "".join(ccn.split(" "))
        ccn = "".join(ccn.split("-"))
        return luhn_validation_algorithm(ccn)
    except Exception as e:
        logger.error(f"Failed to validate credit card number {ccn}. Exception {e}")
        return True


def validate_pesel(pesel):
    """ pesel check sum validation
        https://en.wikipedia.org/wiki/PESEL

        Having a PESEL in the form of ABCDEF GHIJK, one can check the validity of the number by computing the following expression:
        A*1 + B*3 + C*7 + D*9 + E*1 + F*3 + G*7 + H*9 + I*1 + J*3
    """
    try:
        # make sure string
        pesel = str(pesel)
        # convert all digits to integer
        pesel = [int(digit) for digit in pesel if ord(digit) in range(48, 58)]
        multipliers = [1, 3, 7, 9, 1, 3, 7, 9, 1, 3]
        if len(pesel) != len(multipliers) + 1:
            return False
        products_sum = sum([digit * multipliers[idx] for idx, digit in enumerate(pesel[:-1])])
        modulo = products_sum % 10
        last_digit = pesel[-1]
        return (modulo == 0 or modulo == 5) and last_digit == modulo or last_digit == 10 - modulo
    except Exception as e:
        logger.error(f"Failed to validate pesel {pesel}. Exception {e}")
        return True


def validate_austalia_medical_number(medical_number):
    """
        check sum validation for australia medical number
        https://www.clearwater.com.au/code/medicare, 
        https://stackoverflow.com/questions/3589345/how-do-i-validate-an-australian-medicare-number
        example, 20000000000, 2234567891
        regex used by australian_medical_account --> [2-6]\\d{9}\\d? 
        First digit in range of 2 to 6
        Ninth digit is check digit
        Algorithm:
        Pattern --> ABCDEFGHIJK
        (A*1 + B*3 + C*7 + D*9 + E*1 + F*3 + G*7 + H*9) % 10 == I

    """
    try:
        medical_number = str(medical_number)
        weights = [1, 3, 7, 9, 1, 3, 7, 9]
        check_digit = int(medical_number[8])
        checksum = sum([int(digit) * weight for digit, weight in zip(medical_number[:8], weights)])
        return checksum % 10 == check_digit
    except Exception as e:
        logger.error(
            f"Failed to validate australian medical number {medical_number}. Exception {e}"
        )
        return True


def validate_australia_tax_file_number(tax_number):
    """
        check sum validation for australia tax file number
        https://www.clearwater.com.au/code/tfn, 
        https://en.wikipedia.org/wiki/Tax_file_number
        example, 876 543 210
        regex used by australian_tax_file_number --> \\d{3}\\s?\\d{3}\\s?\\d{2}\\d?        
        Algorithm:
        Pattern --> ABCDEFGHI
        (A*1 + B*4 + C*3 + D*7 + E*5 + F*8 + G*6 + H*9 + I*10) % 11 == 0
    """
    try:
        tax_number = str(tax_number)
        tax_number = "".join(tax_number.split(" "))
        n_digits = len(tax_number)
        weights = [1, 4, 3, 7, 5, 8, 6, 9, 10]
        check_digit = 0
        checksum = sum(
            [
                int(digit) * weight
                for digit, weight in zip(tax_number[:n_digits], weights[:n_digits])
            ]
        )
        return checksum % 11 == check_digit
    except Exception as e:
        logger.error(f"Failed to validate australian tax file number {tax_number}. Exception {e}")
        return True


def validate_canadian_sin(canadian_sin):
    """
        check sum validation for canadian social insurance number
        https://answers.laserfiche.com/questions/118797/Validate-a-Canadian-SIN-Social-Insurance-Number, 
        http://www.straightlineinternational.com/docs/vaildating_canadian_sin.pdf
        example, 130 692 544, 130-692-544, 130692544
        regex used by canadian_sin --> \\d{3}[-\\s]?\\d{3}[\\s-]?\\d{3}        
        Algorithm:
        Pattern --> ABC DEF GHI
        luhn validation will be enough here.        
    """
    try:
        canadian_sin = str(canadian_sin)
        canadian_sin = "".join(canadian_sin.split(" "))
        canadian_sin = "".join(canadian_sin.split("-"))
        return luhn_validation_algorithm(canadian_sin)
    except Exception as e:
        logger.error(f"Failed to validate canadian SIN {canadian_sin}. Exception {e}")
        return True


def validate_finnish_hetu(finnish_hetu):
    """
        check sum validation for finland personal identity code also known as hetu
        http://id-check.artega.biz/info-fi.php        
        example, 311280-999J
        regex used by finland_hetu --> [0-9]{2}\\.?[0,1][0-9]\\.?[0-9]{2}[-+A][0-9]{3}[\dA-Z]
        Algorithm:
        Pattern --> DDMMYYCZZZQ
        '0123456789ABCDEFHJKLMNPRSTUVWXY'[(DDMMYYZZZ % 31)] == Q
    """
    try:
        lookup = "0123456789ABCDEFHJKLMNPRSTUVWXY"
        finnish_hetu = str(finnish_hetu)
        finnish_hetu = "".join(finnish_hetu.split(" "))
        finnish_hetu = "".join(finnish_hetu.split("."))
        check_digit = finnish_hetu[-1]
        finnish_hetu = finnish_hetu[:6] + finnish_hetu[7:10]
        return lookup[int(finnish_hetu) % 31].upper() == check_digit.upper()
    except Exception as e:
        logger.error(f"Failed to validate finnish hetu {finnish_hetu}. Exception {e}")
        return True


def validate_french_insee(french_insee):
    """
        check sum validation for french national identification number
        https://en.wikipedia.org/wiki/INSEE_code,
        https://fr.wikipedia.org/wiki/Num%C3%A9ro_de_s%C3%A9curit%C3%A9_sociale_en_France        
        example, 2 95 10 99 126 111 93, 253072B07300470
        regex used by french_insee --> [1-2]\\s?[0-9]{2}\\s?[0-1][0-9]\\s?(?:\\d{2}|\\d[A-B]|\\d{3})\\s?\\d{2,3}\\s?\\d{3}\\s?\\d{2}
        Algorithm:
        Pattern --> ABCDEFGHIJKLM cc
        (97 - (ABCDEFGHIJKLM % 97)) == cc
        if FG == '2A':
            return 19
        elif FG == '2B':
            return 18
    """
    try:
        french_insee = str(french_insee).upper()
        french_insee = "".join(french_insee.split(" "))
        check_digit = int(french_insee[-2:])
        french_insee = french_insee[:-2]
        if french_insee[5:7] == "2A":
            french_insee = french_insee[:5] + "19" + french_insee[7:]
        elif french_insee[5:7] == "2B":
            french_insee = french_insee[:5] + "18" + french_insee[7:]
        return (97 - int(french_insee) % 97) == check_digit
    except Exception as e:
        logger.error(f"Failed to validate french insee {french_insee}. Exception {e}")
        return True


def validate_irish_ppsn(irish_ppsn):
    """
        check sum validation for irish personal public service number
        https://en.wikipedia.org/wiki/Personal_Public_Service_Number        
        example, 1234567FA
        regex used by ireland_ppsn --> ([0-9]{7}[A-Z]{1,2})
        Algorithm:
        Pattern --> ABCDEFGHI
        'A' + ([ A*8 + B*7 + C*6 + D*5 + E*4 + F*3 + G*2 + 9*(alphabet_index of I) ]) % 23 == H
    """
    try:
        irish_ppsn = str(irish_ppsn).upper()
        ninth_digit = (ord(irish_ppsn[8]) - 64) if len(irish_ppsn) == 9 else 0
        check_digit = irish_ppsn[7]
        irish_ppsn = irish_ppsn[:7]
        weights = [8, 7, 6, 5, 4, 3, 2]
        checksum = (
            sum([int(val) * weight for val, weight in zip(irish_ppsn, weights)]) + 9 * ninth_digit
        )
        remainder = checksum % 23
        if remainder == 0:
            remainder = 23
        return chr(64 + remainder) == check_digit
    except Exception as e:
        logger.error(f"Failed to validate irish ppsn {irish_ppsn}. Exception {e}")
        return True


def validate_indian_aadhaar(aadhaar):
    """
        check digit validation based on Verhoeff Algorithm
        https://www.npci.org.in/sites/default/files/circular/Circular_No_9.pdf
        https://en.wikipedia.org/wiki/Verhoeff_algorithm
        example, 4755 8766 9949
        regex used by indian_aadhaar --> (?:\\d{4}[^\\n\\S]?\\d{4}[^\\n\\S]?\\d{4})
    """
    try:
        aadhaar = str(aadhaar)
        aadhaar = "".join(aadhaar.split())
        diahedral_group = [
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            [1, 2, 3, 4, 0, 6, 7, 8, 9, 5],
            [2, 3, 4, 0, 1, 7, 8, 9, 5, 6],
            [3, 4, 0, 1, 2, 8, 9, 5, 6, 7],
            [4, 0, 1, 2, 3, 9, 5, 6, 7, 8],
            [5, 9, 8, 7, 6, 0, 4, 3, 2, 1],
            [6, 5, 9, 8, 7, 1, 0, 4, 3, 2],
            [7, 6, 5, 9, 8, 2, 1, 0, 4, 3],
            [8, 7, 6, 5, 9, 3, 2, 1, 0, 4],
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
        ]
        permutation_table = [
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            [1, 5, 7, 6, 2, 8, 3, 0, 9, 4],
            [5, 8, 0, 3, 7, 9, 6, 1, 4, 2],
            [8, 9, 1, 6, 0, 4, 3, 5, 2, 7],
            [9, 4, 5, 3, 1, 2, 6, 8, 7, 0],
            [4, 2, 8, 6, 5, 7, 3, 9, 0, 1],
            [2, 7, 9, 3, 8, 0, 6, 4, 1, 5],
            [7, 0, 4, 6, 9, 1, 3, 2, 5, 8],
        ]
        # reverse aadhar number
        aadhaar = aadhaar[::-1]
        c = 0
        for i, digit in enumerate(aadhaar):
            c = diahedral_group[c][permutation_table[(i % 8)][int(digit)]]
        return c == 0
    except Exception as e:
        logger.error(f"Failed to validate indian aadhaar {aadhaar}. Exception {e}")
        return True


def filter_phone_entity(phone):
    """
        we are reporting too many false positives when we are simply 
        detecting continuous digits        
    """
    try:
        phone = str(phone)
        num_groups = len(re.split("[- ]", phone))
        if num_groups == 1:
            return False
    except Exception as e:
        return True
    return True

def validate_south_africa_id(south_african_id):
    """
        https://en.wikipedia.org/wiki/South_African_identity_card
        https://mybroadband.co.za/news/security/303812-what-your-south-african-id-number-means-and-what-it-reveals-about-you.html

        Pattern: YYMMDDSSSSCAZ
        Regex defined: (?:\\d{2}[0-1]\\d[0-3]\\d{5}[0-1]\\d{2})
        Validation Algorithm: Luhn validation
    """
    try:
        south_african_id = str(south_african_id)
        south_african_id = "".join([i for i in south_african_id if i.isdigit()])
        return luhn_validation_algorithm(south_african_id)
    except Exception as e:
        logger.error(f"Failed to validate south african ID {south_african_id}. Exception {e}")
        return True

def validate_date(date_):
    """
        need to verify numeric dates for validation
        1. verify that date is actually numeric only
        2. split date and return false if any value is of 3 digits
        3. try to find day, month and year based on their value
        4. if more than one value qualifies for year (drop it)
        5. if more than one value qualifies for day (drop it)        
    """
    def verify_day_month(date_list_idx_1, date_list_idx_2):
        date_list_idx_1 = int(date_list_idx_1)
        date_list_idx_2 = int(date_list_idx_2)
        if date_list_idx_1 == 0 or date_list_idx_2 == 0:
            return False
        if date_list_idx_1 < 32 and date_list_idx_2 < 13:
            return True
        elif date_list_idx_2 < 32 and date_list_idx_1 < 13:
            return True
        return False

    def verify_day_year(date_list_idx_1, date_list_idx_2):
        date_list_idx_1 = int(date_list_idx_1)
        date_list_idx_2 = int(date_list_idx_2)
        if date_list_idx_1 == 0 or date_list_idx_2 == 0:
            return False
        if date_list_idx_1 > 31 and date_list_idx_2 > 31:
            return False    
        return True

    try:
        date_list = re.split("[\\/.-]", date_)
        is_keyword_date = False
        for num in date_list:
            if num.isnumeric() is False:
                is_keyword_date = True
            elif num.isnumeric() is True and len(num) == 3:
                return False
        if is_keyword_date is False:
            if len(date_list) != 3:
                return False
            if int(date_list[2]) > 999: # year is found, verify 0 and 1 now
                return verify_day_month(date_list[0], date_list[1])
            elif int(date_list[0]) > 999: # year is found, verify 1 and 2 now
                return verify_day_month(date_list[1], date_list[2])
            else: # for numeric dates, we will end up with lots of false positive, if combinations of only twos are allowed
                return False
        else:
            date_list = re.split("(?:[ ,\\/.-]|[A-Za-z]+)", date_)
            date_num_list = []
            for word in date_list:
                if word.isnumeric():
                    date_num_list.append(word)
                elif word.isalnum() and not word.isalpha():
                    date_num_list.append(word[:-2])            
            return verify_day_year(date_num_list[0],date_num_list[1])
    except Exception as e:
        logger.debug(f"Failed to classify date {date_}. Exception [{e}]")
        return False