# -*- coding: utf-8 -*-

import ujson
import time
import requests
import sys
import datetime
import threading
from threading import Thread, Event, Timer
import configparser
import os
from core import *
import socket
import hashlib
import operator
import collections
import xml.etree.ElementTree as ET
import time
from CvEEConfigHelper import *
from CvEEMsgQueueHandler import *
from CvCAGenericLogger import *
from CvCAPerfCounters import PerformanceCounter

try:
    from winreg import *
except:
    pass

# Global Variables

PROCESS_ID = os.getpid()
THREAD_ID = threading.current_thread().ident

PROCESSED_BATCHES = {}
CHECK_FOR_NEXT_BATCH = {}
PUBLISH_BATCH = Event()
STOP_PROCESS = Event()
SOLR_ALIVE_TIMER = None
GLOBAL_FP = {}
LOGGER_Publisher = None

SOLR_RUNNING = True
JOB_FINISHED = False
BATCHES = {}

ALL_DOCS = {}
SOLR_REQUEST_SESSIONS = {}


class csConfiguration:
    """
        Class to store configuration details for each subclient
        returned from the DB
    """

    def __init__(self):
        self.solrurl = ""
        self.patterns = {}
        self.entityRulerId = {}
        self.fp = {}
        self.sc = 0
        self.nodeClientId = 0

    def pprint(self):
        module_name = "CvEEJobPublisher"
        func_name = "pprint"
        func_string = "{}::{}() - ".format(module_name, func_name)
        LOGGER_Publisher.info("solrulr : {0}".format(self.solrurl), func_string)
        LOGGER_Publisher.info("patterns : {0}".format(self.patterns), func_string)
        LOGGER_Publisher.info("entityRulerId : {0}".format(self.entityRulerId), func_string)
        LOGGER_Publisher.info("fp : {0}".format(self.fp), func_string)


def getConfiguration(params):
    module_name = "CvEEJobPublisher"
    func_name = "getConfiguration"
    func_string = "{}::{}() - ".format(module_name, func_name)
    # Main function to store configuration details into a config object and return it

    conifg = None
    stopped = Event()

    # Note time of start of Job so as to pick up documents that are content indexed
    # after this in a later job, so as to avoid missing documents due to offset skip mismatch

    currentTime = datetime.datetime.utcnow().isoformat()
    clientName = getClientName()

    stompPort = params["stompPort"]
    check_failures = params["check_failures"]
    docs_per_batch = params["docs_per_batch"]
    batches_per_publish = params["batches_per_publish"]
    perfCounter = params["perfCounter"]

    LOGGER_Publisher.info("Client name " + clientName, func_string)
    LOGGER_Publisher.info("Stomp Port no. is {}".format(stompPort), func_string)

    # Query CS and retrieve XML
    # Then pass XML for parsing

    wt = 0
    while not stopped.wait(wt):
        csxml = queryCS(0, retry=True)
        LOGGER_Publisher.debug("CSXML " + str(csxml), func_string)
        if csxml is None or csxml.strip() == "":
            LOGGER_Publisher.error(
                "Failed to get clients. Please verify that Commserver is reachable."
            )
            wt = 5 * 60
            continue

        config = loadConfiguration(csxml)
        if config != None:
            break
        wt = 60 * 5

    # Iterate over returned configuration details
    # and store them in a meaningful way:
    # config : {
    #   'q_params': ''
    #   'publish_interval': 0,
    #   'solrInfo': {
    #     '<solrurl>': {
    #       'solrurl': '<solrurl>' + /solr
    #       'solrurl1': '<solrurl>'
    #       'scID': [<scIDs>]
    #       'entities': {
    #           <scID>: [<entities>],
    #           ...
    #        }
    #     },
    #     ...
    #   }
    # }

    solrInfo = {}
    for solrurl, attributes in list(config.items()):
        solrDetails = {}
        solrDetails["solrurl"] = solrurl + "/solr"
        solrDetails["solrurl1"] = solrurl
        solrDetails["scID"] = attributes.sc
        solrDetails["fp"] = attributes.fp

        entities = {}
        for p in attributes.patterns:
            entities[p] = []
            for e in attributes.patterns[p]:
                entities[p].append(int(e))
        solrDetails["entities"] = entities

        solrInfo[solrurl] = solrDetails

    localConfig = {
        "q_params": ("127.0.0.1", stompPort, "EEQueue", "EERQueue"),
        "publish_interval": 0,
        "solrInfo": solrInfo,
        "check_failures": check_failures,
        "currentTime": currentTime,
        "docs_per_batch": docs_per_batch,
        "batches_per_publish": batches_per_publish,
        "perfCounter": perfCounter,
    }

    return config, localConfig


def loadConfiguration(configXml):
    # Read response from CS, process it and convert to XML
    # Then read XML and store into csConfiguration() object
    # Then add additional info and pass to getConfiguration()
    configuration = {}
    root = ET.fromstring(configXml)
    subClientIDs = []
    for ed in root.findall("entityDetails"):
        # XML format is as follows:
        #
        # <EntityDetails entityRulerId="2D590023-F439-4220-88D1-4FC5B06E64BF" entityId="9" subclientId="2"
        # appTypeId="33" nodeClientId="2" cloudId="6" forwardRefTime="2900-01-01T00:00:00" backwardRefTime="1900-01-01T00:00:00"
        # extractingClientId="2" flags="1" solrUrl="http://smujjiga-vm3:27000"/>

        solrurls = ed.get("solrUrl")
        if solrurls != None:
            solrurls = solrurls.split(",")
            for solrurl in solrurls:
                solrurl = solrurl.strip()
                scID = ed.get("subclientId")
                nodeClientId = ed.get("nodeClientId")
                if scID not in subClientIDs:
                    subClientIDs.append(scID)

                if solrurl not in configuration:
                    configuration[solrurl] = csConfiguration()

                config = configuration[solrurl]
                config.solrurl = solrurl
                config.nodeClientId = nodeClientId
                if scID not in config.entityRulerId:
                    config.entityRulerId[scID] = []
                if scID not in config.patterns:
                    config.patterns[scID] = []
                config.patterns[scID].append(int(ed.get("entityId")))
                config.entityRulerId[scID].append(ed.get("entityRulerId"))
                fp = datetime.datetime.strptime(ed.get("forwardRefTime"), "%Y-%m-%dT%H:%M:%S")
                if (scID in config.fp and config.fp[scID] > fp) or scID not in config.fp:
                    config.fp[scID] = fp

                config.sc = subClientIDs

    return configuration if len(configuration) > 0 else None


def updateCS(config, solrurl, fp, s):
    module_name = "CvEEJobPublisher"
    func_name = "updateCS"
    func_string = "{}::{}() - ".format(module_name, func_name)
    # Function to update CS DB using CVCIEntityExtractionCsApi.dll
    try:
        msg_skeleton = '<DM2ContentIndexing_EntityDetailsResp><instance instanceName="{0}"/><entity entityRulerId="{1}" nodeClientId="{2}" forwardRefTime="{3}" backwardRefTime="1900-01-01T00:00:00" numItemsProcessed="0"  totalEntitiesFound="0" totalDocsFound="0" /></DM2ContentIndexing_EntityDetailsResp>'
        fpstr = fp.split(".")[0]
        c = config[solrurl]
        mydll = cdll.LoadLibrary(get_helper_dll())
        nodeClientId = config[solrurl].nodeClientId
        for erid in c.entityRulerId[s]:
            LOGGER_Publisher.debug(
                "Update request to DB with SolrUrl {0}, node ID {1}, ForwardPointer {2} and entityRulerID {3}".format(
                    solrurl, nodeClientId, fp, erid
                ),
                func_string,
            )
            msg = msg_skeleton.format(getInstanceName(), erid, nodeClientId, fpstr)
            msg_c = c_char_p(msg.encode("utf-8"))
            LOGGER_Publisher.debug("XML sent to DB is {}".format(msg), func_string)
            r = mydll.SendMsg(msg_c)
            # mydll.freeHeapMemory(r)
    except:
        LOGGER_Publisher.exception("There was an error while updating DB", func_string)


class Publisher(Thread, GenericMsgQueueCommunicator):

    """
        Main class for publishing messages to ActiveMQ
    """

    PUBLISH_INTERVAL = 0
    Q_FULL_SLEEP_INTERVAL = 60
    Q_SIZE = 150
    NOTASK_SLEEP_INTERVAL = 1 * 60  # 30 minutes

    def __init__(self, config):
        Thread.__init__(self)
        GenericMsgQueueCommunicator.__init__(self)
        self._config = config
        self.clients = {"EEQueue": {"subscribe": False, "client": None}}
        self.queue = "EEQueue"
        self.host = config["q_params"][0] if "q_params" in config else "127.0.0.1"
        self.stompPort = config["q_params"][1] if "q_params" in config else 61650

        self.PUBLISH_INTERVAL = (
            config["publish_interval"] if "publish_interval" in config else self.PUBLISH_INTERVAL
        )

        self._stopped = Event()
        self._waitTime = self.PUBLISH_INTERVAL

        self._message_number = 0
        self._sc_message_number = {}

        self._Command = "wakeup"
        self.wakeup = False
        self.NO_DOCS_TO_PROCESS = config["docs_per_batch"]
        self.NO_BATCHES_TO_PROCESS = config["batches_per_publish"]
        self.perfCounter = config["perfCounter"]

    def publishMessage(self, solrurl):
        module_name = "CvEEJobPublisher"
        func_name = "publishMessage"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function to directly contact ActiveMQ and publish messages to EEQueue

        """Given solrurl, get list of subclients associated with the Solr
        Then, for each Subclient, check ALL_DOCS for messages present in offset
        self._sc_message_number, 100 docs at a time. Create a JSON structure for message
        and publish to ActiveMQ
        Then store the published batch info in BATCHES to track for lost batches"""

        try:
            global BATCHES, ALL_DOCS, PUBLISH_BATCH
            if self.wakeup:
                self.wakeup = False

            scIDs = list(GLOBAL_FP[solrurl].keys())[:]
            entities = self._config["solrInfo"][solrurl]["entities"]
            start = GLOBAL_FP.copy()

            if solrurl in ALL_DOCS:
                for s in scIDs[:]:
                    if s in ALL_DOCS[solrurl]:
                        if solrurl not in self._sc_message_number:
                            self._sc_message_number[solrurl] = {}
                        if s not in self._sc_message_number[solrurl]:
                            self._sc_message_number[solrurl][s] = 0

                        while self._sc_message_number[solrurl][s] * self.NO_DOCS_TO_PROCESS < len(
                            ALL_DOCS[solrurl][s]
                        ):
                            self._message_number += 1
                            self._sc_message_number[solrurl][s] += 1
                            LOGGER_Publisher.debug(
                                "Message Number for subclient {} in Solr {} is {} and length of All Docs is {}".format(
                                    s,
                                    solrurl,
                                    self._sc_message_number[solrurl][s],
                                    len(ALL_DOCS[solrurl][s]),
                                ),
                                func_string,
                            )

                            docs = ALL_DOCS[solrurl][s][
                                (self._sc_message_number[solrurl][s] - 1)
                                * self.NO_DOCS_TO_PROCESS : self._sc_message_number[solrurl][s]
                                * self.NO_DOCS_TO_PROCESS
                            ]
                            ids = [doc[0] for doc in docs]
                            times = [doc[1].isoformat() for doc in docs]
                            if len(ids) <= 0:
                                LOGGER_Publisher.debug(
                                    "Offset for subclient {} in Solr {} is {}".format(
                                        s,
                                        solrurl,
                                        (self._sc_message_number[solrurl][s] - 1)
                                        * self.NO_DOCS_TO_PROCESS,
                                    ),
                                    func_string,
                                )
                                self._sc_message_number[solrurl][s] -= 1
                                break

                            message = {}
                            message["batchno"] = self._message_number
                            message["sc_batchno"] = (
                                str(s) + ":" + str(self._sc_message_number[solrurl][s])
                            )
                            message["scID"] = s
                            message["offset"] = (
                                self._sc_message_number[solrurl][s] - 1
                            ) * self.NO_DOCS_TO_PROCESS
                            message["start"] = start[solrurl][s].isoformat()
                            message["end"] = times[-1]
                            message["entities"] = entities[s]
                            message["endTime"] = self._config["currentTime"]
                            message["ids"] = ids
                            message["solrurl"] = solrurl

                            jsonMsg = ujson.dumps(message, ensure_ascii=False)
                            self.perfCounter.start_stopwatch("PublisherMsgQueueEnqueue")
                            self.sendOnConnect(self.queue, jsonMsg.encode("utf-8"))
                            self.perfCounter.stop_stopwatch("PublisherMsgQueueEnqueue")

                            self._waitTime = self.PUBLISH_INTERVAL
                            if solrurl not in BATCHES:
                                BATCHES[solrurl] = {}
                            if s not in BATCHES[solrurl]:
                                BATCHES[solrurl][s] = {}
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])] = {}
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])][
                                "start"
                            ] = start[solrurl][s].isoformat()
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])][
                                "publishedTime"
                            ] = datetime.datetime.utcnow()
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])][
                                "offset"
                            ] = (self._sc_message_number[solrurl][s] - 1) * self.NO_DOCS_TO_PROCESS
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])][
                                "contentIds"
                            ] = ids
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])][
                                "extractingat"
                            ] = times
                            BATCHES[solrurl][s][str(self._sc_message_number[solrurl][s])][
                                "solrurl"
                            ] = solrurl

                            LOGGER_Publisher.debug(
                                "Docs for batch {0} are {1} and end time is: {2}".format(
                                    message["sc_batchno"], ids, message["end"]
                                ),
                                func_string,
                            )
        except:
            LOGGER_Publisher.exception("Couldn't publish message", func_string)

    def stop(self):
        module_name = "CvEEJobPublisher.Publisher"
        func_name = "stop"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Stop event to break out of while loop in the run() function
        LOGGER_Publisher.info("Stopping the producer", func_string)
        self._Command = "stop"
        self._stopped.set()

    def wakeUp(self):
        module_name = "CvEEJobPublisher"
        func_name = "wakeUp"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Not used anymore. Remove after checking all references
        LOGGER_Publisher.info("Wakingup the producer", func_string)
        self._Command = "wakeup"
        self._stopped.set()

    def run(self):
        module_name = "CvEEJobPublisher.Publisher"
        func_name = "run"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Main function of the publishing thread. Connect to ActiveMQ,
        # Call publishMessage() for each SolrUrl infinitely
        # Stop when a stop event is set. Does nothing if PUBLISH_BATCH event is not set.
        global PUBLISH_BATCH, AMQ_CONNECT_WAIT_TIME
        LOGGER_Publisher.info("Started Publisher", func_string)

        self.connectToQueue()

        while True:
            while not PUBLISH_BATCH.isSet():
                pass
            LOGGER_Publisher.debug(
                "Publish batch event is now set. Publishing batches.", func_string
            )
            solrurls = list(self._config["solrInfo"].keys())
            for solrurl in solrurls:
                self.perfCounter.start_stopwatch("PublishAllMessages")
                self.publishMessage(solrurl)
                self.perfCounter.stop_stopwatch("PublishAllMessages")
                LOGGER_Publisher.debug(
                    "Finished Publishing batches for solr {}".format(solrurl), func_string
                )
            LOGGER_Publisher.debug("Clearing Publish Batch event.", func_string)
            PUBLISH_BATCH.clear()

            if self._Command == "stop":
                break
            elif self._Command == "wakeup":
                self._waitTime = 0
                self._stopped = Event()
                self.wakeup = True

        self.disconnectClients()


class TaskManager:

    """
        Class which has several important helper function to help the publisher run smoothly
        and perform various initialisation tasks. Most of the Global variables used throughout the code
        are initialised here.
    """

    def __init__(self, config):
        module_name = "CvEEJobPublisher.TaskManager"
        func_name = "__init__"
        func_string = "{}::{}() - ".format(module_name, func_name)
        self._config = config
        self._lastExtractingat = datetime.datetime.utcfromtimestamp(0)
        self._updateTimer = datetime.datetime.utcfromtimestamp(0)
        self.scIDBackup = {}
        self._forwardPointer = {
            solrurl: config["solrInfo"][solrurl]["fp"] for solrurl in config["solrInfo"]
        }
        # self._forwardPointer = config['fp']
        self._forwardOffset = {}
        self._totalDocs = {}
        self.NO_DOCS_TO_PROCESS = config["docs_per_batch"]
        self.NO_BATCHES_TO_PROCESS = config["batches_per_publish"]
        self.DOCS_PER_QUERY = self.NO_DOCS_TO_PROCESS * self.NO_BATCHES_TO_PROCESS
        self.checkSolrAlive()
        self._gotDocs = Event()
        self.PUBLISH_ALL_SLEEP_TIME = 30 * 60
        self.perfCounter = config["perfCounter"]

        global GLOBAL_FP

        # For each Solr, get oldest doc for each subclient,
        # and store it in self._forwardPointer variable.
        # Then get All docs present in Solr for each subClient
        # and store them in ALL_DOCS variable

        solrurls = self._config["solrInfo"]
        LOGGER_Publisher.info(
            "Current time is {}. Setting this as last time till which documents will be queried for Entity Extraction.".format(
                self._config["currentTime"]
            ),
            func_string,
        )
        for solrurl, attributes in list(solrurls.items()):
            scIDs = attributes["scID"]
            for s in scIDs[:]:
                if s not in attributes["fp"]:
                    tempScID = attributes["scID"][:]
                    tempScID.remove(s)
                    attributes["scID"] = tempScID
                    LOGGER_Publisher.error(
                        "Couldn't find subclient {0} for solr {1} in config".format(s, solrurl),
                        func_string,
                    )
            for s in scIDs[:]:
                LOGGER_Publisher.info(
                    "Retrieving the oldest extracting time for subclient " + s, func_string
                )
                self.getOldestSolrDocument(solrurl, s)
            self.scIDBackup[solrurl] = attributes["scID"][:]
            Thread(name="publishAllWrapper", target=self.getDocsPerSc, args=(solrurl,)).start()
        GLOBAL_FP = self._forwardPointer.copy()

    def getDocsPerSc(self, solrurl):
        module_name = "CvEEJobPublisher"
        func_name = "getDocsPerSc"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function which calls getAllDocs for each subclient
        try:
            LOGGER_Publisher.info("Publisher woke up. Checking for new batches", func_string)
            scIDs = self.scIDBackup[solrurl] if solrurl in self.scIDBackup else []
            for s in scIDs[:]:
                LOGGER_Publisher.info(
                    "Retrieving all docs for subclient {0} in Solr {1}".format(s, solrurl),
                    func_string,
                )
                Thread(name="getDocsBlock", target=self.getAllDocs, args=(solrurl, s)).start()
            # LOGGER_Publisher.debug("All Docs is {0}".format(ALL_DOCS), func_string)
            # Timer(self.PUBLISH_ALL_SLEEP_TIME, self.getDocsPerSc).start()
        except Exception as e:
            LOGGER_Publisher.exception("Error in thread while getting all docs", func_string)

    def getAllDocs(self, solrurl, s):
        module_name = "CvEEJobPublisher"
        func_name = "getAllDocs"
        func_string = "{}::{}() - ".format(module_name, func_name)
        """Get all docs from Solr given a solrurl and a subclient. Docs are queried in chunks
        so that too much RAM is not used while getting everything. Docs are prefetched so that
        time is saved. Docs are fwetched only if CHECK_FOR_NEXT_BATCH event is set.
        It sets the PUBLISH_BATCH event once docs are fetched. """

        global ALL_DOCS, CHECK_FOR_NEXT_BATCH, PROCESSED_BATCHES
        self._gotDocs.clear()
        try:
            s = str(s)
            if solrurl not in ALL_DOCS:
                ALL_DOCS[solrurl] = {}
            if s not in ALL_DOCS[solrurl]:
                ALL_DOCS[solrurl][s] = []
            if solrurl not in PROCESSED_BATCHES:
                PROCESSED_BATCHES[solrurl] = {}
            if s not in PROCESSED_BATCHES[solrurl]:
                PROCESSED_BATCHES[solrurl][s] = 0
            if solrurl not in CHECK_FOR_NEXT_BATCH:
                CHECK_FOR_NEXT_BATCH[solrurl] = {}
            if solrurl not in self._forwardPointer or s not in self._forwardPointer[solrurl]:
                return
            if s not in CHECK_FOR_NEXT_BATCH[solrurl]:
                CHECK_FOR_NEXT_BATCH[solrurl][s] = Event()

            fp = self._forwardPointer[solrurl][s].replace(tzinfo=None)
            conn = SolrConnection(solrurl + "/solr")
            if self._config["check_failures"]:
                raw_query = "((cistate:1 OR cistate:16) AND (datatype: 1 OR datatype:2)) AND apid:{} AND ((extractingat:[{}Z TO {}Z]) OR (entity_state:[1 TO 5]) OR (extractingat:[* TO {}Z] AND (NOT entity_state:*)))"
            else:
                raw_query = "((cistate:1 OR cistate:16) AND (datatype: 1 OR datatype:2)) AND apid:{} AND ((extractingat:[{}Z TO {}Z]) OR (extractingat:[* TO {}Z] AND (NOT entity_state:*)))"
            query = raw_query.format(s, fp.isoformat(), self._config["currentTime"], fp.isoformat())
            LOGGER_Publisher.debug(
                "Solr query to get all docs for subclient {} is {}".format(s, query), func_string
            )
            if solrurl not in self._totalDocs:
                LOGGER_Publisher.error(
                    "Couldn't find total number of documents for solrurl {0}".format(solrurl),
                    func_string,
                )
            if s not in self._totalDocs[solrurl]:
                LOGGER_Publisher.error(
                    "Couldn't find total number of documents for subclient {0}".format(s),
                    func_string,
                )
            current_doc = 0
            response = conn.query(
                query,
                sort="extractingat",
                sort_order="asc",
                fields=["contentid", "extractingat"],
                rows=self.DOCS_PER_QUERY,
                start=current_doc,
            )
            # while current_doc <= len(ALL_DOCS[solrurl][s]):
            while response.numFound > 0:
                CHECK_FOR_NEXT_BATCH[solrurl][s].clear()
                current_doc += self.DOCS_PER_QUERY
                docs = {}
                time = fp
                for doc in response.results:
                    id = doc["contentid"]
                    time = doc["extractingat"]
                    docs[id] = time.replace(tzinfo=None)
                self._forwardPointer[solrurl][s] = time
                ALL_DOCS[solrurl][s].extend(sorted(list(docs.items()), key=operator.itemgetter(1)))
                LOGGER_Publisher.debug(
                    "Length of ALL_DOCS[{0}][{1}] is {2}, and total Docs is {3}".format(
                        solrurl, s, len(ALL_DOCS[solrurl][s]), response.numFound
                    ),
                    func_string,
                )
                self._gotDocs.set()
                PUBLISH_BATCH.set()
                LOGGER_Publisher.debug(
                    "Prefetching next batch for subclient {} in Solr {}".format(s, solrurl),
                    func_string,
                )
                response = conn.query(
                    query,
                    sort="extractingat",
                    sort_order="asc",
                    fields=["contentid", "extractingat"],
                    rows=self.DOCS_PER_QUERY,
                    start=current_doc,
                )
                if not CHECK_FOR_NEXT_BATCH[solrurl][s].isSet():
                    LOGGER_Publisher.debug("Waiting for current batches to finish", func_string)
                    CHECK_FOR_NEXT_BATCH[solrurl][s].wait(timeout=None)
            LOGGER_Publisher.info(
                "Published all jobs for subclient {} in Solr {}. Forward pointer is: {}".format(
                    s, solrurl, self._forwardPointer[solrurl][s]
                ),
                func_string,
            )
        except Exception as e:
            LOGGER_Publisher.exception(
                "Error while retrieving all docs for subclient {0}".format(s), func_string
            )

    def checkSolrAlive(self):
        module_name = "CvEEJobPublisher"
        func_name = "checkSolrAlive"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function that checks periodically to see if Solrs are still up and running.
        # If any Solr goes down, it will set a flag to False, causing the publisher to halt.

        global SOLR_RUNNING, CHECK_SOLR_TIMER, SOLR_REQUEST_SESSIONS, SOLR_ALIVE_TIMER
        solrurls = list(self._config["solrInfo"].keys())
        try:
            for solrurl in solrurls:
                if solrurl not in SOLR_REQUEST_SESSIONS:
                    SOLR_REQUEST_SESSIONS[solrurl] = requests.Session()
                try:
                    s = SOLR_REQUEST_SESSIONS[solrurl].get(solrurl)
                    if not s.status_code == 200:
                        SOLR_RUNNING = False
                        LOGGER_Publisher.error(
                            "Solr {0} is down. Will check again in {1} seconds.".format(
                                solrurl, CHECK_SOLR_TIMER
                            ),
                            func_string,
                        )
                        if SOLR_ALIVE_TIMER == None:
                            SOLR_ALIVE_TIMER = Timer(CHECK_SOLR_TIMER, self.checkSolrAlive).start()
                        return
                except Exception as e:
                    LOGGER_Publisher.exception(
                        f"Failed to ping solr at {solrurl}. Will check again in {CHECK_SOLR_TIMER} seconds. Exception: {e}",
                        func_string,
                    )
                    SOLR_REQUEST_SESSIONS[solrurl] = requests.Session()
                    SOLR_RUNNING = False
                    if SOLR_ALIVE_TIMER == None:
                        SOLR_ALIVE_TIMER = Timer(CHECK_SOLR_TIMER, self.checkSolrAlive).start()
                    return

                SOLR_RUNNING = True
                LOGGER_Publisher.info(
                    "Solr engines are up and running. Will check again in {0} seconds.".format(
                        CHECK_SOLR_TIMER
                    ),
                    func_string,
                )
        except Exception as e:
            SOLR_RUNNING = False
            LOGGER_Publisher.error(
                "Couldn't connect to solr {0}. Exception: {1}. Will check again in {2} seconds".format(
                    solrurl, e, CHECK_SOLR_TIMER
                ),
                func_string,
            )
        if SOLR_ALIVE_TIMER == None:
            SOLR_ALIVE_TIMER = Timer(CHECK_SOLR_TIMER, self.checkSolrAlive).start()

    def doSoftCommit(self):
        global SOLR_REQUEST_SESSIONS, SOLR_RUNNING
        module_name = "CvEEJobPublisher"
        func_name = "doSoftCommit"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function to do softcommit to Solr.
        solrurls = list(self._config["solrInfo"].keys())
        try:
            for solrurl in solrurls:
                query = "{0}/solr/update?commit=true&softcommit=true".format(solrurl)
                if solrurl not in SOLR_REQUEST_SESSIONS:
                    SOLR_REQUEST_SESSIONS[solrurl] = requests.Session()
                try:
                    s = SOLR_REQUEST_SESSIONS[solrurl].get(query)
                    LOGGER_Publisher.info(
                        f"Softcommit returned {s.status_code} for solr - {solrurl}", func_string
                    )
                except Exception as e:
                    SOLR_RUNNING = False
                    LOGGER_Publisher.exception(
                        f"Failed to do soft commit for solr - {solrurl}. Exception: {e}",
                        func_string,
                    )
                    SOLR_REQUEST_SESSIONS[solrurl] = requests.Session()
                    return False
            return True
        except Exception as e:
            SOLR_RUNNING = False
            LOGGER_Publisher.error("Couldn't softcommit due to: {0}".format(e), func_string)
            return False

    def getOldestSolrDocument(self, solrurl, scID):
        module_name = "CvEEJobPublisher"
        func_name = "getOldestSolrDocument"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function which fetches the oldest doc given a Solr and a subClient
        # It also initialises totalDocs present for a subclient in the given solr,
        # and sets _forwardOffset to an empty map.
        global DOCUMENT_RESPONSE_CODES
        no_results = False
        try:
            s = SolrConnection(solrurl + "/solr")
            if scID not in self._forwardPointer[solrurl]:
                return
            fp = self._forwardPointer[solrurl][scID].replace(tzinfo=None)
            if self._config["check_failures"]:
                raw_query = "((cistate:1 OR cistate:16) AND (datatype: 1 OR datatype:2)) AND apid:{} AND ((extractingat:[{}Z TO {}Z]) OR (entity_state:[1 TO 5]) OR (extractingat:[* TO {}Z] AND (NOT entity_state:*)))"
            else:
                raw_query = "((cistate:1 OR cistate:16) AND (datatype: 1 OR datatype:2)) AND apid:{} AND ((extractingat:[{}Z TO {}Z]) OR (extractingat:[* TO {}Z] AND (NOT entity_state:*)))"
            query = raw_query.format(
                scID, fp.isoformat(), self._config["currentTime"], fp.isoformat()
            )
            response = s.query(
                query, sort="extractingat", sort_order="asc", fields=["extractingat"]
            )
            if len(response.results) >= 1 and "extractingat" in response.results[0]:
                oldestTime = response.results[0]["extractingat"].replace(tzinfo=None)
            else:
                no_results = True

            if not no_results:
                if solrurl not in self._forwardPointer:
                    self._forwardPointer[solrurl] = {}
                if solrurl not in self._forwardOffset:
                    self._forwardOffset[solrurl] = {}
                if solrurl not in self._totalDocs:
                    self._totalDocs[solrurl] = {}
                if self._forwardPointer[solrurl][scID].replace(tzinfo=None) < oldestTime:
                    self._forwardPointer[solrurl][scID] = oldestTime
                self._forwardOffset[solrurl][scID] = 0
                self._totalDocs[solrurl][scID] = response.numFound
                LOGGER_Publisher.info(
                    "The forwardPointer for subclient {0} in Solr {1} is set to {2}".format(
                        scID, solrurl, oldestTime.strftime("%Y-%m-%d %H:%M:%S")
                    ),
                    func_string,
                )
            else:
                LOGGER_Publisher.info(
                    "Empty response for subclient {} in Solr {}".format(scID, solrurl), func_string
                )
                tempScID = self._config["solrInfo"][solrurl]["scID"][:]
                tempScID.remove(str(scID))
                self._config["solrInfo"][solrurl]["scID"] = tempScID
        except Exception as e:
            global SOLR_RUNNING
            SOLR_RUNNING = False
            LOGGER_Publisher.exception(
                "There was an exception while getting oldest Solr doc for {0}: {1}".format(
                    solrurl, scID
                ),
                func_string,
            )
            raise


class ForwardPointerManager(Thread, GenericMsgQueueCommunicator):

    """
        Class which listens to responses from ActiveMQ by clients, runs
        a periodic function to update the CS DB according to the docs received,
        and finally handles lost batches (if any) by changing it's entity_state in Solr.
    """

    def __init__(self, csconfig, config, softcommit):
        Thread.__init__(self)
        GenericMsgQueueCommunicator.__init__(self)
        self.config = config
        self.csconfig = csconfig
        self.batch = {}
        self.scIDs = {}
        self.initBatch = {}
        self.lost = False
        self.lostCounter = 0
        self.RQUEUE = config["q_params"][3] if "q_params" in config else "EERQueue"
        self.clients = {self.RQUEUE: {"subscribe": True, "client": None}}
        self.host = config["q_params"][0] if "q_params" in config else "127.0.0.1"
        self.stompPort = config["q_params"][1] if "q_params" in config else 61650
        self._stopped = Event()
        self.NO_DOCS_TO_PROCESS = config["docs_per_batch"]
        self.NO_BATCHES_TO_PROCESS = config["batches_per_publish"]
        self.MAX_TIME_FOR_BATCH = 20
        self.doSoftCommit = softcommit
        self.perfCounter = config["perfCounter"]

    def stop(self):
        self._Command = "stop"
        self.updateFp()
        self._stopped.set()

    def run(self):
        module_name = "CvEEJobPublisher.ForwardPointerManager"
        func_name = "run"
        func_string = "{}::{}() - ".format(module_name, func_name)

        # Listen to ActiveMQ for any messages dropped into REEQueue by clients
        # and according to flag of message, categorise message as pickedUp, or finishedProcessing,
        # and modify BATCHES variable accordingly.

        LOGGER_Publisher.info("Started ForwardPointerManager", func_string)
        self.connectToQueue()

        self.updateFp()

        global PUBLISH_BATCH, CHECK_FOR_NEXT_BATCH

        while not self._stopped.wait(0):
            try:
                frame = self.getFrame(self.clients[self.RQUEUE]["client"])
                if frame == None:
                    continue
                payload = ujson.loads(frame.decode("utf-8"))
                scID, sc_batchno = payload["sc_batchno"].split(":")
                solrurl = payload["solrurl"]
                if payload["flag"] == "0":
                    if solrurl not in self.scIDs:
                        self.scIDs[solrurl] = []
                    if scID not in self.scIDs[solrurl]:
                        self.scIDs[solrurl].append(scID)
                    if (
                        solrurl in BATCHES
                        and scID in BATCHES[solrurl]
                        and sc_batchno in BATCHES[solrurl][scID]
                    ):
                        BATCHES[solrurl][scID][sc_batchno][
                            "pickedUpTime"
                        ] = datetime.datetime.strptime(
                            payload["pickedUpTime"].split(".")[0], "%Y-%m-%dT%H:%M:%S"
                        )
                        BATCHES[solrurl][scID][sc_batchno]["end"] = payload["end"]
                        # BATCHES[scID][sc_batchno]['contentIds'] = payload['ids']
                        LOGGER_Publisher.debug(
                            "Received ids are: {}".format(
                                BATCHES[solrurl][scID][sc_batchno]["contentIds"]
                            ),
                            func_string,
                        )
                    else:
                        LOGGER_Publisher.error(
                            "Something strange happened for subclient {0} in Solr {1}, batch no {2}".format(
                                scID, solrurl, sc_batchno
                            ),
                            func_string,
                        )
                else:
                    if solrurl not in self.scIDs:
                        self.scIDs[solrurl] = []
                    if solrurl not in self.initBatch:
                        self.initBatch[solrurl] = {}
                    if solrurl not in self.batch:
                        self.batch[solrurl] = {}
                    if scID not in self.scIDs[solrurl]:
                        self.scIDs[solrurl].append(scID)
                    if scID not in self.initBatch[solrurl]:
                        self.initBatch[solrurl][scID] = 1
                    if scID not in self.batch[solrurl]:
                        self.batch[solrurl][scID] = {}
                    if int(sc_batchno) >= self.initBatch[solrurl][scID]:
                        self.batch[solrurl][scID][sc_batchno] = (
                            payload["start"],
                            payload["end"],
                            payload["solrurl"],
                            payload["offset"],
                        )
                    LOGGER_Publisher.debug(
                        "Batch {0} for subclient {1} in node {2} came back successfully".format(
                            sc_batchno, scID, solrurl
                        ),
                        func_string,
                    )

                    # Increment Processed batches after each finished batch is received
                    PROCESSED_BATCHES[solrurl][scID] += 1
                    LOGGER_Publisher.debug(
                        "Processed Batches for subclient {} in Solr {} is {} and total no. of Batches published is {}".format(
                            scID,
                            solrurl,
                            PROCESSED_BATCHES[solrurl][scID],
                            len(BATCHES[solrurl][scID]),
                        ),
                        func_string,
                    )
                    if PROCESSED_BATCHES[solrurl][scID] % 100 == 0:
                        LOGGER_Publisher.info(
                            "Batches {}/{} for subclient {} in Solr {} completed".format(
                                PROCESSED_BATCHES[solrurl][scID],
                                len(BATCHES[solrurl][scID]),
                                scID,
                                solrurl,
                            ),
                            func_string,
                        )

                    # If Processed batches is greater than no. of batches published, set event CHECK_FOR_NEXT_BATCH
                    if PROCESSED_BATCHES[solrurl][scID] >= len(BATCHES[solrurl][scID]):
                        CHECK_FOR_NEXT_BATCH[solrurl][scID].set()
                        # PUBLISH_BATCH.set()
            except:
                LOGGER_Publisher.exception("Error while reading from RQueue", func_string)
                self.connected = False
                self.connectToQueue()

        self.disconnectClients()

    def updateFp(self):
        module_name = "CvEEJobPublisher"
        func_name = "updateFp"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Main function that iterates through batches received, serialises them, and updates DB
        # till the last batch received in serial order

        global SOLR_RUNNING, JOB_FINISHED, BATCHES, UPDATE_DB_TIMER, CHECK_FOR_NEXT_BATCH, PROCESSED_BATCHES, PUBLISH_BATCH, STOP_PROCESS

        try:
            if not SOLR_RUNNING:
                LOGGER_Publisher.error(
                    "Solr Engine is down. Please start Solr and restart this process.", func_string
                )
            elif not JOB_FINISHED and not self.doSoftCommit():
                LOGGER_Publisher.error("Not Updating DB since softcommit failed", func_string)
            elif len(self.scIDs) == 0:
                LOGGER_Publisher.info("No nodes to process", func_string)
            else:
                for solrurl in list(self.scIDs.keys())[:]:
                    for s in self.scIDs[solrurl][:]:

                        # For each subclient in each Solr, create a list of tuples of received batches,
                        # sorted by message numbers, to get serial order of docs. And initialise
                        # variable last_updated batch to keep track of progress
                        LOGGER_Publisher.debug(
                            "Processing DB update for solr {0} and subclient {1}".format(
                                solrurl, s
                            ),
                            func_string,
                        )
                        if solrurl not in self.batch or s not in self.batch[solrurl]:
                            break
                        oBatch = collections.OrderedDict(sorted(self.batch[solrurl][s].items()))
                        last_updated_batch = self.initBatch[solrurl][s]

                        # Delete lost batches if any show up, since we've laready moved past
                        # their forward reference times

                        for i in range(1, last_updated_batch):
                            if str(i) in oBatch and str(i) in self.batch[solrurl][s]:
                                LOGGER_Publisher.debug(
                                    "Deleting old index {0} in oBatch for subclient {1} in node {2}".format(
                                        i, s, solrurl
                                    ),
                                    func_string,
                                )
                                del self.batch[solrurl][s][str(i)]

                        # Keep incrementing last_updated_batch if batch with that number
                        # is present in oBatch. If last_updated_batch has increased, do a DB update
                        # If it is not present, check if a certain time interval has elapsed since the
                        # picked up time. Check if batch isn't picked up till a certain time. Mark batch as
                        # lost if any of the other criteria is not met, and increment last_updated_batch for
                        # the corresponding batches

                        while True:
                            if last_updated_batch > len(BATCHES[solrurl][s]):
                                break
                            LOGGER_Publisher.debug(
                                "Solr {}, subclient {} has self.batch as {}".format(
                                    solrurl, s, self.batch[solrurl][s]
                                ),
                                func_string,
                            )

                            if str(last_updated_batch) in oBatch:
                                LOGGER_Publisher.debug(
                                    "Trying to delete index {0}".format(last_updated_batch),
                                    func_string,
                                )
                                del self.batch[solrurl][s][str(last_updated_batch)]
                                last_updated_batch += 1
                            else:
                                current_batch = (
                                    last_updated_batch - 1
                                    if last_updated_batch > 1
                                    else last_updated_batch
                                )
                                batchInfo = BATCHES[solrurl][s][str(current_batch)]
                                if "pickedUpTime" in batchInfo and datetime.datetime.utcnow() < (
                                    batchInfo["pickedUpTime"]
                                    + datetime.timedelta(minutes=self.MAX_TIME_FOR_BATCH)
                                ):
                                    LOGGER_Publisher.info(
                                        "Waiting for batches of subclient {0}, node {1} to be picked up.".format(
                                            s, solrurl
                                        ),
                                        func_string,
                                    )
                                    break
                                elif "pickedUpTime" not in batchInfo and datetime.datetime.utcnow() < (
                                    batchInfo["publishedTime"]
                                    + datetime.timedelta(minutes=self.MAX_TIME_FOR_BATCH)
                                ):
                                    LOGGER_Publisher.info(
                                        "Waiting for batches of subclient {0}, node {1} to be submitted.".format(
                                            s, solrurl
                                        ),
                                        func_string,
                                    )
                                    break
                                else:
                                    if self.handleLostBatches(solrurl, batchInfo["contentIds"]):
                                        if "pickedUpTime" in batchInfo:
                                            LOGGER_Publisher.error(
                                                "Lost batch {0} for subclient {1} in Solr {2}. Will proceed without it".format(
                                                    last_updated_batch, s, solrurl
                                                ),
                                                func_string,
                                            )
                                        else:
                                            LOGGER_Publisher.error(
                                                "Batch {0} for subclient {1} in Solr {2} was never picked up. Will proceed without it".format(
                                                    last_updated_batch, s, solrurl
                                                ),
                                                func_string,
                                            )
                                        last_updated_batch += 1
                                    else:
                                        LOGGER_Publisher.error(
                                            "Unable to mark lost batch {} for Solr {}, subclient {}. Will try in next update".format(
                                                last_updated_batch, solrurl, s
                                            ),
                                            func_string,
                                        )
                                        break

                        LOGGER_Publisher.debug(
                            "Inside updateFp. Value of last updated batch is: {0} and length of oBatch is {1}".format(
                                last_updated_batch, len(oBatch)
                            ),
                            func_string,
                        )
                        next_available_batch = last_updated_batch
                        if len(oBatch) > 0 and last_updated_batch > 1:
                            LOGGER_Publisher.debug(
                                "next_available_batch is {} and last item in OBatch is {} and oBatch is {} in Solr {}".format(
                                    next_available_batch,
                                    BATCHES[solrurl][s][str(last_updated_batch - 1)][
                                        "extractingat"
                                    ][-1],
                                    oBatch,
                                    solrurl,
                                ),
                                func_string,
                            )
                        if last_updated_batch > self.initBatch[solrurl][s]:
                            LOGGER_Publisher.info(
                                "Will update db for subclient {0} in Solr {1} with {2}. No. of processed batches is {3}".format(
                                    s,
                                    solrurl,
                                    BATCHES[solrurl][s][str(last_updated_batch - 1)][
                                        "extractingat"
                                    ][-1],
                                    last_updated_batch - 1,
                                ),
                                func_string,
                            )
                            self.perfCounter.start_stopwatch("DBUpdate")
                            updateCS(
                                self.csconfig,
                                solrurl,
                                str(
                                    BATCHES[solrurl][s][str(last_updated_batch - 1)][
                                        "extractingat"
                                    ][-1]
                                ),
                                s,
                            )
                            self.perfCounter.stop_stopwatch("DBUpdate")
                            self.initBatch[solrurl][s] = last_updated_batch
                        # If last_updated_batch has not increased, check no of items in oBatch
                        # If it is 0, job for subclient in this Solr is finished, and remove the subclient
                        # from processing. Else, wait for more batches
                        else:
                            if len(oBatch) == 0:
                                LOGGER_Publisher.info(
                                    "Nothing to update in DB for subclient {0} in Solr {1}. Removing it from processing.".format(
                                        s, solrurl
                                    ),
                                    func_string,
                                )
                                tempScID = self.scIDs[solrurl][:]
                                tempScID.remove(s)
                                self.scIDs[solrurl] = tempScID
                            else:
                                LOGGER_Publisher.info(
                                    "Waiting for batches for subclient {} in Solr {}. Will update db in next call".format(
                                        s, solrurl
                                    ),
                                    func_string,
                                )

                        # if all subclients have been removed, remove the Solr as well
                        if len(self.scIDs[solrurl]) == 0:
                            JOB_FINISHED = True
                            LOGGER_Publisher.info(
                                "No subclients to process for Solr {}. Removing it from processing".format(
                                    solrurl
                                ),
                                func_string,
                            )
                            del self.scIDs[solrurl]

                # TODO: If all Solrs have been removed, job is finished.
                if len(self.scIDs) == 0:
                    LOGGER_Publisher.info(
                        "No more nodes left to process. Stopping Publisher.", func_string
                    )
                    STOP_PROCESS.set()
                    return

        except Exception as e:
            LOGGER_Publisher.exception("Error while updating DB: {0}".format(e), func_string)

        Timer(UPDATE_DB_TIMER, self.updateFp).start()  # Update the db every 5 minutes

    def updateDocStatus(self, solrurl, cid, status):
        module_name = "CvEEJobPublisher"
        func_name = "updateDocStatus"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function to mark a document with a status, given
        # content id and solrurl, and the status to be marked with

        parameters = {
            "operationtype": "updatefields",
            "q": "contentid:{0}".format(cid),
            "entity_extractedat": datetime.datetime.utcnow().strftime("%Y-%m-%dT%H:%M:%SZ"),
            "entity_state": int(status),
            "fields": "entity_extractedat,entity_state",
        }

        url = "{0}/solr/updatedoc".format(solrurl)
        req = requests.post(url, data=parameters)
        if req.status_code == 200:
            req.close()
            return True
        LOGGER_Publisher.error(
            "Update to mark lost batch for doc id {0} for solr {1} failed with status code {2}".format(
                cid, solrurl, req.status_code
            ),
            func_string,
        )
        req.close()
        return False

    def handleLostBatches(self, solrurl, ids):
        module_name = "CvEEJobPublisher"
        func_name = "handleLostBatches"
        func_string = "{}::{}() - ".format(module_name, func_name)
        # Function to mark documents as lost using the updateDocStatus() function,
        # given a list of ids, and the Solr.
        global DOCUMENT_RESPONSE_CODES
        try:
            for cid in ids:
                if not self.updateDocStatus(solrurl, cid, DOCUMENT_RESPONSE_CODES["timedOut"]):
                    return False
        except Exception as e:
            LOGGER_Publisher.error("Couldn't update lost batch because: {0}".format(e), func_string)
            return False
        return True


class CommandManager(Thread):

    """
        Class dedicated to read the Queue from EntityHandler,
        for any communication sent by handler.
    """

    def __init__(self, commandQ, publisherStopFunction, fpStopFunction, wakeupFunction):
        Thread.__init__(self)

        # Make a list of stop functions, to stop process
        # gracefully when a stop command is received

        self._publisherStopFunction = publisherStopFunction
        self._fpStopFunction = fpStopFunction
        self._wakeupFunction = wakeupFunction
        self._commandQ = commandQ

    def run(self):
        # Listen to queue shared with Handler, and dequeue every message
        # sent, and do task accordingly.

        LOGGER_Publisher.info("Started CommandManager", func_string)
        global STOP_PROCESS
        if self._commandQ == None:
            return
        try:
            while True:
                command = self._commandQ.get()
                if command == "wakeup":
                    self._wakeupFunction()
                elif command == "stop":
                    self._publisherStopFunction()
                    self._fpStopFunction()
                    STOP_PROCESS.set()
                    break

        except KeyboardInterrupt:
            self.stop()

    def stop(self):
        module_name = "CommandManager"
        func_name = "stop"
        func_string = "{}::{}() - ".format(module_name, func_name)
        """Stopping"""
        return


def main_publisher(commandQ=None, conf=None, parent_pid=0):
    module_name = "CvEEJobPublisher"
    func_name = "main_publisher"
    func_string = "{}::{}() - ".format(module_name, func_name)

    # Main function to initialise logging, make instances of every class,
    # get configuration from CS, and start all threads.

    global PUBLISH_BATCH, LOGGER_Publisher, STOP_PROCESS, ROTATING_MAX_BYTES, ROTATING_BACKUP_COUNT

    extra_config_params = {}

    if conf is not None:
        ROTATING_MAX_BYTES = conf["log_max_bytes"]["value"]
        ROTATING_BACKUP_COUNT = conf["log_backup_count"]["value"]
        extra_config_params["stompPort"] = conf["stompPort"]["value"]
        extra_config_params["check_failures"] = conf["check_failures"]["value"]
        extra_config_params["docs_per_batch"] = conf["docs_per_batch"]["value"]
        extra_config_params["batches_per_publish"] = conf["batches_per_publish"]["value"]

    try:
        logger_options = {
            "ROTATING_BACKUP_COUNT": ROTATING_BACKUP_COUNT,
            "ROTATING_MAX_BYTES": ROTATING_MAX_BYTES,
        }

        LOGGER_Publisher = get_logger_handler(
            os.path.join(getBaseDir(), HELPER_DLL), "ContentAnalyzer", logger_options
        )

    except Exception as e:
        print("Error while initialising: {}".format(e))

    PUBLISH_BATCH.clear()
    STOP_PROCESS.clear()

    checkParentAndKill(parent_pid, PROCESS_ID)

    perfCounter = PerformanceCounter(LOGGER_Publisher)
    perfCounter.periodicLogTimer()
    extra_config_params["perfCounter"] = perfCounter

    csconfig, config = getConfiguration(extra_config_params)
    # LOGGER_Publisher.error("{0}".format(config), func_string)

    LOGGER_Publisher.info("Starting Publisher Process", func_string)

    taskManager = TaskManager(config)
    fpManager = ForwardPointerManager(csconfig, config, taskManager.doSoftCommit)
    publisher = Publisher(config)
    cmdManager = CommandManager(commandQ, publisher.stop, fpManager.stop, publisher.wakeUp)

    try:
        cmdManager.start()
        fpManager.start()
        publisher.start()
    except KeyboardInterrupt:
        publisher.stop()
        cmdManager.stop()
        fpManager.stop()
    except Exception as e:
        LOGGER_Publisher.exception("Threads stopped unexpectedly", func_string)

    while not STOP_PROCESS.isSet():
        pass
    publisher.stop()
    cmdManager.stop()
    fpManager.stop()
    LOGGER_Publisher.info("Job finished. Publisher is going to sleep.", func_string)


if __name__ == "__main__":
    main_publisher()
