"""Create a preview of videos by extracting the "key" (loudest) moments
    in video and/or gif formats"""
from struct import unpack
from ffmpy import FFmpeg, FFprobe
from json import loads
import sys, os, shutil
from operator import itemgetter

# import matplotlib.pyplot as plt
import numpy as np
from math import ceil
import imageio
import gevent
import multiprocessing
import logging
from contextlib import closing

from CvEEConfigHelper import *

CLIENT_ID = 1
TEMP_PROCESSING_LOCATION = "../ContentAnalyzer/temp/clips{}"
TEMP_OUTPUT_LOCATION = "../ContentAnalyzer/temp/output{}"
TEMP_INFO_LOCATION = "../ContentAnalyzer/temp/info{}"

# type and config need to be initiated globally so the pools won't have issues with them
config = VIDEO_PREVIEW_CONFIG.copy()  # load config settings

"""Determine whether output should be a gif or video
If both are true, then videos should be processed between the if statements and after the if statements; currently, if both are true it defaults to gif"""
if config["output"]["gif"]:
    type = "gif"
elif config["output"]["vid"]:
    type = "mpg"


def proc_dir(dir):
    global type
    """Process all the videos in directory dir"""
    for f in os.listdir(dir):
        proc_vid(dir, dir + "\\" + f)


def proc_vid(input_dir, input_file, output_dir, output_file):
    """Process a single video file in location input_file"""
    # video duration, audio sample rate
    input_file = os.path.join(input_dir, input_file)
    tim, freq = get_info(input_file)
    if type == "mpg":  # determine number of clips for key clip output
        # length of one clip
        oneclip = (
            config[type]["timend"]["value"]
            + config["framelen"]["value"]
            - config[type]["timstart"]["value"]
        )
        # number of frames relative to full length of video
        framenum = int((tim * config[type]["ratio2original"]["value"]) // oneclip)
        if (
            framenum == 0 or tim < 10
        ):  # check if video is above the minimum reasonable size for preview frame selection
            # full video muted with set resolution if key frame selection is unreasonable
            ff = FFmpeg(
                inputs={input_file: None},
                outputs={
                    "outputs\\small"
                    + input_file.partition(input_dir + "\\")[2].partition(".")[0]
                    + ".mpg": "-vsync 0 -vf scale="
                    + config[type]["scale"]["value"]
                    + config[type]["audiomute"]["value"]
                },
                global_options="-y -loglevel panic -threads 0",
            )
            ff.run()
            return
        if framenum * oneclip < 10:  # ensure preview is at least 10 seconds
            framenum = ceil(10.0 / oneclip)
        endoutop = "-c copy"
    else:  # frame length for a gif of key frames is determined from config options
        framenum = config[type]["framenum"]["value"]
        endoutop = None
    # get input and output parameters for slicing requests
    inop, outop = get_clipoptions(get_n_loudest(get_amps(input_file, freq, tim), freq, framenum))
    # zip together parameters to be used in map
    op = list(
        zip(inop, outop, list(range(len(inop))), [input_file] * len(inop), [CLIENT_ID] * len(inop))
    )
    # extract compressed slices from the original video with each process in parallel
    images = []

    pool = gevent.pool.Pool(multiprocessing.cpu_count())
    results = pool.map_async(make_clips, op)
    pool.join()

    results.get()

    # list of video slices to be combined to a file
    piclist = "concat:"  # making a string that fits ffmpeg option syntax
    for p in os.listdir(TEMP_PROCESSING_LOCATION):
        if config[type]["clip"]["value"] == ".mpg":
            piclist += "clips/" + p + "|"
        else:  # make a list of images to be combined to a gif of key frames
            images.append(imageio.imread(TEMP_PROCESSING_LOCATION + "\\" + p))
    # combine the slices into a stream and copy stream to a video or gif
    if config[type]["clip"]["value"] == ".mpg":  # video or gif of clips
        ff = FFmpeg(
            inputs={piclist[:-1]: None},
            outputs={
                "outputs\\clip"
                + input_file.partition(input_dir + "\\")[2].partition(".")[0]
                + "."
                + type: endoutop
            },
            global_options="-y -loglevel panic",
        )
        # print(ff.cmd)
        ff.run()
    else:  # save a gif of single frames
        # filename = 'frame'+input_file.partition(dir+'\\')[2].partition('.')[0]+'.gif'
        imageio.mimsave(
            "{}\\{}".format(TEMP_OUTPUT_LOCATION, output_file),
            images,
            duration=config[type]["framedur"]["value"],
        )

    # Image has been created. Try copying over to final location

    # First, get a unique filename at the output location

    unique_dest = get_unique_filename(output_dir, output_file)

    # No need to add try-except block here, as it is being handled
    # in the parent function.

    shutil.copyfile("{}\\{}".format(TEMP_OUTPUT_LOCATION, output_file), unique_dest)

    # Since all processing is done, and video file has been copied over
    # (hopefully), we can clear up all temporary processing files

    clean_temporary_items()

    return unique_dest


def get_unique_filename(dir, filename):

    dest_filename = filename
    file_increment = 1
    final_dest_filename = filename

    # To avoid overwriting existing file, check if the
    # filename which will be copied over exists. If it does,
    # Add an incremental counter to the filename till we get
    # a name that does not exist and can be safely written over

    while os.path.isfile(dir + "\\" + final_dest_filename):
        file_arr = dest_filename.split(".")
        name = ".".join(file_arr[0:-1])
        extension = file_arr[-1]
        final_dest_filename = "{}-{}.{}".format(name, file_increment, extension)
        file_increment += 1

    return dir + "\\" + final_dest_filename


def clean_temporary_items():
    list_of_globals = [TEMP_PROCESSING_LOCATION, TEMP_INFO_LOCATION, TEMP_OUTPUT_LOCATION]
    for current_dir in list_of_globals:
        for current_file in os.listdir(current_dir):
            os.remove(current_dir + "\\" + current_file)


def make_clips(op):
    # make timeframe subtitles to add to video
    global TEMP_PROCESSING_LOCATION

    # This is required as this function is spawned as a new process
    # pool, and the global value of this variable has reset
    TEMP_PROCESSING_LOCATION = TEMP_PROCESSING_LOCATION.format(op[4])

    # This is required as the FFMPEG function does not recognize the '\'
    # character, and replaces it with ''. Hence, we change \ to /
    wrangled_location = TEMP_PROCESSING_LOCATION.replace("\\", "/")

    sub(int(op[0].partition(" ")[2]), TEMP_PROCESSING_LOCATION + "\\sub" + str(op[2]) + ".srt")
    # save clips
    ff = FFmpeg(
        inputs={op[3]: op[0]},
        outputs={
            TEMP_PROCESSING_LOCATION
            + "\\thumb_"
            + str(op[2]).zfill(2)
            + config[type]["clip"]["value"]: "-copyts "
            + op[1]
            + " -vsync 0 -vf [in]scale="
            + config[type]["scale"]["value"]
            + ",subtitles={}/sub".format(wrangled_location)
            + str(op[2])
            + ".srt[out]"
            + config[type]["audiomute"]["value"]
        },
        global_options="-y -loglevel panic",
    )
    # print(ff.cmd)
    ff.run()
    # clean up sub file
    os.remove(TEMP_PROCESSING_LOCATION + "\\sub" + str(op[2]) + ".srt")


def sub(tim, f):
    # write timestamps to a subtitle file
    sub = open(f, "w")
    # write in appropriate srt format
    sub.write("1\n")  # only one subtitle
    # subtitle time is an arbitrarily large time frame to save time finding video length
    # because the text is the same throughout the video
    sub.write("00:00:00,000 --> 01:00:00,000\n")
    # convert the time to hr:min:sc as in 00:00:00
    sub.write(
        str(int(tim // (60 * 60))).zfill(2)
        + ":"
        + str(int(tim // 60 % 60)).zfill(2)
        + ":"
        + str(tim % 60).zfill(2)
    )
    sub.close()  # stop writing to subtitle file


def get_clipoptions(loudest):
    times = []
    seek = []
    # make list of beginning and end time options for splitting
    timstart = config[type]["timstart"]["value"]  # Time relative to start of sample to capture
    for t in range(len(loudest)):
        # skip adjacent clip and make next clip start earlier
        if config[type]["clip"]["value"] == ".mpg":
            if t + 1 < len(loudest) - 1 and loudest[t][1] == loudest[t + 1][1] - 1:
                timstart = timstart - config["framelen"]["value"]
                continue
        # start time of clips to capture
        seek += ["-ss " + str(max(0, int(loudest[t][1] * config["framelen"]["value"]) + timstart))]
        if config[type]["clip"]["value"] == ".mpg":
            # end time of clips a given length after start
            times += [
                "-to "
                + str(
                    int(loudest[t][1] * config["framelen"]["value"])
                    + config["framelen"]["value"]
                    + config[type]["timend"]["value"]
                )
            ]
            timstart = config[type]["timstart"][
                "value"
            ]  # reset the relative start time to default in case clips were combined"""
        else:
            times += ["-vframes 1"]  # output options for a gif of key frames
    return seek, times


def get_n_loudest(amps, fps, n):
    """Get the loudest n moments from an array of amps"""
    loudest = []  # list of (amplitudes, times)
    timeframe = int(fps * config["framelen"]["value"])  # number of audio frames in framelen
    i = 0
    samples = np.array(amps) / pow(2, 15)  # normalize audio samples
    # divide all samples into samples of length framelen
    splits = np.cumsum([timeframe] * (samples.size // timeframe))
    samples = np.array_split(samples, splits)
    # get root mean squares of samples
    loudest = [(np.sqrt(sample.dot(sample) / sample.size), i) for i, sample in enumerate(samples)]
    # sort by loudness and grab n loudest
    loudest = sorted(loudest, key=itemgetter(0))
    loudest = loudest[(-1 * n) :]
    loudest = sorted(loudest, key=itemgetter(1))  # sort by time
    return loudest


def get_amps(vid, samplefreq, dur):
    """Get an array of amplitude information"""
    ff = FFmpeg(
        inputs={vid: None},
        outputs={TEMP_INFO_LOCATION + "\\bin.bin": "-ac 1 -map 0:a -c:a pcm_s16le -f data"},
        global_options="-y -loglevel panic",
    )  # get audio samples as raw binary data
    # print(ff.cmd)
    ff.run()
    bina = open(TEMP_INFO_LOCATION + "\\bin.bin", mode="rb").read(
        int(samplefreq * dur) * 2
    )  # read bytes from file
    dat = unpack("%ih" % (int(samplefreq * dur)), bina)  # interpret bytes of signed shorts
    return dat


def get_info(vid):
    """Get (duration, sample rate) from video vid"""
    ff = FFprobe(inputs={vid: "-v error -of json -show_entries stream=duration:stream=sample_rate"})
    ff.run(
        stdout=open(TEMP_INFO_LOCATION + "\\CvCAVideoPreviewInfo.json", "w")
    )  # output duration and rate info to a file
    info = loads(open(TEMP_INFO_LOCATION + "\\CvCAVideoPreviewInfo.json").read())
    return float(info["streams"][1]["duration"]), int(info["streams"][1]["sample_rate"])


def mountFolder(path, domain, username, password):
    mount_command = "net use /user:{}\\{} {} {}".format(domain, username, path, password)
    os.system(mount_command)


def preProcess(params={}):
    global PRE_PROCESSING_DONE, CA_ERROR_CODES
    global TEMP_PROCESSING_LOCATION, TEMP_OUTPUT_LOCATION, CLIENT_ID, TEMP_INFO_LOCATION
    """
        Create folders per client for temporary creation of
        clips which can be combined later for GIF.
    """
    if "clientId" not in params:
        return {
            "ErrorMessage": "Preprocessing VIDEO_PREVIEW failed: Client ID is not present in params",
            "ErrorCode": CA_ERROR_CODES["VPPreProcessingError"],
        }

    try:
        clientId = params["clientId"]
        CLIENT_ID = clientId

        # Check if FFmpeg is added to reg
        FFMPEG_INSTALL_DIR = loadRegValue(FFMPEG_REG_KEY, "", type=str)
        if FFMPEG_INSTALL_DIR.strip() == "":
            return {
                "ErrorMessage": "Preprocessing VIDEO_PREVIEW failed: FFMpeg location is not specified in registry",
                "ErrorCode": CA_ERROR_CODES["VPPreProcessingError"],
            }

        # Add FFMpeg to OS path
        if FFMPEG_INSTALL_DIR not in os.environ["PATH"]:
            old_path = os.environ["PATH"]
            os.environ["PATH"] = "{}{}{}".format(FFMPEG_INSTALL_DIR, os.pathsep, old_path)

        TEMP_PROCESSING_LOCATION = TEMP_PROCESSING_LOCATION.format(clientId)
        TEMP_OUTPUT_LOCATION = TEMP_OUTPUT_LOCATION.format(clientId)
        TEMP_INFO_LOCATION = TEMP_INFO_LOCATION.format(clientId)
        list_of_globals = [TEMP_INFO_LOCATION, TEMP_PROCESSING_LOCATION, TEMP_OUTPUT_LOCATION]
        for d in list_of_globals:
            if not os.path.isdir(d):
                os.makedirs(d)
        PRE_PROCESSING_DONE = True
        return {"ErrorCode": CA_ERROR_CODES["success"]}
    except Exception as e:
        return {
            "ErrorCode": CA_ERROR_CODES["VPPreProcessingError"],
            "ErrorMessage": "Preprocessing VIDEO_PREVIEW failed: {}".format(e),
        }


def checkAndMountFolders(location, params, io="input"):
    global CA_ERROR_CODES
    response = {}
    if not os.path.isdir(location):
        if (
            io + "Path_username" not in params
            or io + "Path_password" not in params
            or io + "Path_domain"
        ):
            response["ErrorCode"] = CA_ERROR_CODES["VPError"]
            response["ErrorMessage"] = io + " folder does not exist or Credentials not provided."

        try:
            mountFolder(
                location,
                params[io + "Path_domain"],
                params[io + "Path_username"],
                params[io + "Path_password"],
            )
        except Exception as e:
            response["ErrorCode"] = CA_ERROR_CODES["VPError"]
            response["ErrorMessage"] = "Error while mounting {} folder: {}".format(io, e)
            return response

    return {"ErrorCode": CA_ERROR_CODES["success"]}


def doAnalysis(input, params={}):
    """Entry Point: Get input and params """
    global PRE_PROCESSING_DONE, CA_ERROR_CODES

    response = {}
    if "outputFilePath" not in params:
        return {
            "ErrorCode": CA_ERROR_CODES["VPError"],
            "ErrorMessage": "Output location is not present in params",
        }
    video_input_location = os.path.dirname(input)
    video_output_location = os.path.dirname(params["outputFilePath"])
    input_file_name = os.path.basename(input)
    output_file_name = os.path.basename(params["outputFilePath"])

    if not PRE_PROCESSING_DONE:
        response["ErrorCode"] = False
        response["ErrorMessage"] = "Preprocessing did not finish."

    response = checkAndMountFolders(video_input_location, params)
    if response == True and input != params["outputFilePath"]:
        response = checkAndMountFolders(video_output_location, params, io="output")

    if "ErrorMessage" not in response:
        try:
            dest = proc_vid(
                video_input_location, input_file_name, video_output_location, output_file_name
            )
            response["ErrorCode"] = CA_ERROR_CODES["success"]
            response["ErrorMessage"] = None
            response["destination"] = dest
        except Exception as e:
            clean_temporary_items()
            response["ErrorCode"] = CA_ERROR_CODES["VPError"]
            response["ErrorMessage"] = "Error while converting file to gif: {}".format(e)

    return response


# a main is necessary so the multiprocessing pools don't have issues
if __name__ == "__main__":
    file = sys.argv[1]
    pre_process_params = {"clientId": 1, "ffmpeg_path": "C:\\ffmpeg\\bin"}
    preProcess(pre_process_params)
    print(doAnalysis(file))
