import time
from threading import Timer


class PerformanceCounter:
    def __init__(self, logger):
        self.performanceStats = {"avgTime": {}, "numTasks": {}, "fileSize": {}, "totalTime": {}, "entity_count": {}}

        self.logger = logger

        # Map for RER will be of the form:
        # {
        #   avgTime: {
        #       email: 0.12,
        #       phone: 0.08,
        #   },
        #   numTasks: {
        #       email: 1100,
        #       phone: 500
        #   },
        # }
        #
        # For NER, we'll simply have 1 key, "NER" for each performance attribute so as to simplify code

        self.start = {}
        self.changed_status = False

    def start_stopwatch(self, key):
        self.start[key] = time.time()

    def stop_stopwatch(self, key, size=0, entity_count=0):

        time_taken = time.time() - self.start[key]
        try:
            prevAvg = self.performanceStats["avgTime"][key]
            prevNum = self.performanceStats["numTasks"][key]
        except:
            self.performanceStats["avgTime"][key] = 0
            self.performanceStats["numTasks"][key] = 0
            self.performanceStats["totalTime"][key] = 0
            self.performanceStats["fileSize"][key] = 0
            self.performanceStats["entity_count"][key] = 0
            prevAvg = 0
            prevNum = 0

        newAvgTime = ((prevAvg * prevNum) + time_taken) / float((prevNum + 1))

        self.performanceStats["avgTime"][key] = newAvgTime
        self.performanceStats["numTasks"][key] += 1
        self.performanceStats["totalTime"][key] += time_taken
        self.performanceStats["fileSize"][key] += size
        self.performanceStats["entity_count"][key] += entity_count

        self.changed_status = True

    def logStats(self):
        module_name = "CvCAPerfCounter"
        func_name = ""
        func_str = "{}::{}() - ".format(module_name, func_name)
        try:
            if not self.changed_status:
                return
            performance_keys = []
            for key in list(self.performanceStats["numTasks"].keys()):
                if key[0].isupper():
                    self.logger.info("Performance stat - {}".format(self.getStat(key)), func_str)
                else:
                    performance_keys.append(key)
            for key in performance_keys:
                self.logger.info("Performance stat - {}".format(self.getStat(key)), func_str)
            self.changed_status = False
        except:
            self.logger.exception("Error while printing perf stats", func_str)

    def periodicLogTimer(self, duration=300, firstCall=True):
        if not firstCall:
            self.logStats()
        Timer(
            duration, self.periodicLogTimer, [], {"duration": duration, "firstCall": False}
        ).start()

    def getStat(self, key):
        try:
            avgTime = self.performanceStats["avgTime"][key]
            numTasks = self.performanceStats["numTasks"][key]
            fileSize = self.performanceStats["fileSize"][key]
            totalTime = self.performanceStats["totalTime"][key]
            entity_count = self.performanceStats["entity_count"][key]

            if totalTime > 0:
                tasksPerSec = float(numTasks) / totalTime
                bytesPerSec = float(fileSize) / totalTime

            return "ID [{}], Documents [{}], Total Time [{}], Total File Size (KB) [{}], Entities Count [{}]".format(
                key, numTasks, round(totalTime, 4), round(fileSize / 1024, 2), entity_count,
            )

        except Exception as e:
            return "Exception while calculating performance: {}".format(e)

    def sampleReports(self, key):
        # This is a test function to show the reports that can be generated
        # This won't be used in production code

        # Print avg time taken by email and number of documents processed
        print(self.performanceStats["email"], self.numTasks["email"])
