import inspect
import logging
import os
import threading
import traceback
from ctypes import c_char_p, cdll
from logging.handlers import RotatingFileHandler

from CvEEConfigHelper import LOG_LEVELS, checkGenericLogLevel, getInstanceName, getLogDir, is_linux

LOGGER_HANDLERS = {}


def get_logger_handler(dll_name, module_name, logger_options):
    global LOGGER_HANDLERS
    if module_name not in LOGGER_HANDLERS:
        LOGGER_HANDLERS[module_name] = GenericLogger(dll_name, module_name, logger_options)
    return LOGGER_HANDLERS[module_name]


class GenericLogger:
    def __init__(self, dll_name, module_name, logger_options):
        try:
            global LOGGER_HANDLERS
            self.logger = cdll.LoadLibrary(dll_name)
            module = c_char_p(module_name.encode())
            instance = c_char_p(getInstanceName().encode())
            self.logger.InitLogger(module, instance)
            self.usingDLL = True
            self.invLogLevels = {v: k for k, v in LOG_LEVELS.items()}
        except:
            import logging

            self.usingDLL = False
            for handler in logging.root.handlers[:]:
                logging.root.removeHandler(handler)
            PROCESS_ID = os.getpid()
            THREAD_ID = threading.current_thread().ident
            ROTATING_MAX_BYTES = logger_options["ROTATING_MAX_BYTES"]
            ROTATING_BACKUP_COUNT = logger_options["ROTATING_BACKUP_COUNT"]
            FILE_NAME = None
            if is_linux() == True:
                FILE_NAME = os.path.join(getLogDir(), module_name + ".log")
            else:
                FILE_NAME = os.getcwd().replace("\\Base", "\\Log Files\\{}.log".format(module_name))
            LOG_FORMAT = (
                str(PROCESS_ID)
                + " "
                + str(THREAD_ID)
                + " %(asctime)s ### %(levelname)s: %(name)s::%(funcName)s() - %(message)s"
            )
            default_level = checkGenericLogLevel(module_name)
            self.logger = logging.getLogger(module_name)
            log_formatter = logging.Formatter(LOG_FORMAT, datefmt="%m/%d %H:%M:%S")
            handler = RotatingFileHandler(
                FILE_NAME, maxBytes=ROTATING_MAX_BYTES, backupCount=ROTATING_BACKUP_COUNT
            )
            handler.setFormatter(log_formatter)
            self.logger.addHandler(handler)
            self.logger.setLevel(logging.getLevelName(LOG_LEVELS[default_level]))

    def getParentFunction(self):
        stack = inspect.stack()
        func = None
        try:
            func = stack[2][3]
        except:
            pass

        module = None
        try:
            module = inspect.getmodule(stack[2][0]).__name__
        except:
            pass

        return module, func

    def log(self, message, level, func_str):
        # if func_str.strip() == "":
        #     try:
        #         frm = inspect.stack()[2]
        #         mod = inspect.getmodule(frm[0])
        #         funn = frm.function
        #         if mod is not None:
        #             func_str = mod.__name__ + "::" + funn + "() - "
        #         else:
        #             func_str = inspect.getfile(frm[0])[:-3] + "::" + funn + "() - "
        #     except Exception as e:
        #         print(e, message)
        message = func_str + message
        log_msg = LOG_LEVELS[level] + ":"
        message = f"{log_msg:<6} {message}"
        self.logger.Log(level, c_char_p(message.encode()))

    def debug(self, message, func_str=""):
        if self.usingDLL:
            self.log("{}".format(message), self.invLogLevels["DEBUG"], func_str)
        else:
            self.logger.debug(message)

    def info(self, message, func_str=""):
        if self.usingDLL:
            self.log("{}".format(message), self.invLogLevels["INFO"], func_str)
        else:
            self.logger.info(message)

    def warning(self, message, func_str=""):
        if self.usingDLL:
            self.log("{}".format(message), self.invLogLevels["WARNING"], func_str)
        else:
            self.logger.warning(message)

    def error(self, message, func_str=""):
        if self.usingDLL:
            self.log("{}".format(message), self.invLogLevels["ERROR"], func_str)
        else:
            self.logger.error(message)

    def exception(self, message, func_str=""):
        if self.usingDLL:
            self.log(
                "{}{}{}".format(message, os.linesep, traceback.format_exc()),
                self.invLogLevels["ERROR"],
                func_str,
            )
        else:
            self.logger.exception(message)

    def verbose(self, message, func_str=""):
        if self.usingDLL:
            self.log("{}".format(message), self.invLogLevels["VERBOSE"], func_str)
        else:
            self.logger.debug(message)
