import os, sys, glob, time
import logging, threading

try:
    import cv2, dlib, pickle
    import cv2.cv as cv
    import numpy as np
    import gridfs
    from pymongo import *
    from bson.binary import Binary
    from CvEEConfigHelper import CA_ERROR_CODES
except Exception as e:
    print("Failed to import module: ".format(e))

PROCESS_ID = os.getpid()
THREAD_ID = threading.current_thread().ident
LOGGER_Generic = None


def configureLog():
    global LOGGER_Generic
    if LOGGER_Generic is not None:
        return
    try:
        for handler in logging.root.handlers[:]:
            logging.root.removeHandler(handler)
        FILE_NAME = os.getcwd().replace("\\Base", "") + "\\Log Files\\GenericCA.log"
        LOG_FORMAT = (
            str(PROCESS_ID)
            + " "
            + str(THREAD_ID)
            + " %(asctime)s ### %(name)s::%(funcName)s() - %(message)s"
        )
        logging.basicConfig(
            filename=FILE_NAME, level=logging.INFO, format=LOG_FORMAT, datefmt="%m/%d %H:%M:%S"
        )
        LOGGER_Generic = logging.getLogger(__name__)
    except Exception as e:
        print("Failed to configure log: ".format(e))


TEMPLATE = np.float32(
    [
        (0.0792396913815, 0.339223741112),
        (0.0829219487236, 0.456955367943),
        (0.0967927109165, 0.575648016728),
        (0.122141515615, 0.691921601066),
        (0.168687863544, 0.800341263616),
        (0.239789390707, 0.895732504778),
        (0.325662452515, 0.977068762493),
        (0.422318282013, 1.04329000149),
        (0.531777802068, 1.06080371126),
        (0.641296298053, 1.03981924107),
        (0.738105872266, 0.972268833998),
        (0.824444363295, 0.889624082279),
        (0.894792677532, 0.792494155836),
        (0.939395486253, 0.681546643421),
        (0.96111933829, 0.562238253072),
        (0.970579841181, 0.441758925744),
        (0.971193274221, 0.322118743967),
        (0.163846223133, 0.249151738053),
        (0.21780354657, 0.204255863861),
        (0.291299351124, 0.192367318323),
        (0.367460241458, 0.203582210627),
        (0.4392945113, 0.233135599851),
        (0.586445962425, 0.228141644834),
        (0.660152671635, 0.195923841854),
        (0.737466449096, 0.182360984545),
        (0.813236546239, 0.192828009114),
        (0.8707571886, 0.235293377042),
        (0.51534533827, 0.31863546193),
        (0.516221448289, 0.396200446263),
        (0.517118861835, 0.473797687758),
        (0.51816430343, 0.553157797772),
        (0.433701156035, 0.604054457668),
        (0.475501237769, 0.62076344024),
        (0.520712933176, 0.634268222208),
        (0.565874114041, 0.618796581487),
        (0.607054002672, 0.60157671656),
        (0.252418718401, 0.331052263829),
        (0.298663015648, 0.302646354002),
        (0.355749724218, 0.303020650651),
        (0.403718978315, 0.33867711083),
        (0.352507175597, 0.349987615384),
        (0.296791759886, 0.350478978225),
        (0.631326076346, 0.334136672344),
        (0.679073381078, 0.29645404267),
        (0.73597236153, 0.294721285802),
        (0.782865376271, 0.321305281656),
        (0.740312274764, 0.341849376713),
        (0.68499850091, 0.343734332172),
        (0.353167761422, 0.746189164237),
        (0.414587777921, 0.719053835073),
        (0.477677654595, 0.706835892494),
        (0.522732900812, 0.717092275768),
        (0.569832064287, 0.705414478982),
        (0.635195811927, 0.71565572516),
        (0.69951672331, 0.739419187253),
        (0.639447159575, 0.805236879972),
        (0.576410514055, 0.835436670169),
        (0.525398405766, 0.841706377792),
        (0.47641545769, 0.837505914975),
        (0.41379548902, 0.810045601727),
        (0.380084785646, 0.749979603086),
        (0.477955996282, 0.74513234612),
        (0.523389793327, 0.748924302636),
        (0.571057789237, 0.74332894691),
        (0.672409137852, 0.744177032192),
        (0.572539621444, 0.776609286626),
        (0.5240106503, 0.783370783245),
        (0.477561227414, 0.778476346951),
    ]
)

TPL_MIN, TPL_MAX = np.min(TEMPLATE, axis=0), np.max(TEMPLATE, axis=0)
MINMAX_TEMPLATE = (TEMPLATE - TPL_MIN) / (TPL_MAX - TPL_MIN)


class AlignDlib:
    #: Landmark indices.
    INNER_EYES_AND_BOTTOM_LIP = [39, 42, 57]
    OUTER_EYES_AND_NOSE = [36, 45, 33]

    def __init__(self, facePredictor):
        """
                Instantiate an 'AlignDlib' object.
                """
        assert facePredictor is not None

        self.detector = dlib.get_frontal_face_detector()
        self.predictor = dlib.shape_predictor(facePredictor)

    def getAllFaceBoundingBoxes(self, rgbImg):
        """
                Find all face bounding boxes in an image.
                """
        assert rgbImg is not None

        try:
            return self.detector(rgbImg, 1)
        except Exception as e:
            print("Warning: {}".format(e))
            # In rare cases, exceptions are thrown.
            return []

    def getLargestFaceBoundingBox(self, rgbImg, skipMulti=False):
        """
                Find the largest face bounding box in an image.
                """
        assert rgbImg is not None

        faces = self.getAllFaceBoundingBoxes(rgbImg)
        if (not skipMulti and len(faces) > 0) or len(faces) == 1:
            return max(faces, key=lambda rect: rect.width() * rect.height())
        else:
            return None

    def findLandmarks(self, rgbImg, bb):
        """
                Find the landmarks of a face.
                """
        assert rgbImg is not None
        assert bb is not None

        points = self.predictor(rgbImg, bb)
        return list([(p.x, p.y) for p in points.parts()])

    def align(
        self,
        imgDim,
        rgbImg,
        bb=None,
        landmarks=None,
        landmarkIndices=INNER_EYES_AND_BOTTOM_LIP,
        skipMulti=False,
    ):
        """
                Transform and align a face in an image.
                """
        assert imgDim is not None
        assert rgbImg is not None
        assert landmarkIndices is not None

        if bb is None:
            bb = self.getLargestFaceBoundingBox(rgbImg, skipMulti)
            if bb is None:
                return

        if landmarks is None:
            landmarks = self.findLandmarks(rgbImg, bb)

        npLandmarks = np.float32(landmarks)
        npLandmarkIndices = np.array(landmarkIndices)

        H = cv2.getAffineTransform(
            npLandmarks[npLandmarkIndices], imgDim * MINMAX_TEMPLATE[npLandmarkIndices]
        )
        thumbnail = cv2.warpAffine(rgbImg, H, (imgDim, imgDim))

        return thumbnail


# Get mongo connection string, and connect to Mongo

try:
    from winreg import *
    import pyodbc
    from ctypes import *
except Exception as e:
    print("Failed to import module: ".format(e))

baseDir = os.path.split(os.path.abspath(__file__))[0]
try:
    CVEEdll = cdll.LoadLibrary(os.path.join(baseDir, "CVCIEntityExtractionCsApi.dll"))
except Exception as e:
    print("Failed to load CVCIEntityExtractionCsApi.dll: ".format(e))


DB_MONGO_PORT = "Mongo DB port"
DB_MONGO_INSTANCE = "Mongo DB Instance"
DB_MONGO_ACCESS = "Mongo DB Access"
mongoClientId = 2


def readRegistry(sectionName, regKey):
    instanceName = "Instance001"

    baseRegistry = "SOFTWARE\\CommVault Systems\\Galaxy\\{0}\\".format(instanceName)

    try:
        aReg = ConnectRegistry(None, HKEY_LOCAL_MACHINE)

        aKey = OpenKey(aReg, baseRegistry + sectionName)
        result = QueryValueEx(aKey, regKey)
        return result[0]
    except EnvironmentError:
        LOGGER_Generic.exception("%s: Unable to read registry [%s]" % (__name__, regKey))


def decryptPassword(encryptedValue):
    # Decrypt value
    encryptedValue1 = c_char_p(str(encryptedValue))
    # Allocate memory for cpp function
    decryptedValue = c_char_p(" " * 1024)
    CVEEdll.decrypt.argtypes = [c_char_p, c_char_p]
    try:
        CVEEdll.decrypt(encryptedValue1, decryptedValue)
    except Exception as e:
        LOGGER_Generic.exception("%s: Error decrypting password [%s]" % (__name__, e))
    returnDecryptValue = decryptedValue.value
    return returnDecryptValue


def getCommServDBConnectionString():
    try:
        DBInstanceName = readRegistry("Database", "sINSTANCE")
        encryptedValue = readRegistry("Database", "pACCESS")
        returnDecryptValue = None
        if encryptedValue:
            returnDecryptValue = decryptPassword(encryptedValue)
        if returnDecryptValue:
            usernameLength = int(returnDecryptValue[0:2])
            userId = returnDecryptValue[2 : usernameLength + 2]
            password = returnDecryptValue[usernameLength + 2 :]
            m_srmdbConnectString = (
                "server="
                + DBInstanceName
                + ";database=commserv;uid="
                + userId
                + ";pwd="
                + password
                + ";"
            )
        return m_srmdbConnectString
    except Exception as e:
        LOGGER_Generic.exception("%s: Error getting from CommServ DB [%s]" % (__name__, e))


def getMongoDbConnectionString(mongoInstance, mongoport, encryptedMongoCred):
    try:
        if encryptedMongoCred:
            mongoCred = decryptPassword(encryptedMongoCred)
            usernameLength = int(mongoCred[0:2])
            mongoUser = mongoCred[2 : 2 + usernameLength]
            mongoPass = mongoCred[2 + usernameLength :]
        if mongoInstance:
            return "mongodb://{0}:{1}@{2}:{3}".format(
                mongoUser, mongoPass, mongoInstance, mongoport
            )
    except Exception as e:
        LOGGER_Generic.exception(
            "%s: Error getting Mongo DB connection string [%s]" % (__name__, e)
        )


def getMongoConnectionStringFromCS():
    csConnectionString = getCommServDBConnectionString()
    try:
        conn = pyodbc.connect("DRIVER={SQL Server};" + csConnectionString)
    except:
        LOGGER_Generic.error("Could not connect to CS DB")
    cursor = conn.cursor()
    try:
        # Get Mongo port
        query = (
            "select attrval from app_clientprop where componentNameId = "
            + str(mongoClientId)
            + " and attrName = '"
            + DB_MONGO_PORT
            + "' and ISNUMERIC(attrval) = 1"
        )
        cursor.execute(query)
        for row in cursor.fetchall():
            r_port = int(row.attrval)
        # Get Mongo instance
        query = (
            "select attrval from app_clientprop where componentNameId = "
            + str(mongoClientId)
            + " and attrName = '"
            + DB_MONGO_INSTANCE
            + "'"
        )
        cursor.execute(query)
        for row in cursor.fetchall():
            r_host = row.attrval
        # Get mongo access
        query = (
            "select attrval from app_clientprop where componentNameId = "
            + str(mongoClientId)
            + " and attrName = '"
            + DB_MONGO_ACCESS
            + "'"
        )
        cursor.execute(query)
        for row in cursor.fetchall():
            r_authCred = row.attrval
        if not ("r_host" in locals() and "r_port" in locals() and "r_authCred" in locals()):
            return
    except Exception as e:
        LOGGER_Generic.exception("%s: Error reading data from CS DB [%s]" % (__name__, e))
    return getMongoDbConnectionString(r_host, r_port, r_authCred)


# Connect to Mongo
def connectToMongo(connStr):
    if not connStr:
        raise Exception("MongoDB is required to do face detection! Please intall MongoDB")
    conn = MongoClient(connStr)
    if conn is None:
        raise Exception("Could not connect to MongoDB")
    return conn


# Get incremental id
def getNextSequence(collection, name):
    return collection.find_one_and_update(
        {"_id": name},
        {"$inc": {"seq": 1}},
        projection={"seq": True, "_id": False},
        upsert=True,
        return_document=ReturnDocument.AFTER,
    )["seq"]


CONN_Str = None
mongoClient = None
db = None
gFs = None


def getMongoConnStr():
    global CONN_Str
    if CONN_Str is None:
        CONN_Str = getMongoConnectionStringFromCS()
    return CONN_Str


# Instantiate dlib class
dlibFacePredictor = "./cvshape_predictor_68_face_landmarks.dat"
DLibAlign = AlignDlib(dlibFacePredictor)


# Detect faces given an image path
def detect_faces(image_path, subclientId=None, documentId=None):
    full_img_path = os.path.normpath(image_path)
    try:
        bgrImg = cv2.imread(full_img_path)
        if bgrImg is None or len(bgrImg.shape) < 3:
            return "No face detected"
        rgbImg = cv2.cvtColor(bgrImg, cv2.COLOR_BGR2RGB)
    except Exception as e:
        LOGGER_Generic.exception(
            "%s: Exception caught, skipping [%s] because of [%s]" % (__name__, full_img_path, e)
        )

    faces = DLibAlign.getAllFaceBoundingBoxes(rgbImg)
    if not faces or len(faces) == 0:
        return "No face detected"
    LOGGER_Generic.info("%s: Found [%d] faces in image [%s]" % (__name__, len(faces), image_path))
    response = save_faces(image_path, bgrImg, faces, subclientId, documentId)
    return response


# Align Face
def alignFace(face, bgrImg):
    rgbImg = cv2.cvtColor(bgrImg, cv2.COLOR_BGR2RGB)
    start = time.time()
    imgDim = 96
    alignedFace = DLibAlign.align(
        imgDim, rgbImg, face, landmarkIndices=AlignDlib.OUTER_EYES_AND_NOSE
    )
    if alignedFace is None:
        raise Exception("Unable to align image")
    LOGGER_Generic.info("Face alignment took {} seconds.".format(time.time() - start))
    return alignedFace


# Save faces to disk and insert to Mongo
def save_faces(image_path, bgrImg, faces, subclientId=None, documentId=None):
    img_id = getNextSequence(db.counters, "image")
    fileName = os.path.splitext(os.path.basename(image_path))[0]
    thumbIds = []
    # Form response message
    response = {}
    response["image_id"] = img_id
    thumbs = []
    dir_path = os.path.dirname(image_path)
    for i in range(len(faces)):
        thumb_path = os.path.normpath(os.path.join(dir_path, fileName + "_face" + str(i) + ".jpg"))
        LOGGER_Generic.info(thumb_path)
        thumb_id = getNextSequence(db.counters, "thumbnail")
        thumbIds.append(thumb_id)
        alignedFace = alignFace(faces[i], bgrImg)
        faceBinary = Binary(pickle.dumps(alignedFace, protocol=2), subtype=128)
        try:
            # Formulate face coordinates, and hanlde when coordinates is negative
            if faces[i].left() > 0:
                x, w = (faces[i].left(), faces[i].width())
            else:
                x, w = (0, faces[i].width() + faces[i].left())

            if faces[i].top() > 0:
                y, h = (faces[i].top(), faces[i].height())
            else:
                y, h = (0, faces[i].height() + faces[i].top())
            face = bgrImg[y : y + h, x : x + w]
            cv2.imwrite(thumb_path, face)
            coordinates = {"left": x, "top": y, "height": h, "width": w}
            faceFsId = gFs.put(open(thumb_path, "rb"))
            db.thumbnail.insert(
                {
                    "_id": thumb_id,
                    "image_id": img_id,
                    "path": thumb_path,
                    "coordinates": coordinates,
                    "aligned_face_binary": faceBinary,
                    "faceFsId": faceFsId,
                }
            )
            curThumb = {"thumbnail_id": thumb_id, "coordinates": coordinates}
            thumbs.append(curThumb)
        except Exception as e:
            LOGGER_Generic.exception("Unable to save face. Warning: {}".format(e))
            if subclientId is not None and documentId is not None:
                db.image.insert(
                    {
                        "_id": img_id,
                        "path": image_path,
                        "thumbIds": thumbIds,
                        "edgeSubclientId": subclientId,
                        "edgeDocumentId": documentId,
                    }
                )
            else:
                db.image.insert({"_id": img_id, "path": image_path, "thumbIds": thumbIds})
    response["faces"] = thumbs
    return response


# Check if a file has been processed or not
def exist(f):
    if db.image.find({"path": f}).count() > 0:
        return True
    else:
        return False


# Get reponse message if the image already in DB
def getResponseMessage(f):
    cur = db.image.find_one({"path": f})
    response = {}
    response["image_id"] = cur["_id"]
    thumbs = []
    for thumbId in cur["thumbIds"]:
        tCur = db.thumbnail.find_one({"_id": thumbId})
        curThumb = {"thumbnail_id": thumbId, "coordinates": tCur["coordinates"]}
        thumbs.append(curThumb)
    response["faces"] = thumbs
    return response


def doAnalysis(input, params=None):
    # Check Mongo connection
    global LOGGER_Generic, mongoClient, db, CONN_Str, gFs, CA_ERROR_CODES
    configureLog()
    if mongoClient is None:
        # Initialize Mongo connection
        CONN_Str = getMongoConnectionStringFromCS()
        mongoClient = connectToMongo(CONN_Str)
        db = mongoClient.InterFace
        gFs = gridfs.GridFS(db)
    LOGGER_Generic.info("FD params is {}".format(params))
    subclientId = None
    documentId = None
    if params is not None:
        if "edgeSubclientId" in params:
            try:
                subclientId = int(params["FDedgeSubclientId"])
            except ValueError:
                LOGGER_Generic.exception("Couldn't parse edge subclient id")
        if "edgeDocumentId" in params:
            try:
                documentId = params["FDedgeDocumentId"]
            except ValueError:
                LOGGER_Generic.exception("Couldn't parse edge document id")
    if exist(input):
        response = getResponseMessage(input)
    else:
        return {
            "ErrorCode": CA_ERROR_CODES["loggerNotFound"],
            "ErrorMessage": "Logger object was not sent in params",
        }
    try:
        if mongoClient is None:
            # Initialize Mongo connection
            CONN_Str = getMongoConnectionStringFromCS()
            mongoClient = connectToMongo(CONN_Str)
            db = mongoClient.InterFace
            gFs = gridfs.GridFS(db)
        LOGGER_Generic.info("FD params is {}".format(params))
        subclientId = None
        documentId = None
        if params is not None:
            if "edgeSubclientId" in params:
                try:
                    subclientId = int(params["FDedgeSubclientId"])
                except ValueError:
                    LOGGER_Generic.exception("Couldn't parse edge subclient id")
            if "edgeDocumentId" in params:
                try:
                    documentId = params["FDedgeDocumentId"]
                except ValueError:
                    LOGGER_Generic.exception("Couldn't parse edge document id")
        if exist(input):
            output = getResponseMessage(input)
        else:
            output = detect_faces(input, subclientId, documentId)

        response = {
            "response": output,
            "ErrorCode": CA_ERROR_CODES["success"],
            "ErrorMessage": None,
        }
    except Exception as e:
        response = {
            "ErrorCode": CA_ERROR_CODES["FDError"],
            "ErrorMessage": "Face Detection failure: {}".format(e),
        }
    return response


if __name__ == "__main__":
    doAnalysis("C:\\catest\\trump8.jpg")
