import ctypes
import inspect
import os
import collections
import json
from contextlib import contextmanager
from CvEEConfigHelper import CA_ERROR_CODES


def _init():
    # Get the path where the current module resides
    modulePath = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))

    # Configure the pipeline DLL
    hllDll = ctypes.CDLL(os.path.join(modulePath, "BtxPipeline.dll"), mode=ctypes.RTLD_GLOBAL)

    # Configure the global _hllProc function
    global _hllProc
    _hllProc = hllDll.pipelinestr_proc
    _hllProc.restype = ctypes.c_void_p
    _hllProc.argtypes = [ctypes.c_void_p, ctypes.c_char_p]

    # Configure the global _hllFree function
    global _hllFree
    _hllFree = hllDll.pipeline_FreePtr
    _hllFree.restype = None
    _hllFree.argtypes = [ctypes.c_void_p]

    # Configure the local hllInit function
    hllInit = hllDll.pipelinestr_init
    hllInit.restype = ctypes.c_void_p
    hllInit.argtypes = [
        ctypes.c_char_p,
        ctypes.c_char_p,
        ctypes.c_int,
        ctypes.c_int,
        ctypes.c_char_p,
        ctypes.c_char_p,
        ctypes.c_int,
        ctypes.c_int,
        ctypes.c_long,
    ]

    # Configure the local _hllFini function
    global _hllFini
    _hllFini = hllDll.pipelinestr_fini
    _hllFini.restype = ctypes.c_void_p
    _hllFini.argtypes = [ctypes.c_void_p]

    # Set the various initialization parameters (see the docs)
    dataPath = "."  # Using a relative path so it works on Cygwin's Python too
    fileConf = os.path.join(dataPath, "cfg_ENG_CONST_REGEX_v05.04.xml")
    fileGram = os.path.join(dataPath, "ENG_CONST_GRAM_v20ENT.bto")
    extractionType = 1057
    gramType = 5
    morph2 = "libmorph_noautoload.so"  # Not used under Windows
    dict2 = dataPath
    processType = 0
    outFlags = 3
    maxtime = 65542

    p1 = ctypes.c_char_p(fileConf.encode("utf8"))
    p2 = ctypes.c_char_p(fileGram.encode("utf8"))
    p3 = ctypes.c_int(extractionType)
    p4 = ctypes.c_int(gramType)
    p5 = ctypes.c_char_p(morph2.encode("utf8"))
    p6 = ctypes.c_char_p(dict2.encode("utf8"))
    p7 = ctypes.c_int(processType)
    p8 = ctypes.c_int(outFlags)
    p9 = ctypes.c_long(maxtime)

    global _cfg
    _cfg = hllInit(p1, p2, p3, p4, p5, p6, p7, p8, p9)


def _cleanup():
    p1 = ctypes.c_void_p(_cfg)
    _hllFini(p1)


def entity_type_text(type):
    # Map entity types to text (cosmetic)
    return {
        "0": "unknown",
        "1": "person",
        "2": "license-plate",
        "3": "place",
        "4": "phone",
        "5": "email",
        "6": "brand",
        "7": "organization",
        "8": "url",
        "9": "ip",
        "10": "date",
        "11": "hour",
        "12": "nif",
        "13": "cif",
        "14": "iban",
        "15": "currency",
        "16": "address",
        "17": "hashtag",
        "18": "username",
        "19": "other",
        "777": "reserved",
    }.get(type, "")


def get_entities(text):
    p1 = ctypes.c_void_p(_cfg)
    p2 = ctypes.c_char_p(text)
    result = _hllProc(p1, p2)

    entities = []
    if result is not None:
        raw_output = ctypes.c_char_p(result).value

        # Parse the output and return a JSON-like array of dictionaries
        for line in raw_output.decode("utf8").split("\n"):
            if line != "":
                (id, entity, type, occurrences, entity_norm) = line.split(";")
                entities.append(
                    {"entity": entity, "type": entity_type_text(type), "entity_norm": entity_norm}
                )
        _hllFree(result)
    return entities


import time


@contextmanager
def TimeIt(label):
    start = time.clock()
    try:
        yield
    finally:
        end = time.clock()
        print("{} : {}".format(label, end - start))


def formatResults(result):
    entities = {}
    for r in result:
        type_result = r["type"]
        if type_result not in entities:
            entities[type_result] = []
        entities[type_result].append(r["entity"])
    return entities


def preProcess(params={}):
    global CA_ERROR_CODES
    try:
        _init()
        return {"ErrorCode": CA_ERROR_CODES["success"]}
    except Exception as e:
        return {
            "ErrorCode": CA_ERROR_CODES["BitextPreProcessingError"],
            "ErrorMessage": "Error while preprocessing Bitext: {}".format(e),
        }


def doAnalysis(processing_input, params={}):
    global CA_ERROR_CODES
    try:
        contentid = ""
        if "contentid" in params:
            contentid = params["contentid"]
        # _init()
        entities = formatResults(get_entities(processing_input))
        # _cleanup()
        entities["ErrorCode"] = 0
        entities["ErrorMessage"] = None
        return entities
    except Exception as e:
        return {
            "ErrorCode": CA_ERROR_CODES["BitextError"],
            "ErrorMessage": "Error during Bitext extraction for contentid {}: {}".format(
                contentid, e
            ),
        }


if __name__ == "__main__":
    # Process stdin input
    print(doAnalysis("His name is Bob. His phone number is 9860412495"))
