'''
/******************************************************************************/
/*  Copyright (c) CommVault Systems                                           */
/*  All Rights Reserved                                                       */
/*                                                                            */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF CommVault Systems          */
/*  The copyright notice above does not evidence any                          */
/*  actual or intended publication of such source code.                       */
/******************************************************************************/
Created on Mar 21, 2012
Revision: $Id: main.py,v 1.72.2.111.4.1 2021/01/21 21:59:07 jge Exp $
$Date: 2021/01/21 21:59:07 $
@author: David Maisonave
'''
from Database import Database
from Common import *
try:
	import winreg as _winreg
except ImportError:
	import unix_winreg as _winreg
import os, sys, datetime, shutil, re
from CustomerFeedbackDatabase import CustomerFeedbackDatabase
from CSSXMLDATA_FileData import CSSXMLDATA_FileData
from xml.dom.minidom import parse, parseString
from CssXmlData import CssXmlData
#from CSSXMLResultDatabase import CSSXMLResultDatabase
import Logger
import logging
from logging.handlers import RotatingFileHandler
import base64
import time
import optparse
import calendar
import xml.etree.ElementTree as ET            
import subprocess
import re
import csv
import json
from urllib.parse import urlencode
from http.client import HTTPSConnection,HTTPConnection
import zipfile
import queue as Q
import glob
if(platform.system() == "Linux"):
    import fcntl
from threading import Lock
from multiprocessing.pool import ThreadPool
from MetricsConfig import MetricsConfig
import os.path

class SurveyProcessor(object):
    def __init__(self, cvdpid, deleteXml, runonce, Log):
        self.Log = Log
        self.MetricsConfigObj = None
        self.CvdPid = cvdpid
        self.IsOpenConnection = False
        
        self.IsCSOpenConnection = False
        self.IsCloudServicesOpenConnection = False
        
        self.Log.debug("Establish connection to main DB and initialize metrics processing")
        
        self.DeleteXmlFileAfterProcessing = deleteXml
        self.RunOneTimeProcessing = runonce
        self.HttpProxyHost = None
        self.HttpProxyPort = 0
        self.HttpProxyUser = None
        self.HttpProxyPasswd = None
        self.UseMetricsServerAsProxy = True
        self._unregisteredCommcells = Q.Queue()
        self.commcelldict = dict()
        self.failedcommcellset = set()
        self.useHttpProxyPrivate = False
        self.LinuxMetricsFileDescriptor = None
        self.ProcessCollectedResultsViaApi = None

    def LockLinuxMetricsFile(self):
        self.LinuxMetricsFileDescriptor = open('lockfile.LOCK', 'w+')
        try:
            fcntl.lockf(self.LinuxMetricsFileDescriptor, fcntl.LOCK_EX | fcntl.LOCK_NB)
            return True
        except BlockingIOError:
            print("ERROR :: " + "Metrics driver process is already running")
            return False

    def isLinuxMetricsFileLocked(self):
        isLocked = self.LockLinuxMetricsFile()
        if isLocked == True:
            print("INFO :: " + "Metrics driver process is NOT running")
            self.LinuxMetricsFileDescriptor.close()     
        return isLocked
        
    def IsDriverRunning(self):
        if(platform.system() == "Linux"):
            return not self.isLinuxMetricsFileLocked()
        GetSuccess= isDriverProcessRunning(self.Log)
        if GetSuccess == True:
            self.Log.error("Another Instance of Base\Cloud\main.py already running.")

        return GetSuccess
    
    def SetDriverRunning(self):
        if(platform.system() == "Linux"):
            GetLock = self.LockLinuxMetricsFile()
            if GetLock == True:
                print("INFO :: " + "Lock set!")
            return GetLock
        self.HMutex = setDriverStartMutex(self.Log)
        if self.HMutex is None:
            self.Log.error("Failed to create driver alive mutex.")
            return False
        self.Log.info("Drive alive mutex created.")
        return True
        
    def ResetDriverRunning(self):
        if(platform.system() == "Linux"):
            self.LinuxMetricsFileDescriptor.close()
            return True
        resetsetDriverStartMutex(self.HMutex, self.Log)
    
    def SetSurveyPathsAndEnv(self):
        self.Log.info("Get environmental variables and settings")
        self.MetricsConfigObj = MetricsConfig('CVSurvey').get_instance()
        
        self.CustomerFeedbackDB = None
        self.CloudDB = None
        self.CSDB    = None
        self.CloudServicesDB    = None
        self.CustomerFeedbackDB = CustomerFeedbackDatabase()
        if self.CustomerFeedbackDB._bOpenConnection == False:
            self.Log.error("Failed to connect to database CVCloud.")
            return False

        self.UNCFolderToUploadFiles = GetEnvVar("UNCFolder_Path", "")
        self.UNCUser = GetEnvVar("UNC_Username", "")
        self.UNCPassword = base64.b64decode(GetEnvVar("UNC_Password", "")).decode("utf-8")
        self.UNCCopyNetUseCommand = False
        if self.UNCFolderToUploadFiles != '':
            command = 'net use %s %s /user:%s /persistent:yes' % (self.UNCFolderToUploadFiles, self.UNCPassword, self.UNCUser)
            retCode = os.system(command)
            if retCode != 0:
                self.Log.error("Error executing net use remote command - retcode :: " + str(retCode))
                self.Log.error("Copy to local folder instead of UNC path as 'net use' command failed to initialize")  
                self.UNCCopyNetUseCommand = False
            else:
                self.Log.info('Successfully executed net use remote command')
                self.UNCCopyNetUseCommand = True      
        
        self.SimpanaCloudRegPath = self.MetricsConfigObj.sSimpanaCloudRegPath
        self.SimpanaBaseRegPath = self.MetricsConfigObj.sSimpanaBaseRegPath 
        self.SimpanaCloudDBRegPath = self.MetricsConfigObj.sSimpanaDatabaseRegPath 

        self.Log.info("Get the simpana base directory path.")
        try:
            aReg = _winreg.ConnectRegistry(None,_winreg.HKEY_LOCAL_MACHINE)
            cloudRegKey = _winreg.OpenKey(aReg, self.SimpanaCloudRegPath)
            baseRegKey = _winreg.OpenKey(aReg, self.SimpanaBaseRegPath)
            dbRegKey = _winreg.OpenKey(aReg, self.SimpanaCloudDBRegPath)
            
            value, Success = _winreg.QueryValueEx(baseRegKey, r"dBASEHOME")
            self.SimpanaBaseFolder = value
            Log.debug("SimpanaBaseFolder %s" % (self.SimpanaBaseFolder))
            
            try:
                value, Success = _winreg.QueryValueEx(baseRegKey, r"dGALAXYHOME")
                self.SimpanaHomeFolder = value
            except Exception as err:
                self.SimpanaHomeFolder = os.path.abspath(os.path.join(self.SimpanaBaseFolder, os.pardir))
            Log.debug("SimpanaHomeFolder %s" % (self.SimpanaHomeFolder))
            
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nXMLPATH")
                self.FolderContainingXmlFile = value
            except Exception as err:
                self.FolderContainingXmlFile = self.SimpanaBaseFolder + "\\Reports\\MetricsUpload\\Upload"
            
            Log.debug("FolderContainingXmlFile %s" % (self.FolderContainingXmlFile))
            
            #Create the folder if does not exist
            if not os.path.exists(self.FolderContainingXmlFile):
                os.makedirs(self.FolderContainingXmlFile)

            try:
                value, type = _winreg.QueryValueEx(baseRegKey, r"dGALAXYTEMPDIR")
                self.SimpanaTempFolder = value
            except Exception as err:
                self.SimpanaTempFolder = self.SimpanaBaseFolder + "\\Temp"
            Log.debug("SimpanaTempFolder %s" % (self.SimpanaTempFolder))

            self.CloudDbName = self.MetricsConfigObj.sCLOUDDBNAME 
            self.CloudDbDSN = self.MetricsConfigObj.sCLOUDCONNECTION
            self.CloudDbConnStr = self.MetricsConfigObj.sCLOUDDBCONNSTR
            self.sz_cloudusername = self.MetricsConfigObj.sCLOUDDBUser
            self.sz_cloudpassword = self.MetricsConfigObj.sCLOUDDBPassword
            Log.debug("Cloud DB name [%s]. Cloud DSN [%s]" % (self.CloudDbName, self.CloudDbDSN))
            self.CloudDB = Database(ConnStr = self.CloudDbConnStr, DSNName = self.CloudDbDSN, Dbname = self.CloudDbName)
            
            CSDbName = self.MetricsConfigObj.sCSDBNAME  
            CSDbDSN = self.MetricsConfigObj.sCONNECTION 
            CSDbConnStr = self.MetricsConfigObj.sCSDBCONNSTR 
            Log.debug("CommServe DB name [%s]. Commserv DSN [%s]" % (CSDbName, CSDbDSN))
            self.CSDB = Database(ConnStr = CSDbConnStr, DSNName = CSDbDSN, Dbname = CSDbName)
            
            CloudServicesDbName = self.MetricsConfigObj.sCLOUDSERVICESDBNAME   
            CloudServicesDbDSN = self.MetricsConfigObj.sCLOUDSERVICESCONNECTION
            CloudServicesDbConnStr = self.MetricsConfigObj.sCLOUDSERVICESCONNSTR
            Log.debug("CloudServices DB name [%s]. CloudServices DSN [%s]" % (CloudServicesDbName, CloudServicesDbDSN))
            self.CloudServicesDB = Database(ConnStr = CloudServicesDbConnStr, DSNName = CloudServicesDbDSN, Dbname = CloudServicesDbName)

            self.PruneScheduleHrs = 24
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nPruneScheduleHours")
                self.Log.info("regKey nPruneScheduleHours found. Run pruning every %s hours"%value)
                self.PruneScheduleHrs = int(value)
            except Exception as err:
                self.Log.info("regKey nPruneScheduleHours not found. Going with default 24 hours")
                self.PruneScheduleHrs = 24
            Log.debug("PruneScheduleHrs %d" % (self.PruneScheduleHrs))
               
            self.ProcessFrequency = 1 
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nProcessFrequencyMins")
                self.Log.info("regKey nProcessFrequencyMins found %s minutes"%value)
                self.ProcessFrequency = int(value)
            except Exception as err:
                self.Log.info("regKey nProcessFrequencyMins not found. Going with default 1 minute")
                self.ProcessFrequency = 1
            Log.debug("ProcessFrequency %d" % (self.ProcessFrequency))
            
            self.CCGroupSyncFreq = 1440 #Sync once in 24 hours
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nCCGroupSyncFreqMins")
                self.Log.info("regKey nCCGroupSyncFreqMins found %s minutes"%value)
                self.CCGroupSyncFreq = int(value)
                if self.CCGroupSyncFreq == 0:
                    self.Log.info("CommCell Groups sync operation is disabled")
            except :
                self.Log.info("regKey nCCGroupSyncFreqMins not found. Going with default 24 hours")
                self.CCGroupSyncFreq = 1440
            Log.debug("CommCell group sync frequency is set to [%d] minutes" % (self.CCGroupSyncFreq))            
            
            self.UpdateLocationFreq = 1440 #Update commcell locations in 24 hours
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nUpdateCCLocFreqMins")
                self.Log.info("regKey nUpdateCCLocFreqMins found %s minutes"%value)
                self.UpdateLocationFreq = int(value)
                if self.UpdateLocationFreq == 0:
                    self.Log.info("Update Commcell Location is disabled")
            except :
                self.Log.info("regKey nUpdateCCLocFreqMins not found. Going with default 24 hours")
                self.UpdateLocationFreq = 1440
            Log.debug("Update commcell location is set to [%d] minutes" % (self.UpdateLocationFreq))

            self.LastMetricsPruningTime = 0
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nLastMetricsPruningTime")
                self.LastMetricsPruningTime = int(value)
            except Exception as err:
                self.LastMetricsPruningTime = 0
            Log.debug("LastMetricsPruningTime %d" % (self.LastMetricsPruningTime))

            self.EnableDBMaintenance = 0
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nEnableDBMaintenance")
                self.EnableDBMaintenance = int(value)
            except Exception as err:
                self.EnableDBMaintenance = 0
            Log.debug("EnableDBMaintenance %d" % (self.EnableDBMaintenance))
            
            self.DBInstanceName = None
            try:
                value, type = _winreg.QueryValueEx(dbRegKey, r"sCLOUDINSTANCE")
                self.DBInstanceName = value
            except Exception as err:
                self.DBInstanceName = None

            if self.EnableDBMaintenance != 0:
                
                if self.DBInstanceName == None:
                    Log.error("Failed to get sCLOUDINSTANCE, DB maintainance will be skipped")
                    self.EnableDBMaintenance = 0
                else:              
                    self.DBMaintenanceFullInterval = 0
                    try:
                        value, Success = _winreg.QueryValueEx(cloudRegKey, r"nDBMaintenanceFullInterval")
                        self.DBMaintenanceFullInterval  = int(value)
                    except Exception as err:
                        self.DBMaintenanceFullInterval  = 7
                    Log.debug("DBMaintenanceFullInterval %d" % (self.DBMaintenanceFullInterval ))
                   
                    #setting LastDBMaintenanceFull to 86400 because datetime.datetime.fromtimestamp fails for 0 - 86399 (windows bug)
                    #86400 defaults to Jan 02 1970 in timestamp format whereas 0 defaults to Jan 01 1970
                    self.LastDBMaintenanceFull = 86400
                    try:
                        value, Success = _winreg.QueryValueEx(cloudRegKey, r"nLastDBMaintenanceFull")
                        self.LastDBMaintenanceFull  = int(value)
                    except Exception as err:
                        self.LastDBMaintenanceFull  = 86400
                    Log.debug("DBMaintenanceFullInterval %d" % (self.LastDBMaintenanceFull ))
            
            self.nDisablePruning = 0        
            try:
                value, Success = _winreg.QueryValueEx(cloudRegKey, r"nDisablePruning")
                self.nDisablePruning = value
            except Exception as err:
                self.nDisablePruning = 0;
            
            Log.debug("nDisablePruning: %s" % ("TRUE" if self.nDisablePruning>0 else "FALSE"))                            

        except Exception as err:
            self.Log.exception("Exception reading registry keys.")
            return False
            
        self.ArchiveFolder = os.path.join(self.FolderContainingXmlFile , "Archive")
        self.FailedFolder = os.path.join(self.FolderContainingXmlFile , "Failed")
        self.BlockedFolder = os.path.join(self.FolderContainingXmlFile , "Blocked")
        self.ZipArchiveFolder = os.path.join(self.FolderContainingXmlFile , "ZipArchive")
            
        self.UpgradeArchiveFolder = os.path.join(self.FolderContainingXmlFile , "UpgradeReadiness_Archive")
        self.UpgradeFailedFolder = os.path.join(self.FolderContainingXmlFile , "UpgradeReadiness_Failed")
        self.UpgradeBlockedFolder = os.path.join(self.FolderContainingXmlFile , "UpgradeReadiness_Blocked")

        for folder in [self.ArchiveFolder, self.FailedFolder, self.BlockedFolder,
                       self.UpgradeArchiveFolder, self.UpgradeFailedFolder,
                       self.UpgradeBlockedFolder, self.ZipArchiveFolder]:
            if not os.path.exists(folder):
                os.makedirs(folder)

        self.Log.info(__name__ + " Establish cloud db connection")
    
        try:
            if self.CloudDB.OpenSimpleDSN() == True:
                self.IsOpenConnection = True
        except Exception as err:
            if self.IsOpenConnection == False:
                self.Log.info(__name__ + " :: Failed to open DB '%s', with error %s" %(self.CloudDbName, self.CloudDB.GetErrorErrStr()))
                return False

        if  CSDbName != None :
            try:
                if self.CSDB.OpenSimpleDSN() == True:
                    self.IsCSOpenConnection = True
            except Exception as err:
                if self.IsCSOpenConnection == False:
                    self.Log.info(__name__ + " :: Failed to open DB '%s', with error %s" %(CSDbName, self.CSDB.GetErrorErrStr()))
                    return False					
                   
        if CloudServicesDbName != None and len(CloudServicesDbName) > 0:
            try:
                if self.CloudServicesDB.OpenSimpleDSN() == True:
                    self.IsCloudServicesOpenConnection = True
            except Exception as err:
                if self.IsCloudServicesOpenConnection == False:
                    self.Log.info(__name__ + " :: Failed to open DB '%s', with error %s" %(CloudServicesDbName, self.CloudServicesDB.GetErrorErrStr()))
                    return False           
    
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH(NOLOCK) WHERE Name = 'isEvalAndNFRBlocked';" #isEvalAndNFRBlocked stands for only Eval CommCells
    
        self.isEvalCCBlocked = False
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row and row.Value == '1':
                    self.isEvalCCBlocked = True
                break
    
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH(NOLOCK) WHERE Name = 'BlockNFRCC';"
    
        self.isNFRCCBlocked = False
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row and row.Value == '1':
                    self.isNFRCCBlocked = True
                break
    
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH(NOLOCK) WHERE Name = 'isPrivate';"
    
        self.isPrivate = False
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row and row.Value == '1':
                    self.isPrivate = True
                break
            
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH(NOLOCK) WHERE Name = 'PendingXMLResultThreshold';"
        self.PendingXMLResultThreshold = 10000
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.PendingXMLResultThreshold = int(row.Value)
                break
        self.Log.info('PendingXMLResultThreshold: %d ' % (self.PendingXMLResultThreshold))                
    
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH (NOLOCK) WHERE Name = 'ArchiveFilePruneInterval';"
        self.ArchiveFilePruneInterval = 15
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.ArchiveFilePruneInterval = int(row.Value)
                break
        self.Log.info('ArchiveFilePruneInterval: %d ' % (self.ArchiveFilePruneInterval))				
                    
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH (NOLOCK) WHERE Name = 'BlockedFilePruneInterval';"    
        self.BlockedFilePruneInterval = 15
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.BlockedFilePruneInterval = int(row.Value)
                break
        self.Log.info('BlockedFilePruneInterval: %d ' % (self.BlockedFilePruneInterval))				
                    
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH (NOLOCK) WHERE Name = 'FailedFilePruneInterval';"
        self.FailedFilePruneInterval = 60
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.FailedFilePruneInterval = int(row.Value)
                break
        self.Log.info('FailedFilePruneInterval: %d ' % (self.FailedFilePruneInterval))				

        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH (NOLOCK) WHERE Name = 'ProcessFilePruneInterval';"
        self.ProcessFilePruneInterval = 60
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.ProcessFilePruneInterval = int(row.Value)
                break
        self.Log.info('ProcessFilePruneInterval: %d ' % (self.ProcessFilePruneInterval))
        self._upgradedCSInfo = dict()
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH (NOLOCK) WHERE Name = 'CheckUpgradedCommcells';"
        self.CheckUpgradedCommcells = 0
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.CheckUpgradedCommcells = int(row.Value)
                break
        self.Log.info('CheckUpgradedCommcells: %d ' % (self.CheckUpgradedCommcells))
        
        query = "SELECT Value FROM dbo.cf_SurveyConfig WITH (NOLOCK) WHERE Name = 'NoOfProcessingThreads';"
        self.NoOfProcessingThreads = 0
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row:
                    self.NoOfProcessingThreads = int(row.Value)
                break
        self.Log.info('NoOfProcessingThreads: %d ' % (self.NoOfProcessingThreads))                                
        
        self.IncludedIPList = []
        self.BlockedIPList = []
        self.CopytoHttpServerlist = dict()
        self.RestrictQuery = dict()
        self.ScrubCC = dict()
        self.RegEx = ''
        self.ExclusionDict = dict()
        self.TransferXMLFileConstraint = TransferXMLFileConstraint()
        self.MaxXMLFilesToProcess = 5000
         
        #Maintain the list of queries which are valid for processing in the metrics server
        self.ValidQueryIds = []                    
        
        #Retrieve timeout value for unzipping subprocessor
        self.UnzipFileTimeout = 300;        #set to 5 mins
        query = "SELECT Value FROM cf_SurveyConfig WITH (NOLOCK) WHERE name LIKE 'UnzipFileTimeout';"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row: 
                    self.UnzipFileTimeout = int(row.Value)
                break
        self.Log.debug("UnzipFileTimeout:%d" % (self.UnzipFileTimeout))

        #Retrieve limit to the number of unregistered commcells that can be registered in one go through workflow
        self.UnregisteredCommcellsLimit = 10;        #set to 10
        query = "SELECT Value FROM cf_SurveyConfig WITH (NOLOCK) WHERE name LIKE 'UnregisteredCommcellsLimit';"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row: 
                    self.UnregisteredCommcellsLimit = int(row.Value)
                break
        self.Log.debug("UnregisteredCommcellsLimit:%d" % (self.UnregisteredCommcellsLimit))

        #Retrieve optional regex to be scrubbed
        query = "SELECT Value FROM cf_SurveyConfig WITH(NOLOCK) WHERE name LIKE 'CommservSurveyScrubRegex';"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row: 
                    self.RegEx = row.Value
                break
        if len(self.RegEx) <= 0:
            self.RegEx = "(\B(?<!<)/[\w./$-]+(?: [\w./$-]+)*)|((\\\\\\\\[\w.]+)(\\\\([a-zA-Z0-9_.-\\\\$]+ ?)+)?\\\\?)|(([a-zA-Z]:\\\\)+(([a-zA-Z0-9_.-\\\\$]+ ?)+)*\\\\?)"
            
        #Retrieve maximum number of files to process in single iteration
        query = "SELECT Value FROM cf_SurveyConfig WITH(NOLOCK) WHERE name LIKE 'MaxXMLFilesToProcess';"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            row = self.CloudDB.m_cursor.fetchone()
            if (row and (int(row.Value) > 0)):
                self.MaxXMLFilesToProcess = int(row.Value)                        

        #Retrieve Attribute Names and Element names which are to be excluded from scrubbing
        query = "SELECT Value FROM cf_SurveyConfig WITH(NOLOCK) WHERE name LIKE 'CommservSurveyExcludeScrubList';"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row: 
                    tmpRoot = ET.fromstring(row.Value)
                    for child in tmpRoot.findall('Name'):
                        if child.text is not None:
                            self.ExclusionDict[child.text] = child.text
                break        
          
        #Retrieve optional RetryIntervalTransferXMLFile value to specify minimum retry interval for secondary metric server in minutes
        query = "SELECT Value FROM cf_SurveyConfig WITH(NOLOCK) WHERE name LIKE 'RetryIntervalTransferXMLFile';"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            while 1:
                row = self.CloudDB.m_cursor.fetchone()
                if row: 
                    self.TransferXMLFileConstraint.RetryIntervalMinutes = int(row.Value)
                break
           
        self.Log.debug("ForwardingXML RetryIntervalMinutes [%d] " % (self.TransferXMLFileConstraint.RetryIntervalMinutes) )
        try:
            SimpanaConfigDir = self.SimpanaHomeFolder
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "Reports")
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "MetricsUpload")
            IncludedIPListFile = os.path.join(SimpanaConfigDir, "IncludeIPs.txt")
            
            with open(IncludedIPListFile) as IPs:
                self.IncludedIPList = IPs.read().splitlines()                
        except IOError:
            self.Log.info("Cannot Access Include IP list file IncludeIPs.txt... Proceeding...")
    
        try:
            SimpanaConfigDir = self.SimpanaHomeFolder
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "Reports")
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "MetricsUpload")
            BlockedIPListFile = os.path.join(SimpanaConfigDir, "BlockedIPs.txt")

            with open(BlockedIPListFile) as IPs:
                self.BlockedIPList = IPs.read().splitlines()                
        except IOError:
            self.Log.info("Cannot Access Exclude IP list file BlockedIPs.txt... Proceeding...")
        
        self.CurRelPreUpgradeCCIDList = []
        self.LastRelPreUpgradeCCIDList = []
        self.PreLastRelPreUpgradeCCIDList = []
        self.CurRelPreUpgrade = []
        self.LastRelPreUpgrade = []
        self.PreLastRelPreUpgrade = []
        try:
            CurRelPreUpgrade = os.path.dirname(os.path.realpath(__file__)) + '\CurRelPreUpgrade.lst'
            LastRelPreUpgrade = os.path.dirname(os.path.realpath(__file__)) + '\LastRelPreUpgrade.lst'
            PreLastRelPreUpgrade = os.path.dirname(os.path.realpath(__file__)) + '\PreLastRelPreUpgrade.lst'
            with open(CurRelPreUpgrade) as CurRelPreUpgradeList:
                self.CurRelPreUpgrade = CurRelPreUpgradeList.read().splitlines()
            with open(LastRelPreUpgrade) as LastRelPreUpgradeList:
                self.LastRelPreUpgrade = LastRelPreUpgradeList.read().splitlines()
            with open(PreLastRelPreUpgrade) as PreLastRelPreUpgradeList:
                self.PreLastRelPreUpgrade = PreLastRelPreUpgradeList.read().splitlines()
        except IOError:
            self.Log.info("Cannot Access PreUpgrade lst files... Proceeding...")
            
        useHttpProxy = False
        ForwardingXML = None
        if  self.IsCSOpenConnection == True:
          try:
             query = "SELECT value FROM GXGlobalParam WITH(NOLOCK) where name = 'CommservSurveyForward' and modified=0;"
             if self.CSDB.Execute(query) == True:
                row = self.CSDB.m_cursor.fetchone()
                if row and len(row.value) > 0:  #MR 120881
                    ForwardingXML=row.value
                    self.Log.debug("ForwardingXML value [%s]" % (ForwardingXML) )
                else:
                    ForwardingXML = None

             query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='SendLogsUseHTTPProxy' and modified=0;"
             if self.CSDB.Execute(query) == True:
                while 1:
                    row = self.CSDB.m_cursor.fetchone()
                    if row and row.value == '1':
                        useHttpProxy = True
                    break

             if useHttpProxy == True:
                self.useHttpProxyPrivate = False
                query = "select value FROM GXGlobalParam WITH(NOLOCK) WHERE name ='SendLogsUseHTTPProxyPrivate' and modified=0;"
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        if row and row.value == '1':
                            self.useHttpProxyPrivate = True
                            self.Log.info("HttpProxy is enabled for private uploading")
                        break

             if self.isPrivate == False:
                 self.fetchConstantsForLocation()


             if useHttpProxy == True:
                 self.Log.info("Using HTTP Proxy" )
                 
                 query = "SELECT value FROM GXGlobalParam WITH(NOLOCK) where name = 'SendLogsHTTPProxyPassword' and modified=0;"
                 if self.CSDB.Execute(query) == True:
                    row = self.CSDB.m_cursor.fetchone()
                    if row:
                        self.HttpProxyPasswd=GetMetricsSimpleData(row.value,self.SimpanaBaseFolder,Log)

                 query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='SendLogsHTTPProxySite' and modified=0;"
                 if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        if row and len(row.value) > 0:
                            self.HttpProxyHost = row.value
                        break
                 query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='SendLogsHTTPProxyPort' and modified=0;"
                 if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        if row :
                            self.HttpProxyPort = int(row.value)
                        break
                 self.HttpProxyUser = None
                 query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='SendLogsHTTPProxyUser' and modified=0;"
                 if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        if row :
                            self.HttpProxyUser = row.value
                        break   
                 Log.debug("Use HTTP Proxy Site [%s] Port [%s] User [%s] " % (str(self.HttpProxyHost),str(self.HttpProxyPort),str(self.HttpProxyUser)) )
          except:
             self.HttpProxyHost = None  
             self.Log.exception("Proxy information can't be obtained - query %s failed" % query)

        if self.isPrivate == True:
            SimpanaConfigDir = self.SimpanaHomeFolder
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "Reports")
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "MetricsUpload")
            ReadHttpServerInfo(self.RestrictQuery, self.ScrubCC, self.CopytoHttpServerlist, self.Log, ForwardingXML, SimpanaConfigDir)
        else:
            Log.debug("Skipping forwarding of XML files as the setup is Public.")

        #get the user creds required to execute workflow on cloud
        self.workflowUserCreds = {}
        if self.isPrivate == False:
            self.workflowUserCreds = self.getUserToExecuteWorkflow()
            if len(self.workflowUserCreds) < 3:
                Log.error("Couldn't find information in database to execute workflows on cloud.")
            else:
                Log.debug("Fetched information to execute workflow on cloud: username=[%s], webconsoleURL=[%s]" % (self.workflowUserCreds['username'],self.workflowUserCreds['WebConsoleURL']))

        #get the properties related to metrics server being used as proxy
        if  self.IsCSOpenConnection == True:
            query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='CommservSurveyMetricsServerUsedAsInternetGateway' and modified=0;"
            if self.CSDB.Execute(query) == True:
                while 1:
                    row = self.CSDB.m_cursor.fetchone()
                    if row and row.value == '0':
                        self.UseMetricsServerAsProxy = False
                    break
            
            if self.UseMetricsServerAsProxy == True:
                query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='CommservSurveyUploadsite' and modified=0;"
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        self.PublicMetricsUploadSite = row.value
                        if self.PublicMetricsUploadSite[len(self.PublicMetricsUploadSite) - 1] != '/':
                            self.PublicMetricsUploadSite += '/'
                        self.PublicMetricsUploadSite += 'upload.asp'
                            
                        break
                
                query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='CommservSurveyUploadsiteUser' and modified=0;"
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        self.PublicMetricsUploadSiteUser = row.value
                        break
                
                query = "select value FROM  GXGlobalParam WITH(NOLOCK) WHERE name ='CommservSurveyUploadsitePasswd' and modified=0;"
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        self.PublicMetricsUploadSitePasswd = GetMetricsSimpleData(row.value,self.SimpanaBaseFolder,self.Log)
                        break
                
                #get sendlogs upload url credentials
                oemId = '0'        
                query = "select value FROM SimOemProp WITH(NOLOCK) WHERE name ='SendLogsDefaultHTTPSite' and OemId=%s;" % oemId
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        self.SendlogsUploadSite = row.value
                        break
        
                query = "select value FROM SimOemProp WITH(NOLOCK) WHERE name ='SendLogsDefaultHTTPUser' and OemId=%s;" % oemId
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        self.SendlogsUploadSiteUser = row.value
                        break
        
                query = "select value FROM SimOemProp WITH(NOLOCK) WHERE name ='SendLogsDefaultHTTPPassword' and OemId=%s;" % oemId
                if self.CSDB.Execute(query) == True:
                    while 1:
                        row = self.CSDB.m_cursor.fetchone()
                        self.SendlogsUploadSitePassword = GetMetricsSimpleData(row.value,self.SimpanaBaseFolder,self.Log)
                        break
            
                if self.SendlogsUploadSite[len(self.SendlogsUploadSite) - 1] != '/':
                    self.SendlogsUploadSite += '/'
                self.SendlogsUploadSite += 'httplogupload.do'
        #get the public metrics sendlogs
        parentDir = os.path.dirname(self.FolderContainingXmlFile)
        self.PublicMetricsUploadFolder = os.path.join(parentDir, 'Public')
        self.SendlogsUploadFolder = os.path.join(parentDir, 'SendlogFiles')
        
        self.LastProcessFileBufferTimeSec = 60*60 #give a buffer of 1 hour 
        #Create the folder if does not exist
        if not os.path.exists(self.PublicMetricsUploadFolder):
            os.makedirs(self.PublicMetricsUploadFolder)
        if not os.path.exists(self.SendlogsUploadFolder):
            os.makedirs(self.SendlogsUploadFolder)


        self.csvErrorsList = []
        csvErrorsListFileName = 'csvErrorsList.xml'  
        try:             
            SimpanaConfigDir = self.SimpanaHomeFolder
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "Reports")
            SimpanaConfigDir = os.path.join(SimpanaConfigDir, "MetricsUpload")
            csvErrorsListFile = os.path.join(SimpanaConfigDir, csvErrorsListFileName)  
            if os.path.isdir(csvErrorsListFile) or os.path.exists(csvErrorsListFile) == False:
                self.Log.debug("Skipping check for invalid CSVs as file [%s] not found." % (csvErrorsListFileName))
            else:                                
                with open(csvErrorsListFile, "r") as fp:  
                    try:
                        xmlDoc = parseString(fp.read())                    
                        for err in xmlDoc.getElementsByTagName('error'):
                            self.csvErrorsList.append(err.getAttribute("val"))
                        self.Log.debug("Successfully loaded CSV error list from file [%s]." % (csvErrorsListFileName))
                    except:
                        self.Log.exception("Unable to parse XML in file [%s]" % csvErrorsListFileName)                                    

                    fp.close()                
        except Exception as err:
            self.Log.exception("Exception while reading file [%s] for checking error messages. Error: [%s]" % (csvErrorsListFileName, str(err)))


        return True

    def PostProcessFile(self, iFileName, success=True,upgradeReadiness=False,isErrOutputFile=False, disabled=False, cvcloudDBObj = None):
        if cvcloudDBObj is None:
            cvcloudDBObj = self.CustomerFeedbackDB
        ipFile = iFileName + '.txt'
        sourceFile = os.path.join(self.FolderContainingXmlFile , iFileName)
        sourceIpFile = os.path.join(self.FolderContainingXmlFile , ipFile)
        
        if success is False and disabled is False:
            if isErrOutputFile == False:
                self.Log.error(__name__ + "File %s failed processing"%iFileName)
            else:
                self.Log.debug(__name__ + "File %s moving to failed folder"%iFileName)
        
        if self.DeleteXmlFileAfterProcessing is True:
            if os.path.exists(sourceFile):
                try:
                    os.remove(sourceFile) #xml File
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%sourceFile)
            if os.path.exists(sourceIpFile): #ip file
                try:
                    os.remove(sourceIpFile)
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%sourceIpFile)
        else:        
            if upgradeReadiness is False:
                if sourceIpFile.endswith('.zip.txt'):
                    archiveFolder = self.ZipArchiveFolder  
                else:
                    archiveFolder = self.ArchiveFolder
                failedFolder  = self.FailedFolder
            else:
                archiveFolder = self.UpgradeArchiveFolder
                failedFolder = self.UpgradeFailedFolder
                       
            if success is True:
                destFolder = archiveFolder
            else:
                destFolder = failedFolder

            destFile = os.path.join(destFolder , iFileName)
            destIpFile = os.path.join(destFolder , ipFile)
                           
            if os.path.exists(destFile):
                try:
                    os.remove(destFile)
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%destFile)
            if os.path.exists(destIpFile):
                try:
                    os.remove(destIpFile)
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%destIpFile)
                
            retVal = True
            if self.isPrivate == True: 
                if(len (self.CopytoHttpServerlist)>0):
                    fName, fExtension = os.path.splitext(sourceFile)
                    if(fExtension.lower() != ".xml" and fExtension.lower() != ".csv"):
                        self.Log.debug(__name__ + "File [%s] is not required to be uploaded " % (iFileName))
                    else:
                        if self.useHttpProxyPrivate == True:
                            retVal = cvcloudDBObj.copytoHttpServer(sourceFile, iFileName,
                            self.CopytoHttpServerlist, self.RestrictQuery,
                            self.ScrubCC, self.RegEx, self.ExclusionDict,
                            self.SimpanaTempFolder, self.HttpProxyHost, self.HttpProxyPort, self.HttpProxyUser, self.HttpProxyPasswd,
                            self.TransferXMLFileConstraint)
                        else:
                            retVal = cvcloudDBObj.copytoHttpServer(sourceFile, iFileName,
                            self.CopytoHttpServerlist, self.RestrictQuery,
                            self.ScrubCC, self.RegEx, self.ExclusionDict,
                            self.SimpanaTempFolder, None, None, None, None, 
                            self.TransferXMLFileConstraint)
            
            #MR#111777 - if copy to secondary metrics server is failed, keep it in current location so that it will be retried next time.
            if retVal == True:
                try:
                    if os.path.exists(sourceFile): #ip file
                        shutil.move(sourceFile, destFolder)
                    if os.path.exists(sourceIpFile):
                        # delete ip files of parsed csv files
                        if sourceIpFile.endswith('.csv.txt'):
                            os.remove(sourceIpFile)
                        else:
                            shutil.move(sourceIpFile, destFolder)
                except OSError as oserr:
                    if oserr.errno == 32:
                        self.Log.info(__name__ + "Failed to move file due to WinError %s, as file is still being copied. Driver will retry this file after some time" %oserr.errno)
                    else:
                        self.Log.error(__name__ + "Failed to move file with error %s" %oserr)
                        
            else:
                self.Log.exception(__name__ + "Failed to copy file to next metrics report server. File will be retained for retry")

    def ProcessBlockedFile(self, ccID, ccIP, iFileName, upgradeReadiness=False, cvcloudDBObj = None):
        if cvcloudDBObj is None:
            cvcloudDBObj = self.CustomerFeedbackDB        
        failProcessing = False
        #No need to process xml files from blocked CommServers (Staging server)
        if ccIP:
            ccIPMatchedInIncludeIPs = False
            for inIP in self.IncludedIPList:
                # MR 126509 - skip empty lines to avoid matching everything
                if len(inIP.strip())>0:
                    regex = re.compile(inIP)
                    ms = regex.match(ccIP)
                    if ms or inIP == '*.*.*.*':
                        ccIPMatchedInIncludeIPs = True
                        break

            if ccIPMatchedInIncludeIPs is False:
                for bIP in self.BlockedIPList:
                    # MR 126509 - skip empty lines to avoid matching everything
                    if len(bIP.strip())>0:
                        regex = re.compile(bIP)
                        ms = regex.match(ccIP)
                        if ms:
                            self.Log.debug(__name__ + "File '%s' coming from %s is blocked based on IP rule %s." % (iFileName,ccIP,bIP))
                            failProcessing = True
                            break
        if failProcessing is False:
            if (ccID == -1 and self.isEvalCCBlocked is True):
                #No need to process xml files from Evaluation CommCells
                self.Log.debug(__name__ + "File '%s' is blocked based on CommCell ID!!!" % iFileName)
                failProcessing = True

            elif (ccID >= 10001 and ccID <= 655360 and self.isNFRCCBlocked == True):
                #No need to process xml files from NFR CommCells
                self.Log.debug(__name__ + "File '%s' is blocked since it is from NFR CommCell!!!" % iFileName)
                failProcessing = True

        if failProcessing is False:
            return False
                
        ipFile = iFileName + '.txt'
        sourceFile = os.path.join(self.FolderContainingXmlFile, iFileName)
        sourceIpFile = os.path.join(self.FolderContainingXmlFile, ipFile)
        
        if self.DeleteXmlFileAfterProcessing == True:
            if os.path.exists(sourceFile):
                try:
                    os.remove(sourceFile) #xml File
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%sourceFile)
            if os.path.exists(sourceIpFile): #ip file
                try:
                    os.remove(sourceIpFile)
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%sourceIpFile)
        else:
            if self.UNCFolderToUploadFiles != '' and self.UNCCopyNetUseCommand == True:
                destFolder = self.UNCFolderToUploadFiles
                destFile = os.path.join(self.UNCFolderToUploadFiles, iFileName)
                destIpFile = os.path.join(self.UNCFolderToUploadFiles, ipFile)
            else:
                if self.UNCFolderToUploadFiles != '' and self.UNCCopyNetUseCommand == False:
                    self.Log.debug(__name__ + "Copy to local folder instead of UNC path as 'net use' command failed to initialize")
                if upgradeReadiness == False:
                    destFolder = self.BlockedFolder
                else:
                    destFolder = self.UpgradeBlockedFolder
                destFile = os.path.join(destFolder, iFileName)
                destIpFile = os.path.join(destFolder, ipFile)
                               
            if os.path.exists(destFile):
                try:
                    os.remove(destFile)
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%destFile)
            if os.path.exists(destIpFile):
                try:
                    os.remove(destIpFile)
                except Exception as err:
                    self.Log.exception(__name__ + "Failed to delete file %s"%destIpFile)
    
            retVal = True
            if self.isPrivate == True:                
                if(len (self.CopytoHttpServerlist)>0):
                    fName, fExtension = os.path.splitext(sourceFile)
                    if(fExtension.lower() != ".xml" and fExtension.lower() != ".csv" and fExtension.lower() != ".zip"):
                        self.Log.debug(__name__ + "File [%s] is not required to be uploaded " % (iFileName))
                    else:
                        if self.useHttpProxyPrivate == True:
                            retVal = cvcloudDBObj.copytoHttpServer(sourceFile, iFileName,
                            self.CopytoHttpServerlist, self.RestrictQuery,
                            self.ScrubCC, self.RegEx, self.ExclusionDict,
                            self.SimpanaTempFolder, self.HttpProxyHost, self.HttpProxyPort, self.HttpProxyUser, self.HttpProxyPasswd,
                            self.TransferXMLFileConstraint)
                        else:
                            retVal = cvcloudDBObj.copytoHttpServer(sourceFile, iFileName,
                            self.CopytoHttpServerlist, self.RestrictQuery,
                            self.ScrubCC, self.RegEx, self.ExclusionDict,
                            self.SimpanaTempFolder, None, None, None, None, 
                            self.TransferXMLFileConstraint)
            else:
                Log.debug("Skipping forwarding of XML files as the setup is Public.")            
            #MR#111777 - if copy to secondary metrics server is failed, keep it in current location so that it will be retried next time.
            if retVal == True:
                try:
                    if os.path.exists(sourceFile): #ip file
                        shutil.move(sourceFile, destFolder)
                    if os.path.exists(sourceIpFile): #ip file
                        shutil.move(sourceIpFile, destFolder)
                except OSError as oserr:
                    if oserr.errno == 32:
                        self.Log.info(__name__ + "Failed to move file due to WinError %s, as file is still being copied. Driver will retry this file after some time" %oserr.errno)
                    else:
                        self.Log.error(__name__ + "Failed to move file with error %s" %oserr)
            else:
                self.Log.exception(__name__ + "Failed to copy file to next metrics report server. File will be retained for retry")           
        return True

    #this function is not used anymore
    def ProcessHeartbeat(self, commcellid, cssxmldata_element):
        #surveyenabled = 0 Survey Disabled
        #surveyenabled = 1 Public Survey enabled
        #surveyenabled = 2 Private Survey enabled
        surveyenabled = 0
        
        #surveyservices = 0 no services
        #surveyservices = 1 HealthCheck
        #surveyservices = 2 Upgrade
        #surveyservices = 4 Activity
        #surveyservices = 8 Audit
        #surveyservices = 16 ChargeBack
        surveyservices = 0
        enableddiags = None
        #commcelliddec = int(commcellid,16)
        heartbeattime = 0
        
        try:
            heartbeattime = int(cssxmldata_element.getAttribute('LogDate'))
            private = cssxmldata_element.getElementsByTagName('Rpt_PrivateDiagnostics')
            if private.length != 0:
                privateenabled = int(private[0].getAttribute("Enabled"))
                if privateenabled == 1:
                    surveyenabled = 2
                    enableddiags = private[0]
            
            public = cssxmldata_element.getElementsByTagName('Rpt_PublicDiagnostics')
            if public.length != 0:
                privateenabled = int(public[0].getAttribute("Enabled"))
                if privateenabled == 1:
                    surveyenabled = 1
                    enableddiags = public[0]
            
            if enableddiags is not None:
                services = enableddiags.getElementsByTagName("Rpt_DiagnosticServices")
                if services[0].hasAttribute("HealthCheck"):
                    serv = services[0].getAttribute("HealthCheck")
                    if serv == '1':
                        surveyservices |= 1
                if services[0].hasAttribute("Upgrade"):
                    serv = services[0].getAttribute("Upgrade")
                    if serv == '1':
                        surveyservices |= 2
                if services[0].hasAttribute("Activity"):
                    serv = services[0].getAttribute("Activity")
                    if serv == '1':
                        surveyservices |= 4
                if services[0].hasAttribute("Audit"):
                    serv = services[0].getAttribute("Audit")
                    if serv == '1':
                        surveyservices |= 8
                if services[0].hasAttribute("ChargeBack"):
                    serv = services[0].getAttribute("ChargeBack")
                    if serv == '1':
                        surveyservices |= 16
        except Exception as ex:
            self.Log.exception("Failed during processing heartbeat")
            
        self.CustomerFeedbackDB.UpdateCommcellHeartbeat(commcellid, surveyenabled, surveyservices, heartbeattime)
        
    def ProcessSurveyDisableXML(self, filename, ip, cvcloudDBObj = None):
        if cvcloudDBObj is None:
            cvcloudDBObj = self.CustomerFeedbackDB    	
        try:
            fullFileName = self.FolderContainingXmlFile + "\\" + filename
            fName, fExtension = os.path.splitext(fullFileName) 
            if os.path.isdir(fullFileName):
                return 
            elif fExtension.lower() != ".xml":
                return 

            try:
                xmlDoc = ET.parse(fullFileName)
                rootElement = xmlDoc.getroot()
            except Exception as err:
                self.Log.exception("Failed to parse file '%s'!!!" % fullFileName)
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return 
            
            if rootElement.tag != "Rpt_CSSXMLDATAROOT":
                self.Log.exception("'%s' is not valid disable commserv xml!!!" % fullFileName)
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return 
            
            ccGUID = rootElement.get("commcellGUID")

            child = rootElement.find("CVMetricsInfo")
            
            if child is None:
                self.Log.exception("'%s' is not valid disable commserv xml!!!" % fullFileName)
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return
            
            ret_val = False
            is_enabled = True
            if child.get("isEnabled") == "0":
                self.Log.info("Disabling Commserv for request file %s" % (filename))
                is_enabled = False
            else:
                self.Log.info("Enabling Commserv for request file %s" % (filename))
                is_enabled = True

            commcellNamemapId = cvcloudDBObj.GetCommCellNameMapId(None, ip, ccGUID)    #argument ccid is not used
            if commcellNamemapId <= 0:
                self.Log.debug("Unable to get commserv information for %s" % (filename))
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
                return 
            ret_val = cvcloudDBObj.EnableCommcell(commcellNamemapId, is_enabled)

            self.PostProcessFile(filename, ret_val, cvcloudDBObj = cvcloudDBObj)

        except Exception as err:
            self.Log.exception("Exception processing the xml file %s" % (filename))
            self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
            return 

    def ProcessAlertsXML(self, filename, ip, cvcloudDBObj = None):
        if cvcloudDBObj is None:
            cvcloudDBObj = self.CustomerFeedbackDB         
        try:
            fullFileName = self.FolderContainingXmlFile + "\\" + filename
            
            #get the commserv guid from filename
            fnamesplit = filename.split("_",4)
            extsplit = fnamesplit[3].split('.',2)
            ccGuid = extsplit[0]
            
            #get the commserv UniqueId
            commcellNamemapId = cvcloudDBObj.GetCommCellNameMapId(None, ip, ccGuid)
            if commcellNamemapId <= 0:
                self.Log.debug("Unable to get commserv information for %s" % (filename))
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
                return 

            #parse XML data to get information
            xmlDoc = parse(fullFileName)
            
            #push the data to cloud database
            rootElements = xmlDoc.getElementsByTagName("App_LiveFeedListResp")
            if rootElements == None:
                Log.error("File %s doesn't have element App_LiveFeedListResp!!!" % filename)
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return 
            
            ProcessXMLQuery = "exec ParsebrNTFormatLiveFeedsNotifications N'%s', N'%s'" % (str(commcellNamemapId), xmlDoc.toxml().replace('\'', '\'\'')) 
            
            self.Log.debug("Executing query '%s'" % ProcessXMLQuery.encode('utf-8'))
            if self.CloudDB.Execute(ProcessXMLQuery) == False:
                self.Log.debug("Query '%s' failed with error: '%s'" % (ProcessXMLQuery.encode('utf-8'), self.CloudDB.GetErrorErrStr()))
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return 

            self.PostProcessFile(filename, True, cvcloudDBObj = cvcloudDBObj)          

        except Exception as err:
            self.Log.exception("Exception processing the xml file %s" % (filename))
            self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
            return 

    def ProcessSurveyXML(self, filename, ip, cvcloudDBObj = None):
        if cvcloudDBObj is None:
            cvcloudDBObj = self.CustomerFeedbackDB
        try:
            fullFileName = os.path.join(self.FolderContainingXmlFile, filename)
            
            fName, fExtension = os.path.splitext(fullFileName) 
            if os.path.isdir(fullFileName):
                return 0, None
            elif fExtension.lower() != ".xml":
                return 0, None
                
            #check if the file is for a specific query CSS<TimeStamp>_<CCID>_<QID>.xml 
            pattern = 'CSS\d{10}_.[a-fA-F0-9]*_(.[0-9]*)\.xml'
            queryId = re.findall(pattern, filename, re.I)
            if (len(queryId) == 1 and (int(queryId[0]) not in self.ValidQueryIds)):
                self.Log.debug("QueryId [%i] is disabled for processing !!! No need to process [%s]" % (int(queryId[0]), filename))
                self.PostProcessFile(filename, False, disabled=True, cvcloudDBObj = cvcloudDBObj)
                return 0, None                 

            cssxmldata_filedata = None
            xmlDoc = None
            cssxmldata_filedata = CSSXMLDATA_FileData()
            cssxmldata_filedata.FileName = fullFileName
            cssxmldata_filedata.FileTime = datetime.datetime.fromtimestamp(os.path.getmtime(fullFileName)).strftime("%m/%d/%Y  %H:%M")
                    
            if cvcloudDBObj.GetId(cssxmldata_filedata) != 0:
                self.Log.debug("Skipping file %s, because it has been previously processed..."  % (filename))
                self.PostProcessFile(filename, True, cvcloudDBObj = cvcloudDBObj)
                return -1, None
                    
            cssxmldata_filedata.rpt_cssxmldataroot.CommServIP = ip

            self.Log.debug(
                __name__ + " :: found file '%s' with time-stamp (%s)" % (
                    cssxmldata_filedata.FileName,
                    cssxmldata_filedata.FileTime
                )
            )
            try:
                xmlDoc = parse(fullFileName)
            except Exception as err:
                if err.args[0] == 'no element found: line 1, column 0':
                    self.Log.info("XML File %s is empty." % fullFileName)
                    self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)  
                else:
                    self.Log.exception("Failed to parse file '%s'!!!" % fullFileName)
                    self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return 0, None
            
            rootElements = xmlDoc.getElementsByTagName("Rpt_CSSXMLDATAROOT")
            
            if rootElements is None:
                Log.error("File %s doesn't have element Rpt_CSSXMLDATAROOT!!!" % filename)
                self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                return 0, None
                    
            for rootElement in rootElements:
                failProcessing = False
                if rootElement is None:
                    return 0, None
                
                if rootElement.hasAttribute("commcellID") is False:
                    self.Log.error("Could not find commcellID attribute in XML file %s!!!" % (filename))
                    self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)          
                    return 0, None

                cc_id = rootElement.getAttribute("commcellID")
                cssxmldata_filedata.rpt_cssxmldataroot.commcellID = str(cc_id)

                if rootElement.hasAttribute("emailRecipients") is False:
                    self.Log.warning("Could not find emailRecipients attribute in XML file %s!!!" % (filename))
                    cssxmldata_filedata.rpt_cssxmldataroot.emailRecipients = ""   
                else:
                    cssxmldata_filedata.rpt_cssxmldataroot.emailRecipients = rootElement.getAttribute("emailRecipients")
        
                CSRegInfo = None
                CSRegInfo = rootElement.getElementsByTagName('EVGui_RegistrationInformation')
                if not CSRegInfo:
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCommCellName = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCompanyName = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCustomerName = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoAddress = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoPhone = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoDescription = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoEmailAddress = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoInstallTime = "0"
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMajorBrand = "0"
                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMiniBrand = "0"            
                else:            
                    for csRegistrationInfo in CSRegInfo:
                        if csRegistrationInfo is None:
                            self.Log.error("There is no Registration Information!!!" % filename)
                            break
                        else:
                            if csRegistrationInfo.hasAttribute("commcellName") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCommCellName = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCommCellName = csRegistrationInfo.getAttribute("commcellName")
                                
                            if csRegistrationInfo.hasAttribute("companyName") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCompanyName = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCompanyName = csRegistrationInfo.getAttribute("companyName")
        
                            if csRegistrationInfo.hasAttribute("customerName") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCustomerName = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoCustomerName = csRegistrationInfo.getAttribute("customerName")
        
                            if csRegistrationInfo.hasAttribute("address") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoAddress = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoAddress = csRegistrationInfo.getAttribute("address")
                                
                            if csRegistrationInfo.hasAttribute("phoneNumber") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoPhone = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoPhone = csRegistrationInfo.getAttribute("phoneNumber")
                                
                            if csRegistrationInfo.hasAttribute("description") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoDescription = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoDescription = csRegistrationInfo.getAttribute("description")
        
                            if csRegistrationInfo.hasAttribute("emailAddress") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoEmailAddress = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoEmailAddress = csRegistrationInfo.getAttribute("emailAddress")
        
                            if csRegistrationInfo.hasAttribute("ccguid") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID = csRegistrationInfo.getAttribute("ccguid")
        
                            #if csRegistrationInfo.hasAttribute("ccinstalltime") == False:
                            #    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoInstallTime = "0"   
                            #else:
                            #    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoInstallTime = csRegistrationInfo.getAttribute("ccinstalltime")
                            #    try:
                            #        val = int(cssxmldata_filedata.rpt_cssxmldataroot.RegInfoInstallTime)
                            #    except:
                            #        cssxmldata_filedata.rpt_cssxmldataroot.RegInfoInstallTime = "0"
        
                            if csRegistrationInfo.hasAttribute("majorbrandid") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMajorBrand = "0"   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMajorBrand = csRegistrationInfo.getAttribute("majorbrandid")
                                try:
                                    val = int(cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMajorBrand)
                                except Exception as err:
                                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMajorBrand = "0"
        
                            if csRegistrationInfo.hasAttribute("minibrandid") is False:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMiniBrand = "0"   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMiniBrand = csRegistrationInfo.getAttribute("minibrandid")
                                try:
                                    val = int(cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMiniBrand)
                                except Exception as err:
                                    cssxmldata_filedata.rpt_cssxmldataroot.RegInfoMiniBrand = "0"
        
                            break                                
                
                Cssxmldatas = rootElement.getElementsByTagName("Rpt_CSSXMLDATA")
                if len(Cssxmldatas) == 0:
                    Cssxmldatas = rootElement.getElementsByTagName("Rpt_CSSXMLATA")
                    
                childno = 0
                
                commServerInfoResults = {}
                commServerAliasName = {}
                
                # print "Processing each child element in file " + filename
                del cssxmldata_filedata.rpt_cssxmldataroot.CssXmlDatas[:]
                for cssxmldata_element in Cssxmldatas:
                    childno += 1
                    cssxmldata = CssXmlData()
                    cssxmldata.logDate = cssxmldata_element.getAttribute("LogDate")
                    if cssxmldata.logDate is None:
                        self.Log.error("Could not find LogDate element in XML file %s for child %i!!!" % (filename, childno))
                        continue
                    
                    cssxmldata.QueryId = cssxmldata_element.getAttribute("QueryId")
                    if cssxmldata.QueryId is None:
                        self.Log.error("Could not find QueryId element in XML file %s for child %i!!!" % (filename, childno))
                        continue
                    
                    if int(cssxmldata.QueryId) not in self.ValidQueryIds:
                        self.Log.debug("QueryId [%i] is disabled for processing !!!" % (int(cssxmldata.QueryId)))
                        continue                        
                    
                    if cssxmldata.QueryId == 'heartbeat':
                        self.ProcessHeartbeat(cc_id, cssxmldata_element)
                        self.Log.info("Processed heartbeat information")
                        self.PostProcessFile(filename, True, cvcloudDBObj = cvcloudDBObj)
                        xmlDoc.unlink()
                        return 0, None
                    
                    if cssxmldata_element.hasAttribute("QueryRunningTime") == False:
                        cssxmldata.RunningTime = -1
                        #cssxmldata_filedata.Logging +=  "Error: Could not find QueryRunningTime element in XML file %s for child %i!!!" % (filename, childno)
                        #continue
                    else: 
                        cssxmldata.RunningTime = cssxmldata_element.getAttribute("QueryRunningTime")
                        if cssxmldata.RunningTime is None:
                            cssxmldata.RunningTime = -1
                            #cssxmldata_filedata.Logging +=  "Error: Could not find QueryRunningTime element in XML file %s for child %i!!!" % (filename, childno)
                            #continue                
                    
                    if cssxmldata_element.firstChild is None:
                        #Log.warn("No collected result found for QueryId %s, child %i!!!" % (cssxmldata.QueryId, childno))
                        cssxmldata.CollectedResults = ''
                    else:
                        cssxmldata.CollectedResults = cssxmldata_element.firstChild.nodeValue
                    
                    if cssxmldata.QueryId == '2':
                        commServerInfoResults[int(cssxmldata.logDate)] = cssxmldata.CollectedResults
                    if cssxmldata.QueryId == '57':
                        commServerAliasName[int(cssxmldata.logDate)] = cssxmldata.CollectedResults.encode('utf-8')
                                         
                    #print "logDate = %s, QueryId = %s, Node value = '%s'" % (cssxmldata.logDate, cssxmldata.QueryId, cssxmldata.CollectedResults)
                    #print "logDate = %s, QueryId = %s" % (cssxmldata.logDate, cssxmldata.QueryId)
                    
                    cssxmldata_filedata.rpt_cssxmldataroot.CssXmlDatas.append(cssxmldata)
                
                #print "Number of items :: %s" % (len(cssxmldata_filedata.rpt_cssxmldataroot.CssXmlDatas))               
                #Get CommServ Timezone and CommServer Version from Collected Results of Query Id = 2
                recentCSInfoResult = ''
                if len(commServerInfoResults) > 0:
                    recentCSInfoResult = commServerInfoResults.get(max(commServerInfoResults))
                recentCSAliasName = ''
                if len(commServerAliasName) > 0:
                    recentCSAliasName = commServerAliasName.get(max(commServerAliasName))                
                
                if len(recentCSInfoResult) != 0:
                    try:
                        csInfoXMLDoc = parseString(recentCSInfoResult)
                        
                        csInfoRootElements = csInfoXMLDoc.getElementsByTagName('CommServerInfo')
                         
                        for csInfoRootElement in csInfoRootElements:
                            if csInfoRootElement == None:
                                continue
                            if cssxmldata_filedata.rpt_cssxmldataroot.CommServIP == '':
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServIP = "N/A"

                            if csInfoRootElement.hasAttribute("CommServGUID") == False:
                                self.Log.warning("Could not find CommServGUID attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID = "N/A"   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID = csInfoRootElement.getAttribute("CommServGUID") 
                            
                            if csInfoRootElement.hasAttribute("CommServTZ") == False:
                                self.Log.warning("Could not find CommServTZ attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServTZ = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServTZ = csInfoRootElement.getAttribute("CommServTZ")          
                        
                            if csInfoRootElement.hasAttribute("CommServVersion") == False:
                                self.Log.warning("Could not find CommServVersion attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServVersion = "Not Found"   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServVersion = csInfoRootElement.getAttribute("CommServVersion")                
        
                            if csInfoRootElement.hasAttribute("CommServEdition") == False:
                                self.Log.warning("Could not find CommServEdition attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServEdition = ""   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServEdition = csInfoRootElement.getAttribute("CommServEdition")
        
                            if csInfoRootElement.hasAttribute("MetricServices") == False:
                                self.Log.warning("Could not find MetricServices attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.MetricServices = '0'   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommcellMetricServices = csInfoRootElement.getAttribute("MetricServices")
                                
                            if csInfoRootElement.hasAttribute("adminEmail") == True:
                                cssxmldata_filedata.rpt_cssxmldataroot.RegInfoEmailAddress = csInfoRootElement.getAttribute("adminEmail")

                    except Exception as err:
                        self.Log.exception("Failed to parse and get CommServTZ and CommServVersion info from string %s !!!" % str(recentCSInfoResult))
                else:
                    cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID = "N/A"
                    cssxmldata_filedata.rpt_cssxmldataroot.CommServTZ = ""
                    cssxmldata_filedata.rpt_cssxmldataroot.CommServVersion = "Not Found"
                    cssxmldata_filedata.rpt_cssxmldataroot.CommServEdition = ""
                    self.Log.warning("No Collected results available for CommServTZ and CommServVersion!!!")
                
                if cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID == "N/A":
                    self.Log.error("CommServGUID is not found, skipping the XML file " + filename)
                    self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
                    return 0, None
           
                if len(queryId) <= 0: #do not do anything for frequency files like CSSTS_CCID_QueryId.xml
                    #check css timestamp and skip those that are already processed
                    fileUTCTime = self.getTimestampFromFileName(filename)
                    intccid = int(cc_id)
                    self.Log.debug(__name__ + "Check if file [%s] with csguid [%s], ccid [%s] and timestamp [%d] is to be skipped" % (filename, cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID, intccid, fileUTCTime))
                    lastFileProcessedTime = self.CustomerFeedbackDB.getlastprocessedtime(cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID, intccid)
                    if lastFileProcessedTime >= fileUTCTime + self.LastProcessFileBufferTimeSec:
                        self.Log.info(__name__ + "Latest file is already processed. Skip [%s]", filename)
                        self.PostProcessFile(filename, True)
                        return -1, None
                
                if len(recentCSAliasName) != 0:
                    try:
                        cssxmldata_filedata.rpt_cssxmldataroot.CommServAliasName = ''
                        csInfoXMLDoc = parseString(recentCSAliasName)
                        csInfoRootElements = csInfoXMLDoc.getElementsByTagName('CommServInfo')
                         
                        for csInfoRootElement in csInfoRootElements:
                            if csInfoRootElement == None:
                                continue
                            
                            if csInfoRootElement.hasAttribute("AliasName") == False:
                                self.Log.warning("Could not find AliasName attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServAliasName = ''   
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServAliasName = csInfoRootElement.getAttribute("AliasName").encode('utf-8')
        
                            if csInfoRootElement.hasAttribute("IPAddress") == False:
                                self.Log.warning("Could not find IPAddress attribute !!!")
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServDBIPAddr = ''
                            else:
                                cssxmldata_filedata.rpt_cssxmldataroot.CommServDBIPAddr = csInfoRootElement.getAttribute("IPAddress")
                    except Exception as err:
                        self.Log.exception("Failed to parse and get AliasName from string %s !!!" % recentCSAliasName)
                else:
                    cssxmldata_filedata.rpt_cssxmldataroot.CommServAliasName = ''
                    self.Log.debug("No Collected results available for CS Alias Name!!!")                

                # Check if the CS is upgraded or updated to new version/service pack 
                hexccId = hex(int(cc_id))[2:]
                if self.isPrivate is False and self.CheckUpgradedCommcells == 1:
                    retCode, retData = cvcloudDBObj.checkForCommcellUpgrade(cssxmldata_filedata.rpt_cssxmldataroot)
                    if retCode == True and hexccId not in self._upgradedCSInfo:
                        self._upgradedCSInfo[hexccId] = retData #TODO: acquire lock
                        
                retVal, fileId = cvcloudDBObj.Add_CSSXMLDATA_FileData(cssxmldata_filedata)
                
                if retVal is True:
                    # if self.isPrivate is False:
                        # self.CheckForCommcellRegistration(cssxmldata_filedata.rpt_cssxmldataroot)

                    self.PostProcessFile(filename, True, cvcloudDBObj = cvcloudDBObj)
                else:
                    self.Log.error("Failed to add xml data for file " + filename)
                    self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
                xmlDoc.unlink()
                return fileId, cssxmldata_filedata.rpt_cssxmldataroot.CommServGUID
        except Exception as err:
            self.Log.exception("Exception processing the xml file %s" % (filename))
            self.PostProcessFile(filename, False, cvcloudDBObj = cvcloudDBObj)
            return 0, None
	
	#not used anymore
    def CheckForCommcellRegistration(self, CSInfo):
            
        try:

            if CSInfo.commcellID == "-1":
                self.Log.debug(__name__ + ":: Skipping commcell registration check for Eval commcell [%s]" % (CSInfo.CommServGUID))         
                return

            hexccId = hex(int(CSInfo.commcellID))[2:]

            retStatus,accGlobalId = self.CustomerFeedbackDB.getCommcellRegistrationInfo(CSInfo)        

            #self.Log.debug("CheckForCommcellRegistration:: Got Global Account Id: [%d], Status: [%r]",accGlobalId, retStatus)

            if self.IsCSOpenConnection == False or retStatus == False or accGlobalId == -1:
                return                    

            query = """SELECT CC.clientId AS clientId, CC.id AS Id
                                FROM UMGroupsProp UP WITH(NOLOCK)
                                       INNER JOIN UMSecurityASsociations S WITH(NOLOCK)
                                              ON attrName = 'Customer User Group' AND CAST((CASE WHEN ISNULL(UP.attrVal,'') = '' THEN '0' Else attrVal END ) AS BIGINT) = %d 
                                                     AND userOrGroupId = UP.componentNameId and isUser = 0 and entityType1 = 3
                                       INNER JOIN App_Client AC WITH(NOLOCK)
                                              ON S.entityId1 = AC.Id AND AC.GUID = '%s'
                                       INNER JOIN APP_CommCell CC WITH(NOLOCK)
                                              ON CC.clientId = AC.id""" % (accGlobalId, CSInfo.CommServGUID)
            self.Log.debug("Executing query '%s'" % query)
            if self.CSDB.Execute(query) == False:
                self.Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
                return
            
            row = self.CSDB.m_cursor.fetchone()
            if row:                
                self.CustomerFeedbackDB.UpdateAPPCommcellId(CSInfo.CommServGUID, row.Id)
            else:
                self._unregisteredCommcells.put(hexccId)            

        except Exception as err:
            self.Log.exception("Caught Exception while checking for commcell registration. [%s]", str(err))

    def ProcessSurveyCSV(self, csvfile, fileid = 0, csguid = '', cvcloudDBObj = None):
        if cvcloudDBObj is None:
            cvcloudDBObj = self.CustomerFeedbackDB
        try:
            fullFileName = os.path.join(self.FolderContainingXmlFile, csvfile)
            fnamesplit = csvfile.split("_",4)
            scc_id = fnamesplit[1]                   
            cc_id = self.get_ccid_int(scc_id)
            
            if csguid is None or len(csguid) <= 0:
                csguid = fnamesplit[2]
            qidsplit = fnamesplit[3].split(".", 2)
            logdate = fnamesplit[0].split("CSS")[1]

            if (qidsplit[0].isdigit()):
                queryid = qidsplit[0]
                
                #get the query id and check if this is valid to be processed locally
                if (int(queryid) not in self.ValidQueryIds):
                    self.Log.debug("QueryId [%i] is disabled for processing !!! No need to process [%s]" % (int(qidsplit[0]), csvfile))
                    self.PostProcessFile(csvfile, True, cvcloudDBObj = cvcloudDBObj)
                    return
    
                if fileid is None or fileid <= 0:
                    if scc_id in self.failedcommcellset:
                        self.Log.error("Main XML data failed to process for file " + csvfile)
                        self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)
                        return
                                     
                    if csguid in self.commcelldict:
                        fileid = self.commcelldict[csguid]
                    else:
                        self.Log.debug("Associated xml not found.Skipping csv processing.")
                        return
    
                #### Check if CSV file is invalid and contains some error message instead of comma separated values #####
                csvfileFound = False
                for x in range(0, 3):   #retry 3 times as the file is not found sometimes
                    if os.path.exists(fullFileName) == False:
                        self.Log.error("File not found [%s]. Retry counter [%d]" % (fullFileName, x))
                        time.sleep(1) #sleep for a sec
                        continue
                    try:         
                        statinfo  = os.stat(fullFileName)
                        csvfileFound = True
                        break
                    except FileNotFoundError as e1:
                        self.Log.error("File not found exception while trying to get stat for csv file [%s]. Retry counter [%d]. Exception [%s]" % (fullFileName, x, str(e1)))
                        time.sleep(1) #sleep for a sec
                        continue
                    except Exception as e2:
                        self.Log.error("Exception while trying to get stat for csv file [%s]. Fail processing. Exception [%s]" % (fullFileName, str(e2)))
                        break                      
                        
                if csvfileFound == False:
                    self.Log.error("Fail processing csv file [%s] as it is not found" % (csvfile))
                    self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)                                                        
                    return                    
                
                try:
                    if statinfo.st_size < 1024 and statinfo.st_size > 0 and any(self.csvErrorsList):    #Check for file size between 0 and 1KB, also check if there are any errors to compare
                        fp = open(fullFileName, "r", encoding ="utf-16")
                        try:
                            data = fp.read()
                            #csv file is utf-16 encoded
                        except UnicodeError:
                            #we got the error because the file is ASCII encoded
                            fp = open(fullFileName, "r")
                            data = fp.read()
                        fp.close()
                        for err in self.csvErrorsList:
                            if re.search(err, data, re.IGNORECASE) is not None:                                
                                newFileName = fullFileName + ".output"                                    
                                try:
                                    self.Log.info("Found invalid CSV [%s]. Renaming it to [%s] and moving to Failed folder." % (fullFileName,newFileName))
                                    os.rename(fullFileName, newFileName)
                                    csvfile = csvfile + ".output"
                                    #delete corresponding IP files
                                    ipFileName = fullFileName + ".txt"  #delete *.csv.txt
                                    if os.path.exists(ipFileName):
                                        os.remove(ipFileName)
                                    ipFileNameTxt = ipFileName + ".txt" #delete *.csv.txt.txt
                                    if os.path.exists(ipFileNameTxt):
                                        os.remove(ipFileNameTxt)                                    
                                except Exception as err:
                                    self.Log.exception("Exception while renaming errorneous csv file [%s]: Error: [%s]" % (csvfile, str(err)))

                                self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)                                                        
                                return
                except Exception as err:
                    self.Log.exception("Exception while reading csv file [%s] for checking error messages. Error: [%s]" % (csvfile, str(err)))
                    self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)                                                        
                    return

                filepath = os.path.join(self.ArchiveFolder, csvfile)
                if cvcloudDBObj.insertCsvResult(queryid, fileid, logdate, filepath) == True:
                    self.PostProcessFile(csvfile, True, cvcloudDBObj = cvcloudDBObj)
                else:
                    self.Log.error("Failed to add file data for file " + csvfile)
                    self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)                 
            else:
                if cvcloudDBObj.parseCsvResult(fullFileName, logdate, qidsplit[0], cc_id, csguid) == True:
                    self.PostProcessFile(csvfile, True, cvcloudDBObj = cvcloudDBObj)
                else:
                    self.Log.error("Failed to add file data for file " + csvfile)
                    self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)                                  

        except Exception as err:
            self.Log.exception("Exception processing the csv file %s" % (csvfile))
            self.PostProcessFile(csvfile, False, cvcloudDBObj = cvcloudDBObj)
    
    def ProcessMetricsRemoteOperationsReq(self, filename):
        try:
            fileContent = ""
            ccid = -1
            msgid = -1
            msgidflag = -1
            regcode = -1
            enddate = -1
            PREUPGRADEREQUESTED = 3
            PREUPGRADEREQUESTFAILED = 4
            UPGRADECONFIGURATIONFAILED  = 12
            UPGRADECOMMSERVER = 13
            fullFileName = os.path.join(self.FolderContainingXmlFile,filename)
            fName, fExtension = os.path.splitext(fullFileName)
            fnamesplit = filename.split("_")
            if fnamesplit[3].lower() == "CommServSurveyQuery".lower():
               return
            #skipping files with name :  CSS1497246154_10916D_6BC374A6-37EC-4DC1-84CF-9FEE7876E92C_CommServ_CommServSurveyQuery_21.sql.output
            if (len(fnamesplit) >= 5 and fnamesplit[4].lower() == "CommServSurveyQuery".lower()):
               return
            
            ccid = fnamesplit[1]
            queryname =  fnamesplit[4].split(".")[0]
            
            try:
                fFile = open(fullFileName,"r")
                fileContent = fFile.readlines()
                fFile.close()

            except Exception as err:
                self.Log.exception("ProcessMetricsRemoteOperationsReq :: Failed to open/read file %s!!!" % (fullFileName))
                return
            
            for line in fileContent:
                if(queryname.lower() == "EnablePreUpgradReadiness".lower()):
                    if("<RegCode>" in line):
                        regcode = line.split("<RegCode>")[1].split("</RegCode>")[0]
                    if("<isEnabled>1</isEnabled>" in line):
                        msgid = PREUPGRADEREQUESTED
                        msgidflag = PREUPGRADEREQUESTED
                        break
                elif(queryname.lower() == "EnablePreUpgradReadiness".lower()):
                    if("<RegCode>" in line):
                        regcode = line.split("<RegCode>")[1].split("</RegCode>")[0]
                    if("<isEnabled>0</isEnabled>" in line):
                        msgid = PREUPGRADEREQUESTFAILED
                        msgidflag = PREUPGRADEREQUESTED
                        break
                elif(queryname.lower() == "DisablePreUpgradReadiness".lower()):
                    if("<isEnabled>0</isEnabled>" in line):
                        msgid = UPGRADECOMMSERVER
                        break
                elif(queryname.lower() == "DisablePreUpgradReadiness".lower()):
                    if("<isEnabled>1</isEnabled>" in line):
                        msgid = UPGRADECONFIGURATIONFAILED
                        break
                elif(queryname.lower() == "UpdateGxGlobalParam".lower()):
                    if("<modified>" in line):
                        enddate = line.split("<modified>")[1].split("</modified>")[0]
                    if enddate == "0":
                        if("<created>" in line):
                            enddate = line.split("<created>")[1].split("</created>")[0]
                    if("<name>AUTHCODEFORUPGRADE</name>" in line):
                        msgid = UPGRADECOMMSERVER
                        break
                elif(queryname.lower() == "UpdateGxGlobalParam".lower()):
                    if("[<GXV11UpgradeValue></GXV11UpgradeValue>]" in line):
                        msgid = UPGRADECONFIGURATIONFAILED
                        break
            
            self.PostProcessFile(filename, True, True)
        except Exception as err:
            self.Log.exception("ProcessMetricsRemoteOperationsReq :: Exception " + str(err))
            self.PostProcessFile(filename, False, True)
            return
        finally:
            #update the db with status
            if(ccid != -1 and msgid != -1):
                self.UpdateV11DBUpgradeStatus(ccid,msgid,-1,msgidflag)
            if(ccid != -1 and msgid == UPGRADECOMMSERVER):
                self.UpdateCloudRequestStatus(ccid)
            if(regcode != -1):
                self.UpdateV11DBUpgradeRegCode(ccid, regcode)
            if(enddate != -1):
                self.UpdateV11DBUpgradeExpiryDate(ccid, enddate)

    def getPreUpgradeReadinessStatus(self, ccid):
        try:
            status = True
            Query = "select Comments from cf_SurveyUpgradeReadiness WITH(NOLOCK) where CommServUniqueId  = (select Top 1 ID from cf_CommcellIdNameMap WITH(NOLOCK) where commcellid = "+str(int(ccid,16))+") and Status <> 0 and CheckName not in ('tblDBUAll','tblScriptDBUAll_V2','tblScriptDBUAll','SIMTableDrop','CheckSpacesInClientNames','SIMTmpTblForDepPkgCheck','SIMVSAInstanceCheck_CS')"
            self.Log.debug("getPreUpgradeReadinessStatus :: Query : %s " % Query)
            
            if(self.CloudDB.Execute(Query) == True):
                cur = self.CloudDB.m_cursor
                rows = cur.fetchall()
                if(len(rows) > 0 ):
                    for row in rows:
                        comments = row[0]
                        if (comments.find("###ERROR###") != -1):
                            status = False
                            break
                else:
                    status = True
            else:
                self.Log.error("Non-fatal error, SELECT cf_SurveyUpgradeReadiness :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (Query, self.CloudDB.GetErrorErrStr()))
            
            return status
        except Exception as err:
            self.Log.exception("getPreUpgradeReadinessStatus :: Exception " + str(err))
            return False
    
    def isV11DBUpgradeCommcell(self, ccid):
        try:
            Query = "SELECT 1 from CommcellIDUpgradePhaseMap WITH(NOLOCK) Where CommCellID like '" + str(ccid) +"' and PhaseID not in (1,13) and (CloudRequestID is not null) "
            self.Log.debug("isV11DBUpgradeCommcell :: Query : %s " % Query)
            
            if(self.CloudServicesDB.Execute(Query) == True):
                cur = self.CloudServicesDB.m_cursor
                rows = cur.fetchall()
                if(len(rows) > 0 ):
                    return True
                else:
                    return False
            else:
                self.Log.error("Non-fatal error, SELECT CommcellIDUpgradePhaseMap :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (Query, self.CloudServicesDB.GetErrorErrStr()))
                return False

        except Exception as err:
            self.Log.exception("isV11DBUpgradeCommcell :: Exception " + str(err))
            return False  

    def UpdateV11DBUpgradeStatus(self, ccid, msgid, regcode, msgidflag):
        try:
            regcodeflags = "";
            if regcode != -1:
                regcodeflags = ", regcode = '"+str(regcode) + "'"
            
            if msgidflag != -1:
                Query = "UPDATE CommcellIDUpgradePhaseMap set PhaseID = " +str(msgid)+ " , logdate = getdate() "+ regcodeflags + " Where CommCellID = '" + str(ccid) +"' and  PhaseID < " + str(msgidflag)
            else:
                Query = "UPDATE CommcellIDUpgradePhaseMap set PhaseID = " +str(msgid)+ " , logdate = getdate() "+ regcodeflags + " Where CommCellID = '" + str(ccid) +"'"
            self.Log.debug("UpdateV11DBUpgradeStatus :: Set Message ID Query : %s " % Query)
            
            if(self.CloudServicesDB.Execute(Query) == False):
                self.Log.error("Non-fatal error, Update CommcellIDUpgradePhaseMap :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (Query, self.CloudServicesDB.GetErrorErrStr()))
            
            cur = self.CloudServicesDB.m_cursor
            return cur.rowcount

        except Exception as err:
            self.Log.exception("UpdateV11DBUpgradeStatus :: Exception " + str(err))
            return  
    
    def UpdateV11DBUpgradeRegCode(self, ccid, regcode):
        try:
            
            Query = "UPDATE CommcellIDUpgradePhaseMap set regcode = '" +str(regcode) + "' Where CommCellID = '" + str(ccid) +"'"
            
            self.Log.debug("UpdateV11DBUpgradeRegCode :: Update RegCode Query : %s " % Query)
            
            if(self.CloudServicesDB.Execute(Query) == False):
                self.Log.error("Non-fatal error, Update CommcellIDUpgradePhaseMap :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (Query, self.CloudServicesDB.GetErrorErrStr()))

        except Exception as err:
            self.Log.exception("UpdateV11DBUpgradeRegCode :: Exception " + str(err))
            return
    
    def UpdateV11DBUpgradeExpiryDate(self, ccid, enddate):
        try:
            
            Query = "UPDATE CommcellIDUpgradePhaseMap set ExpiryDate = '" +str(enddate) + "' Where CommCellID = '" + str(ccid) +"'"
            
            self.Log.debug("UpdateV11DBUpgradeExpiryDate :: Update ExpiryDate Query : %s " % Query)
            
            if(self.CloudServicesDB.Execute(Query) == False):
                self.Log.error("Non-fatal error, Update CommcellIDUpgradePhaseMap :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (Query, self.CloudServicesDB.GetErrorErrStr()))

        except Exception as err:
            self.Log.exception("UpdateV11DBUpgradeExpiryDate :: Exception " + str(err))
            return
    
    def UpdateCloudRequestStatus(self, commcellid):
        try:
            PHASECOMPLETED = 3
            Query = "UPDATE CloudRequest set PhaseStatusId = "+str(PHASECOMPLETED)+",SubPhaseStatusId= "+str(PHASECOMPLETED)+" Where RequestId = (Select Top 1 CloudRequestID From CommcellIDUpgradePhaseMap WITH(NOLOCK) Where CommCellID ='"+str(commcellid)+"')"
            
            self.Log.debug("UpdateCloudRequestStatus :: Set Cloud Request Status Query : %s " % Query)
            
            if(self.CloudServicesDB.Execute(Query) == False):
                self.Log.error("Non-fatal error, Update CloudRequest :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (Query, self.CloudServicesDB.GetErrorErrStr()))

        except Exception as err:
            self.Log.exception("UpdateV11DBUpgradeStatus :: Exception " + str(err))
            return

    def RunUpgradeReadiness(self, upg_files):
        self.Log.debug(" ############   START Upgrade Readiness Parsing   ##############")
        metricsOperationsList = ["EnablePreUpgradReadiness", "DisablePreUpgradReadiness",
                                 "UpdateGxGlobalParam"]
        PreUpgradeReadinessReqCommcellIDList = []
        self.CurRelPreUpgradeCCIDList = []
        self.LastRelPreUpgradeCCIDList = []
        self.PreLastRelPreUpgradeCCIDList = []
        for each_upg in upg_files:
            if self.IsCloudServicesOpenConnection == True:
                fNameList = each_upg.split("_")
                ccid = fNameList[1]
                if (len(fNameList) == 5 and ((fNameList[4].split(".")[
                    0]) in metricsOperationsList)):  # process Metrics Operation Requests : CSS1496196995_FA2AB_EB63CC8E-224F-46DB-B0CB-10E62D0E800C_CommServ_DisablePreUpgradReadiness.sql
                    self.ProcessMetricsRemoteOperationsReq(upgfile)
                if (not PreUpgradeReadinessReqCommcellIDList.__contains__(ccid)):
                    PreUpgradeReadinessReqCommcellIDList.append(ccid)
            self.ProcessUpgradeReadiness(each_upg)

        # clean deleted pre upgrade checks from db.
        CurRelPreUpgrade = "','".join(self.CurRelPreUpgrade)
        LastRelPreUpgrade = "','".join(self.LastRelPreUpgrade)
        PreLastRelPreUpgrade = "','".join(self.PreLastRelPreUpgrade)
        CurRelPreUpgradeCCIDList = "','".join(self.CurRelPreUpgradeCCIDList)
        LastRelPreUpgradeCCIDList = "','".join(self.LastRelPreUpgradeCCIDList)
        PreLastRelPreUpgradeCCIDList = "','".join(self.PreLastRelPreUpgradeCCIDList)
        deleteQuery = ""
        if CurRelPreUpgrade != '' and CurRelPreUpgradeCCIDList != '':
            deleteQuery += "delete from cf_surveyupgradereadiness where checkname+'.sql' not in ('" + CurRelPreUpgrade + "') and CommServUniqueId in ( '" + str(
                CurRelPreUpgradeCCIDList) + "' );"
        if LastRelPreUpgrade != '' and LastRelPreUpgradeCCIDList != '':
            deleteQuery += "delete from cf_surveyupgradereadiness where checkname+'.sql' not in ('" + LastRelPreUpgrade + "') and CommServUniqueId in ( '" + str(
                LastRelPreUpgradeCCIDList) + "' );"
        if PreLastRelPreUpgrade != '' and PreLastRelPreUpgradeCCIDList != '':
            deleteQuery += "delete from cf_surveyupgradereadiness where checkname+'.sql' not in ('" + PreLastRelPreUpgrade + "') and CommServUniqueId in ( '" + str(
                PreLastRelPreUpgradeCCIDList) + "' );"

        if deleteQuery != "":
            self.Log.debug("clean deleted pre upgrade scripts :: Query : %s " % deleteQuery)
            if (self.CloudDB.Execute(deleteQuery) == False):
                self.Log.error(
                    "Non-fatal error, clean deleted pre upgrade scripts :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (
                        deleteQuery, self.CloudDB.GetErrorErrStr()))

        if self.IsCloudServicesOpenConnection == True:
            # update commcell db upgrade status in db.
            PREUPGRADEREADINESPASSED = 6
            PREUPGRADEREADINESFAILED = 5
            status = PREUPGRADEREADINESFAILED
            for commcellid in PreUpgradeReadinessReqCommcellIDList:
                if (self.isV11DBUpgradeCommcell(commcellid)):
                    if (self.getPreUpgradeReadinessStatus(commcellid)):
                        status = PREUPGRADEREADINESPASSED
                    else:
                        status = PREUPGRADEREADINESFAILE
                    rowcount = self.UpdateV11DBUpgradeStatus(commcellid, PREUPGRADEREADINESPASSED, -1,
                                                             status)
                    if rowcount != 0:
                        self.UpdateCloudRequestStatus(commcellid)
        self.Log.debug(" ############   END Upgrade Readiness Parsing   ##############")

    def ProcessUpgradeReadiness(self, filename):
      try:
        fullFileName = os.path.join(self.FolderContainingXmlFile,filename)
        fnamesplit = filename.split("_",3)
        if (len(fnamesplit[2])) == 36:  #  v11 file name format : CSS1497246154_10916D_6BC374A6-37EC-4DC1-84CF-9FEE7876E92C_CommServ_SIMMacLicCheck.sql.output
            fnamesplit = filename.split("_",4)
            fDBName = fnamesplit[3]
            fCheckName = fnamesplit[4].split(".")[0]
            fCommServGUID = fnamesplit[2]
        else: # V9/V10 file name format CSS1496147274_F9331_CommServ_SIMiftSPCheck_9_0.sql.output
            fDBName = fnamesplit[2]
            fCheckName = fnamesplit[3].split(".")[0]
            fCommServGUID = ""
                        
        fileUTCTime = datetime.datetime.fromtimestamp(int(fnamesplit[0][3:])).strftime('%Y-%m-%d %H:%M:%S')

        strCCID = self.GetCCIDFromFileName(filename)
        fccid = 0
        if strCCID is not None and len(strCCID) > 0:
            fccid = self.get_ccid_int(strCCID)
        fileMTime = int(fnamesplit[0][3:])
        fComment = ''
        
        try:
            fFile = open(fullFileName,"r")
            lst = fFile.readlines()
            fFile.close()
            
            for line in lst:
                if(line != lst[len(lst)-1]):
                    #remove extra logging in comments
                    if ((line.find("] [RecNum") != -1) and (line.find("INFO: [") != -1) ):
                        while (line.find("] [RecNum") != -1):
                            tline = line.split("] [RecNum",1)
                            line = tline[1]
                            if(tline[0].find("INFO: [") != -1):
                                fComment += tline[0].split("INFO: [",1)[1]
                            else:
                                fComment += tline[0]
                    elif (line.find("INFO: [") != -1):
                        tline = line.split("INFO: [",1)
                        fComment += tline[1]
                    elif (line.find("] [RecNum") != -1):
                        pass#do nothing
                    else:
                        fComment += line
                else:
                    if(line.find("Return Value :: [") == -1):
                        fStatus = 1
                        if(fComment == ''):
                            fComment = line
                    else:
                        fStatus = (line.split("Return Value :: [")[1]).split("]")[0]
        except Exception as err:
            fFile.close()
            self.Log.exception("UpgradeReadiness :: Failed to open/read file %s!!!" % (fullFileName))
            self.PostProcessFile(filename, False, True)
            return
       
        sCommCellid = "CommCellid = '"+str(fccid)+"'"
        sqlCommServGUID = ""
        if (fCommServGUID != ""):
            sqlCommServGUID = " and CommServGUID = '"+str(fCommServGUID)+"'"

        ccidQuery = "select top 1 ID, CommServTZ, ModifiedTime, CommServVersion from cf_CommcellIdNameMap with (nolock) where "+str(sCommCellid)+str(sqlCommServGUID)+" and (Flags & 1) = 0 order by ModifiedTime desc"
        self.Log.debug("UpgradeReadiness :: CommServ Unique ID Query : %s " % ccidQuery)
        CommServTZ=""
        ccid=""
        commServeLastModTime = ""
        CommServVersion = ""
        if(self.CloudDB.Execute(ccidQuery) == True):
            cur = self.CloudDB.m_cursor
            rows = cur.fetchall()
            if(len(rows) > 0 ):
                rst = rows[0]
                self.Log.debug("Cursor Results  : %s " % rst)
                ccid = rst[0] 
                CommServTZ = rst[1]
                commServeLastModTime = rst[2]
                CommServVersion = rst[3].strip()
            else:
                self.Log.error("Failed to get CommCell Unique ID in cf_CommcellIdNameMap Table. CommCellid : %s " % (ccid))
                self.PostProcessFile(filename, False, True)
                return

        
        if(CommServTZ == "" and ccid != ""):
            CommServTZ = "Eastern Standard Time"
        
        fComment = fComment.replace("\r","<br/>")
        fComment = fComment.strip(' \t\n\r')
        if(fComment.find("'") != -1):
            fComment = fComment.replace("'","''")

        CommServVersion = CommServVersion[:3]
        
        if ((float(CommServVersion) == 11.0) and (not self.CurRelPreUpgradeCCIDList.__contains__(str(ccid)))):
            self.CurRelPreUpgradeCCIDList.append(str(ccid))
        elif ((float(CommServVersion) == 10.0) and (not self.LastRelPreUpgradeCCIDList.__contains__(str(ccid)))):
            self.LastRelPreUpgradeCCIDList.append(str(ccid))
        elif ((float(CommServVersion) == 9.0) and (not self.PreLastRelPreUpgradeCCIDList.__contains__(str(ccid)))):
            self.PreLastRelPreUpgradeCCIDList.append(str(ccid))

        iQuery = "delete from cf_surveyupgradereadiness where CheckName like '"+fCheckName+"' and CommServUniqueId = "+str(ccid)+"; insert into cf_surveyupgradereadiness values("+str(ccid)+", '"+fileUTCTime+"', dbo.UTCToLocalDateTimeConversion(dateadd(S, "+str(fileMTime)+", '1970-01-01'), '"+CommServTZ+"'), '"+fCheckName+"', "+str(fStatus)+", '"+fComment+"', '"+fDBName+"')"
        self.Log.debug("UpgradeReadiness :: INSERT QUERY : %s" % iQuery)
        if self.CloudDB.Execute(iQuery) == False:
            self.Log.error("UpgradeReadiness :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (iQuery, self.CloudDB.GetErrorErrStr()))
            return        
        
        try:
            commServeLastModTime = datetime.datetime.strptime(str(commServeLastModTime),'%Y-%m-%d %H:%M:%S')            
            timeDiff = datetime.datetime.utcnow() - commServeLastModTime
            days, seconds = timeDiff.days, timeDiff.seconds
            minSinceLastUpdate = days * 24 * 60 + seconds / 60
            self.Log.debug("cf_CommcellIdNameMap last update-> Days : %s, Seconds:%s, Minutes:%s" % (days,seconds,minSinceLastUpdate))
            if minSinceLastUpdate > 15:               
                iUpdateCNP = "update dbo.cf_CommcellIdNameMap SET ModifiedTime = '%s' Where ID = '%s'" % (time.strftime("%Y-%m-%d %H:%M:%S",time.gmtime()),str(ccid))
                Log.debug("UPDATE cf_CommcellIdNameMap: %s " % iUpdateCNP)
                if self.CloudDB.Execute(iUpdateCNP) == False:
                    self.Log.error("Non-fatal error, Update cf_CommcellIdNameMap :: " + __name__ + " :: Query '%s' failed with error: '%s'" % (iQuery, self.CloudDB.GetErrorErrStr()))
        except Exception as err:
            self.Log.exception("UPDATE cf_CommcellIdNameMap :: Exception ")
        self.PostProcessFile(filename, True, True)
      except Exception as err:
        self.Log.exception("UpgradeReadiness :: Exception ")
        self.PostProcessFile(filename, False, True)
        return
    
    def getUploadedFileIP(self, fullFileName):
        ip = ''
        ipFileName = fullFileName + ".txt"
        if self.isPrivate == False:
            ipFileNameTxt = ipFileName + ".txt"     #When a private metrics server is used as gateway for uploading to public cloud, "*.txt.txt" file received for the source CS that is to be used. 
            if os.path.exists(ipFileNameTxt):
                try:
                    #delete the .txt file
                    if os.path.exists(ipFileName):
                        os.remove(ipFileName)
                    
                    #move the .txt.txt file to .txt file
                    os.rename(ipFileNameTxt, ipFileName)
                except Exception as err:
                    # in case of exception, assume the .txt.txt file is to ipfile
                    ipFileName = ipFileNameTxt        
        
        if os.path.exists(ipFileName):
            try:
                with open(ipFileName, "r") as r:
                    ip = r.read().strip()
            except Exception as err:
                Log.exception("Failed to open ip file [%s]"  % ipFileName)
                ip = ''                            
        else:
            self.Log.debug("IP file [%s] doesn't exist" % ipFileName)
            ip = ''

        return ip
        
    def generate_ip_file(self, file, ip):
        """Create IP file for the given file"""
        try:
            txtFile = os.path.join(self.FolderContainingXmlFile, file) + ".txt"
            with open(txtFile, "w+") as ipFile:
                ipFile.write(ip)
            self.Log.debug("Created ipFile %s" % txtFile)
        except Exception as e:
            self.Log.error("Generation of IP files failed for the file %s" % file)
            raise

    def unzipFiles(self, zip_fullname, ip, cvcloudDBObj):
        try:
        	#verify that the zip archive contains only xml, csv and output files only before proceeding
            for file in zipfile.ZipFile(zip_fullname).namelist():
                if (not  ( file.endswith('.xml') or file.endswith('.output') or file.endswith('.csv') ) ):
                    raise ValueError('File ['+file+'] with unexpected extention found in zip archive ['+zip_fullname+']')
            with zipfile.ZipFile(zip_fullname, 'r') as zip_ref:
                zip_ref.extractall(self.FolderContainingXmlFile)
            self.Log.debug(__name__ + " Unzipping [%s] done " % zip_fullname)                
            xml_files = []
            csv_files = []
            upg_files = []
             
            for file in zipfile.ZipFile(zip_fullname).namelist():
                if file.endswith('.csv'):
                    csv_files.append(file)
                    continue    #no need to generate ip file for error csv files
                elif file.endswith('.xml'):
                    xml_files.append(file)
                    continue    #no need to generate ip file for xml files
                #no need to generate ip file for error output files
                fName, fExtension = os.path.splitext(file)
                fnamesplit = fName.split("_")
                #skipping files with name :  CSS1497246154_10916D_6BC374A6-37EC-4DC1-84CF-9FEE7876E92C_CommServ_CommServSurveyQuery_21.sql.output
                if (len(fnamesplit) >= 5 and (fnamesplit[4]).lower() == "CommServSurveyQuery".lower()):
                    self.PostProcessFile(file, False, False, True, cvcloudDBObj = cvcloudDBObj) #skip output files
                    continue
                if (fnamesplit[3]).lower() == "CommServSurveyQuery".lower():
                    self.PostProcessFile(file, False, False, True, cvcloudDBObj = cvcloudDBObj) #skip output files
                    continue
                if file.endswith('.output'):
                    upg_files.append(file)
                    continue    #no need to generate ip file for upg files
                 
            if len(xml_files) <= 0:
                self.Log.error(__name__ + " No xml files in zip [%s]. Fail processing.", zip_fullname)
                self.PostProcessFile(os.path.basename(zip_fullname), False, cvcloudDBObj = cvcloudDBObj)
            else:        
                self.PostProcessFile(os.path.basename(zip_fullname), True, cvcloudDBObj = cvcloudDBObj)
            return xml_files, csv_files, upg_files

        except Exception as e:
            self.Log.error(__name__ + " Encountered exception %s" % (e))
            self.Log.error(__name__ + " Moving the zip file to failed folder: %s" % zip_fullname)
            self.PostProcessFile(os.path.basename(zip_fullname), False, cvcloudDBObj = cvcloudDBObj)

    def process_main_xml(self, filename, ip, cvcloudDBObj = None):
        """Process the main survey xml"""
        fileid = 0
        csguid = None
        fileid, csguid = self.ProcessSurveyXML(filename, ip, cvcloudDBObj = cvcloudDBObj)
        #print ("fileid = [%d], csguid = [%s]" % (fileid, csguid))
        if fileid is not None and fileid != 0 and csguid is not None:
            self.commcelldict[csguid] = fileid
            return True, fileid, csguid
        else:
            if not (fileid is not None and fileid == -1 and csguid is None):    #fileid = -1 means it is skipped processing now as it is already processed
                self.failedcommcellset.add(self.GetCCIDFromFileName(filename))
            return False, fileid, csguid
        

    def process_zipfile(self, filename, ip, cvcloudDBObj = None):
        '''
        we need to create db connection objects per thread.
        CustomerFeedbackDB
        '''
        islocalcvdbobj = False
        if cvcloudDBObj is None:
            cvcloudDBObj = CustomerFeedbackDatabase()
            islocalcvdbobj = True

        if cvcloudDBObj._bOpenConnection == False:
            self.Log.error(__name__ + " Failed to connect to database CVCloud. ")
            return False
                
        fullFileName = os.path.join(self.FolderContainingXmlFile, filename)
        retVal = False
        fileId = None
        csguid = None
                
        try:
            #unzip
            xml_files, csv_files, upg_files = self.unzipFiles(zip_fullname = fullFileName, ip = ip, cvcloudDBObj = cvcloudDBObj)
            if len(xml_files) <= 0:
                self.Log.error(__name__ + " No xml files in zip [%s]", filename)
                return retVal
            self.Log.debug(__name__ + " Unzip [%s] completed" % filename)
            
            #Process xml files. Ideally there should be only one xml file
            for each_xml in xml_files:
                retVal, fileId, csguid = self.process_main_xml(each_xml, ip, cvcloudDBObj = cvcloudDBObj)
            
            if retVal == True:
                self.Log.debug(__name__ + " Completed processing xml file in [%s] " % filename)
                #Process csv files
                for each_csv in csv_files:
                    self.ProcessSurveyCSV(each_csv, fileId, csguid, cvcloudDBObj = cvcloudDBObj)
                self.Log.debug(__name__ + " Completed processing csv files in [%s] " % filename)
                # Process upg files
                if upg_files:
                    self.RunUpgradeReadiness(upg_files)
            else:
                self.Log.error(__name__ + " Failed to process xml file in [%s]", filename)
                #Process csv files
                for each_csv in csv_files:
                    self.PostProcessFile(each_csv, False, cvcloudDBObj = cvcloudDBObj)
                self.Log.info(__name__ + " Moved csv files in [%s] to failed folder" % filename)
                # Process upg files
                for each_upg in upg_files:
                    self.PostProcessFile(each_upg, False, cvcloudDBObj=cvcloudDBObj)
                self.Log.info(__name__ + " Moved upg files in [%s] to failed folder" % filename)
                
        except Exception as e:
            self.Log.error(__name__ + " Caught an exception while unzipping file [%s]. Exception [%s]", filename, e)
            retVal = False
        if islocalcvdbobj == True:
            del cvcloudDBObj
        return retVal
    
    def setProcessedFileCount(self, isSuccess):
        if isSuccess == True:
            self.numxmls += 1
        else:
            self.numfailxmls += 1
    
    def RunSurvey(self):

        self.Log.debug("Start Processing uploaded files.....")

        # deleting file which may come due to copying from metrics server
        # in case of forwarding, the ip in .txt file may be ip from internal ip address range
        # on public setup, we should retain these files and use these as the .txt.txt files will contain the ip in external range
        if self.isPrivate == True:
            if(platform.system() == "Linux"):
                if len(glob.glob('*.txt.txt')) > 0:
                    command = 'rm ' + self.FolderContainingXmlFile + '/*.txt.txt'
                    os.system(command)
            else:
                command = 'del /Q "' + self.FolderContainingXmlFile + '\\*.txt.txt"'
                os.system(command)
        
        csvfiles = list()
        upgfiles = list()
        self.commcelldict = dict()
        self.failedcommcellset = set()
        self.numxmls = 0
        self.numfailxmls = 0

        bXMLsPresent = False
        trdPool = None
        if self.NoOfProcessingThreads > 0:
            self.Log.info("Creating thread pool with [%d] threads" % (self.NoOfProcessingThreads))
            trdPool = ThreadPool(self.NoOfProcessingThreads)
        else:
            self.Log.debug("Do not create thread pool")
        
        self.Log.debug("Iterate through all the files in path " + self.FolderContainingXmlFile + " and process them.")
        for filename in os.listdir(self.FolderContainingXmlFile):
            
            if filename.startswith('Part'):
                self.Log.debug("Skipping file as it is being copied from remote server " + filename)
                continue
            if filename.startswith('CSS_RegistrationInfo'):
                self.Log.debug("Skipping Registration file " + filename)
                continue
            if filename.startswith('CSS_SendLogsStatus'):
                self.Log.debug("Skipping SendLogStatus file " + filename)
                continue
            if filename.startswith('CSS_ProactiveSupportStatus'):
                self.Log.debug("Skipping ProactiveSupportStatus file " + filename)
                continue
            
            #check if the file is output file
            pattern = 'CSS.[0-9]*_.[a-fA-F0-9-_]*CommServ_CommservSurveyQuery_.[0-9]*\.sql\.output$'
            if re.match(pattern, filename, re.I):
                self.PostProcessFile(filename, False, False, True)
                continue
                
            fullFileName = os.path.join(self.FolderContainingXmlFile, filename)
            fName, fExtension = os.path.splitext(fullFileName)      
    
            if os.path.isdir(fullFileName):
                continue
            elif filename == "." or filename == "..":
                continue
            elif fExtension.lower() not in ['.xml', '.zip', '.csv', ".output"]:
                continue
                                
            #check if the file is to be blocked
            ip = self.getUploadedFileIP(fullFileName)
            if len(ip) <= 0:
                if fExtension.lower() in ['.xml', '.zip']:
                    self.Log.error(__name__ + "Failed to get uploaded server IP for file [%s]", filename)
                self.PostProcessFile(filename, False, False, True)
                continue
            strCCID = self.GetCCIDFromFileName(filename)
            intCCID = 0
            if strCCID is not None and len(strCCID) > 0:
                intCCID = self.get_ccid_int(strCCID)    
                    
            if self.ProcessBlockedFile(intCCID, ip, filename) is True:
                continue
                    
            fileUTCTime = self.getTimestampFromFileName(filename)
                                
            if fExtension.lower() == ".csv":
                #check css timestamp and skip those that are already processed
                csguid = self.getGUIDFromCsvFileName(filename)
                self.Log.debug(__name__ + "Check if file [%s] with csguid [%s], ccid [%d] and timestamp [%d] is to be skipped" % (filename, csguid, intCCID, fileUTCTime))
                lastFileProcessedTime = self.CustomerFeedbackDB.getlastprocessedtime(csguid, intCCID)
                if lastFileProcessedTime > fileUTCTime + self.LastProcessFileBufferTimeSec: 
                    self.Log.info(__name__ + "Latest file is already processed. Skip [%s]", filename)
                    self.PostProcessFile(filename, True)
                    continue
                                
                csvfiles.append(filename)
                self.Log.debug(__name__ + "Adding CSV File %s" % (filename))
                continue

            if fExtension.lower() == ".output":
                upgfiles.append(filename)
                self.Log.debug(__name__ + "Adding pre upgrade file %s" % (filename))

            if bXMLsPresent == False:
                self.Log.info("#######################################")
                self.Log.info("# Begin processing metrics collection #")
                self.Log.info("#######################################")
                bXMLsPresent = True
            
            self.Log.debug(__name__ + " Processing file [%s], CCID [%s], IP [%s]" %(filename, strCCID, ip))
            
            if fExtension.lower() == ".zip":
                #check css timestamp and skip those that are already processed
                csguid = self.getGUIDFromZipFileName(filename)
                self.Log.debug(__name__ + "Check if file [%s] with csguid [%s], ccid [%d] and timestamp [%d] is to be skipped" % (filename, csguid, intCCID, fileUTCTime))
                lastFileProcessedTime = self.CustomerFeedbackDB.getlastprocessedtime(csguid, intCCID)
                if lastFileProcessedTime >= fileUTCTime + self.LastProcessFileBufferTimeSec:
                    self.Log.info(__name__ + "Latest file is already processed. Skip [%s]", filename)
                    self.PostProcessFile(filename, True)
                    continue            	
                if self.NoOfProcessingThreads <= 0: #do not spawn multiple threads.
                    try:
                        self.setProcessedFileCount(self.process_zipfile(filename = filename, ip = ip, cvcloudDBObj = self.CustomerFeedbackDB))
                    except Exception as e:
                        self.Log.error(__name__ + "Caught an exception while processing zip file [%s]. Exception [%s]", filename, e)
                        self.numfailxmls += 1                	
                else: #spawn multiple threads
                    try:
                        trdPool.apply_async(self.process_zipfile, args = (filename, ip, None, ), callback = self.setProcessedFileCount)
                    except Exception as e:
                        self.Log.error(__name__ + "Caught an exception while processing zip file [%s]. Exception [%s]", filename, e)
                        self.numfailxmls += 1
            else:
                #xml files
                try:
                    if filename.startswith('CSS_DisableCSMessage'):
                        self.ProcessSurveyDisableXML(filename, ip)
                    elif filename.startswith('CSS_AlertsMessage'):
                        self.ProcessAlertsXML(filename, ip)
                    else:
                        success = False
                        fileId = None
                        csguid = None                    
                        success, fileId, csguid = self.process_main_xml(filename, ip)
                        if success == True:
                            self.numxmls += 1
                        else:
                            self.numfailxmls += 1
                except Exception as err:
                    self.Log.exception(__name__ + 'Exception While processing XML file %s' % (filename))
                    self.numfailxmls += 1
                
            if (self.numxmls >= self.MaxXMLFilesToProcess):
                self.Log.info(__name__ + "Reached max limit [%d] for processing xml file." % (self.MaxXMLFilesToProcess))
                break
        if trdPool is not None:
            trdPool.close()
            trdPool.join()
        self.Log.debug(__name__ + " :: Done Processing uploaded files for current batch. Success [%d], failed [%d]" % (self.numxmls, self.numfailxmls))
        
        # Call workflow asynchronously
        if self.isPrivate == False and self.CheckUpgradedCommcells == 1:
            self.CallUpgradeWorkflow()
            # self.CallCommcellRegisterationWorkflow()

        #Below csv processing is needed for global metrics server that receives xml/csv separately and older commcells which are uploading xml/csv separately. 
        if csvfiles: 
            if len(self.commcelldict) > 0: #successful main xml parsed, and csv file exists. 
                self.Log.info("# Begin processing CSV files")
                for csvfile in csvfiles:
                    self.ProcessSurveyCSV(csvfile, 0, '')
                self.Log.info("# Completed processing CSV files")
            else:	#Move these csv files to failed folder as corresponding xml is not founf
                self.Log.info("# Start moving dangled csv files to failed folder")
                for csvfile in csvfiles:
                    self.PostProcessFile(csvfile, False)
                self.Log.info("# End moving dangled csv files to failed folder")          	

        if upgfiles:
            self.RunUpgradeReadiness(upgfiles)

        if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
            self.Log.info("cvd with process id [%d] is no longer running. Quit" % (self.CvdPid))
            return            

        if bXMLsPresent == True:
            retVal = False
            retVal = self.CallProcessCollectedResults()
            if retVal == True:
                self.Log.info("Completed processing for [%d] commcells from [%d] successful xmls. [%d] xmls failed to process." % ( len(self.commcelldict), self.numxmls, self.numfailxmls))
                self.Log.info("#####################################")
                self.Log.info("# End processing metrics collection #")
                self.Log.info("#####################################")
            else:    
                self.Log.info("Failed processing for [%d] commcells from [%d] xmls." % ( len(self.commcelldict), self.numxmls+self.numfailxmls))
                self.Log.info("########################################")
                self.Log.info("# Failed processing metrics collection #")
                self.Log.info("########################################")
                               
        return
    
    def WebserverLogin(self):

        conn = None
        authToken = ""
        baseUrl = ''

        try:
            if len(self.workflowUserCreds) < 3:
                self.Log.error("Failed to get user info to execute workflow webserver API")
            else:
                webConsole = self.workflowUserCreds['WebConsoleURL']

                baseUrl = webConsole[(webConsole.find('//')+2):webConsole.find('/',webConsole.find('//')+2)]
                protocol = webConsole[:webConsole.find('//')]
                requestUrl = u"/webconsole/api/Login"
                body = json.dumps({'password': self.workflowUserCreds['password'],'username': self.workflowUserCreds['username']})
                header = {"Content-Type": "application/json"}

                if protocol.lower() == 'http:':
                    conn = HTTPConnection(baseUrl)
                else:
                    conn = HTTPSConnection(baseUrl)

                conn.request("POST", requestUrl,body,header)
                response = conn.getresponse()

                data = response.read()

                # create element tree object
                tree = ET.ElementTree(ET.fromstring(data))
                # get root element
                root = tree.getroot()
                authToken = root.attrib['token']

        except Exception as err:
                Log.exception(__name__ + 'Exception while calling Login API: [%s]. Error: [%s]' % (baseUrl+requestUrl, str(err)))
                
        return baseUrl,conn, authToken

    def IsProcessCollectedResultsCallViaApi(self):
        if (self.ProcessCollectedResultsViaApi is not None):
            return self.ProcessCollectedResultsViaApi
        #this will return if true if we have to call using api, and false if we have to call the sp using python driver
        if self.IsOpenConnection == True:
            query = "select Value from cf_surveyConfig WITH(NOLOCK) where name = 'CallProcessCollectedResultsViaApi'"
            try:
                if self.CloudDB.Execute(query) == True:
                    row = self.CloudDB.m_cursor.fetchone()
                    if row and row.Value == '0':
                        self.Log.debug("Call ProcessCollectedResults SP using python driver")
                        self.ProcessCollectedResultsViaApi = False
                        return False
            except:
                self.Log.exception("IsProcessCollectedResultsCallViaApi:: Caught Exception while querying CVCloud DB")
        self.Log.debug("Call ProcessCollectedResults SP through the API")
        self.ProcessCollectedResultsViaApi = True
        return True

    def CallProcessCollectedResults(self):
        if(not self.IsProcessCollectedResultsCallViaApi()):
            return self.CustomerFeedbackDB.parseSurverResults()

        dllpath = ''
        if(platform.system() == "Linux"):
            dllpath = os.path.join(self.SimpanaBaseFolder, 'libEvExternal.so')
        else:
            dllpath = os.path.join(self.SimpanaBaseFolder, 'EvExternal.dll')

        mydll = ctypes.CDLL(dllpath)
        try:
            #we need to pass the instance name to the function
            instanceName = 'Instance001'
            if(platform.system() == "Windows"):
                QinetixVM = os.path.dirname(os.path.realpath(__file__))
                startindex = QinetixVM.find('Cloud');
                if(startindex<0):
                    self.Log.error("Could not obtain the instance name, proceeding with default value")
                else:
                    QinetixVM = QinetixVM[0:startindex]
                    QinetixVM = QinetixVM+'\QinetixVM'
                    with open(QinetixVM) as instancefile:
                        for line in instancefile.readlines():
                            instanceName = line
                            break
            else:
                dir_path = os.path.dirname(os.path.realpath(__file__ + '/../../'))
                with open(dir_path + "/galaxy_vm") as fp:
                    lines = fp.read().split("\n")
                    for line in lines:
                        if "GALAXY_INST" in line:
                            instanceName = line.split('"')[1]
                    fp.close()

            instance_c = ctypes.c_char_p(instanceName.encode('utf-8'))
            mydll = ctypes.CDLL(dllpath)
            mydll.ProcessCollectedResults.restype = ctypes.c_int
            self.Log.info("Calling ProcessCollectedResults...")
            resp = mydll.ProcessCollectedResults(instance_c)
            if (resp != 0):
                return False
            return True
        except Exception as err:
            self.Log.error("CallProcessCollectedResults failed with error: " + str(err))
            return False

    def CallCommcellRegisterationWorkflow(self):
        if self._unregisteredCommcells.qsize() == 0:
            self.Log.debug("No unregistered Commcells to proceed with.")
            return

        base_url,conn,authToken = self.WebserverLogin()
        if conn is None or len(authToken) == 0:            
            return

        requestUrl = u"/webconsole/api/wapi/RegisterBeamingCommserver"        
        header = {"Content-Type": "application/xml", "Authtoken":authToken}   

        try:     
            commcellIds = ""
            i = 0
            qLen = self._unregisteredCommcells.qsize()
            while i < self.UnregisteredCommcellsLimit and i < qLen:
                commcellIds += "<commcellIds>%s</commcellIds>" % self._unregisteredCommcells.get_nowait()    
                i += 1            

            body = "<inputs>%s</inputs>" % commcellIds                
            
            conn.request("POST", requestUrl,body,header)
            response = conn.getresponse()

            data = response.read()
            self.Log.debug("Response: [%s]" % data)
            tree = ET.ElementTree(ET.fromstring(data))            
            root = tree.getroot() 
            isError = 0
            if root.tag == 'Workflow_StartWorkflowResponse':
                if int(root.attrib['jobId']) <= 0:
                    isError = 1
            else:
                isError = 1
                    
            if isError == 1:
                self.Log.error("Workflow failed to start with error: [%s]" % (data))                

        except Exception as err:
            self.Log.exception(__name__ + 'Exception while calling workflow execution API for Upgrade: [%s]. Error: [%s]' % (base_url+requestUrl, str(err)))

    def CallUpgradeWorkflow(self):
        
        if len(self._upgradedCSInfo) == 0:
            self.Log.debug("No Upgraded Commcells to proceed with.")
            return

        base_url,conn,authToken = self.WebserverLogin()
        if conn is None or len(authToken) == 0:
            return


        requestUrl = u"/webconsole/api/wapi/UpdateCommcellProductInfo"
        header = {"Content-Type": "application/xml", "Authtoken":authToken}

        for r in self._upgradedCSInfo:
                   
            try:     
                body = ('''<inputs>
                                <serialCode>%s</serialCode>
                                <commCellID>%s</commCellID>
                                <registrationCode>%s</registrationCode>
                                <productVersion>%s</productVersion>
                                <servicePackVersion>%s</servicePackVersion>
                            </inputs>''' % (self._upgradedCSInfo[r]['serial'],self._upgradedCSInfo[r]['commcellId'],self._upgradedCSInfo[r]['registrationCode'],self._upgradedCSInfo[r]['version'],self._upgradedCSInfo[r]['servicePack']))
                
                conn.request("POST", requestUrl,body,header)
                response = conn.getresponse()

                data = response.read()

                tree = ET.ElementTree(ET.fromstring(data))            
                root = tree.getroot() 
                isError = 0
                if root.tag == 'Workflow_StartWorkflowResponse':
                    if int(root.attrib['jobId']) < 0:
                        isError = 1
                else:
                    isError = 1
                        
                if isError == 1:
                    self.Log.error("Workflow failed to start with error: [%s]" % (data))                

            except Exception as err:
                self.Log.exception(__name__ + 'Exception while calling workflow execution API for Upgrade: [%s]. Error: [%s]' % (base_url+requestUrl, str(err)))

        self._upgradedCSInfo = {}

    def DeleteOldFiles(self,FolderName="",FilePruneInterval=60):
        PathToArchiveFiles = os.path.join(self.FolderContainingXmlFile, FolderName)
        if not os.path.exists(PathToArchiveFiles):
            return True
        listOfFiles = os.listdir(PathToArchiveFiles)
        for ArchiveFile in listOfFiles:
            if ArchiveFile.startswith('Part_'):
                self.Log.debug("Skipping temporary file %s" % ArchiveFile)
                continue            
            fullfilename = os.path.join(PathToArchiveFiles, ArchiveFile)
            if os.path.isdir(fullfilename):
                continue;        
            try:
                createTimeEpoch = os.path.getctime(fullfilename)
                createTime = datetime.datetime.fromtimestamp(createTimeEpoch)
                timeDifference = self.Tmnow - createTime
                daysSinceCreate = int(timeDifference.days)
                if( daysSinceCreate >= FilePruneInterval):
                    self.Log.debug("Deleting file %s"%fullfilename)
                    os.remove(fullfilename)
            except Exception as err:
                self.Log.exception("Failed to delete file %s"%fullfilename)
                
        return True

    def get_ccid_int(self, cc_id):
        """
        returns int of commcell id
        Args:
            ccid (str): hexadecimal ccid in string format
        """
        try:
            if cc_id == 'FFFFF':
                cc_id = -1
            else:
                cc_id = int(cc_id, 16)
        except ValueError:
            cc_id = -1
        return cc_id

    def GetCCIDFromFileName(self, filename):
        ccid = None

        self.FileNamePatterns = ['CSS\d{10}_(.[a-fA-F0-9]*)\.xml'  #CSS1379387259_F4A19.xml
                                 , 'CSS\d{10}_(.[a-fA-F0-9]*)_.[0-9]*\.xml'  #CSS1459096822_F7299_137.xml
                                 , 'CSS_RegistrationInfo_\d{10}_(.[a-fA-F0-9]*)\.xml'  #CSS_RegistrationInfo_1431796374_FB5A4.xml
                                 , 'CSS_SendLogsStatus_\d{10}_(.[a-fA-F0-9]*)\.xml'  #CSS_SendLogsStatus_1431796374_FB5A4.xml
                                 , 'CSS_DisableCSMessage_\d{10}_(.[a-fA-F0-9]*)\.xml'  #CSS_DisableCSMessage_1431796374_FB5A4.xml
                                 , 'CSS\d{10}_(.[a-fA-F0-9]*)_CommServ_?'  #CSS1434403657_F9FD8_CommServ_SIMTableDrop.sql.output
                                 , 'CSS\d{10}_(.[a-fA-F0-9]*)_.[a-fA-F0-9\-]*_CommServ_?'  # CSS1609108568_FC473_814F5DCE-C464-462A-BA4B-96C1E48FDDC4_CommServ_CheckCCMStoragePolicies.sql.output
                                 , 'sendLogFiles_(.[a-fA-F0-9]*)_?'  #sendLogFiles_1297BF0_2015_06_15_16_32_00_8366.7z
                                 , 'Logs_.[a-zA-Z0-9]*_(.[a-fA-F0-9]*)_?'  #Logs_BHUSHANTEST5_1297BF0_2015_06_15_16_32_00_8366.7z
                                 , 'Index_.[a-zA-Z0-9]*_(.[a-fA-F0-9]*)_?'  #Index_BKCLOUDTEST01_1297BF0_2015_06_15_16_32_00_8366.7z
                                 , 'CSS\d{10}_(.[a-fA-F0-9]*)_.[a-fA-F0-9\-]*_.[0-9]*\.csv'  #CSS1445348906_10165B_B9D55776-1CDF-4CD3-9499-B1238A9EAC9C_134.csv
                                 , 'CSS_ProactiveSupportStatus_(.[a-fA-F0-9]*)_RequestId_\d{1,10}\.xml'  #CSS_ProactiveSupportStatus_109580_RequestId_1234.xml
                                 , 'CSS\d{10}_(.[a-fA-F0-9]*)_.[a-fA-F0-9\-]*\.zip'  #CSS1589445302_FFFFF_BAA553476-1CDF-4CD3-9499-B1238A9EAC9C.zip
                                 , 'CSS0_(.[a-fA-F0-9]*)_.[a-fA-F0-9\-]*\.zip'  #CSS0_FFFFF_BAA553476-1CDF-4CD3-9499-B1238A9EAC9C.zip
                                 ]
        
        for pattern in self.FileNamePatterns:
            ccids = re.findall(pattern, filename, re.I)
            if len(ccids) == 1:
                return ccids[0]
                
        return ccid

    def getTimestampFromFileName(self, filename):
        retVal = -1        
        try:
            fnamesplit = filename.split("_",1)
            timestamp = fnamesplit[0].split("CSS")[1]
            if timestamp.isdigit():
                retVal = int(timestamp)
            else: 
                retVal = -1
        except Exception as err:
            self.Log.debug("Cannot find timestamp from file [%s]" % filename)
            retVal = -1
        return retVal
    
    def getGUIDFromZipFileName(self, filename):
        csguid = '' 
        #get the commserv guid from filename
        fnamesplit = filename.split("_",3)
        extsplit = fnamesplit[2].split('.',2)
        csguid = extsplit[0]
        return csguid

    def getGUIDFromCsvFileName(self, filename):
        csguid = '' 
        #get the commserv guid from filename
        fnamesplit = filename.split("_",4)       
        csguid = fnamesplit[2]
        return csguid    

    def ParsePruning(self):
            Printon = 0
            
            if self.nDisablePruning > 0:
                return False, -1, "Pruning is disabled via Registry key: [nDisablePruning] under [Cloud] registry";
            try:
                ProcessXMLQuery = "EXEC PruneSurveyResults %s" % (Printon)
                self.Log.debug("Executing query '%s'" % ProcessXMLQuery)
                if self.CloudDB.Execute(ProcessXMLQuery) == False:
                    self.Log.info("Query '%s' failed with error: '%s'" % (ProcessXMLQuery, self.CloudDB.GetErrorErrStr()))
                    return False, -1, self.CloudDB.GetErrorErrStr()
                else:
                    row = self.CloudDB.m_cursor.fetchone()
                    if row:
                        if row.ErrorString and row.ErrorCode == 0:
                            self.Log.info(row.ErrorString)
 
                            ProcessXMLQuery = "SELECT DISTINCT CommcellIdNameMap_ID from cf_CommservSurveyDeletedCommcells WITH (NOLOCK) WHERE DeleteSchedules = 0"
                            self.Log.debug("Executing query '%s'" % ProcessXMLQuery)
                            if self.CloudDB.Execute(ProcessXMLQuery) == False:
                                self.Log.info("Query '%s' failed with error: '%s'" % (ProcessXMLQuery, self.CloudDB.GetErrorErrStr()))
                            else:
                                delccs = self.CloudDB.m_cursor.fetchall()
                                for delcc in delccs:
                                    self.Log.info( 'Deleted  Commcell with unique ID [%d] ' % (delcc.CommcellIdNameMap_ID))
                                    if self.IsCSOpenConnection == False:
                                        self.Log.info( 'Unable to delete Alarms/ schedules for Commcell with unique ID [%d]. There is no connection to CSDB ' % (delcc.CommcellIdNameMap_ID))
                                    else:
                                        self.Log.debug( 'Deleting Alarms and schedules for Commcell with unique ID [%d].' % (delcc.CommcellIdNameMap_ID))
                                        bDeletedScheds = True
                                        #AlarmAction.DELETE = 3
                                        ProcessXMLQuery = 'exec AlrmAddEditAlarm 1, \'<CVGui_AlarmRequest action="3" commUniId="%d"></CVGui_AlarmRequest>\', 0' % (delcc.CommcellIdNameMap_ID)
                                        self.Log.debug("Executing query '%s'" % ProcessXMLQuery)
                                        if self.CSDB.Execute(ProcessXMLQuery) == False:
                                            self.Log.info("Query '%s' failed with error: '%s'" % (ProcessXMLQuery, self.CSDB.GetErrorErrStr()))
                                            bDeletedScheds = False

                                        #AlarmAction.DELETE_COMMCELL = 9
                                        ProcessXMLQuery = 'exec AlrmAddEditAlarm 1, \'<CVGui_AlarmRequest action="9" commUniId="%d"></CVGui_AlarmRequest>\', 0' % (delcc.CommcellIdNameMap_ID)
                                        self.Log.debug("Executing query '%s'" % ProcessXMLQuery)
                                        if self.CSDB.Execute(ProcessXMLQuery) == False:
                                            self.Log.info("Query '%s' failed with error: '%s'" % (ProcessXMLQuery, self.CSDB.GetErrorErrStr()))
                                            bDeletedScheds = False

                                        if bDeletedScheds:
                                            ProcessXMLQuery = 'update cf_CommservSurveyDeletedCommcells set DeleteSchedules = 1 where CommcellIdNameMap_ID = %d' % (delcc.CommcellIdNameMap_ID)
                                            self.Log.debug("Executing query '%s'" % ProcessXMLQuery)
                                            if self.CloudDB.Execute(ProcessXMLQuery) == False:
                                                self.Log.info("Query '%s' failed with error: '%s'" % (ProcessXMLQuery, self.CloudDB.GetErrorErrStr()))
                            return True, row.ErrorCode, row.ErrorString
                        else:
                            self.Log.info(row.ErrorString)
                            return False, row.ErrorCode, row.ErrorString
                    return False, -1, ""
            except Exception as err:
                self.Log.exception("Caught Exception while pruning data")
                self.Log.exception(sys.exc_info()[0])
                return False, -1, ""
            
    def UploadPublicMetricsData(self):
        try:
            for file in os.listdir(self.PublicMetricsUploadFolder):
                #skip uploading .txt files
                filepath = os.path.join(self.PublicMetricsUploadFolder, file)
                fName, fExtension = os.path.splitext(file)
                if fExtension.lower() == ".txt":
                    try:
                        os.remove(filepath)
                    except Exception as err:
                        self.Log.exception("Failed to delete file %s" % (filepath))
                    continue
                
                # get ccid from file
                ccid = self.GetCCIDFromFileName(file)
                
                try:
                    if os.path.isfile(filepath):
                        uploadFile(filepath, self.PublicMetricsUploadSite, self.PublicMetricsUploadSiteUser,self.PublicMetricsUploadSitePasswd, ccid, self.Log, self.HttpProxyHost, self.HttpProxyPort, self.HttpProxyUser, self.HttpProxyPasswd)
                except Exception as e:
                    Log.exception("Exception uploading public metrics file")
        except Exception as e:
            Log.exception("Exception uploading public metrics data")
        return

    def UploadSendlogFilesData(self):
        try:
            for file in os.listdir(self.SendlogsUploadFolder):
                #skip uploading .txt files
                filepath = os.path.join(self.SendlogsUploadFolder, file)
                fName, fExtension = os.path.splitext(file)

                if fExtension.lower() == ".txt":
                    try:
                        os.remove(filepath)
                    except Exception as err:
                        self.Log.exception("Failed to delete file %s" % (filepath))
                    continue

                # get ccid from file
                ccid = self.GetCCIDFromFileName(file)              

                try:
                    ulurl = self.SendlogsUploadSite
                    uluser = self.SendlogsUploadSiteUser
                    ulpassword = self.SendlogsUploadSitePassword
                    if os.path.isfile(filepath):
                        #check if the the upload info is to be read from .url file
                        urlfilepath = filepath + '.url'
                        if os.path.exists(urlfilepath) and os.path.isfile(urlfilepath):
                            f = open(urlfilepath, 'r')
                            lines = f.readlines()
                            f.close()
                            os.unlink(urlfilepath)
                            if len(lines) >= 3:
                                ulurl = lines[0].strip()
                                uluser = lines[1].strip()
                                ulpassword = GetMetricsSimpleData(lines[2].strip(),self.SimpanaBaseFolder,self.Log)
                                if ulurl[len(ulurl) - 1] != '/':
                                    ulurl += '/'
                                ulurl += 'httplogupload.do'                    
                        uploadFile(filepath, ulurl, uluser, ulpassword, ccid, self.Log, self.HttpProxyHost, self.HttpProxyPort, self.HttpProxyUser, self.HttpProxyPasswd)
                except Exception as e:
                        Log.exception("Exception uploading send logs file %s" % filepath)
        except Exception as e:
            Log.exception("Exception uploading send logs files data")
                
        return

    def RunMetricsAsProxyProcessing(self):
        #check if the metrics server is being used as proxy
        if self.UseMetricsServerAsProxy == False:
            return
        
        #upload the public metrics data
        self.UploadPublicMetricsData()
        
        #upload sendlogs data
        self.UploadSendlogFilesData()

            
    def RunSurveyPruning(self):
        dd = datetime.datetime
        tmlastprune = dd.utcfromtimestamp(self.LastMetricsPruningTime)
        self.Tmnow = dd.now()
        cloudRegKey = None
        
        runPruning = False
        if (self.PruneScheduleHrs % 24 == 0) and (self.Tmnow.hour <= 1) and (tmlastprune + datetime.timedelta(hours=self.PruneScheduleHrs) < self.Tmnow):
            runPruning = True #Pruning interval set in days. Run between 12:00 AM to 1:00 AM.
        elif self.PruneScheduleHrs > 0 and self.PruneScheduleHrs % 24 != 0 and (tmlastprune + datetime.timedelta(hours=self.PruneScheduleHrs) < self.Tmnow):
            runPruning = True #Pruning interval set in hours (not daily). Run as per the specified interval.
        elif (self.RunOneTimeProcessing == True):
            runPruning = True #one time execution. initiate pruning now.
            
        if runPruning == True:
            Log.info("Start Pruning.....")
            
            #do pruning
            Folderlist =[{"name":"Archive","FilePruneInterval":self.ArchiveFilePruneInterval},
                         {"name":"Blocked","FilePruneInterval":self.BlockedFilePruneInterval},
                         {"name":"Failed","FilePruneInterval":self.FailedFilePruneInterval},
                         {"name":"ZipArchive","FilePruneInterval":self.ArchiveFilePruneInterval},
                         {"name":"UpgradeReadiness_Archive","FilePruneInterval":self.ArchiveFilePruneInterval},
                         {"name":"UpgradeReadiness_Blocked","FilePruneInterval":self.BlockedFilePruneInterval},
                         {"name":"UpgradeReadiness_Failed","FilePruneInterval":self.FailedFilePruneInterval},
                         {"name":"","FilePruneInterval":self.ProcessFilePruneInterval}
                         ]
 
            for folder in Folderlist:
                self.Log.info(" Started to prune old ['%s'] files "% (folder))
                retVal = self.DeleteOldFiles(folder["name"],folder["FilePruneInterval"])
                if retVal == True:
                    self.Log.info(" Successfully pruned old ['%s'] files "% (folder))
                else:
                    self.Log.info(" Files pruning failed for ['%s'] files "% (folder))
            
            retVal, ErrorCode, szErrorString = self.ParsePruning()
            if retVal == True:
                self.Log.info("Successfully completed pruning ['%s']" % (szErrorString))
            else:
                self.Log.error("Failed in pruning ErrorCode: ['%d' : '%s']" % (ErrorCode, szErrorString))
            
            if (self.Tmnow.hour <= 1) and self.EnableDBMaintenance != 0: #run db maintenance only between 12:00 to 1:00 midnight
                tmdbmaintfull = datetime.datetime.fromtimestamp(self.LastDBMaintenanceFull)
                runfulldbmaint = False
    
                if(tmdbmaintfull + datetime.timedelta(days=self.DBMaintenanceFullInterval) < self.Tmnow):
                    runfulldbmaint = True
                
                dbmaintcmd = "DBMaintenance.exe  -N %s -S %s -U %s -P %s " % (self.CloudDbName, self.DBInstanceName, self.sz_cloudusername.decode('utf-8'), self.sz_cloudpassword.decode('utf-8'))
                
                if runfulldbmaint == True:
                    dbmaintcmd += '-full'
                    Log.info("Start Full DB Maintainance.....")
                else:
                    dbmaintcmd += '-shrinkdb -reindexrecommended'
                    Log.info("Start Shrink DB Maintainance.....")
                
                childproc = subprocess.Popen(args = dbmaintcmd, cwd = self.SimpanaBaseFolder)
                childproc.wait()
                
                if childproc.returncode != 0:
                    self.Log.error("DbMaintainance failed with error code %d" % (childproc.returncode))                
                else:
                    try:
                        aReg = _winreg.ConnectRegistry(None,_winreg.HKEY_LOCAL_MACHINE)
                        cloudRegKey = _winreg.OpenKey(aReg, self.SimpanaCloudRegPath, 0, _winreg.KEY_ALL_ACCESS)
                        
                        if runfulldbmaint == True:
                            self.LastDBMaintenanceFull = calendar.timegm( self.Tmnow.utctimetuple())
                            _winreg.SetValueEx(cloudRegKey, r"nLastDBMaintenanceFull", 0, _winreg.REG_DWORD, self.LastDBMaintenanceFull)
                            Log.info("End Full DB Maintainance.....")
                        else:
                            Log.info("End Shrink DB Maintainance.....")
                    except Exception as err:
                        self.Log.exception("Exception while trying to set full db maintainance time")
            else:
                if self.CustomerFeedbackDB.runDefragmentIndexes() == True:
                    self.Log.info("Successfully executed DefragmentIndexes")
                else:    
                    self.Log.info("Failed to execute DefragmentIndexes") 
               
            #set the last pruning time in reg
            try:
                if cloudRegKey is None:
                    aReg = _winreg.ConnectRegistry(None,_winreg.HKEY_LOCAL_MACHINE)
                    cloudRegKey = _winreg.OpenKey(aReg, self.SimpanaCloudRegPath, 0, _winreg.KEY_ALL_ACCESS)

                self.LastMetricsPruningTime = calendar.timegm( self.Tmnow.utctimetuple())
                _winreg.SetValueEx(cloudRegKey, r"nLastMetricsPruningTime", 0, _winreg.REG_DWORD, self.LastMetricsPruningTime)
            except Exception as err:
                self.Log.exception("Exception while trying to set last pruning time")
        else:
            self.Log.debug("Pruning interval not met. Skipping pruning.")
            
        return
    
    def DoesMetricsNeedReinitialize(self):
        if self.IsCSOpenConnection == True:            
            query = "select value from GXGlobalParam WITH(NOLOCK) where name = 'ReinitializeMetricsParameters' and modified = 0"
            try:                
                if self.CSDB.Execute(query) == False:
                    self.Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
                    return False
                row = self.CSDB.m_cursor.fetchone()
                if row:
                    value = int(row.value)
                    if value == 1:
                        self.Log.debug("Found key[ReinitializeMetricsParameters] in GxGlobalParam on CommServ")
                        return True            
            except:
                self.Log.exception("DoesMetricsNeedReinitialize:: Caught Exception while querying CommServ DB")                
        
        if self.IsOpenConnection == True:
            query = "select Value from cf_surveyConfig WITH(NOLOCK) where name = 'ReinitializeMetricsParameters'"
            try:
                if self.CloudDB.Execute(query) == True:            
                    row = self.CloudDB.m_cursor.fetchone()
                    if row and row.Value == '1':
                        self.Log.debug("Found key[ReinitializeMetricsParameters] in cf_surveyConfig on CVCloud.")
                        return True                    
            except:
                self.Log.exception("DoesMetricsNeedReinitialize:: Caught Exception while querying CVCloud DB")           

        return False
    
    def getUserToExecuteWorkflow(self):
         
        userCreds = {}

        if self.IsCSOpenConnection == True:            
            query = "select name, value from GxGlobalParam WITH(NOLOCK) where Name IN ('cloud workflow user name', 'cloud workflow user password', 'WebConsoleURL') AND modified = 0"
            try:                        
                if self.CSDB.Execute(query) == False:
                    self.Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
                    return False
                for row in self.CSDB.m_cursor.fetchall():
                    if row.name == 'cloud workflow user name':
                        userCreds['username'] = row.value
                    elif row.name == 'cloud workflow user password':
                        userCreds['password'] = row.value  
                    elif row.name == 'WebConsoleURL':
                        userCreds['WebConsoleURL'] = row.value                           
            except:
                self.Log.exception("getUserToExecuteWorkflow:: Caught Exception while querying CommServ DB")
        
        return userCreds

    def ResetDoesMetricsNeedRinitialize(self):
        if self.IsCSOpenConnection == True:            
            query = "Update GXGlobalParam set value = '0' where name = 'ReinitializeMetricsParameters' and modified = 0"
            try:                
                #self.Log.debug("Executing query '%s'" % query)
                if self.CSDB.Execute(query) == False:
                    self.Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
            except:
                self.Log.exception("ResetDoesMetricsNeedRinitialize:: Caught Exception while querying CommServ DB")            

        if self.IsOpenConnection == True:
            query = "Update cf_surveyConfig set value = '0' where name = 'ReinitializeMetricsParameters'"
            try:
                if self.CloudDB.Execute(query) == False:            
                    self.Log.error("Query '%s' failed with error: '%s'" % (query, self.CloudDB.GetErrorErrStr()))              
            except:
                self.Log.exception("ResetDoesMetricsNeedRinitialize:: Caught Exception while querying CVCloud DB")
                
    
    def RollbackPendingTransaction(self):
        foundPendingTransaction = False
        quitProcessing = False
        sleepCounter = 1
        
        #check for pending transaction and give 5 mins time.
        while (sleepCounter > 0):
            foundPendingTransaction = self.CustomerFeedbackDB.isTransactionPending() 
            if (foundPendingTransaction == True):
                Log.info("Wait for 30 seconds for transaction to finish. Timer counter [%d]" % (11-sleepCounter))
                time.sleep(30)
                sleepCounter = sleepCounter - 1
                if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                    self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                    quitProcessing = True
                    return quitProcessing
                continue
            else:
                break
            
        if (foundPendingTransaction == True):       
            try:
                if (self.CustomerFeedbackDB.rollbackTransaction() == True):
                    Log.info("CVCloud transaction Rollback done.")
                    self.CloudDB.CloseDSN()
                    if self.SetSurveyPathsAndEnv() != True:
                        Log.info("Failed to reinitialize database connection")
                        quitProcessing = True
                    Log.info("Successfully reinitialized database connection. Metrics processing will continue.")
                else:
                    Log.error("Failed to rollback transaction in CVCloud. Metrics processing will stop.")
                    quitProcessing = True
                                                                    
            except Exception as err:
                self.Log.exception("Caught exception while trying to rollback pending transaction. Metrics processing will stop.")
                quitProcessing = True        
            
        return quitProcessing

    def upgradeConfigFile(self):
        try:
            CloudCopyScripts = ["csvErrorsList.xml", "BlockedIPs.txt", "IncludeIPs.txt", "HttpServer.xml", "Parameters.config", "CommCellGroups.csv"]
            SimpanaObsoleteCloudScriptsDir = self.SimpanaBaseFolder
            SimpanaObsoleteCloudScriptsDir = os.path.join(SimpanaObsoleteCloudScriptsDir, "Cloud")
            SimpanaNewCloudScriptsDir = self.SimpanaHomeFolder
            SimpanaNewCloudScriptsDir = os.path.join(SimpanaNewCloudScriptsDir, "Reports")
            SimpanaNewCloudScriptsDir = os.path.join(SimpanaNewCloudScriptsDir, "MetricsUpload")
            for item in CloudCopyScripts:
                oldPath = os.path.join(SimpanaObsoleteCloudScriptsDir, item)
                newpath = os.path.join(SimpanaNewCloudScriptsDir, item)
                if os.path.isfile(oldPath) and os.path.isfile(newpath) == False:
                    if not os.path.exists(SimpanaNewCloudScriptsDir):
                        os.makedirs(SimpanaNewCloudScriptsDir)
                    shutil.move(oldPath,newpath)
        except:
            e = sys.exc_info()[0]
            self.Log.exception("Exception in upgradeConfigFile.[%s]" % e )

    def Process(self):
        if self.IsDriverRunning() == True:
            return
        
        if self.SetDriverRunning() == False:
            return
              
        if self.SetSurveyPathsAndEnv() != True:
            self.ResetDriverRunning()
            return
        
        if self.RunOneTimeProcessing == True:
            self.Log.info("Starting one time metrics processing.")
        else:
            self.Log.info("Starting metrics processing loop.")

        #upgrade config file before run survey script
        self.upgradeConfigFile()

        lastReinitCheck = datetime.datetime.now()
        CCGroupLastSyncTime = 0
        UpdateLocationLastTime = 0
        while True:
            if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                break

            flag, logfilefullpath, fileSize, fileVersions, loglevelset = Logger.GetRegistryInfo('CVSurvey', self.Log)
            if flag == True:
                self.Log = Logger.ModifyHandler(logfilefullpath, fileSize, fileVersions, loglevelset, self.Log)
            now = datetime.datetime.now()
            if (lastReinitCheck + datetime.timedelta(hours=1) < now) or self.RunOneTimeProcessing == False:
                lastReinitCheck = now
                if self.DoesMetricsNeedReinitialize() == True:
                    self.Log.info("Reinitializing metrics environment.")
                    if self.SetSurveyPathsAndEnv() != True:
                        break
                    CCGroupLastSyncTime = 0 #sync commcell group
                    self.ResetDoesMetricsNeedRinitialize()
                
            self.Log.debug("Metrics processing will continue as cvd with process id [%d] is running." % (self.CvdPid))
            
            quitProcessing = self.RollbackPendingTransaction() 
            if (quitProcessing == True):
                self.Log.info("Metrics processing will stop")
                break
            
            #Check if there are xml results pending for processing in database. Call process collected results before inserting more data.
            noOfResults = self.CustomerFeedbackDB.countPendingXMLResults()
            self.Log.debug("Number of xml results pending for processing [%d]" % (noOfResults)) 
            if (noOfResults > 0 and noOfResults >= self.PendingXMLResultThreshold):
                self.Log.info("Unprocessed XML results are greater than the threshold. Begin ProcessCollecedresults sp.")
                if self.CallProcessCollectedResults() == True:
                    self.Log.info("Completed processing xmls in CVCloud database")
                else:    
                    self.Log.error("Failed processing xmls in CVCloud database")
                continue #start the while loop again to check if cvd running, pending transaction etc.,.                          
                        
            CCGroupCurrentTime = int(time.time())
            if self.CCGroupSyncFreq > 0:
                if (CCGroupLastSyncTime == 0 or (CCGroupLastSyncTime + self.CCGroupSyncFreq*60 <= CCGroupCurrentTime)):
                    self.Log.info("Starting CommCell Groups sync...")
                    if self.SyncGroupData() == True:
                        self.Log.info("CommCell Groups sync successful")
                        CCGroupLastSyncTime = CCGroupCurrentTime
                    else:
                        self.Log.error("CommCell Groups sync failed")
                       
                    self.Log.info("Update Available Service Pack Info...")
                    if self.UpdateAvailableMajorSP() is True:
                        self.Log.info("Update Available Service Pack Info successful")
                    else:
                        self.Log.error("Update Available Service Pack Info failed")                            
            
            self.initValidQueries()
             
            try:
                self.RunSurvey()
            except Exception as err:
                self.Log.exception("Exception during RunSurvey")
                break

            if self.isPrivate == False:
                UpdateLocationCurrentTime = int(time.time())
                if self.UpdateLocationFreq > 0:
                    if(UpdateLocationLastTime == 0 or (UpdateLocationLastTime + self.UpdateLocationFreq*60 <= UpdateLocationCurrentTime)):
                        self.Log.info("Starting Commcells location update")
                        self.RunUpdateCommcellLocation()
                        UpdateLocationLastTime = UpdateLocationCurrentTime
                        self.Log.info("Completed Commcells location update")

            if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                break             
            
            if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                break
                            
            try:
                self.RunSurveyPruning()
            except Exception as err:
                self.Log.exception("Exception during RunSurveyPruning")
                break

            if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                break                
            
            try:
                self.RunMetricsAsProxyProcessing()
            except Exception as err:
                self.Log.exception("Exception during RunMetricsAsProxyProcessing")
                break                

            if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                break                

            if self.RunOneTimeProcessing == True:
                self.Log.info("Metrics processing is completed")
                break
            else:
                sleeptime = 60 * self.ProcessFrequency
                while sleeptime > 0:
                    if isCVDRunning(self.MetricsConfigObj.sSimpanaInstallPath, self.CvdPid, self.Log) == False:
                        self.Log.info("Metrics processing will stop as cvd with process id [%d] is no longer running." % (self.CvdPid))
                        self.ResetDriverRunning()
                        return
                    sleeptime -= 10
                    time.sleep(10)        
                    
        self.ResetDriverRunning()
    
    # Sync from APP_ClientGroup and APP_ClientGroupAssoc tables to CVCloud database.        
    def SyncGroupData(self):
        query = "EXEC RptGetCommCellGroupsXML '', 1, 0"
        cloudpath = 'cloud'
        if(platform.system() == "Linux"):
            cloudpath = 'Cloud'
        SimpanaConfigDir = self.SimpanaHomeFolder
        SimpanaConfigDir = os.path.join(SimpanaConfigDir, "Reports")
        SimpanaConfigDir = os.path.join(SimpanaConfigDir, "MetricsUpload")
        grpFileName = os.path.join(SimpanaConfigDir, "CommCellGroups.csv")
        try:
            if self.CSDB.Execute(query) == True:
                with open(grpFileName, 'w', newline='', encoding='utf-16') as csvfile:
                    writer = csv.writer(csvfile, delimiter='~')
                    groups = self.CSDB.m_cursor.fetchall()
                    for grp in groups:
                        writer.writerow(grp)
            else:
                Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
                return False
        except:
            Log.exception("Caught Exception while getting Group Association from CommServ DB")    
            return False
        
        try:
            finalQuery = "EXEC UpdateCommCellGroupAssociation '', 1, N'%s'" % grpFileName
           
            if self.CloudDB.Execute(finalQuery) == True:
                row = self.CloudDB.m_cursor.fetchone()
                if row[0] == 0:
                    Log.debug("Add or Update commcell group association success")
                else:
                    Log.error("Failed to Add/update Commcell group association. Error String [%s]" % row[1])
                    return False            
            else:
                Log.error("Query '%s' failed with error: '%s'" % (finalQuery, self.CloudDB.GetErrorErrStr()))
                return False           
        except:
            Log.exception("Caught Exception while updating Group and CommCell with CVCloud DB with Query [%s]" % finalQuery)    
            return False
        
        return True
    
    # Update available major service packs for V10 and V11 from Forevercell to CVCloud server
    def UpdateAvailableMajorSP(self):
        
        query = "SELECT Release, MAX(tmp.HighestSP) AS HighestSP \
                    FROM \
                    (SELECT Release, MAX(SPMajor) AS HighestSP FROM PatchSPVersion WITH (NOLOCK) WHERE Release >= 15 AND SPMajor<100 AND LEN(DVDName) > 0 GROUP BY Release \
                    UNION \
                    SELECT (SELECT MAX(id) FROM simAllGalaxyRel WITH (NOLOCK)) AS Release, MAX(HighestSP) AS HighestSP FROM simInstalledPackages WITH (NOLOCK) WHERE ClientId = (SELECT TOP 1 clientId FROM APP_Platform WITH (NOLOCK) WHERE platformType = 1)) AS tmp \
                    GROUP BY tmp.Release"
        
        try:
            if self.CSDB.Execute(query) == True:
                rows = self.CSDB.m_cursor.fetchall()
                if(len(rows) > 0 ):
                    for row in rows:
                        release = row[0]
                        highestSP = row[1]  
                        if release == 15: #v10
                            updateQuery = "UPDATE cf_SurveyConfig SET Value = N'%s' WHERE Name = '10_0_LatestSPMajorNo'" % highestSP
                            if self.CloudDB.Execute(updateQuery) == False:
                                Log.error("Query '%s' failed with error: '%s'" % (updateQuery, self.CloudDB.GetErrorErrStr()))
                                return False        
                        elif release == 16: #V11
                            updateQuery = "UPDATE cf_SurveyConfig SET Value = N'%s' WHERE Name = '11_0_LatestSPMajorNo'" % highestSP
                            if self.CloudDB.Execute(updateQuery) == False:
                                Log.error("Query '%s' failed with error: '%s'" % (updateQuery, self.CloudDB.GetErrorErrStr()))
                                return False                            
            else:
                Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
                return False
        except:
            Log.exception("Caught Exception while updating Available MajorSP")    
            return False
        
        query = "SELECT sub.Release AS '@Release', sub.SPMajor AS '@SPMajor', sub.UPNumber AS '@UPNumber', sub.FriendlyName AS '@UPFriendlyName', CASE WHEN (sub.Release >= 16 AND sub.SPMajor >= 16 AND sub.SPMajor%4 = 0) THEN 1 ELSE 0 END AS '@IsLTS'  \
                FROM \
                (SELECT S.Release, S.SPMajor, S.TransactionID, U.UPNumber, U.MaxTransactionID, U.FriendlyName, S.nLTSFlag, ROW_NUMBER() OVER(PARTITION BY S.release, S.SPMajor ORDER BY S.TransactionID DESC, U.MaxTransactionID DESC) rn \
                FROM PatchUPVersion U WITH(NOLOCK) INNER JOIN PatchSPVersion S WITH(NOLOCK) ON U.SPVersionID = S.id AND ((S.Release = 16 AND S.SPMajor >= 14) OR S.Release > 16) AND U.MaxTransactionID > S.TransactionID AND U.bIsAvailableForDownload = 1) AS sub \
                WHERE sub.rn = 1 \
                FOR XML PATH('SPUPInfo')"        
        
        SPUPInfoXML = ''
        try:
            if self.CSDB.Execute(query) == True:
                results = self.CSDB.m_cursor.fetchall()
                for res in results:
                    if res[0] is not None:
                        SPUPInfoXML = SPUPInfoXML + res[0]
            else:
                Log.error("Query '%s' failed with error: '%s'" % (query, self.CSDB.GetErrorErrStr()))
                return False
        except:
            Log.exception("Caught Exception while getting Available Update Pack information from CommServ DB")
            return False        
        
        try:
            query = "EXEC SaveAvailableUpdatePack N'%s'" % SPUPInfoXML

            if self.CloudDB.Execute(query) == True:
                row = self.CloudDB.m_cursor.fetchone()
                if row[0] == 0:
                    Log.debug("Successfully saved available Update Pack information")
                else:
                    Log.error("Failed to save available Update Pack information with error [%s]" % row[1])
                    return False
            else:
                Log.error("Query '%s' failed with error: '%s'" % (query, self.CloudDB.GetErrorErrStr()))
                return False
        except:
            Log.exception("Caught Exception while saving available Update Pack information in CVCloud DB with Query [%s]" % query)
            return False   
        
        return True    
    
    #retrieve the list of valid metrics queries to be pushed into the table
    def initValidQueries(self):
        self.ValidQueryIds = []
        query = "SELECT QueryId FROM dbo.RptGetMetricsQueryFlags(0) WHERE QueryEnabled = 1;"
        if self.IsOpenConnection == True and self.CloudDB.Execute(query) == True:
            rows = self.CloudDB.m_cursor.fetchall()
            for row in rows: 
                if row:
                    qid = int(row[0])
                    if qid > 0:
                        self.ValidQueryIds.append(qid)
        #Query 57 should be included
        if 57 not in self.ValidQueryIds:
            self.ValidQueryIds.append(57)

    def fetchConstantsForLocation(self):
        query = "SELECT Value FROM GxGlobalParam WITH (NOLOCK) WHERE Name = 'SkyhookSite';"
        self.SkyhookSite = ''
        if self.CSDB.Execute(query) == True:
            row = self.CSDB.m_cursor.fetchone()
            if row:
                self.SkyhookSite = row.Value
        self.Log.debug('Skyhook Site: %s' % (self.SkyhookSite))

        query = "SELECT Value FROM GxGlobalParam WITH (NOLOCK) WHERE Name = 'SkyhookUser';"
        self.SkyhookUser = ''
        if self.CSDB.Execute(query) == True:
            row = self.CSDB.m_cursor.fetchone()
            if row:
                self.SkyhookUser = row.Value
        self.Log.debug('Skyhook User: %s' % (self.SkyhookUser))

        query = "SELECT Value FROM GxGlobalParam WITH (NOLOCK) WHERE Name = 'SkyhookKey';"
        self.SkyhookKey = ''
        if self.CSDB.Execute(query) == True:
            row = self.CSDB.m_cursor.fetchone()
            if row:
                self.SkyhookKey = row.Value
        self.Log.debug('Skyhook Key: %s' % (self.SkyhookKey))

        query = "SELECT Value FROM GxGlobalParam WITH (NOLOCK) WHERE Name = 'SkyhookEndPoint';"
        self.SkyhookEndPoint = ''
        if self.CSDB.Execute(query) == True:
            row = self.CSDB.m_cursor.fetchone()
            if row:
                self.SkyhookEndPoint = row.Value
        self.Log.debug('Skyhook Endpoint: %s' % (self.SkyhookEndPoint))
        
        query = "SELECT Value FROM GxGlobalParam WITH (NOLOCK) WHERE Name = 'geoLocationURL';"
        self.GeoLocationURL = ''
        if self.CSDB.Execute(query) == True:
            row = self.CSDB.m_cursor.fetchone()
            if row:
                self.GeoLocationURL = row.Value
        self.Log.debug('GeoLocationURL: %s' % (self.GeoLocationURL))

    def fetchCountryForIPUsingSkyhook(self, IP):
        latitude = None
        longitude = None
        country = ''
        returnVal = False
        params = urlencode({
            'version': '2.0',
            'ip': IP,
            'key': self.SkyhookKey,
            'user': self.SkyhookUser
        })
        response = {}
        try:
            baseUrl = self.SkyhookSite
            requestUrl = self.SkyhookEndPoint+'?'+params
            conn = HTTPSConnection(baseUrl)
            conn.request("GET", requestUrl)
            responseObj = conn.getresponse()
            response = json.loads(responseObj.read())
            try:
                latitude = response["data"]["location"]["latitude"]
                longitude = response["data"]["location"]["longitude"]
                country = response["data"]["civic"]["countryIso"]
                returnVal = True
            except Exception as e:
                self.Log.debug("Failed fetching data for IP {0}".format(IP))
                (latitude, longitude, country) = (None, None, '')
        except Exception as e:
            self.Log.error("Error while requesting location details %s" , e)
            (latitude, longitude, country) = (None, None, '')
        return (returnVal, latitude, longitude, country)
    
    def fetchCountryForIPUsingStaticDB(self, IP):
        latitude = None
        longitude = None
        country = ''
        return_val = False
        try:
            rUrl = self.GeoLocationURL.split("//")[1]
            baseUrl = rUrl.split("/")[0]  # baseUrl = 'edc.commvault.com' or the host name in the URL
            requestUrl = "/" + "/".join(rUrl.split("/")[1:3]) + IP #get the request endpoint 
            conn = HTTPSConnection(baseUrl)
            conn.request("GET", requestUrl)
            responseObj = conn.getresponse()
            response = json.loads(responseObj.read())
            status = response["status_code"]
            if status != "SUCCESS":
                self.Log.debug("Unable to fetch the data for IP from static DB {0}".format(IP))
                (latitude, longitude, country) = (None, None, '')
            else:
                latitude = response["latitude"]
                longitude = response["longitude"]
                country = response["countrycode"]
                return_val = True;
        except Exception as e:
            self.Log.error("Error while requesting location details using the static DB %s" , e)
            (latitude, longitude, country) = (None, None, '')
        return (return_val, latitude, longitude, country)
    
    def fetchCountryForIP(self, IP):
        retVal = False
        latitude = None
        longitude = None
        countryCode = ''
        (retVal, latitude, longitude, countryCode) = self.fetchCountryForIPUsingSkyhook(IP)
        if not retVal:
            (retVal, latitude, longitude, countryCode) = self.fetchCountryForIPUsingStaticDB(IP)
        return (retVal, latitude, longitude, countryCode)

    def fetchCommcellsForLocation(self):
        query = "SELECT CommservUniqueId, C.ClientId, CommServIP " \
                "FROM cf_SurveyClient C WITH(NOLOCK) " \
                "INNER JOIN cf_CommcellIdNameMap M WITH(NOLOCK) " \
                "ON C.CommservUniqueId = M.ID " \
                "AND Type & 256 = 256 " \
                "WHERE (Latitude IS NULL OR Longitude IS NULL OR CountryCode = '')"
        try:
            if self.CloudDB.Execute(query) == True:
                commcells = self.CloudDB.m_cursor.fetchall()
                return commcells
            else:
                return None
        except Exception as e:
            self.Log.error("Error while fetching commcells for updating location %s", e)
            return None

    def updateCommcellLocationData(self, lat, long, countryCode, commservUniqueId, clientId):
        query = "UPDATE cf_SurveyClient " \
                "SET CountryCode = '{0}', Latitude = {1}, Longitude = {2} " \
                "WHERE CommservUniqueId = {3} " \
                "AND ClientId = {4}".format(countryCode, lat, long, commservUniqueId, clientId)
        try:
            if self.CloudDB.Execute(query) == True:
                self.Log.debug("Successfully updated the location details for the commcell %s", commservUniqueId)
                return True
            else:
                self.Log.error("Error while executing %s", query)
                return False
        except Exception as e:
            self.Log.error("Error with query %s", query)
            return False

    def RunUpdateCommcellLocation(self):
        commcells = self.fetchCommcellsForLocation()
        if commcells is not None:
            for (commservUniqueId, clientId, commservIP) in commcells:
                (retVal, lat, long, country) = self.fetchCountryForIP(commservIP)
                if retVal:
                    self.updateCommcellLocationData(lat, long, country, commservUniqueId, clientId)
                else:
                    self.Log.error("Fetching details for commcell %s failed", commservIP)

if __name__ == '__main__':
    parser = optparse.OptionParser()
    parser.add_option("-p", "--cvdpid", dest="cvdpid", default=0)
    parser.add_option("-d", "--delete", action='store_true', dest="deletexml", default=False)
    parser.add_option("-r", "--runonce", action='store_true', dest="runonce", default=False)
    options,args = parser.parse_args()
    
    
    #initialize logger
    Log = Logger.InitLogger('CVSurvey')
    Log.info("Starting metrics python driver")
    
    deletexml = bool(options.deletexml)
    cvdpid = int(options.cvdpid)
    runonce = bool(options.runonce)
    if cvdpid == 0 and runonce == False:
        Log.error("Process not launched by cvd. Blocking the processing...")
        exit(0)
    
    Log.info("CVD PID [%d]" % (cvdpid))
    
    Log.debug("Delete Files after processing: [%r]" % (deletexml))
    Log.debug("One time processing enabled: [%r]" % (runonce))
    
    try:
        worker = SurveyProcessor(cvdpid, deletexml, runonce, Log)
        
        worker.Process()

    except BaseException as err:
        if(err.args[0] != 0):
            Log.exception("Exception while processing metrics data." )
    
    except:
        Log.exception("Exception while processing metrics data.")