'''
/******************************************************************************/
/*  Copyright (c) CommVault Systems                                           */
/*  All Rights Reserved                                                       */
/*                                                                            */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF CommVault Systems          */
/*  The copyright notice above does not evidence any                          */
/*  actual or intended publication of such source code.                       */
/******************************************************************************/
Created on Mar 22, 2012
Revision: $Id: Common.py,v 1.23.14.26.8.1 2021/01/21 21:59:09 jge Exp $
$Date: 2021/01/21 21:59:09 $
@author: David Maisonave
'''
import platform

if(platform.system() == "Windows"):
    import win32com.client
    import win32event
    import win32con
    import win32api
    import winerror

import subprocess
import string
import unicodedata
import urllib
import urllib.request as req
import email, mimetypes
import os, stat
import ctypes
import base64

try:
    import winreg as _winreg
except ImportError:
    import unix_winreg as _winreg


import re
from xml.dom.minidom import parse, parseString
import xml.etree.ElementTree as ET
from AsciiDammit import *
from io import StringIO

class TransferXMLFileConstraint:
    '''
    classdocs
    '''
    MaxRetry = 10
    RetryIntervalMinutes = 60
    RetentionDays = 7

    def __init__(self):
        '''
        Constructor
        '''
        
class Callable:
    def __init__(self, anycallable):
        self.__call__ = anycallable

# Controls how sequences are uncoded. If true, elements may be given multiple values by
#  assigning a sequence.
doseq = 1

class MultipartPostHandler(req.BaseHandler):
    handler_order = urllib.request.HTTPHandler.handler_order - 10 # needs to run first

    def http_request(self, request):
        data = request.get_data()
        if data is not None and type(data) != str:
            v_files = []
            v_vars = []
            try:
                 for(key, value) in data.items():
                     if type(value) == file:
                         v_files.append((key, value))
                     else:
                         v_vars.append((key, value))
            except TypeError:
                systype, value, traceback = sys.exc_info()
                raise TypeError ("not a valid non-string sequence or mapping object", traceback)

            if len(v_files) == 0:
                data = urllib.urlencode(v_vars, doseq)
            else:
                boundary, data = self.multipart_encode(v_vars, v_files)

                contenttype = 'multipart/form-data; boundary=%s' % boundary
                if(request.has_header('Content-Type')
                   and request.get_header('Content-Type').find('multipart/form-data') != 0):
                    print( "Replacing %s with %s" % (request.get_header('content-type'), 'multipart/form-data'))
                request.add_unredirected_header('Content-Type', contenttype)

            request.add_data(data)
        
        return request

    def multipart_encode(vars, files, boundary = None, buf = None):
        if boundary is None:
            boundary = email.generator._make_boundary()
        if buf is None:
            buf = StringIO()
        for(key, value) in vars:
            buf.write('--%s\r\n' % boundary)
            buf.write('Content-Disposition: form-data; name="%s"' % key)
            buf.write('\r\n\r\n' + value + '\r\n')
        for(key, fd) in files:
            file_size = os.fstat(fd.fileno())[stat.ST_SIZE]
            filename = fd.name.split('/')[-1]
            contenttype = mimetypes.guess_type(filename)[0] or 'application/octet-stream'
            buf.write('--%s\r\n' % boundary)
            buf.write('Content-Disposition: form-data; name="%s"; filename="%s"\r\n' % (key, filename))
            buf.write('Content-Type: %s\r\n' % contenttype)
            fd.seek(0)
            buf.write('\r\n' + fd.read() + '\r\n')
        buf.write('--' + boundary + '--\r\n\r\n')
        buf = buf.getvalue()
        return boundary, buf
    multipart_encode = Callable(multipart_encode)

    https_request = http_request
    
def replaceFilePathRegex(strIp, RegEx):
    return re.sub(RegEx, 'Masked Data', strIp)

def scrubXML(XMLRoot, RegEx, ExclusionDict, Log):
    retVal = True
    
    try:        
        for key in XMLRoot.attrib.keys():
            if key not in ExclusionDict and XMLRoot.attrib[key] != 'N/A':
                XMLRoot.attrib[key] = replaceFilePathRegex(XMLRoot.attrib[key], RegEx)
        
        if XMLRoot.text is not None and XMLRoot.text != 'N/A' and XMLRoot.tag not in ExclusionDict:
                XMLRoot.text = replaceFilePathRegex(XMLRoot.text, RegEx)         
             
        for child in XMLRoot:
            retVal = scrubXML(child, RegEx, ExclusionDict, Log)
            if retVal == False:
                return False
                   
        return retVal

    except Exception as inst:
        Log.exception(inst)
        return False 

def uploadFile(sourceFile, uploadurl, user, passwd, ccid, Log, httpProxyHost, httpProxyPort, httpProxyUser,httpProxyPasswd):
    isCopyFailed = False
    if not hasattr(uploadFile, 'sslver'):
        uploadFile.sslver = ''
    
    try:
        if  httpProxyHost is not None:
            Log.debug("Using Proxy Host "+httpProxyHost + ':' + str(httpProxyPort))
            proxycommand =  ' -x ' + httpProxyHost + ':' + str(httpProxyPort) 
            if (httpProxyUser is not None) and (httpProxyPasswd is not None) and (len(httpProxyUser)>0):
               Log.debug("Using Proxy User Authentication  Proxy [http://%s:%s] proxyUser [%s] " %(str(httpProxyHost),str(httpProxyPort),str(httpProxyUser)))
               proxycommand = proxycommand + ' -U ' + httpProxyUser +':'+httpProxyPasswd
    except:
         Log.debug("HTTP proxy information can't be loaded")
         httpProxyHost = None

    try:
       # without -f, HTTP errors will result in a successful return code
       precommand = 'curl -f -F "mode=go" -F "ccid=%s" -F "username=%s" -F "password=%s" -F "logfile=@' % (ccid, user, passwd)
       passwrd = ''
       if passwd is not None:
          passwrd = passwd.decode('utf-8')
       postcommand ='" '+ uploadurl + ' -k -s -u '+ user + ':'+ passwrd + uploadFile.sslver

       if (httpProxyHost is None) or (len(httpProxyHost)==0):
           command = precommand + sourceFile + postcommand 
       else:
           command = precommand + sourceFile + postcommand + proxycommand

       ret = os.system(command)
       if ret == 35:
          Log.error("Retrying source File %s with sslv3" % (sourceFile))
          uploadFile.sslver = ' -3 '
          command += uploadFile.sslver
          postcommand += uploadFile.sslver
          ret = os.system(command)
       if ret != 0:
          Log.error("File copy failed for source file %s to URL [%s] - error code %s " % (sourceFile,uploadurl, str(ret)))
          isCopyFailed = True
       else:         
          Log.debug("XML File [%s] uploaded successfully to URL [%s]" % (sourceFile, uploadurl))
    except:
        Log.exception("Failed to upload file [%s] to URL [%s] " % (sourceFile, uploadurl))
        isCopyFailed = True
    
    if not isCopyFailed:
        os.unlink(sourceFile)
            
def FilteredQuerySourceFiles(sourceFile, sourceoutFile, RestrictQuery, doScrubbing, RegEx, ExclusionDict, Log):

    try:
        tree = ET.parse(sourceFile)
        root = tree.getroot()
        
        for child in root:
            if 'QueryId' in child.attrib.keys() and child.attrib['QueryId'] in RestrictQuery:
                root.remove(child)
                continue
            
            if doScrubbing == True:
                for key in child.attrib.keys():
                    if key not in ExclusionDict and child.attrib[key] != 'N/A':
                        child.attrib[key] = replaceFilePathRegex(child.attrib[key], RegEx)    
                
                if child.text is not None:
                    tempRoot = None                 
                    try:
                        txtValue = u'<?xml version="1.0" encoding="UTF-8"?><TempRoot>'
                        dataText = child.text
                        if isinstance(dataText, bytes):
                            dataText = dataText.decode('utf-8')
                        txtValue = txtValue + dataText
                        txtValue = txtValue + '</TempRoot>' #include dummy root since the text may not be properly formatted
                        tempRoot = ET.fromstring(txtValue)
                        retValue = scrubXML(tempRoot, RegEx, ExclusionDict, Log)
                        if retValue == True:
                            txtValue = ET.tostring(tempRoot, 'UTF-8')
                            #remove dummy root
                            txtValue = txtValue.decode('utf-8');
                            txtValue = txtValue.replace("<TempRoot>", '')
                            txtValue = txtValue.replace("</TempRoot>", '')
                            child.text = txtValue
                        else:
                            return False
                    except Exception as inst:
                        Log.exception("Failed to scrub below element")
                        Log.exception(child.text)                    
                        Log.exception(inst)
                        return False
        outputContent = ET.tostring(root, 'utf-8')
        with open(sourceoutFile,'wb') as fd:
           fd.write(outputContent)
           Log.info("Filter file written to %s" % (sourceoutFile))
        return True

    except Exception as inst:
        if inst.args[0] == 'no element found: line 1, column 0':
           Log.info("XML File %s is empty." % sourceFile)
        else:
           Log.exception(inst)
        return False            

def ReadHttpServerInfo(RestrictQuery, ScrubCC, copytoHttpServerlist, Log, forwardingXML, SimpanaConfigDir):
    root=None
    try:
        copyFilestoHttpServer = os.path.join(SimpanaConfigDir, "HttpServer.xml")
                
        if os.path.isdir(copyFilestoHttpServer):
            return
        if os.path.exists(copyFilestoHttpServer)== False:
            # allow xml file to override database entry
            copyFilestoHttpServer=forwardingXML
            if copyFilestoHttpServer is None or len(copyFilestoHttpServer)<1:
                return
            try:
                xmlDoc = parseString(copyFilestoHttpServer)
                root = xmlDoc.getElementsByTagName("EVGui_HttpServerConfig")
            except:
                Log.exception("Unable to parse forwarding XML")
                return
        else:
            xmlDoc = parse(copyFilestoHttpServer)
            root = xmlDoc.getElementsByTagName("HttpServerInfo")
    except:
        Log.exception("Cannot access forwarding HttpServer list file %s... Proceeding..." % copyFilestoHttpServer)

        if root == None:
            #print "File %s doesn't have element HttpServerInfo!!!" % copyFilestoHttpServer
            Log.error("File %s doesn't have element HttpServerInfo!!!" % (copyFilestoHttpServer))
            return

    try:
        blockQuery = xmlDoc.getElementsByTagName("RestrictQuery")
        if blockQuery.length == 0 :
            #print "RestrictQuery token is not found in xml..."
            Log.info("RestrictQuery token is not found in xml...")                  
        elif blockQuery[0].hasAttribute('QueryIds') == False:
            #print "RestrictQuery token is not having any QueryIds attribute..."
            Log.info("RestrictQuery token is not having any QueryIds attribute...")
        else:
            RestrictQueryarray = blockQuery[0].getAttribute("QueryIds").split(",")
            for element in RestrictQueryarray:
                if(len(element)>0):
                    RestrictQuery[element.strip()]=element.strip()
    except:
        Log.exception("Cannot process forwarding query restriction.")
                    
    try:
        scrubCommCells = xmlDoc.getElementsByTagName("ScrubCommCells")
        if scrubCommCells.length == 0 :
            Log.info("ScrubCommCells token is not found in xml...")                  
        elif scrubCommCells[0].hasAttribute('CCIDs') == False:
            Log.info("ScrubCommCells token is not having any CCIDs attribute...")
        else:
            scrubCommCellsArray = scrubCommCells[0].getAttribute("CCIDs").split(",")
            for element in scrubCommCellsArray:
                if(len(element)>0):
                    ScrubCC[element.strip().upper()]=element.strip().upper()                    
    except:
        Log.exception("Cannot process forwarding commcell scrubbing.")
                    
    try:
        for httpserver in xmlDoc.getElementsByTagName('httpServer'):
            ForwardingCommCells = []
            if httpserver.hasAttribute("httpServerURL") == False:
                #print "httpserver in not having httpServerURL attribute..."
                Log.info("httpserver is not having httpServerURL attribute...")
                continue;
            httpServerURL = httpserver.getAttribute("httpServerURL")
            #print httpServerURL
            if httpserver.hasAttribute("urlUser") == False or httpserver.getAttribute("urlUser") == '':
                user = 'customer'
            else:
                user = httpserver.getAttribute("urlUser")
               
            if httpserver.hasAttribute("urlPwd") == False or httpserver.getAttribute("urlPwd") == '':
                pwd = '2GGFiggFaggmgggJaggJbggJTggTmggJSggmgggJ7ggFigg'
            else:
                pwd = httpserver.getAttribute("urlPwd")
              
            if httpserver.hasAttribute("UNCPath") == False or httpserver.getAttribute("UNCPath") == '':
                UNCPath = ''
            else:
                UNCPath = httpserver.getAttribute("UNCPath")

            if httpserver.hasAttribute("UNCUser") == False or httpserver.getAttribute("UNCUser") == '':
                UNCUser = 'gbuilder'
            else:
                UNCUser = httpserver.getAttribute("UNCUser")

            if httpserver.hasAttribute("UNCPwd") == False or httpserver.getAttribute("UNCPwd") == '':
                UNCPwd = 'builder!12'
            else:
                UNCPwd = httpserver.getAttribute("UNCPwd")

            if httpserver.hasAttribute("name") == False or httpserver.getAttribute("name") == '':
                serverName = ''
                m = re.match(r'([\w:]*)(//)([\w\.]+)(.*)', httpServerURL)
                if m is not None:
                  serverName = m.group(3)
                else:
                  serverName = httpServerURL
            else:
                serverName = httpserver.getAttribute("name")

            isPublic = 0
            if httpserver.hasAttribute("isPublic") == True:
               isPublic = int(httpserver.getAttribute("isPublic"))

            #we do no want to forward to public cloud anymore
            if isPublic ==1: 
                continue
            
            CCList = httpserver.getElementsByTagName('CommCell')
            if CCList != None:
                for commcell in CCList:
                    forwardingCC = commcell.getAttribute("ID")
                    if forwardingCC not in ForwardingCommCells:
                        ForwardingCommCells.append(forwardingCC)
                              
            httpserverprop = {'httpServerURL': httpServerURL, 'user':user, 'pwd':pwd, 'isPublic':isPublic, 'ForwardingCommCells':ForwardingCommCells, 'UNCPath':UNCPath, 'UNCUser':UNCUser, 'UNCPwd':UNCPwd, 'serverName':serverName}
            if httpserver in copytoHttpServerlist:
                continue
            copytoHttpServerlist[httpserver] = httpserverprop
            Log.info("Entered httpServer forward URL=["+httpServerURL+"] "+("UNC=["+UNCPath+"]" if (len(UNCPath)>0) else "")+" user=["+user+"] isPublic=["+str(isPublic)+"]")
        xmlDoc.unlink()  

    except:
        Log.exception("Cannot process httpServer XML entries. Proceeding...")
        
def GetEnvVar(Name, DefaultValue = ''):
    try:
        return os.environ[Name]
    except:
        return DefaultValue

def GetLastWriteTime(ftWrite):
    return None

def wtoa(unistr):
    #return unicodedata.normalize('NFKD', unistr).encode('ascii','ignore')
    return asciiDammit(unistr)

def Replace(org_value, old, new):
    return string.replace(org_value,old, new)

def GetProcessList():
    fn = "GetProcessList"
    try:
        WMI = win32com.client.GetObject('winmgmts:')
        p = WMI.ExecQuery('select * from Win32_Process where Name="python.exe"')
        procList = list()
        for prop in p:
            procList.append((prop.Properties_('ProcessId').Value, prop.Properties_('Caption').Value, prop.Properties_('CommandLine').Value))
        p = WMI.ExecQuery('select * from Win32_Process where Name="cvd.exe"')
        for prop in p:
            procList.append((prop.Properties_('ProcessId').Value, prop.Properties_('Caption').Value, prop.Properties_('CommandLine').Value))
        return procList
    except:
        return None
    
def isDriverProcessRunning(Log):
    mutexName = 'Global\CVMETRICSDRIVER' 
    
    try:
        hmutex = win32event.OpenMutex(win32con.SYNCHRONIZE, False, mutexName)
        if hmutex:
            win32api.CloseHandle(hmutex)
        else:
            Log.debug("Could not find driver alive mutex [%s]." % (mutexName))
            return False
    except:
        Log.exception("Could not find driver alive mutex [%s]." % (mutexName))
        return False
    Log.info("Metrics driver is already running")
    return True

def setDriverStartMutex(Log):
    mutexName = 'Global\CVMETRICSDRIVER' 
    try:
        hMutex = win32event.CreateMutex(None, True, mutexName)
        lasterror = win32api.GetLastError()
        if lasterror != winerror.ERROR_SUCCESS:
            Log.info("Failed to create driver alive mutex name [%s]. Error value [%d]" % (mutexName,lasterror))
            return None
        Log.info("Created driver alive mutex name [%s]. Error value [%d]" % (mutexName,lasterror))
        return hMutex
    except:
        Log.exception("Could not create driver alive mutex mutex [%s]. CVD is not running." % (mutexName))
        return None

def resetsetDriverStartMutex(hMutex, Log):
    try:
        if hMutex is not None: 
            retval = win32api.CloseHandle(hMutex)
            if retval != True:
                lasterror = win32api.GetLastError()
                if lasterror != winerror.ERROR_SUCCESS:
                    Log.debug("Failed to close driver alive mutex. Error value [%d]" % (lasterror))
                    return False
            Log.info("Released driver alive mutex name.")
            return True
    except:
        Log.exception("Could not close driver alive mutex mutex. CVD is not running.")
        return False
        

def isCVDRunning(simpanaInstallLocation, cvdpid, Log):
    if(platform.system() == "Linux"):
        processpath = os.path.abspath(os.path.join(__file__ ,"../../..")) + "/Base/cvd"
        try:
            ret = subprocess.check_output(["pidof",processpath])
            Log.debug("CVD is running ["+ret.decode("utf-8").rstrip()+"]")
            return True
        except:
            Log.error("CVD is not running")
        return False

    instanceName = simpanaInstallLocation[simpanaInstallLocation.find("\\Instance")+1:]
    
    mutexName = 'Global\CVDALIVE-' + instanceName
    
    try:
        hmutex = win32event.OpenMutex(win32con.SYNCHRONIZE, False, mutexName)
        if hmutex:
            win32api.CloseHandle(hmutex)
        else:
            Log.debug("Could not find cvd alive mutex [%s]. CVD is not running." % (mutexName))
    except:
        Log.exception("Could not find cvd alive mutex [%s]. CVD is not running." % (mutexName))
        return False
    
    if cvdpid != 0:
        try:
            cvdpidStr = str(cvdpid)
            pidString = 'PID eq ' + cvdpidStr
            result = os.popen('tasklist /FI "%s" | find /C "cvd"' % pidString).read() 
            #os.popen returns a os._wrap_close object which has a read() function defined
            #unlike subprocess.Popen which requires the developer to manually convert the data read from bytes to str
            #changes in command now results in command generating 1 or 0 as output rather than string which needs to be parsed
            if '1' in result:
                Log.debug("CVD process [%d] running" % (cvdpid))
            else:
                Log.debug("CVD process [%d] not running" % (cvdpid))
                return False

        except Exception as ex:
            Log.exception("Exception caught while checking if cvd (%s) is running : %s" % (cvdpid,ex))
            return False
        
    return True

def getSimpanaRegPath():
    if not hasattr(getSimpanaRegPath, 'SimpanaInstallPath'):
        getSimpanaRegPath.SimpanaInstallPath = None
        try:
            QinetixVM = os.path.dirname(os.path.realpath(__file__))
            startindex = QinetixVM.find('Cloud');
            if(startindex<0):
                print ( "Cloud not find in current dir path..." + QinetixVM )
                return
            QinetixVM = QinetixVM[0:startindex]
            QinetixVM = QinetixVM+'\QinetixVM'
            instanceName = ''
            with open(QinetixVM) as instancefile:
                for line in instancefile.readlines():
                    instanceName = line
                    break
            getSimpanaRegPath.SimpanaInstallPath = "SOFTWARE\\CommVault Systems\\Galaxy\\" + instanceName
        except IOError:
            print('Cannot Access QinetixVM file in Base folder Procceeding with Instance001...')
            getSimpanaRegPath.SimpanaInstallPath = "SOFTWARE\\CommVault Systems\\Galaxy\\Instance001"
    return getSimpanaRegPath.SimpanaInstallPath

def GetMetricsSimpleData(encrStr, baseFolder, Log):
    try:
        if(baseFolder == ''):
            dllpath = os.getcwd()
            if(platform.system() == "Linux"):
                dllpath = os.path.join(dllpath, 'Base')
            else:
                dllpath = os.path.dirname(dllpath)
            if(platform.system() == "Linux"):
                dllpath = os.path.join(dllpath, 'libEvExternal.so')
            else:
                dllpath = os.path.join(dllpath, 'EvExternal.dll')
        else:
            if(platform.system() == "Linux"):
                dllpath = os.path.join(baseFolder, 'libEvExternal.so')
            else:
                dllpath = os.path.join(baseFolder, 'EvExternal.dll')

        mydll = ctypes.CDLL(dllpath)
        mydll.CVMetricsProc.restype = ctypes.c_int
        
        c_s = ctypes.c_char_p(encrStr.encode('utf-8'))
        c_p = ctypes.create_string_buffer(b'',1000)
        
        i = 1000
        c_len = ctypes.c_void_p(i)

        if mydll.CVMetricsProc(c_s, c_p, c_len) != 0:
            Log.error("Error obtaining metrics string")
            return None
        return c_p.value
    except:
        Log.exception("Failed to obtain metrics string")
        return None

def GetMetricsData(encrStr, Log, baseFolder=''):
    try:
        c_p=GetMetricsSimpleData(encrStr, baseFolder, Log)
        if c_p != None:
            #exp = re.compile()
            usernamelenstr = re.match(r'\d+', c_p.decode('utf-8')).group()
            usernamelen = int(usernamelenstr)
            username = c_p[len(usernamelenstr):len(usernamelenstr)+usernamelen]
            password = c_p[len(usernamelenstr)+usernamelen:]
            return True, username, password
    except Exception as err:
        Log.exception("Caught exception while fetching Metrics string: %s", err)
    return False, None, None

# Execute an arbitrary qCommand
# returns a tuple
# First value: True when successful and False when an error is encountered
# Second value: String containing console output, if any
def executeQCommand(CSDB, Log, command):
    try:
        # Get credentials to execute command
        query = "SELECT u.login, u.password FROM dbo.UMUsers u WITH (NOLOCK) WHERE u.id='1';"
        if CSDB.Execute(query) == True:
            row = CSDB.m_cursor.fetchone();
            if row:
                username = row[0]
                password = row[1]
            else:
                Log.debug("User credentials not found in database")
                return False, ""
        else:
            Log.error(__name__ + " :: Error: Query '%s' failed with error: '%s'" % (query, CSDB.GetErrorErrStr()))
            return False, ""


        # Login, execute command, and logout
        cmdProcess = subprocess.Popen('qlogin -u "' + username + '" -ps "' + password + '"',
                                    shell=True,
                                    stdout=subprocess.PIPE)
        cmdProcess.wait()
        output = cmdProcess.stdout.read()
        if not output.startswith("User logged in successfully"):
            Log.error("Problem encountered executing qlogin script: " + output)
            return False, ""

        cmdProcess = subprocess.Popen(command, shell=True, stdout=subprocess.PIPE)
        cmdProcess.wait()
        output = cmdProcess.stdout.read()

        cmdProcess = subprocess.Popen('qlogout', shell=True, stdout=subprocess.PIPE)
        cmdProcess.wait()

        return True, output
    except:
        Log.exception("Unexpected exception encountered executing qCommand")
        return False, ""